/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libdos.so
 *
 *   File: segs.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>

#include "ptables.h"
#include "defsegmgr.h"
#include "checks.h"
#include "segs.h"
#include "display.h"
#include "os2dlat.h"
#include "sn.h"
#include "resolver.h"

// list for keeping disk private data areas
list_anchor_t  Disk_PrivateData_List=NULL;
#define DISK_PDATA_TAG  0x08080808


/*
 *  A freespace has a name that has 3 parts to it.
 *  (1) a parent object name like hda
 *  (2) the word freespace
 *  (3) a number used to keep freespace seg names different
 *
 *  Examples:  hda_freespace1
 *             hda_freespace2
 *             dasdh3_freespace4
 *
 *  This routine returns the number found at the end
 *  of the freespace segment name, ignoring any differences
 *  that may be found in other parts of the name.
 *
 *  Returns: success: integer > 0
 *           failure: -1
 *
 */
static int get_freespace_number(DISKSEG *freespace)
{
	int number=-1;
	int i;

	LOG_ENTRY();

	if (freespace) {

		if (freespace->name) {

			// pretty liberal here...
			// minimum string has length of 2 ... freespac[eN]
			if (strlen(freespace->name)>=2) {

				// walk backwards
				for (i=strlen(freespace->name)-1; i>0; i--) {

					// first e we find ends the search
					if ( freespace->name[i-1] == 'e' ) {

						// convert ascii number to integer
						number =  atoi( &freespace->name[i] );
						break;

					}

				}

			}

		}

	}

	LOG_EXIT_INT(number);
	return number;
}


/*
 *  Called by the get_name_for_disk_segment() routine to come up
 *  with the next available number to use when naming freespace
 *  segments on this drive.
 */
static int  get_next_avail_freespace_number(LOGICALDISK *ld)
{
	int freespace_number=0;
	DISKSEG *seg;
	list_element_t iter;

	LOG_ENTRY();

	LIST_FOR_EACH( ld->parent_objects, iter, seg) {

		if (seg->data_type == FREE_SPACE_TYPE) {

			if ( get_freespace_number(seg) > freespace_number ) {
				freespace_number = get_freespace_number(seg);
			}

		}

	}

	++freespace_number;

	LOG_EXIT_INT(freespace_number);
	return freespace_number;
}

/*
 *  Function: get device name
 *
 *  On a devfs system ... we need to change the name
 *  from ide/..../disc   to ide/..../part before appending
 *  the partition number.
 */
static void get_device_name(LOGICALDISK *ld, char *device_name)
{
	int i;

	LOG_ENTRY();

	if (ld && device_name) {

		strcpy(device_name, ld->name);

		for (i=strlen(device_name)-1; i>=0; i--) {
			if (device_name[i] == '/') {
				if (strncmp(&device_name[i],"/disc",5)==0) {
					strcpy(&device_name[i+1],"");
					LOG_EXIT_VOID();
					return;
				}
			}
		}


	}

	LOG_EXIT_VOID();
}

boolean isa_partition_number(char *number)
{
	LOG_ENTRY();

	if (number && strlen(number)) {
		if (*number=='1' || *number=='2' || *number=='3' ||
		    *number=='4' || *number=='5' || *number=='6' ||
		    *number=='7' || *number=='8' || *number=='9') {
			LOG_EXIT_BOOL(TRUE);
			return TRUE;
		}
	}
	LOG_EXIT_BOOL(FALSE);
	return FALSE;
}

/*
 *  Called by insert_DiskSeg_Into_List() to get a name
 *  for the segment storage object being created.
 */
int  get_name_for_disk_segment( DISKSEG *seg )
{
	int                 rc = EINVAL;
	LOGICALDISK        *ld  = get_logical_disk(seg);
	DISK_PRIVATE_DATA  *disk_pdata;
	SEG_PRIVATE_DATA   *pdata;
	char                name[EVMS_VOLUME_NAME_SIZE+1];

	LOG_ENTRY();

	disk_pdata = get_disk_private_data(ld);

	if (ld && disk_pdata) {

		pdata      = (SEG_PRIVATE_DATA *)seg->private_data;

		strcpy(name,"");
		get_device_name(ld,name);
		if (strlen(name)==0) {
			LOG_EXIT_INT(EINVAL);
			return EINVAL;
		}

		if (seg->data_type == DATA_TYPE) {

			if (ld->object_type != DISK ) {
				if (isa_partition_number(&name[strlen(name)-1])==TRUE) {
					strcat(name, ".");  // must be embedded dos partition
				}
			}
			else if (name[strlen(name)-1] == '/') {
				strcat(name, "part");
			}
			else {
				// Special case for HP/Compaq disk arrays.
				// These disks have names like "cciss/c0d0",
				// and the segments need to be named like
				// "cciss/c0d0p1". Search for the pattern
				// "name/c<digits>d<digits>".
				char junk1[64];
				unsigned int junk2, junk3;

				rc = sscanf(name, "%63[^/]/c%ud%u",
					    junk1, &junk2, &junk3);
				if (rc == 3) {
					strcat(name, "p");
				}
				rc = 0;
			}

			if ( pdata->flags & SEG_IS_PRIMARY_PARTITION ) {

				sprintf(seg->name, "%s%d", name, pdata->part_number );

			}
			else if ( ( pdata->flags & SEG_IS_BSD_PARTITION ) ||
				  ( pdata->flags & SEG_IS_UNIXWARE_PARTITION ) ||
				  ( pdata->flags & SEG_IS_SOLARIS_X86_PARTITION )) {

				sprintf(seg->name, "%s%d", name, pdata->part_number );

			}
			else {

				DISKSEG *ebr;
				SEG_PRIVATE_DATA *ebr_pdata;

				ebr = pdata->ebr;
				if (ebr) {
					ebr_pdata = (SEG_PRIVATE_DATA *)ebr->private_data;
					if (ebr_pdata) {
						sprintf(seg->name, "%s%d", name, ebr_pdata->ebr_number+5   );
					}
				}

			}

			rc = 0;

		}
		else if (seg->data_type == META_DATA_TYPE) {

			if (ld->object_type != DISK ) {
				strcat(name, ".");
			}
			else {
				if (name[strlen(name)-1] != '/') {
					strcat(name, "_");
				}
			}

			if ( pdata->flags & SEG_IS_MBR) {
				sprintf(seg->name, "%smbr", name );
				rc = 0;
			}
			else if ( pdata->flags & SEG_IS_EBR) {
				sprintf(seg->name, "%sebr%d", name, pdata->ebr_number );
				rc = 0;
			}
			else {
				rc = 0;	// must be embedded metadata segment
			}

		}
		else if (seg->data_type == FREE_SPACE_TYPE) {

			if (ld->object_type != DISK ) {
				strcat(name, ".");
			}
			else {
				if (name[strlen(name)-1] != '/') {
					strcat(name, "_");
				}
			}

			sprintf(seg->name, "%sfreespace%d", name, get_next_avail_freespace_number(ld) );
			rc = 0;

		}
		else {
			LOG_ERROR("segment has unknown data type (type=%d)\n", seg->data_type );
		}

	}


	LOG_EXIT_INT(rc);
	return rc;
}


/*
 *  Called to allocate a segment storage object
 */
DISKSEG *  allocate_disk_segment( LOGICALDISK *ld )
{
	int   rc;
	DISKSEG *seg=NULL;


	LOG_ENTRY();


	rc = EngFncs->allocate_segment( NULL, &seg );
	if (rc==0) {

		// insert logical disk into new segment's child list
		if (EngFncs->insert_thing(seg->child_objects,ld,INSERT_BEFORE,NULL)) {
			rc = 0;
		}
		else {
			rc = EPERM;
		}

		if (rc == 0) {

			seg->plugin      = Seg_My_PluginRecord_Ptr;
			seg->object_type = SEGMENT;

			seg->flags      &= ~SOFLAG_DIRTY;

			memcpy(&seg->geometry, &ld->geometry, sizeof(geometry_t));

			seg->private_data = calloc(1, sizeof(SEG_PRIVATE_DATA));
			if (seg->private_data) {

				((SEG_PRIVATE_DATA *)seg->private_data)->signature = DOS_SEG_MGR_PDATA_SIGNATURE;
				((SEG_PRIVATE_DATA *)seg->private_data)->logical_disk = ld;

			}
			else {
				LOG_ERROR("call to malloc segment private storage area failed\n");
				EngFncs->free_segment( seg );
				seg = NULL;
			}
		}
		else {
			LOG_ERROR("call to insert DISK storage object in segment child_objects list failed, RC= %d\n", rc );
		}

	}
	else {
		LOG_ERROR("call to engine_allocate_segment failed, RC= %d\n", rc);
		seg = NULL;
	}

	LOG_EXIT_PTR(seg);
	return seg;
}


/*
 * Called to free a segment storage object
 */
void free_disk_segment( DISKSEG *seg )
{
	LOG_ENTRY();

	LOG_DEBUG("segment name= %s\n", seg->name);

	if (seg) {

		if (seg->private_data) free(seg->private_data);
		seg->private_data = NULL;
		EngFncs->free_segment( seg );

	}

	LOG_EXIT_VOID();

}


/*
 *  Called to allocate a DISK_PRIVATE_DATA struct and link it into the
 *  list we maintain for disks we manage disk segments on.
 */
int create_disk_private_data( LOGICALDISK *ld )
{
	int rc=ENOMEM;
	DISK_PRIVATE_DATA *disk_pdata;


	LOG_ENTRY();

	// make sure list is created
	if (Disk_PrivateData_List == NULL) {

		Disk_PrivateData_List = EngFncs->allocate_list();

		if (Disk_PrivateData_List == NULL) {
			LOG_EXIT_INT(rc);
			return rc;
		}

	}

	if ( get_disk_private_data(ld ) == NULL ) {

		disk_pdata = (DISK_PRIVATE_DATA *) calloc(1, sizeof(DISK_PRIVATE_DATA) );

		if ( disk_pdata ) {

			disk_pdata->signature               = DOS_SEG_MGR_PDATA_SIGNATURE;
			disk_pdata->key                     = (void *)ld;
			disk_pdata->container_segs          = EngFncs->allocate_list();
			disk_pdata->deactivate_object_list  = EngFncs->allocate_list();
			disk_pdata->vsectors_per_block = ld->geometry.bytes_per_sector >> EVMS_VSECTOR_SIZE_SHIFT;
			memcpy(&disk_pdata->geometry, &ld->geometry, sizeof(geometry_t));


			if (disk_pdata->container_segs) {

				if (EngFncs->insert_thing(Disk_PrivateData_List,disk_pdata,INSERT_BEFORE,NULL)) {
					rc = 0;
				}
				else {
					rc = EPERM;
				}

			}
			else {
				rc = ENOMEM;
			}

			if ( rc ) {
				free(disk_pdata);
			}

		}
		else {
			rc = ENOMEM;
		}
	}
	else {
		rc = 0;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

static void dos_prune_deactivate_segments( list_anchor_t list )
{
	list_element_t iter,iter2;
	DISKSEG *seg;

	LOG_ENTRY();

	LIST_FOR_EACH_SAFE(list,iter,iter2,seg) {
		if ( seg->private_data)	free(seg->private_data);
		free(seg);
		EngFncs->delete_element(iter);
	}

	LOG_EXIT_VOID();
}

/*
 *  Called to delete the specified disk private data by pruning it from the double
 *  linked list.
 */
int delete_disk_private_data( LOGICALDISK *ld )
{
	int                rc = EINVAL;
	DISK_PRIVATE_DATA *disk_pdata = get_disk_private_data(ld );


	LOG_ENTRY();

	// make sure list exists
	if ( Disk_PrivateData_List == NULL ) {
		LOG_EXIT_INT(rc);
		return rc;
	}

	// delete object from list and free its memory
	if (disk_pdata) {

		EngFncs->remove_thing( Disk_PrivateData_List, disk_pdata );

		if ( disk_pdata->container_segs ) EngFncs->destroy_list(disk_pdata->container_segs);
		if ( disk_pdata->deactivate_object_list ) {
			dos_prune_deactivate_segments( disk_pdata->deactivate_object_list );
			EngFncs->destroy_list(disk_pdata->deactivate_object_list);
		}

		free(disk_pdata);
		rc = 0;
	}
	else {
		rc = EINVAL;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 *  Returns the private data area for the specified disk.
 */
DISK_PRIVATE_DATA * get_disk_private_data( LOGICALDISK *ld )
{
	DISK_PRIVATE_DATA  *disk_pdata = NULL;
	list_element_t      iter;

	LOG_ENTRY();

	if ( Disk_PrivateData_List ) {

		LIST_FOR_EACH( Disk_PrivateData_List, iter, disk_pdata) {
			if ( disk_pdata->key == ld ) {
				LOG_EXIT_PTR(disk_pdata);
				return disk_pdata;
			}
		}

	}

	LOG_EXIT_PTR(NULL);
	return NULL;
}


/*
 *  Returns TRUE if the disk has an extended partition
 */
boolean   disk_has_extended_partition(LOGICALDISK *ld)
{
	SEG_PRIVATE_DATA   *pdata;
	boolean   rc=FALSE;
	DISKSEG   *mbr;

	LOG_ENTRY();

	mbr = get_mbr_from_seglist( ld->parent_objects );
	if (mbr) {

		pdata = (SEG_PRIVATE_DATA *)mbr->private_data;
		if (pdata) {
			// if the mbr chains to an ebr then YES ...
			if (pdata->next_ebr) {
				rc=TRUE;
			}
		}

	}

	LOG_EXIT_BOOL(rc);
	return rc;
}


/*
 *  Returns TRUE if the disk segment list has a META_DATA segment
 *  that is an MBR track.
 */
boolean   seglist_has_mbr_segment( list_anchor_t seglist )
{
	LOG_ENTRY();

	if (seglist != NULL) {
		if ( get_mbr_from_seglist(seglist)==NULL ) {
			LOG_EXIT_BOOL(FALSE);
			return FALSE;
		}
	}

	LOG_EXIT_BOOL(TRUE);
	return TRUE;
}


/*
 *  Returns an MBR DISKSEG if one is found in the list.
 */
DISKSEG * get_mbr_from_seglist( list_anchor_t seglist )
{
	DISKSEG          *seg;
	SEG_PRIVATE_DATA *pdata;
	list_element_t    iter;

	LOG_ENTRY();

	if (seglist != NULL) {

		LIST_FOR_EACH( seglist, iter, seg ) {
			pdata = (SEG_PRIVATE_DATA *) seg->private_data;
			if (pdata) {
				if ( pdata->flags & SEG_IS_MBR ) {
					LOG_EXIT_PTR(seg);
					return seg;
				}
			}
		}

	}

	LOG_EXIT_PTR(NULL);
	return NULL;
}


/*
 *  Called to find the 1st unused entry in a partition table.  This is normally called
 *  when we are createing new partitions and need to add a partition record to a
 *  table.
 *
 *  If successful:  RC = 0...3  a valid partition table index
 *
 *  If errors:      RC = -1  either there are no free partition table entries or else
 *                  we had difficulties operating on the segment LIST.
 *
 */
int  get_first_unused_ptable_entry( list_anchor_t seglist, DISKSEG *ebr )
{
	DISKSEG            *seg;
	int                 rc = 0;
	SEG_PRIVATE_DATA   *pdata;
	boolean             ptable_entry_in_use[4] = {FALSE,FALSE,FALSE,FALSE};
	int                 i;
	LOGICALDISK        *ld = get_logical_disk(ebr);
	DISK_PRIVATE_DATA  *disk_pdata;
	list_element_t      iter;


	LOG_ENTRY();

	if (seglist) {

		LIST_FOR_EACH( seglist, iter, seg ) {

			pdata = (SEG_PRIVATE_DATA *) seg->private_data;
			if (pdata) {

				if ( pdata->ebr == ebr ) {
					if ( pdata->ptable_index >=0  && pdata->ptable_index < 4) {
						ptable_entry_in_use[pdata->ptable_index] = TRUE;
					}
					else {
						rc = -1;
					}
				}

			}
			else {
				rc = -1;
			}

			if (rc)	break;
		}

	}
	else {
		rc = -1;
	}

	/*
	 * also walk the container seg list, looking for primary partitions
	 */
	if (ld  &&  rc != -1) {

		disk_pdata = get_disk_private_data(ld);

		if (disk_pdata) {

			LIST_FOR_EACH( disk_pdata->container_segs, iter, seg ) {

				if (seg!=NULL) {

					pdata = (SEG_PRIVATE_DATA *) seg->private_data;
					if (pdata) {

						if ( pdata->ebr == ebr ) {
							if ( pdata->ptable_index >=0  && pdata->ptable_index < 4) {
								ptable_entry_in_use[pdata->ptable_index] = TRUE;
							}
							else {
								rc = -1;
							}
						}

					}
					else {
						rc = -1;
					}

				}

				if (rc)	break;
			}

		}

	}


	/*
	 * if successful in walking the seglist then look to see which
	 * is the first unused slot in the partition table.
	 */
	if (rc != -1) {

		rc = -1;

		for (i=0; i < 4; i++) {
			if (ptable_entry_in_use[i] == FALSE) {
				rc = i;
				break;
			}
		}
	}

	/*
	 * we are going to return either -1 (error) or else the first
	 * free partition table entry.
	 */
	LOG_EXIT_INT(rc);
	return rc;
}



/*
 *  Called to find an entry in the mbr partition table for the extended
 *  partition record.
 *
 *  If successful:  RC = 0...3  a valid partition table index
 *
 *  If errors:      RC = -1  either there are no free partition table entries or else
 *                  we had difficulties operating on the segment LIST.
 *
 */
static int  get_extd_partition_ptable_entry( list_anchor_t seglist, DISKSEG *mbr )
{
	DISKSEG            *seg;
	int                 rc = -1;
	SEG_PRIVATE_DATA   *pdata;
	boolean             ptable_entry_in_use[4] = {FALSE,FALSE,FALSE,FALSE};
	int                 i;
	LOGICALDISK        *ld = get_logical_disk(mbr);
	DISK_PRIVATE_DATA  *disk_pdata;
	list_element_t      iter;

	LOG_ENTRY();

	if (seglist) {

		LIST_FOR_EACH( seglist, iter, seg ) {

			pdata = (SEG_PRIVATE_DATA *) seg->private_data;

			if ( (pdata) && (pdata->ebr == mbr) ) {

				if ( pdata->ptable_index >=0  && pdata->ptable_index < 4) {

					if ( (pdata->flags & SEG_IS_EBR)==FALSE) {
						ptable_entry_in_use[pdata->ptable_index] = TRUE;
					}

				}

			}

		}

	}

	/*
	 * also walk the container seg list, looking for primary partitions
	 */
	if (ld) {

		disk_pdata = get_disk_private_data(ld);
		if (disk_pdata) {

			LIST_FOR_EACH( disk_pdata->container_segs, iter, seg ) {

				pdata = (SEG_PRIVATE_DATA *) seg->private_data;

				if ( (pdata) && (pdata->ebr == mbr) ) {
					if ( pdata->ptable_index >=0  && pdata->ptable_index < 4) {
						ptable_entry_in_use[pdata->ptable_index] = TRUE;
					}
				}
			}

		}

	}


	/*
	 * if successful in walking the seglist then look to see which
	 * of the ptable entries we can use.
	 */
	rc = -1;

	for (i=3; i >= 0; i--) {

		if (ptable_entry_in_use[i] == FALSE) {
			rc = i;
			break;
		}

	}

	/*
	 * we are going to return either -1 (error) or else the index
	 * of the mbr partition table entry to be used for the extd
	 * partition record.
	 */

	LOG_EXIT_INT(rc);
	return rc;
}



/*
 *  Called to get a count of used entries in either the mbr
 *  or ebr partition tables. These tables only hold 4 entries
 *  apiece.
 */
int  get_ptable_entry_count( list_anchor_t seglist, DISKSEG *ebr )
{
	DISKSEG            *seg;
	int                 count = 0;
	SEG_PRIVATE_DATA   *pdata;
	list_element_t      iter;

	LOG_ENTRY();

	if (seglist != NULL) {

		LIST_FOR_EACH( seglist, iter, seg ) {
			pdata = (SEG_PRIVATE_DATA *) seg->private_data;
			if (pdata) {
				if (pdata->ebr == ebr ) {
					++count;
				}
			}
		}

	}

	LOG_EXIT_INT(count);
	return count;
}


/*
 *  Called by Assign_SegmentManager_ToDisk() to get a freespace
 *  segment from the segment list.
 */
DISKSEG * get_first_freespace_seg_in_list( list_anchor_t seglist )
{
	DISKSEG *seg;
	list_element_t iter;

	LOG_ENTRY();

	if (seglist != NULL) {

		LIST_FOR_EACH( seglist, iter, seg ) {
			if (seg->data_type == FREE_SPACE_TYPE) {
				LOG_EXIT_PTR(seg);
				return seg;
			}
		}

	}

	LOG_EXIT_PTR(NULL);
	return NULL;
}


/*
 *   Returns the DISKSEG struct ptr to the freespace seg following the
 *   specified DISKSEG.  Returns NULL if freespace is not found.
 *
 *  Consider the following seglist ...
 *
 *    Seg_1     Seg2       Seg3       Seg4      Seg5
 *    MBR_SEG   DATA_SEG   Free_Seg   EBR_SEG   DATA_SEG
 *
 *   If we were asked for the freespace seg following Seg2 we would
 *   respond by returning Seg3.  The freespace seg must immediately
 *   follow the specified DISKSEG.
 *
 */
DISKSEG * get_freespace_following_seg(DISKSEG *seg)
{
	list_anchor_t      seglist=NULL;
	DISKSEG     *prev = NULL;
	DISKSEG     *tseg = NULL;
	LOGICALDISK  *ld=NULL;
	list_element_t iter;

	LOG_ENTRY();

	// get logical disk and the entire segment list for the disk
	ld = get_logical_disk( seg );
	if (ld) {
		seglist = ld->parent_objects;
	}

	if ( ld && seglist ) {

		LIST_FOR_EACH( seglist, iter, tseg ) {

			if (prev==NULL) {
				prev = tseg;
			}
			else {
				if (prev == seg) {

					if ( tseg->data_type==FREE_SPACE_TYPE ) {
						LOG_EXIT_PTR(tseg);
						return tseg;
					}
					else {
						LOG_EXIT_PTR(NULL);
						return NULL;
					}

				}

				prev=tseg;
			}

		}

	}

	LOG_EXIT_PTR(NULL);
	return NULL;
}



/*
 *   Called to try and find free space between disk data segments, returning a
 *   new free space DISKSEG struct if a gap between existing segments is found.
 */

DISKSEG * find_freespace_in_seglist( list_anchor_t seglist )
{
	DISKSEG            *prev = NULL;
	DISKSEG            *this = NULL;
	DISKSEG            *freeseg  = NULL;
	LOGICALDISK        *ld;
	DISK_PRIVATE_DATA  *disk_pdata;
	int64_t             gap;
	list_element_t      iter;


	LOG_ENTRY();

	if (seglist) {

		prev = EngFncs->first_thing( seglist, &iter);
		if (prev != NULL) {

			// need disk private data area
			ld = get_logical_disk( prev );
			disk_pdata = get_disk_private_data( ld );
			if (disk_pdata==NULL) {
				LOG_EXIT_PTR(NULL);
				return NULL;
			}

			do {
				this = EngFncs->next_thing( &iter );
				if ( this != NULL ) {

					gap = this->start - ( prev->start + prev->size );

					if ( gap > 0 ) {

						// allocate a segment storage object from the engine
						freeseg = allocate_disk_segment(ld);

						if (freeseg ) {

							freeseg->data_type    = FREE_SPACE_TYPE;
							freeseg->size         = gap;
							freeseg->start        = prev->start + prev->size;

							// dont expose freespace in bsd,unixware and solaris containers
							if ( seg_is_within_container_segment( freeseg ) == TRUE ) {
								free_disk_segment( freeseg );
								prev = this;
								continue;
							}

							// and dont expose any part of the container seg that wasnt
							// allocated to embedded partitions. you can easilly show
							// freespace that overlaps a container seg ... and a container
							// seg is a primary partition ... yikes!  dont do it.
							else if ( seg_overlaps_container_segment( freeseg ) == TRUE ) {

								// call routine to try and adjust the size of the freespace
								if ( remove_container_seg_overlap(freeseg) ) {
									free_disk_segment( freeseg );
									prev = this;
									continue;
								}

							}



							LOG_EXIT_PTR(freeseg);
							return freeseg;

						}
						else {
							LOG_EXIT_PTR(NULL);
							return NULL;
						}

					}
					else {
						prev = this;
					}

				}

			} while ( this!=NULL );
		}
	}


	LOG_EXIT_PTR(NULL);
	return NULL;
}


/*
 *  Called to look for free space on the logical disk and to create
 *  segments which will expose the free space on the disk.
 */
int  find_freespace_on_disk( LOGICALDISK *ld )
{
	DISKSEG            *freeseg              = NULL;
	DISKSEG            *seg                  = NULL;
	DISKSEG            *segaddr              = NULL;
	sector_count_t      sectors_left_on_disk = 0;
	int                 rc;
	lba_t               freespace_start_lba;
	DISK_PRIVATE_DATA  *disk_pdata;
	list_anchor_t             seglist = ld->parent_objects;



	LOG_ENTRY();

	// need disk private data area
	disk_pdata = get_disk_private_data( ld );
	if (disk_pdata==NULL) {
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	if ( EngFncs->list_count(seglist) > 0 ) {	 // IF ... we have disk segments

		do {					// THEN ... look for gaps between them
			seg = find_freespace_in_seglist( seglist );

			if (seg != NULL) {

				segaddr = insert_diskseg_into_list( seglist, seg);
				if (segaddr==NULL) {
					free_disk_segment( seg );
					rc = ENOMEM;
					LOG_EXIT_INT(rc);
					return rc;
				}

			}

		} while (seg != NULL);


		seg = EngFncs->last_thing( seglist, NULL );

		sectors_left_on_disk = ld->size - ( seg->start + seg->size );

		if ( sectors_left_on_disk > 0 ) {
			freespace_start_lba  =  seg->start + seg->size;
		}
		else {
			sectors_left_on_disk = 0;
			freespace_start_lba  = 0;
		}
	}
	else {	// there are no segments on the disk at all so just create a
		// single free space segment for the disk!
		freespace_start_lba  = 0;
		sectors_left_on_disk = ld->size;
	}

	if ( sectors_left_on_disk > 0 ) {

		freeseg = allocate_disk_segment(ld);
		if (freeseg) {

			freeseg->data_type          = FREE_SPACE_TYPE;
			freeseg->size               = sectors_left_on_disk;
			freeseg->start              = freespace_start_lba;

			((SEG_PRIVATE_DATA *)freeseg->private_data)->flags |= SEG_IS_FREE_SPACE_PARTITION;

			// and dont expose any part of the container seg that wasnt
			// allocated to embedded partitions. you can easilly show
			// freespace that overlaps a container seg ... and a container
			// seg is a primary partition ... yikes!  dont do it.
			remove_container_seg_overlap(freeseg);

			segaddr = insert_diskseg_into_list( seglist, freeseg);
			if (segaddr==NULL) {

				int i;

				for (i=1; i<10; i++) {

					segaddr = insert_diskseg_into_list( seglist, freeseg);
					if (segaddr==NULL) {
						LOG_DEBUG("error, insert_DiskSeg_Into_List returned an error\n");
						free_disk_segment(freeseg);
						LOG_EXIT_INT(EIO);
						return EIO;
					}
					else {
						break;
					}
				}

			}

		}
		else {
			LOG_EXIT_INT(EIO);
			return EIO;
		}
	}

	// lastly, merge adjacent free areas
	merge_adjacent_freedisksegs_in_list( ld->parent_objects );

	LOG_EXIT_INT(0);
	return 0;
}





/*
 *   Called to try and join neighboring free segments together.
 *
 *   NOTE:  Assumes that the DISKSEG structs are kept in an ordered list
 *
 */
static int  merge_freespace_segments( list_anchor_t seglist )
{
	DISKSEG            *prev;
	DISKSEG            *this;
	DISK_PRIVATE_DATA  *disk_pdata;
	LOGICALDISK        *ld;
	list_element_t      iter;

	LOG_ENTRY();

	prev = EngFncs->first_thing( seglist, &iter);
	if (prev != NULL) {

		// need disk private data area
		ld = ((SEG_PRIVATE_DATA *)prev->private_data)->logical_disk;
		disk_pdata = get_disk_private_data( ld );
		if (disk_pdata==NULL) {
			LOG_EXIT_INT(EINVAL);
			return EINVAL;
		}

		do {
			this = EngFncs->next_thing( &iter );
			if ( this != NULL ) {

				if ( ( prev==NULL ) ||
				     ( this->data_type != FREE_SPACE_TYPE )||
				     ( prev->data_type != FREE_SPACE_TYPE )) {
					prev = this;
					continue;
				}

				if (get_freespace_number(prev) > get_freespace_number(this) ) {
					EngFncs->remove_thing( seglist, prev );
					this->start -= prev->size;
					this->size  += prev->size;
					free_disk_segment( prev );
					LOG_DEBUG("        kept seg: %s  start: %"PRIu64"  size: %"PRIu64"\n", this->name, this->start, this->size );
					LOG_EXIT_INT(0);
					return 0;
				}
				else {
					EngFncs->remove_thing( seglist, this );
					prev->size  += this->size;
					free_disk_segment( this );
					LOG_DEBUG("        kept seg: %s  start: %"PRIu64"  size: %"PRIu64"\n", prev->name, prev->start, prev->size );
					LOG_EXIT_INT(0);
					return 0;
				}

			}

		} while (this != NULL);
	}

	LOG_EXIT_INT(ENODATA);
	return ENODATA;
}





/*
 *   Called to try and join neighboring free segments together.
 *
 *   NOTE:  Assumes that the DISKSEG structs are kept in an ordered list
 *
 */
int  merge_adjacent_freedisksegs_in_list( list_anchor_t seglist )
{
	int rc;

	LOG_ENTRY();

	do {
		rc = merge_freespace_segments(seglist);
	} while (rc==0);  /* continue while there are segs to merge */

	LOG_EXIT_INT(0);
	return 0;
}



/*
 *  Called to add a segment to the logical disk's segment LIST. First, convert the
 *  partition record to a DISKSEG struct.  Then, insert the DISKSEG struct into
 *  the LIST hanging off the logical disk.
 */
DISKSEG * build_mbr_disk_segment( LOGICALDISK *ld )
{
	Partition_Record   pr;
	DISKSEG            *mbr=NULL;
	DLA_Table_Sector   *dlat_buffer = NULL;
	DISK_PRIVATE_DATA  *disk_pdata = get_disk_private_data(ld);

	LOG_ENTRY();

	if (disk_pdata) {

		// for OS2 disks we need to get the Drive Link Address Table
		if ( disk_pdata->flags & DISK_HAS_OS2_DLAT_TABLES ) {

			dlat_buffer = Read_Dlat_Sector( ld, 0 );

			if (dlat_buffer) {

				if ( seg_register_serial_number( dlat_buffer->Disk_Serial_Number ) == 0) {

					if ( EngFncs->register_name( (char *) dlat_buffer->Disk_Name ) ) {

						seg_unregister_serial_number( dlat_buffer->Disk_Serial_Number);
						LOG_EXIT_PTR(NULL);
						return NULL;

					}

				}
				else {
					LOG_EXIT_PTR(NULL);
					return NULL;
				}

				// save disk name in convenient place
				strncpy( disk_pdata->disk_name, &dlat_buffer->Disk_Name[0], DISK_NAME_SIZE );
			}
		}



		// now hang the MBR meta data segment off the evms logical drive
		memset( &pr, 0, sizeof(Partition_Record));
		pr.start_sect = 0;
		pr.nr_sects   = CPU_TO_DISK32( disk_pdata->geometry.sectors_per_track );
		pr.sys_ind    = MBR_PARTITION;

		mbr = build_diskseg_from_partition_record(ld, &pr, NULL, 0, FALSE );

		if (mbr==NULL) {
			if (dlat_buffer != NULL) free(dlat_buffer);
		}
		else {
			((SEG_PRIVATE_DATA *)mbr->private_data)->dlat = dlat_buffer;
		}

	}

	LOG_EXIT_PTR(mbr);
	return mbr;
}


/*
 *  Called to add a segment to the logical disk's segment LIST. First, convert the
 *  partition record to a DISKSEG struct.  Then, insert the DISKSEG struct into
 *  the LIST hanging off the logical disk.
 */
DISKSEG * build_ebr_disk_segment( LOGICALDISK      *ld,
				  Partition_Record *part,
				  DISKSEG          *ebr,
				  lba_t             ebr_lba,
				  u_int32_t         ptable_index,
				  boolean           primary_partition_flag  )
{
	Partition_Record            pr;
	DISKSEG                    *new_ebr=NULL;
	DLA_Table_Sector           *dlat_buffer = NULL;
	DISK_PRIVATE_DATA          *disk_pdata = get_disk_private_data(ld);
	sector_count_t              ebr_size;
	Extended_Boot_Record        ebr_buffer;
	struct plugin_functions_s  *dft;
	Partition_Record           *tpart=NULL;
	int                         rc, i;


	LOG_ENTRY();
	LOG_DEBUG("ebr lba= %"PRIu64"\n", ebr_lba );

	if (disk_pdata) {

		// for OS2 disks we need to get the Drive Link Address Table
		if ( disk_pdata->flags & DISK_HAS_OS2_DLAT_TABLES ) {

			dlat_buffer = Read_Dlat_Sector( ld, ebr_lba );

			if (dlat_buffer==NULL) {
				LOG_EXIT_PTR(NULL);
				return NULL;
			}

		}

		// initialize size to a full track
		ebr_size = disk_pdata->geometry.sectors_per_track;

		// now try to get actual sector allocation for the EBR
		// this will be the distance between the EBR sector and
		// the start of the logical partition.
		dft = (struct plugin_functions_s *)ld->plugin->functions.plugin;
		if (dft==NULL) {
			LOG_ERROR("error, logical disk has no plugin function table\n");
			LOG_EXIT_PTR(NULL);
			return NULL;
		}
		else {

			rc = dft->read(ld, ebr_lba, 1, (void *) &ebr_buffer );
			if ( rc == 0 ) {

				// find logical partition record
				for (i=0; i<4; i++) {

					tpart = &ebr_buffer.Partition_Table[i];

					if ( isa_ebr_partition_record(tpart) == FALSE ) {

						// we only shrink them ...
						if ( DISK_TO_CPU32(START_LBA(tpart)) < ebr_size ) {
							ebr_size = DISK_TO_CPU32(START_LBA(tpart));
						}
						break;
					}

				}

			}

		}


		// build an extended partition EBR meta data segment
		memset( &pr, 0, sizeof(Partition_Record));
		pr.nr_sects   = CPU_TO_DISK32(ebr_size);
		pr.start_sect = START_LBA(part);
		pr.sys_ind    = SYS_IND(part);
		new_ebr = build_diskseg_from_partition_record( ld, &pr, ebr, ptable_index, primary_partition_flag );

		if (new_ebr==NULL) {
			if (dlat_buffer) free(dlat_buffer);
		}
		else {
			((SEG_PRIVATE_DATA *)new_ebr->private_data)->dlat = dlat_buffer;
		}
	}

	LOG_EXIT_PTR(new_ebr);
	return new_ebr;
}


/*
 *  Called to add a segment to the logical disk's segment LIST. First, convert the
 *  partition record to a DISKSEG struct.  Then, insert the DISKSEG struct into
 *  the LIST hanging off the logical disk.
 *
 *  Note: Because this routine converts from partition records to storage objects it
 *        is only called during discovery when we are reading partition tables.
 *
 */
DISKSEG * build_diskseg_from_partition_record(  LOGICALDISK       *ld,
						Partition_Record  *part,
						DISKSEG           *ebr,
						u_int32_t          ptable_index,
						boolean            primary_partition )
{
	DISKSEG           *seg        = NULL;
	SEG_PRIVATE_DATA  *pdata      = NULL;
	DISK_PRIVATE_DATA *disk_pdata = get_disk_private_data(ld);


	LOG_ENTRY();

	// allocate a new disk segment storage object
	seg = allocate_disk_segment( ld );
	if (seg==NULL) {
		LOG_EXIT_PTR(NULL);
		return NULL;
	}
	else {
		pdata      = (SEG_PRIVATE_DATA *) seg->private_data;
	}

	// copy disk geometry to storage object
	memcpy(&seg->geometry, &disk_pdata->geometry, sizeof(geometry_t) );

	seg->size              = DISK_TO_CPU32(NR_SECTS(part));
	seg->start             = DISK_TO_CPU32(START_LBA(part));

	pdata->sys_id          = (u_int32_t) SYS_IND(part);
	pdata->boot_ind        = (u_int32_t) BOOT_IND(part);


	/* mbr, ebr, primary or logical ? */
	if ( isa_ebr_partition_record(part) ) {
		pdata->flags      |= SEG_IS_EBR;
	}
	else if ( isa_mbr_partition_record(part) ) {
		pdata->flags      |= SEG_IS_MBR;
	}
	else if ( primary_partition == TRUE ) {
		pdata->flags      |= SEG_IS_PRIMARY_PARTITION;
		// seg->flags        |= SOFLAG_PRIMARY;
	}
	else {
		pdata->flags      |= SEG_IS_LOGICAL_PARTITION;
	}

	/*  Data, MetaData segment type */
	if ( ( isa_ebr_partition_record(part) ) ||
	     ( isa_mbr_partition_record(part) )) {
		seg->data_type   = META_DATA_TYPE;
	}
	else {
		seg->data_type   = DATA_TYPE;
	}

	/*  Look for special kind of partitions and mark them */
	if ( SYS_IND(part) == LINUX_RAID_PARTITION ) {
		pdata->flags  |= SEG_IS_LINUX_RAID_PARTITION;
	}
	else if ( isa_linux_swap_partition_record(ld, part, disk_pdata->extd_partition_lba)==TRUE ) {
		pdata->flags  |= SEG_IS_LINUX_SWAP_PARTITION;
	}

	/* Bootable or not */
	if ( BOOT_IND(part) == ACTIVE_PARTITION) {
		seg->flags |= SOFLAG_BIOS_READABLE ;
	}


	pdata->ptable_index = ptable_index; // remember where we were found in the EBR or MBR
					    // this is the index in the ptable where this seg
					    // was found.

	pdata->ebr          = ebr;	    // if primary partition then this pts to the MBR
					    // if logcial partition then this pts to the EBR
					    // if EBR meta data seg then this is the same as
					    // pdata->prev_ebr.
					    // if MBR meta data seg then this is NULL.


	// since logical drives are relative to the start of the extended
	// partition ... we need to add in the extended partitions lba to
	// get an address that is relative to the start of the disk. Also,
	// any EBR segs we lay down must also have their starting LBA
	// adjusted to be disk relative.
	if ( pdata->flags & SEG_IS_LOGICAL_PARTITION ) {
		seg->start += ebr->start;
	}
	else if ( ( pdata->flags & SEG_IS_EBR ) &&
		  ( disk_has_extended_partition(ld) == TRUE ) ) {
		seg->start += disk_pdata->extd_partition_lba;
	}

	/*  Only logical and primary partitions have entries in the DLAT.
	 *  Here we find the DLAT entry and save a ptr to the DLA_Entry
	 *  in the DISKSEG struct.
	 */
	if ( ( disk_pdata->flags & DISK_HAS_OS2_DLAT_TABLES ) &&
	     ( seg->data_type == DATA_TYPE ) ) {

		pdata->dla_entry = Get_Dlat_Entry_Matching_DiskSegment(ebr,seg );

		if ( pdata->dla_entry == NULL ) {
			LOG_ERROR("disk partition was not found in corresponding DLA Table\n");
			free_disk_segment(seg);
			seg=NULL;
		}

	}


	// look for efi system partition record and mark segment as non divideable
	if ( pdata->sys_id == GPT_ESP_PARTITION ) {
		pdata->cflags |= SEG_CFLAG_TOP_SEGMENT;
	}


	LOG_EXIT_PTR(seg);

	return seg;
}


/*
 *  Wrapper routine that is called to remove a segment storage object from a list.
 */
int  remove_diskseg_from_list( list_anchor_t seglist, DISKSEG *seg )
{
	LOGICALDISK       *ld = get_logical_disk(seg);
	DISK_PRIVATE_DATA *disk_pdata;
	SEG_PRIVATE_DATA  *seg_pdata;

	disk_pdata = get_disk_private_data(ld);
	seg_pdata  = (SEG_PRIVATE_DATA *)seg->private_data;


	LOG_ENTRY();
	LOG_DEBUG("segment name= %s\n",  seg->name );

	EngFncs->remove_thing( seglist, seg );

	EngFncs->unregister_name( seg->name );

	// unregister the os2 partition name
	if ( ( disk_pdata->flags & DISK_HAS_OS2_DLAT_TABLES ) &&
	     ( seg->data_type == DATA_TYPE ) ) {

		char pname[PARTITION_NAME_SIZE+12];

		if (seg_pdata->dla_entry->Partition_Name[0]!=0x00) {

			strcpy(pname, "os2_seg_");
			strncat( pname, (char *) &seg_pdata->dla_entry->Partition_Name[0], PARTITION_NAME_SIZE);

			EngFncs->unregister_name( pname );
		}

	}

	// test if we need to unregister any serial numbers for this segment
	if ( seg_pdata->flags & SEG_HAS_DLAT_SERIAL_NUMBERS_REGISTERED ) {

		seg_unregister_serial_number( seg_pdata->dla_entry->Partition_Serial_Number );
		seg_unregister_serial_number( seg_pdata->dla_entry->Volume_Serial_Number );

	}

	// if this is the mbr segment then test if we need to unregister os2 disk info
	if ( ( seg_pdata->flags & SEG_IS_MBR ) &&
	     ( disk_pdata->flags & DISK_HAS_OS2_DLAT_TABLES )) {

		seg_unregister_serial_number( seg_pdata->dlat->Disk_Serial_Number );
		EngFncs->unregister_name( (char *) seg_pdata->dlat->Disk_Name );

	}


	LOG_EXIT_INT(0);
	return 0;
}


/*
 *   Called to name a new disk segment and insert it into the
 *   ordered segment list for the logical disk.
 */
void *  insert_diskseg_into_list( list_anchor_t seglist, DISKSEG *seg)
{
	int                rc=EINVAL;
	SEG_PRIVATE_DATA  *seg_pdata = (SEG_PRIVATE_DATA *) seg->private_data;
	LOGICALDISK       *ld = get_logical_disk(seg);
	DISK_PRIVATE_DATA *disk_pdata=NULL;
	void              *result=NULL;


	LOG_ENTRY();
	LOG_DEBUG("seg start= %"PRIu64"   size= %"PRIu64"\n", seg->start, seg->size);

	// need our private disk data
	disk_pdata = get_disk_private_data(ld);
	if (disk_pdata == NULL) {
		LOG_EXIT_PTR(NULL);
		return NULL;
	}

	/*
	 *  Storage objects must all have unique names. First get a name
	 *  for the disk segment and then register it with the engine.
	 *  if successful ... continue.
	 */
	rc = get_name_for_disk_segment( seg );
	if (rc) {
		LOG_ERROR("error, get_name_for_disk_segment failed, RC= %d\n", rc);
		LOG_EXIT_PTR(NULL);
		return NULL;
	}
	else {
		rc = EngFncs->register_name( seg->name );
		if (rc) {
			LOG_ERROR("error, get_name_for_disk_segment failed, RC= %d\n", rc);
			LOG_EXIT_PTR(NULL);
			return NULL;
		}
	}

	/*
	 * Check if we need to register any DLAT serial numbers
	 */
	if ( disk_pdata->flags & DISK_HAS_OS2_DLAT_TABLES ) {

		if ( (seg_pdata->flags & SEG_IS_PRIMARY_PARTITION) ||
		     (seg_pdata->flags & SEG_IS_LOGICAL_PARTITION)) {


			// OS/2 partition names must be unique. But make
			// sure you have a name to register!
			if ( seg_pdata->dla_entry->Partition_Name[0]!=0x00) {

				char pname[PARTITION_NAME_SIZE+12];

				strcpy(pname, "os2_seg_");
				strncat(pname, seg_pdata->dla_entry->Partition_Name, PARTITION_NAME_SIZE);

				rc = EngFncs->register_name( pname );
				if (rc) {
					LOG_ERROR("error, register os2 partition name failed\n");
					LOG_EXIT_PTR(NULL);
					return NULL;
				}

			}

			// OS/2 dlat entries will not have serial numbers if the partitions
			// have not been made into volumes yet.
			if ( (seg_pdata->dla_entry->Partition_Serial_Number != 0 ) &&
			     (seg_pdata->dla_entry->Volume_Serial_Number != 0 ) ) {

				rc  = seg_register_serial_number( seg_pdata->dla_entry->Partition_Serial_Number );

				// dont register LVM serial numbers

				if (rc) {
					LOG_ERROR("error, register dlat entry serial numbers failed\n");
					LOG_EXIT_PTR(NULL);
					return NULL;
				}
				else {
					seg_pdata->flags |= SEG_HAS_DLAT_SERIAL_NUMBERS_REGISTERED;
				}
			}

		}

	}

	result =  insert_diskseg_into_ordered_list( seglist, seg );

	// if we just created an extended partition then chain it to the
	// mbr so that the routine disk_has_extended_partition() will work.
	if ( ( result != NULL) &&
	     ( seg_pdata->flags & SEG_IS_EBR ) &&
	     ( disk_has_extended_partition(ld) == FALSE ) ) {

		storage_object_t *mbr;

		mbr = get_mbr_from_seglist( ld->parent_objects );

		if (mbr) {
			((SEG_PRIVATE_DATA *)mbr->private_data)->next_ebr = result;
		}

	}


	LOG_DEBUG("returning %p\n", result);
	LOG_EXIT_PTR(result);
	return result;
}


/*
 *   Called to insert a disk segment into a segment list that we
 *   are trying to keep ordered by the segments starting LBA
 *   number.
 */
void *  insert_diskseg_into_ordered_list( list_anchor_t seglist, DISKSEG *seg )
{
	DISKSEG           *seg2;
	int                rc=0;
	LOGICALDISK       *ld = get_logical_disk(seg);
	SEG_PRIVATE_DATA  *seg_pdata;
	SEG_PRIVATE_DATA  *seg2_pdata;
	lba_t              seg2_end_lba;
	lba_t              seg_end_lba;
	boolean            overlapping=FALSE;
	DISK_PRIVATE_DATA *disk_pdata=NULL;
	list_element_t     iter,e;
	char               number_buffer[64];


	LOG_ENTRY();

	seg_end_lba = seg->start+seg->size-1;

	LOG_DEBUG("seg name= %s   seg start= %"PRIu64"  ends= %"PRIu64"  size= %"PRIu64"\n",
		  seg->name,
		  seg->start,
		  seg_end_lba,
		  seg->size );

	rc = -1;

	LIST_FOR_EACH( seglist, iter, seg2 ) {

		seg2_end_lba = seg2->start + seg2->size - 1;

		// test and set ... overlapping segments flag
		if ( (  seg->start >= seg2->start )&&
		     (  seg->start <= seg2_end_lba)) {
			overlapping = TRUE;
		}
		else if ( ( seg->start  <  seg2->start ) &&
			  ( seg2->start <= (seg->start + seg->size - 1)) ) {
			overlapping = TRUE;
		}
		else {
			overlapping = FALSE;
		}

		// try and fix overlapping segments ...
		if ( overlapping == TRUE ) {

			LOG_DEBUG("Error ... Overlapping Segments ...\n");
			LOG_DEBUG("seg2:   name: %s\n", seg2->name );
			LOG_DEBUG("       start: %"PRIu64"\n", seg2->start );
			LOG_DEBUG("        size: %"PRIu64"\n", seg2->size );
			LOG_DEBUG("         end: %"PRIu64"\n", seg2_end_lba );
			LOG_DEBUG(" overlap lba: %"PRIu64"\n", seg->start );

			seg_pdata  = (SEG_PRIVATE_DATA *) seg->private_data;
			seg2_pdata = (SEG_PRIVATE_DATA *) seg2->private_data;

			disk_pdata = get_disk_private_data(ld);

			// Test if we simply created the mbr metadata seg too big
			// some tools will start partitions immediately following
			// the MBR sector, instead of leaving an entire track for the
			// MBR.
			if ( ( seg2_pdata->flags & SEG_IS_MBR ) &&
			     ( seg->start > seg2->start  ) ) {

				seg2->size = seg->start;   // solution ... just shrink
							   // the mbr segment
			}

			// Test if we have an ebr metadata seg that is too big.
			// Usually, we give the ebr segment an entire track like
			// msdos would. Sometimes though, we find an ebr segment
			// that larger than a track and we need to shrink it down
			// to fit in another logical drive.  Usually, you will
			// find the logical drive starting in the sector immediately
			// following the ebr segment we are overlapping.  This
			// happens when the ebr is in the 1st sector of the extended
			// partition but the data partition is at the end of the drive.
			// Then, other logical partitions are added in between the two.
			else if ( ( seg2_pdata->flags & SEG_IS_EBR ) &&
				  ( seg2->size > disk_pdata->geometry.sectors_per_track) &&
				  ( seg->start > seg2->start       ) ) {

				seg2->size = seg->start - seg2->start; // shrink ebr segment

				if ( seg2->size > disk_pdata->geometry.sectors_per_track ) {
					seg2->size = disk_pdata->geometry.sectors_per_track;
				}

			}

			// Test if we have an ebr metadata seg that is being used
			// to anchor an ebr chain and along comes a logical drive
			// that sits on top of the anchor seg. In this situation we
			// simply move and shrink the new ebr segment as you will
			// see. This situation occurs during a create and we catch
			// it here to help ease the create seg work.
			else if ( ( seg_pdata->flags & SEG_IS_EBR  ) &&
				  ( seg2_pdata->flags & SEG_IS_EBR ) &&
				  ( seg->start == seg2->start      ) ) {

				seg2->size = 1;
				if (seg->size > 1) seg->size -= 1;
				++seg->start;

			}

			// Test if we have an ebr metadata seg that is in the last
			// sector of a track that has an ebr sector in the first
			// sector and just fixup sizes ... the first ebr segment
			// will be track size minus 1.  The second ebr will be just
			// 1 sector in size and left alone.room
			else if ( ( seg_pdata->flags & SEG_IS_EBR  ) &&
				  ( seg2_pdata->flags & SEG_IS_EBR ) &&
				  ( seg->start == seg2_end_lba ) &&
				  ( seg2->size > 1) ) {

				seg2->size -= 1;

			}

			else {
				rc = EINVAL;	// must be genuine partition overlap
				break;
			}

		}


		// test for ... valid insertion point
		if ( seg2->start > seg->start ) {
			rc = 0;
			break;
		}
	}


	switch (rc) {
	
	case EINVAL:

		sprintf(number_buffer, "%"PRIu64, seg->start);
		if ( (seg->data_type == DATA_TYPE) &&
		     (seg2->data_type == DATA_TYPE)) {

			MESSAGE(_("Found overlapping partitions on drive %s.\n"
				  "Partition %s overlaps partition %s at lba %s.\n"),
				ld->name, seg->name, seg2->name, number_buffer);
		}
		else {
			MESSAGE(_("Found overlapping segment storage objects for drive %s.\n"
				  "EVMS segment %s overlaps segment %s at lba %s.\n"),
				ld->name, seg->name, seg2->name, number_buffer);
		}
		break;

	case 0:	 /* Ok, found a segment we should insert in front of */

		e = EngFncs->insert_thing( seglist,
					   seg,
					   INSERT_BEFORE|EXCLUSIVE_INSERT,
					   EngFncs->find_in_list(seglist, seg2, NULL, NULL) );
		if (e != NULL) {
			rc = 0;
		}
		else {
			rc = EPERM;
		}
		break;

	case -1: /* This new segment lays out on the disk at a higher */
		/* LBA than any other existing segment.  So, just    */
		/* insert at end of the segment list.                */

		e = EngFncs->insert_thing( seglist,
					   seg,
					   EXCLUSIVE_INSERT,
					   NULL);
		if (e) {
			rc = 0;
		}
		else {
			rc = EPERM;
		}
		break;

	default:     /* REAL ERROR ... return NULL ptr */
		LOG_ERROR("error, insertion failed ... RC= %d\n",  rc);
		break;
	}

	if (rc) {
		LOG_EXIT_PTR(NULL);
		return NULL;
	}
	else {
		LOG_EXIT_PTR(seg);
		return seg;
	}
}


/*
 *  Called to walk an MBR/EBR chain and calculate the size for each
 *  EBR found. This is only called by fixup_EBR_Chain() because this
 *  routine is dependant on the EBR chaining ptrs which are setup by
 *  the routine fixup_disk_extd_partition_dimensions().
 *
 *  The purpose of the routine is to save away the size of each EBR
 *  so that the commit code can use the EBR_SECTOR_COUNT field in
 *  seg private data when committing an EBR table.  This is needed
 *  in advance because the commit code would have to read ahead to
 *  get this info and it screws up the commit.  Look at the routine
 *  Build_EBR_PartitionTable(), found in commit.c.  When it is building
 *  an ebr table, and it encounters an extended partition record, it
 *  needs to know how big the extd partitoin is ... that the record
 *  points to.  Rather than do read aheads ... I just keep the info
 *  in the ebr seg private data.
 *
 */
void fixup_EBR_Sizes( LOGICALDISK *ld )
{
	DISKSEG          *seg;
	SEG_PRIVATE_DATA *pdata;

	DISKSEG          *ebr;
	SEG_PRIVATE_DATA *ebr_pdata;

	DISKSEG           *mbr = get_mbr_from_seglist(ld->parent_objects);
	DISK_PRIVATE_DATA *disk_pdata = get_disk_private_data( ld );
	list_element_t     iter;

	LOG_ENTRY();

	// advance MBR -> extended partition
	ebr_pdata = (SEG_PRIVATE_DATA *) mbr->private_data;
	ebr   = ebr_pdata->next_ebr;

	// return if no extended partition exists
	if (ebr==NULL) {
		LOG_DEBUG("no extended partition on this disk\n");
		LOG_EXIT_VOID();
		return;
	}

	// advance extd partition -> 1st logical drive
	ebr_pdata                   = (SEG_PRIVATE_DATA *) ebr->private_data;
	ebr_pdata->ebr_sector_count = disk_pdata->extd_partition_size;
	ebr                         = ebr_pdata->next_ebr;

	// walk EBR chain
	while (ebr != NULL) {

		ebr_pdata = (SEG_PRIVATE_DATA *) ebr->private_data;

		// initial size is just the EBR track itself
		ebr_pdata->ebr_sector_count = ebr->size;

		// now add in logical partitions to get total size.
		LIST_FOR_EACH( ld->parent_objects, iter, seg ){

			pdata = (SEG_PRIVATE_DATA *) seg->private_data;

			if ( ( pdata->ebr == ebr ) &&
			     ( pdata->flags & SEG_IS_LOGICAL_PARTITION )) {
				ebr_pdata->ebr_sector_count += seg->size;
			}



		}

		ebr = ebr_pdata->next_ebr;
	};

	LOG_EXIT_VOID();
}


/*
 *  Called to fixup names of logial and embedded partitions because their names will
 *  change if the EBR chain is modified by a delete or create.
 */
int   fixup_logical_partition_names( LOGICALDISK *ld )
{
	int                rc;
	DISKSEG           *seg=NULL;
	DISKSEG           *embedded_seg=NULL;
	DISKSEG           *ebr=NULL;
	SEG_PRIVATE_DATA  *pdata=NULL;
	SEG_PRIVATE_DATA  *ebr_pdata=NULL;
	u_int32_t          embedded_minor;
	list_anchor_t      embedded_seg_list;
	int                embedded_partition_count=0;
	int                logical_partition_count=0;
	int                i;
	int                lowest_number;
	char               old_name[EVMS_NAME_SIZE+1];
	list_element_t     iter;

	LOG_ENTRY();

	embedded_seg_list = EngFncs->allocate_list();
	if (embedded_seg_list == NULL) {
		rc = EPERM;
		LOG_ERROR("error: create embedded seglist failed\n");
		LOG_EXIT_INT(rc);
		return rc;
	}

	// unregister all logical partition names and embedded partition names
	LIST_FOR_EACH( ld->parent_objects, iter, seg ){

		pdata = (SEG_PRIVATE_DATA *) seg->private_data;

		if ( pdata->flags & SEG_IS_LOGICAL_PARTITION ) {

			ebr = pdata->ebr;
			if (ebr) {

				ebr_pdata = (SEG_PRIVATE_DATA *) ebr->private_data;
				if (ebr_pdata) {

					if (strlen(seg->name))	EngFncs->unregister_name( seg->name );

					// temp copy of old name
					strcpy(old_name, seg->name);

					// build a name for this logical partition
					pdata->part_number  = ebr_pdata->ebr_number + 5;
					get_name_for_disk_segment( seg );

					// test if name has changed and then mark the object so we
					// can do a dm rename of the kernel object at commit time.
					if (  strlen(old_name) > 0  &&
					      strncmp(seg->name, old_name, EVMS_NAME_SIZE)!=0) {

						dos_schedule_dm_rename(seg);


					}

					++logical_partition_count;
				}


			}
			else {
				LOG_ERROR("error, found a logical partition that has no ebr associated with it.\n");
				// but continue to get logical partitions named correctly
			}


		}
		else if ( pdata->flags & SEG_IS_EMBEDDED ) {

			list_element_t e;

			if (strlen(seg->name)) EngFncs->unregister_name( seg->name );

			e = EngFncs->insert_thing( embedded_seg_list,
						   seg,
						   EXCLUSIVE_INSERT,
						   NULL );
			if (e) {
				rc = 0;
				++embedded_partition_count;
			}
			else {
				rc = EPERM;
				LOG_ERROR("error, list errors constructing embedded seglist\n");
				// but continue to get logical partitions named correctly
			}
		}
	}

	embedded_minor = logical_partition_count + 5;

	// build new embedded partition names
	rc = 0;
	for (i=0; (i<embedded_partition_count)&&(rc==0); i++) {

		lowest_number   = MAX_EBR_NUMBER;
		embedded_seg    = NULL;

		LIST_FOR_EACH( embedded_seg_list, iter, seg ) {

			pdata = (SEG_PRIVATE_DATA *) seg->private_data;

			if ( pdata->part_number < lowest_number ) {
				embedded_seg = seg;
				lowest_number = pdata->part_number;
			}

		}

		if (embedded_seg != NULL) {

			pdata = (SEG_PRIVATE_DATA *) embedded_seg->private_data;

			// temp copy of old name
			strcpy(old_name, embedded_seg->name);

			pdata->part_number  = embedded_minor;
			get_name_for_disk_segment( embedded_seg );

			// test if name has changed and then mark the object so we
			// can do a dm rename of the kernel object at commit time.
			if (  strlen(old_name) > 0  &&
			      strncmp(embedded_seg->name, old_name, EVMS_NAME_SIZE)!=0) {
				dos_schedule_dm_rename(embedded_seg);
			}

			++embedded_minor;
			EngFncs->remove_thing( embedded_seg_list, embedded_seg );
		}
		else {
			rc = ENODEV;
		}

	}

	// register all logical and embedded partition names
	LIST_FOR_EACH( ld->parent_objects, iter, seg ) {

		pdata = (SEG_PRIVATE_DATA *) seg->private_data;

		if ( ( pdata->flags & SEG_IS_LOGICAL_PARTITION ) ||
		     ( pdata->flags & SEG_IS_EMBEDDED ) ) {
			EngFncs->register_name( seg->name );
		}
	}

	EngFncs->destroy_list(embedded_seg_list);
	LOG_EXIT_INT(0);
	return 0;
}



/*
 *  Called when changes are made to the EBR chain to fixup
 *  the names of EBR segments.  These names, and those of
 *  logical partitions, will change as you add and delete
 *  logical drives.
 */
void fixup_EBR_Names( LOGICALDISK *ld ){
	u_int32_t         ebr_number = 0;
	DISKSEG          *ebr;
	DISKSEG          *mbr;
	SEG_PRIVATE_DATA *pdata;


	LOG_ENTRY();

	mbr = get_mbr_from_seglist(ld->parent_objects);
	if ( mbr ) {

		// unregister old names and build new names
		// new names depend on position in ebr chain
		ebr = ((SEG_PRIVATE_DATA *)mbr->private_data)->next_ebr;
		while ( ebr ) {

			pdata = (SEG_PRIVATE_DATA *)ebr->private_data;

			if (pdata) {

				// unregister old name if we have one
				if (strlen(ebr->name)) EngFncs->unregister_name( ebr->name );

				// set new ebr number
				pdata->ebr_number = ebr_number;

				// build new segment name
				get_name_for_disk_segment( ebr );

				// increment for next ebr
				++ebr_number;

				// chain to next ebr
				ebr = pdata->next_ebr;


			}
			else {
				LOG_EXIT_VOID();
				return ; // quit ...
			}
		};

		// now register ebr names
		ebr = ((SEG_PRIVATE_DATA *)mbr->private_data)->next_ebr;
		while ( ebr ) {

			pdata = (SEG_PRIVATE_DATA *)ebr->private_data;

			if (pdata) {

				EngFncs->register_name( ebr->name );

				ebr = pdata->next_ebr;


			}
			else {
				LOG_EXIT_VOID();
				return;	// quit ...
			}
		}


	}


	LOG_EXIT_VOID();
}


static int do_os2_ebr_chaining( LOGICALDISK       *ld,
				DISK_PRIVATE_DATA *disk_pdata,
				DISKSEG           *mbr,
				list_anchor_t            ebr_seg_list,
				int                ebr_count ){
	int                rc = 0;
	SEG_PRIVATE_DATA  *pdata;
	DISKSEG           *last_ebr=NULL;
	DISKSEG           *ebr;
	list_element_t     iter;

	LOG_ENTRY();


	// ebr chain is anchored off mbr disk segment so setup the mbr
	// for hangingin an ebr chain off it.

	pdata = (SEG_PRIVATE_DATA *) mbr->private_data;

	pdata->ebr          = NULL;
	pdata->prev_ebr     = NULL;
	pdata->next_ebr     = NULL;

	disk_pdata->logical_drive_count = ebr_count;
	disk_pdata->flags  |= DISK_HAS_PHYS_ORDERED_LOGICAL_PARTITIONS;

	last_ebr            = mbr;


	//
	// Now, chain EBRs in the order in which they occur in the working
	// list.  This is because they are in the correct order already and
	// so all we need to do is to walk the list and touch up ptrs.
	//
	LIST_FOR_EACH( ebr_seg_list, iter, ebr ) {

		//  fixup ... ptable index if prev ebr ptr is actually mbr
		if (last_ebr == mbr) {

			int mbr_ptable_index = get_extd_partition_ptable_entry(ld->parent_objects, mbr);

			if ( mbr_ptable_index == -1 ) {
				LOG_ERROR("error, there are no unused entries in MBR partition table\n");
				rc = EINVAL;
				LOG_EXIT_INT(rc);
				return rc;
			}
			else {
				pdata->ptable_index = mbr_ptable_index;
			}
		}

		/* fix up THIS EBR chaining ptrs */
		pdata = (SEG_PRIVATE_DATA *) ebr->private_data;
		pdata->ebr          = last_ebr;
		pdata->prev_ebr     = last_ebr;
		pdata->next_ebr     = NULL;

		/* fix up PREV EBR forward ptrs */
		pdata = (SEG_PRIVATE_DATA *) last_ebr->private_data;
		pdata->next_ebr     = ebr;

		/* setup for next pass */
		last_ebr            = ebr;

	}

	LOG_EXIT_INT(rc);
	return rc;
}




static int do_linux_ebr_chaining( LOGICALDISK       *ld,
				  DISK_PRIVATE_DATA *disk_pdata,
				  DISKSEG           *mbr,
				  list_anchor_t            ebr_seg_list,
				  int                ebr_count ){
	int                rc = 0;
	int                i;
	SEG_PRIVATE_DATA  *pdata;
	DISKSEG           *last_ebr=NULL;
	DISKSEG           *seg;
	DISKSEG           *ebr;
	u_int32_t          lowest_ebr_number;
	lba_t              last_lba;
	list_element_t     iter;

	LOG_ENTRY();

	// ebr chain is anchored off mbr disk segment so setup the mbr
	// for hangingin an ebr chain off it.

	pdata = (SEG_PRIVATE_DATA *) mbr->private_data;

	pdata->ebr          = NULL;
	pdata->prev_ebr     = NULL;
	pdata->next_ebr     = NULL;

	disk_pdata->logical_drive_count = ebr_count;

	last_ebr            = mbr;

	//
	// Now, chain EBRs in the order in which they were discovered on the drive.
	// the EBR chaining ptrs are used by commit code, allowing the code to
	// walk the EBR chain and commit partition tables.
	//
	for (i=0; (i<ebr_count)&&(rc==0); i++) {

		lowest_ebr_number   = MAX_EBR_NUMBER;
		ebr                 = NULL;

		LIST_FOR_EACH( ebr_seg_list, iter, seg ) {

			pdata = (SEG_PRIVATE_DATA *) seg->private_data;

			if ( pdata->ebr_number < lowest_ebr_number ) {
				ebr = seg;
				lowest_ebr_number = pdata->ebr_number;
			}

		}

		if (ebr != NULL) {

			EngFncs->remove_thing( ebr_seg_list, ebr );

			//  fixup ... ptable index if prev ebr ptr is actually mbr
			if (last_ebr == mbr) {

				int mbr_ptable_index = get_extd_partition_ptable_entry(ld->parent_objects, mbr);

				if ( mbr_ptable_index == -1 ) {
					LOG_ERROR("error, there are no unused entries in MBR partition table\n");
					rc = EINVAL;
					LOG_EXIT_INT(rc);
					return rc;
				}
				else {
					pdata->ptable_index = mbr_ptable_index;
				}
			}

			/* fix up THIS EBR chaining ptrs */
			pdata = (SEG_PRIVATE_DATA *) ebr->private_data;
			pdata->ebr          = last_ebr;
			pdata->prev_ebr     = last_ebr;
			pdata->next_ebr     = NULL;

			/* fix up PREV EBR forward ptrs */
			pdata = (SEG_PRIVATE_DATA *) last_ebr->private_data;
			pdata->next_ebr     = ebr;

			/* setup for next pass */
			last_ebr            = ebr;


		}
		else {
			rc = ENODEV;
		}

	}


	// Test if the logical drives are layed down on the disk in
	// physical order.  This is easy to check ... just walk the
	// ebr chain and each LOGICAL PARTITION that we encounter in
	// the chain should start at a higher LBA than the previous
	// LOGICAL PARTITION.
	if (rc == 0 && ebr_count > 0) {

		ebr                 = mbr;
		last_lba            = 0;
		disk_pdata->flags  |= DISK_HAS_PHYS_ORDERED_LOGICAL_PARTITIONS;


		while ( (ebr != NULL) && (disk_pdata->flags & DISK_HAS_PHYS_ORDERED_LOGICAL_PARTITIONS) ) {

			LIST_FOR_EACH( ld->parent_objects, iter, seg ) {

				pdata = (SEG_PRIVATE_DATA *) seg->private_data;

				if ( ( pdata->ebr == ebr ) &&
				     ( pdata->flags & SEG_IS_LOGICAL_PARTITION ) ) {

					if (last_lba > seg->start) {
						disk_pdata->flags  &= ~DISK_HAS_PHYS_ORDERED_LOGICAL_PARTITIONS;
						break;
					}

					last_lba = seg->start;
				}
			}

			ebr = ((SEG_PRIVATE_DATA *)ebr->private_data)->next_ebr;

			// this new ebr segment must also be in physical order on the disk
			if (ebr != NULL) {

				if ( ebr->start < ((SEG_PRIVATE_DATA *)ebr->private_data)->prev_ebr->start ) {
					disk_pdata->flags  &= ~DISK_HAS_PHYS_ORDERED_LOGICAL_PARTITIONS;
				}

			}

		}

		rc=0;
	}


	LOG_EXIT_INT(rc);
	return rc;
}


/*
 *  Called to walk an MBR/EBR chain and fixup the forward and backward pointers
 *  that are kept in the segment managers private data area.
 */
int  fixup_EBR_Chain( LOGICALDISK *ld ){
	int                rc = EINVAL;
	DISKSEG           *seg;
	SEG_PRIVATE_DATA  *pdata;
	DISKSEG           *mbr;
	DISK_PRIVATE_DATA *disk_pdata;
	int                ebr_count=0;
	list_anchor_t      ebr_seg_list;
	list_element_t     iter;

	LOG_ENTRY();

	//
	// Test callers parms
	//
	mbr        = get_mbr_from_seglist(ld->parent_objects);
	disk_pdata = get_disk_private_data( ld );

	if ((mbr==NULL)||(disk_pdata==NULL)) {
		LOG_ERROR("error: bad parms, mbr_ptr= %p  disk_pdata_ptr= %p\n", mbr, disk_pdata);
		LOG_EXIT_INT(rc);
		return rc;
	}

	// Get a temp list to use
	ebr_seg_list = EngFncs->allocate_list();
	if (ebr_seg_list == NULL) {
		LOG_ERROR("error: bad parms, mbr_ptr= %p  disk_pdata_ptr= %p\n", mbr, disk_pdata);
		LOG_EXIT_INT(rc);
		return rc;
	}

	//
	// Copy the EBR segments to our working LIST
	//
	LIST_FOR_EACH( ld->parent_objects, iter, seg ) {

		pdata = (SEG_PRIVATE_DATA *) seg->private_data;

		if (pdata->flags & SEG_IS_EBR) {
			if (EngFncs->insert_thing(ebr_seg_list,seg,INSERT_AFTER,NULL)) {
				rc = 0;
			}
			else {
				rc = EPERM;
			}
		}
	}

	//
	// see if there are any logical drives to chain together
	//
	ebr_count = EngFncs->list_count(ebr_seg_list);

	//
	// OS/2 disks keep logical drives in physical order.
	// A linux disk does not. So, we use two different
	// ordering algorithms to support this.
	//
	if ( disk_pdata->flags & DISK_HAS_OS2_DLAT_TABLES ) {
		rc = do_os2_ebr_chaining( ld, disk_pdata, mbr, ebr_seg_list, ebr_count );
	}
	else {
		rc = do_linux_ebr_chaining( ld, disk_pdata, mbr, ebr_seg_list, ebr_count );
	}


	if (rc == 0) {

		// make sure we have a sys_ind value for the extended partition
		// in case we just created the extd partition itself
		if ( (ebr_count > 0) &&
		     (disk_pdata->extd_partition_sys_ind == 0)) {
			disk_pdata->extd_partition_sys_ind = DOS_EXTENDED_PARTITION;  // use MSDOS default
		}

		// recalc the size of the extended partition
		fixup_disk_extd_partition_dimensions( ld );

		// if logical drives are offset in the extd partition we need to
		// keep an anchor EBR in the 1st sector of the extd partition.
		fixup_disk_extd_partition_anchor(ld);

		// calc sizes of each std ebr partition
		fixup_EBR_Sizes(ld);

		// rename EBR segments due to shifting
		fixup_EBR_Names(ld);


	}


	EngFncs->destroy_list(ebr_seg_list);

	LOG_EXIT_INT(rc);
	return rc;
}

/*
 *  Function: prune zero length freespace segs
 */
static void prune_zero_length_freespace_segs( list_anchor_t list )
{
	DISKSEG *seg;
	list_element_t iter,iter2;
	boolean prune;

	LOG_ENTRY();

	LIST_FOR_EACH_SAFE( list, iter, iter2, seg ) {

		prune = FALSE;

		if ( seg->data_type == FREE_SPACE_TYPE &&
		     seg->size      == 0) {
			EngFncs->unregister_name( seg->name );
			free_disk_segment( seg );
			prune = TRUE;
		}

		if (prune==TRUE) EngFncs->delete_element(iter);
	}

	LOG_EXIT_VOID();
}


/*
 *  Called to inspect the extended partition on a drive and make sure we have
 *  an anchor EBR in the 1st sector of the extended partition.  This might entail
 *  moving the EBR from the 1st logical drive to this new location.  This really
 *  only happens when we delete the 1st logical drive in a chain and its ebr was
 *  being used to anchor the extd partition.
 *
 *  Simple EBR Chain occurs when logical drives are in physical order ...
 *
 *      MBR -> LD0 -> LD1 -> LD2
 *
 *  More complex EBR chain occcurs when user creates the 1st logical drive, using
 *  an offset into the freespace segment.
 *
 *      MBR -----------------> LD0
 *             LD2 <-- LD1<--- +
 *
 *  This results in requiring an EBR sector in the 1st sector of the extd partition
 *  AT ALL TIMES!  So, if you later delete LD0 you still need to maintain the EBR
 *  anchor.
 *
 *  The extended partition will look kinda like this ...
 *
 *         +----------------------------------------------------------------+
 *         |  +-----------------------------+                               |
 *         |  |     +---------------------+ | +---+                         |
 *         |  |     |  +---+              | | |   |                         |
 *         |  |     V  |   V              | V |   V                         V
 *         +------+-------+----------------+------+---------------------------+------------+
 *  MBR--> | ebr0 |  ebr2 | ld2            | ebr1 | ld1                       | ld0        |
 *         +------+-------+----------------+------+---------------------------+------------+
 *
 *
 *  You can see that the extended partition record, from the MBR partition table, expects
 *  to find an EBR in the 1st sector of the extended partition.  If we deleted LD0 and its
 *  EBR sector there would no longer be an EBR at this location.
 *
 *
 *  SO ... the purpose of this routine is to preserve the anchor EBR if we need one!!!!
 *
 *  Obviously,if the logical drives are in physical order then as we delete drives we'll
 *  simply shrink the extended partition.
 *
 */
void  fixup_disk_extd_partition_anchor( LOGICALDISK *ld ){
	DISK_PRIVATE_DATA *disk_pdata = get_disk_private_data(ld);
	DISKSEG           *mbr=NULL;
	DISKSEG           *ebr=NULL;
	DISKSEG           *ebr2=NULL;
	DISKSEG           *anchor_ebr=NULL;
	DISKSEG           *lp;
	DISKSEG           *first_lp=NULL;

	DISKSEG           *seg=NULL;
	DISKSEG           *last_seg=NULL;
	SEG_PRIVATE_DATA  *pdata=NULL;
	sector_count_t     offset;
	sector_count_t     track_size;
	boolean            cleanup_freespace=FALSE;
	lba_t              track_ending_lba=0;
	lba_t              track_starting_lba=0;
	lba_t              ebr_start=0;
	sector_count_t     ebr_size=0;
	int                rc;
	list_element_t     iter;

	LOG_ENTRY();

	// OS2 partitions are in physical order ... so we never have to
	// anchor them with an ebr at sector Zero of the ext partition.
	if ( (disk_pdata->flags & DISK_HAS_OS2_DLAT_TABLES) == FALSE) {

		LOG_DEBUG("extended start lba: %"PRIu64"\n", disk_pdata->extd_partition_lba );
		LOG_DEBUG("extended   end lba: %"PRIu64"\n", disk_pdata->extd_partition_end_lba );
		LOG_DEBUG("extended      size: %"PRIu64"\n", disk_pdata->extd_partition_size );

		mbr = get_mbr_from_seglist(ld->parent_objects);

		if ( mbr ) {

			ebr = ((SEG_PRIVATE_DATA *)mbr->private_data)->next_ebr;

			if ( ebr ) {

				// if we have an existing extended partition AND ...
				// we need to move the anchor ebr THEN ...
				// move ebr to new location on drive
				if ( ( disk_pdata->extd_partition_size != 0 ) &&
				     ( ebr->start != disk_pdata->extd_partition_lba )) {

					LOG_DEBUG("need to anchor the extd partition ... moving ebr0 to new location\n");

					// save ebr info
					ebr_start = ebr->start;
					ebr_size  = ebr->size;

					// look for an ebr already anchoring the extd partition
					LIST_FOR_EACH(ld->parent_objects, iter, ebr2){
						if (ebr2->start == disk_pdata->extd_partition_lba) {
							anchor_ebr = ebr2;
						}
					}

					// look for the first logical partition in current ebr chain
					LIST_FOR_EACH(ld->parent_objects, iter, lp){
						if ( ( lp->data_type == DATA_TYPE ) &&
						     ( ((SEG_PRIVATE_DATA *)lp->private_data)->ebr == ebr )) {
							first_lp = lp;
						}
					}

					// remove it from the disk seglist
					EngFncs->remove_thing( ld->parent_objects, ebr );

					// set new location and size. whenever we split off
					// the ebr segment like this well only use a single
					// sector like other partitioning tools.
					if (anchor_ebr) {
						++anchor_ebr->start;
						--anchor_ebr->size;
					}

					// anchor ebr always goes in 1st sector of ext partition
					ebr->start = disk_pdata->extd_partition_lba;

					// now check if we size the new anchor ebr to 1 sector because there
					// is considerable distance to the logical partition it anchors
					// ... or ...
					// if we resize the new anchor ebr to the number of sectors
					// between the ebr and its logical partition.
					//
					// this happens when the ebr is currently in the 2nd sector of its
					// ebr track because we were anchoring an ebr chain in the extd
					// partition ... and we just destroyed a logical partition ...
					// leaving a single logical partition in the extd partition.
					// anchor.
					if ( ( first_lp != NULL) &&
					     ( first_lp->start > ebr->start ) &&
					     ( (first_lp->start - ebr->start) <= disk_pdata->geometry.sectors_per_track) ) {
						ebr->size = first_lp->start - ebr->start;
					}
					else {
						ebr->size = 1;
					}

					// insert updated ebr in list ... if any errors ... recover original ebr
					if ( insert_diskseg_into_ordered_list( ld->parent_objects, ebr )==NULL) {
						ebr->start = ebr_start;
						ebr->size  = ebr_size;
						insert_diskseg_into_ordered_list( ld->parent_objects, ebr );
					}


				}

				// Every EBR should be in the very 1st sector of the EBR track ...
				//
				// So, walk the ebr chain and see if any ebr sector can be moved
				// back into the very 1st sector of the ebr track. Sometimes you
				// will wind up with an EBR that is NOT in the very 1st sector of
				// the cylinder due to some EBR anchoring movements with the EBRs.
				//
				// Some window installs seem to be putting EBR metadata sectors in
				// the last sector of a track. Leave them!
				track_size = ld->geometry.sectors_per_track * disk_pdata->vsectors_per_block;
				LIST_FOR_EACH(ld->parent_objects, iter, seg ){

					pdata = (SEG_PRIVATE_DATA *)seg->private_data;

					// if we have an EBR and it doesn't start on a cyl bdy ...
					if ( (pdata->flags & SEG_IS_EBR) &&
					     starts_on_cylinder_boundary(ld,seg->start) == FALSE ) {

						track_starting_lba = rounddown_to_track_boundary(ld, seg->start);
						track_ending_lba   = roundup_to_track_boundary(ld,seg->start);

						LOG_DEBUG("EBR %s does not start on cyl bdy\n", seg->name);
						LOG_DEBUG("     seg start lba = %"PRIu64"\n", seg->start);
						LOG_DEBUG("     seg size      = %"PRIu64"\n", seg->size);
						LOG_DEBUG("     trk start lba = %"PRIu64"\n", track_starting_lba);
						LOG_DEBUG("     trk end   lba = %"PRIu64"\n", track_ending_lba);

						// and it is not in the last sector of the track ...
						if ( seg->start != track_ending_lba ) {

							LOG_DEBUG("     will try to move ebr that is not in last sector of trk\n");

							offset = seg->start - track_starting_lba;

							// see if this sector is available for moving the EBR seg
							// if any errors or misgivings ... just dont move it.
							if (last_seg && (offset < track_size) ) {

								// if previous segment doesnt spill into the EBR track ...
								if ( (last_seg->start+last_seg->size-1) < track_starting_lba) {
									LOG_DEBUG("     track is clear ... moving to 1st sector of track\n");
									seg->start -= offset;
									seg->size  += offset;
								}
								else {	// something is already in the EBR track ...
									LOG_DEBUG("     track has something else on it\n");

									// freespace on the track ... but starts before track ...
									if ( last_seg->data_type == FREE_SPACE_TYPE &&
									     last_seg->start < track_starting_lba) {
										LOG_DEBUG("     freespace but ok to shrink it and move ebr\n");
										seg->start -= offset;
										seg->size  += offset;
										last_seg->size -= offset;
									}
								}


							}
							else if (offset < track_size) {
								LOG_DEBUG("     ok to move cuz no previous segments\n");
								seg->start -= offset;
								seg->size  += offset;
							}


						}


					}

					last_seg = seg;
				}

				// if zero length freespace seg(s) were detected we'll need to
				// prune them from the list.
				if (cleanup_freespace==TRUE) {
					prune_zero_length_freespace_segs( ld->parent_objects );
				}

				rc = 0;
			}


		}


	}

	LOG_EXIT_VOID();
}


/*
 *  Called to inspect the dimensions of an extended partition and adjust the size
 *  to match the EBR chain. It doesn't matter if the disk flags dont show an
 *  extended partition because we will look at the segment list and determine if
 *  an extended partition exists, updating the disk flags before exiting.
 */
void  fixup_disk_extd_partition_dimensions( LOGICALDISK *ld ){
	DISK_PRIVATE_DATA *disk_pdata = get_disk_private_data(ld);
	DISKSEG           *seg;
	SEG_PRIVATE_DATA  *pdata;
	DISKSEG           *last_logical = NULL;
	DISKSEG           *first_ebr = NULL;
	lba_t              extd_start;
	sector_count_t     extd_size;
	int                rc;
	list_element_t     iter;


	LOG_ENTRY();

	/* walk the list looking for an extended partition */
	LIST_FOR_EACH( ld->parent_objects, iter, seg ) {

		pdata = (SEG_PRIVATE_DATA *) seg->private_data;

		if ( (pdata->flags & SEG_IS_EBR) && (first_ebr == NULL)) {

			first_ebr = seg;
		}
		else if ( pdata->flags & SEG_IS_LOGICAL_PARTITION ) {

			last_logical = seg;
		}
	}


	/* test if we found the start and end of the extended partition */
	if (first_ebr  && last_logical) {

		extd_start = first_ebr->start;

		// special check ... if the extd partition is being anchored
		// because the logical drives are out of physical order
		// then we might need to preserve the start of the ext partition.
		// this only occurs when we have freed a single sector at the
		// start of the extd partition, due to destroying a logical
		// parition, and we are moving the start of the extd partition
		// by 1 sector.  However, the anchor sector came from an EBR track
		// and we should not do this. So ... check for it.
		if (extd_start == disk_pdata->extd_partition_lba + 1) {
			extd_start = disk_pdata->extd_partition_lba;
		}

		extd_size  = (last_logical->start + last_logical->size) - extd_start;

		disk_pdata->flags                 |= DISK_HAS_EXTENDED_PARTITION;
		disk_pdata->extd_partition_lba     = extd_start;
		disk_pdata->extd_partition_size    = extd_size;
		disk_pdata->extd_partition_end_lba = extd_start + extd_size - 1;

		rc = 0;


	}
	else {

		disk_pdata->flags                 &= ~DISK_HAS_EXTENDED_PARTITION;
		disk_pdata->extd_partition_lba     = 0;
		disk_pdata->extd_partition_end_lba = 0;
		disk_pdata->extd_partition_size    = 0;


	}


	LOG_DEBUG("extended start lba: %"PRIu64"\n", disk_pdata->extd_partition_lba );
	LOG_DEBUG("extended   end lba: %"PRIu64"\n", disk_pdata->extd_partition_end_lba );
	LOG_DEBUG("extended      size: %"PRIu64"\n", disk_pdata->extd_partition_size );


	LOG_EXIT_VOID();
}

/*
 *  Called to create a Logical Drive. A logical drive consists of
 *  an EBR track and the logical segment.  Well create the EBR segment
 *  for the logical drive first and then the data segment, inserting
 *  both EBR and logical partition into the DISKSEG LIST which
 *  hangs off the LOGICALDISK.
 *
 *  (1) actually create an EBR DISKSEG
 *  (2) hang the EBR on the logical disk
 *  (3) adjust the starting lba and size of the DISKSEG
 *      to reflect that we used a track of disk space to
 *      hold the EBR.
 */
int create_primary_partition( LOGICALDISK *ld, DISKSEG *seg, DLA_Entry *dla ){
	SEG_PRIVATE_DATA  *pdata      = (SEG_PRIVATE_DATA *)seg->private_data;
	DISK_PRIVATE_DATA *disk_pdata = get_disk_private_data(ld);
	DISKSEG           *mbr        = get_mbr_from_seglist(ld->parent_objects);
	int                rc         = 0;
	void              *handle     = NULL;
	DLA_Entry         *newdla     = NULL;
	int                i;

	LOG_ENTRY();

	// check create primary partition parms
	if ( ld==NULL || seg==NULL || pdata==NULL || mbr == NULL ) {
		LOG_ERROR("error, invalid parms\n");
		rc = EINVAL;
		LOG_EXIT_INT(rc);
		return rc;
	}

	//  A primary partition points back to the mbr
	pdata->ebr = mbr;

	//  Look for an unused entry in the MBR partition table.
	i = get_first_unused_ptable_entry(ld->parent_objects, mbr);
	if ( i == -1 ) {
		LOG_ERROR("error, there are no unused entries in MBR partition table\n");
		rc = EINVAL;
		LOG_EXIT_INT(rc);
		return rc;
	}

	// found unused entry in partition table so assign
	// this new partition to this ptable entry
	pdata->ptable_index = i;
	pdata->part_number  = i+1;


	// if OS2 partition ... setup dlat entry info
	if ( disk_pdata->flags & DISK_HAS_OS2_DLAT_TABLES ) {

		/*  A primary partition's DLAT ptr will point to the DLAT
		 *  owned by the MBR meta data segment.
		 */
		pdata->dlat = ((SEG_PRIVATE_DATA *)(mbr->private_data))->dlat;

		/*  A primary partition will have a DLA entry in the DLAT
		 *  found in the MBR track.  This is the dlat owned by our
		 *  MBR meta data segment. So, the dla entry for this new
		 *  primary partition we are creating will be the first
		 *  available entry in the DLA table owned by the MBR.
		 */
		pdata->dla_entry = NULL;

		for (i=0; i<4; i++) {

			newdla = &pdata->dlat->DLA_Array[i];

			if ( (newdla->Partition_Size  == 0 ) &&
			     (newdla->Partition_Start == 0 )) {

				/* found an unused dla entry to use, so copy our dla entry
				 * to this location and set the dla entry ptr in this data
				 * segment.
				 */
				pdata->dla_entry = newdla;
				memcpy( pdata->dla_entry, dla, sizeof(DLA_Entry));
				break;
			}
		}

		if (pdata->dla_entry == NULL) {
			LOG_ERROR("error, need a DLA entry but there are none available\n");
			rc = ENOMEM;
			LOG_EXIT_INT(rc);
			return rc;
		}


	}

	/*
	 *  Now we can insert this data segment in the segment list that
	 *  hangs off the LOGICALDISK struct.
	 */
	handle = insert_diskseg_into_list( ld->parent_objects, seg );
	if (handle==NULL) {
		LOG_ERROR("error, some kind of list insert error");
		rc = ENOMEM;
	}


	LOG_EXIT_INT(rc);
	return rc;
}


/*
 *  Called to create a Logical Drive. A logical drive consists of
 *  an EBR track and the logical segment.  Well create the EBR segment
 *  for the logical drive first and then the data segment, inserting
 *  both EBR and logical partition into the DISKSEG LIST which
 *  hangs off the LOGICALDISK.
 *
 *  (1) actually create an EBR DISKSEG
 *  (2) hang the EBR on the logical disk
 *  (3) adjust the starting lba and size of the DISKSEG
 *      to reflect that we used a track of disk space to
 *      hold the EBR.
 *  (4) if we are offsetting within freespace for the 1st
 *      logical drive then we need to split apart the EBR sector
 *      from its data partition.
 */
int create_logical_partition( LOGICALDISK *ld, DISKSEG *seg, DLA_Entry *dla, DISKSEG *freespace, sector_count_t offset ){
	DISKSEG           *ebr;
	SEG_PRIVATE_DATA  *pdata=NULL;
	SEG_PRIVATE_DATA  *ebr_pdata=NULL;
	int                rc=0;
	void              *handle;
	DISK_PRIVATE_DATA *disk_pdata = get_disk_private_data( ld );
	DLA_Table_Sector  *dlat_buffer = NULL;
	boolean            split_logical_drive = FALSE;
	lba_t              seg_end = seg->start + seg->size - 1;

	LOG_ENTRY();

	LOG_DEBUG("seg: start= %"PRIu64"  size= %"PRIu64"\n", seg->start, seg->size );
	LOG_DEBUG("free: start= %"PRIu64" size= %"PRIu64"\n", freespace->start, freespace->size );

	/* first malloc a new EBR storage object */
	ebr = allocate_disk_segment(ld);
	if (ebr==NULL) {
		rc = ENOMEM;
		LOG_ERROR("alloc of new segment storage object failed\n");
		LOG_EXIT_INT(rc);
		return rc;
	}
	else {
		ebr_pdata = (SEG_PRIVATE_DATA *) ebr->private_data;
	}

	// if OS2 partition ... malloc a DLAT buffer and hang it off
	// the new EBR diskseg.
	if ( disk_pdata->flags & DISK_HAS_OS2_DLAT_TABLES ) {

		dlat_buffer = Allocate_Dlat( ld );
		if (dlat_buffer==NULL) {
			free_disk_segment(ebr);
			rc = ENOMEM;
			LOG_ERROR("alloc of dlat buffer failed\n");
			LOG_EXIT_INT(rc);
			return rc;
		}


	}

	// figure out where to put the ebr and how big the metadata
	// segment should be.  If we are offsetting the logical partition
	// and no extd partition exists yet ... then anchor the extd
	// by placing the ebr at sector 1 in the extd partition
	if ( ( disk_pdata->logical_drive_count == 0 ) &&
	     ( (disk_pdata->flags & DISK_HAS_OS2_DLAT_TABLES) == FALSE) &&
	     ( seg->start > freespace->start ) ) {

		if (freespace->size > (disk_pdata->geometry.sectors_per_track*disk_pdata->vsectors_per_block)) {

			ebr->start  = freespace->start;
			ebr->size   = 1; //disk_pdata->geometry.sectors_per_track;
			freespace->start += ebr->size;
			freespace->size  -= ebr->size;
			split_logical_drive = TRUE;


		}
		else {
			free_disk_segment(ebr);
			rc = ENOMEM;
			MESSAGE(_("Error:  If you absolutely must use an offset when creating a logical drive, it cannot be less than 1 track in size.\n"
				  "Try an offset of at least %d sectors.\n"),
				disk_pdata->geometry.sectors_per_track );
			LOG_EXIT_INT(rc);
			return rc;
		}


	}
	else {	// ensure the logical partition starts on a track bdy by
		// making sure the ebr metadata segment ends on track bdy

		lba_t ebr_end = roundup_to_track_boundary( ld, seg->start );

		if (ebr_end == seg->start) {  // already on track bdy then just add track
			ebr_end = roundup_to_track_boundary(ld, seg->start + 1);
		}

		ebr->start  = seg->start;
		ebr->size   = ebr_end - seg->start + 1;


	}

	// finish building the EBR segment
	ebr->data_type          = META_DATA_TYPE;
	ebr->flags             |= SOFLAG_DIRTY;

	ebr_pdata->dlat         = dlat_buffer;
	ebr_pdata->sys_id       = DOS_EXTENDED_PARTITION ;
	ebr_pdata->flags        = SEG_IS_EBR;
	ebr_pdata->ebr_number   = MAX_EBR_NUMBER-1;

	sprintf(ebr->name, "%s_ebr%d", ld->name, ebr_pdata->ebr_number);

	// insert the EBR segment into the ordered list so we can
	// tell where it is in the EBR chain and name it correctly.
	handle = insert_diskseg_into_ordered_list( ld->parent_objects, ebr );
	if (handle==NULL) {
		rc = ENOMEM;
		if (                     dlat_buffer)			     free(                       dlat_buffer);
		if ( split_logical_drive == TRUE ) {
			freespace->size  += ebr->size;
			freespace->start -= ebr->size;
		}
		free_disk_segment(ebr);
		LOG_EXIT_INT(rc);
		return rc;
	}

	// Fixup the EBR chain before adding the logical partition
	fixup_EBR_Chain( ld );

	/*
	 * the EBR meta data segment exists as an extended partition record in
	 * some partition table. Since we just added an EBR segment to the segment
	 * LIST we must find out which partition table the extended partition
	 * record will reside within and also which partition table entry will be
	 * used.  The partition table is identified by the bkwd pointer in our
	 * segments private data area.  And the exact index will be returned by
	 * calling first_unused_ptable_entry() with this info.
	 */
	if (ebr_pdata->ptable_index == 0) {

		ebr_pdata->ptable_index = get_first_unused_ptable_entry( ld->parent_objects,
									 ebr_pdata->prev_ebr );

		if (ebr_pdata->ptable_index == -1) {
			if ( remove_diskseg_from_list(ld->parent_objects, ebr ) == 0) {
				if (dlat_buffer) free(dlat_buffer);
				if (split_logical_drive == TRUE) {
					freespace->size  += ebr->size;
					freespace->start -= ebr->size;
				}
				free_disk_segment(ebr);
				--disk_pdata->logical_drive_count;
				fixup_EBR_Chain( ld );
			}
			rc = ENOMEM;
		}
	}

	if (rc == 0) {

		/* subtract EBR track from size of logical partition and insert the
		 * segment into the LIST.
		 */
		if (split_logical_drive == FALSE) {
			seg->start  = ebr->start + ebr->size;
			seg->size   = seg_end - seg->start + 1;
		}

		// seg->flags |= SOFLAG_EXTENDED;

		/* setup seg private data */
		pdata = (SEG_PRIVATE_DATA *) seg->private_data;
		pdata->ptable_index = 0;      // this logical partition will be the 1st partition record in table
		pdata->ebr          = ebr;
		pdata->dlat         = dlat_buffer;

		/* update dla entry info and copy dla to dlat */
		if ( disk_pdata->flags & DISK_HAS_OS2_DLAT_TABLES ) {

			pdata->dla_entry     = &pdata->dlat->DLA_Array[0];

			dla->Partition_Start = seg->start;
			dla->Partition_Size  = seg->size;

			memcpy(pdata->dla_entry, dla, sizeof(DLA_Entry));


		}

		handle = insert_diskseg_into_ordered_list( ld->parent_objects, seg );
		if ( handle ) {

			// because partitions shift when we add logical partitions
			// we need to fixup their names.
			fixup_logical_partition_names(ld);

			// update count
			++disk_pdata->logical_drive_count;

			// set the size of this logical drive in the EBR private data
			// area. it is simply the size of the EBR (1 track) plus the
			// size of the logical partition itself.
			ebr_pdata->ebr_sector_count = ebr->size + seg->size;

			/*
			 *  Lastly, fixup the size of the extended partition if we changed its size
			 *  when adding this logical drive.
			 */

			fixup_disk_extd_partition_dimensions(ld);
		}
		else {

			// recover the space used by the EBR segment
			if (split_logical_drive == TRUE) {
				freespace->size  += ebr->size;
				freespace->start += ebr->size;
			}
			else {
				seg->start -= ebr->size;
				seg->size  += ebr->size;
			}

			remove_diskseg_from_list( ld->parent_objects, ebr );
			if (dlat_buffer) free(dlat_buffer);
			free_disk_segment(ebr);
			fixup_EBR_Chain( ld );
			rc = ENOMEM;
		}


	}


	if (rc == 0) {

		LOG_DEBUG("New EBR ...\n");
		LOG_DEBUG("    Start LBA: %"PRIu64"\n", ebr->start);
		LOG_DEBUG("         Size: %"PRIu64"\n", ebr->size);
		LOG_DEBUG("         Name: %s\n", ebr->name );
		LOG_DEBUG("New Logical Partition ...\n");
		LOG_DEBUG("    Start LBA: %"PRIu64"\n", seg->start);
		LOG_DEBUG("         Size: %"PRIu64"\n", seg->size);
		LOG_DEBUG("         Name: %s\n", seg->name);
		LOG_DEBUG("FreeSpace ...\n");
		LOG_DEBUG("    Start LBA: %"PRIu64"\n", freespace->start);
		LOG_DEBUG("         Size: %"PRIu64"\n", freespace->size);
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 *  Called to find out if the specified DISKSEG will fall adjacant to an extended partition
 *  on the drive.  This question is asked by SEG create code to help it determine if the
 *  new SEG should be a logical drive or not.
 */
boolean seg_is_within_or_adjacant_to_extended_partition( LOGICALDISK *ld, DISKSEG *seg ){
	DISK_PRIVATE_DATA *disk_pdata = get_disk_private_data(ld);
	DISKSEG           *this_seg;
	SEG_PRIVATE_DATA  *pdata;
	boolean            rc = FALSE;
	lba_t              extd_start;
	lba_t              extd_end;
	list_element_t     iter;


	LOG_ENTRY();

	if (disk_pdata == NULL) {
		LOG_DEBUG("disk has no private data\n");
		LOG_EXIT_BOOL(FALSE);
		return FALSE;
	}


	if ( disk_has_extended_partition( ld ) == TRUE ) {

		/* need starting and ending LBA of the extended partition */
		extd_start = disk_pdata->extd_partition_lba;
		extd_end   = disk_pdata->extd_partition_end_lba;

		if ( (  seg->start >= extd_start ) &&
		     ( (seg->start+seg->size) <= extd_end )) {

			/*  seg falls entirely within extended partition */
			rc = TRUE;
		}
		else if (  (  seg->start >= extd_start ) &&
			   (  seg->start <  extd_end ) &&
			   ( (seg->start+seg->size) > extd_end ) ) {

			/*  seg starts within the extended partition but ends beyond the
			 *  end of the extended partition. So the extended partition can
			 *  just be expanded.
			 */

			rc = TRUE;
		}
		else if ( seg->start < extd_start ) {

			/*
			 * seg starts before extended partition so see if there are interveneing
			 * segments between this guy and the extended partition.
			 */
			this_seg = EngFncs->first_thing( ld->parent_objects, &iter );
			if (this_seg) {

				rc = TRUE;

				while ( this_seg &&
					this_seg->start < extd_start ) {

					if ( this_seg->start > seg->start ) {

						pdata = (SEG_PRIVATE_DATA *) this_seg->private_data;

						/* ebr segs and freespace segs are Ok ... all else fail */
						if ( ( pdata->flags & SEG_IS_PRIMARY_PARTITION ) ||
						     ( pdata->flags & SEG_IS_LOGICAL_PARTITION )) {
							rc = FALSE;
							break;
						}
					}
					this_seg = EngFncs->next_thing( &iter );
				}
			}
		}
		else {

			/*
			* seg starts beyond the end of the extended partition so look for
			* interveneing segments between the extended partition and this guy.
			*/
			this_seg = EngFncs->first_thing( ld->parent_objects, &iter );
			if (this_seg) {

				rc = TRUE;

				while ( this_seg &&
					this_seg->start < seg->start ) {

					if ( this_seg->start > extd_end  ) {

						pdata = (SEG_PRIVATE_DATA *) this_seg->private_data;

						/* ebr segs and freespace segs are Ok ... all else fail */
						if ( ( pdata->flags & SEG_IS_PRIMARY_PARTITION ) ||
						     ( pdata->flags & SEG_IS_LOGICAL_PARTITION )) {
							rc = FALSE;
							break;
						}
					}

					this_seg = EngFncs->next_thing( &iter );
				}

			}

		}
	}


	LOG_EXIT_INT(rc);
	return rc;
}


/*
 *  Called because the disk doesn't have an MBR yet.  So, we create one, hang
 *  it off the disk, and fill in a DLAT sector.
 */
int  create_mbr_For_Disk( LOGICALDISK *ld, char *DiskName, boolean isa_os2_disk ){
	DISKSEG           *mbr;
	DISKSEG           *freeseg;
	int                rc=EINVAL;
	DLA_Table_Sector  *dlat_buffer = NULL;
	DISK_PRIVATE_DATA *disk_pdata = get_disk_private_data(ld);

	LOG_ENTRY();
	LOG_DEBUG("disk= %s disk_name= %s os2_flag= %d\n", ld->name, DiskName, isa_os2_disk );

	// get the existing freespace segment
	freeseg = EngFncs->first_thing( ld->parent_objects, NULL);
	if ( freeseg == NULL) {
		rc = EPERM;
		LOG_ERROR("cant create MBR, no freespace segments found on disk %s\n", ld->name);
		LOG_EXIT_INT(rc);
		return rc;
	}

	/* Ok, we got the first diskseg on the logical drive.  It must be freespace
	 * because this is where we need to lay down the MBR track. It must also
	 * start at LBA 0 else some non FREESPACE seg exists where the MBR track
	 * must reside.
	 */
	if ( ( freeseg->data_type != FREE_SPACE_TYPE ) ||
	     ( freeseg->start != 0 ) ) {
		rc = EPERM;
		LOG_ERROR("cant create MBR, first segment on disk (%s) not FREESPACE seg or not at LBA 0\n", ld->name);
		LOG_EXIT_INT(rc);
		return rc;
	}

	/* hang an MBR meta data segment off the logical drive */
	disk_pdata->flags = 0;
	mbr = build_mbr_disk_segment( ld );
	if (mbr == NULL) {
		rc = EPERM;
		free(dlat_buffer);
		LOG_ERROR("cant create MBR, build MBR storage object failed\n");
		LOG_EXIT_INT(rc);
		return rc;
	}


	if ( isa_os2_disk ) { // register Disk Serial Number and Disk Name

		dlat_buffer = Allocate_Dlat(ld);
		if (dlat_buffer == NULL) {
			free_disk_segment(mbr);
			rc = ENOMEM;
			LOG_ERROR("cant create dlat for MBR track\n");
			LOG_EXIT_INT(rc);
			return rc;
		}

		// copy os2 disk name to dlat and disk pdata
		strcpy(dlat_buffer->Disk_Name, DiskName);
		strcpy(disk_pdata->disk_name, DiskName );

		// gen a serial number for the disk
		dlat_buffer->Disk_Serial_Number = seg_gen_serial_number( (u_int32_t) &dlat_buffer->Disk_Serial_Number );
		if (dlat_buffer->Disk_Serial_Number != BAD_SERIAL_NUMBER) {
			rc = seg_register_serial_number( dlat_buffer->Disk_Serial_Number);
		}
		else {
			rc = ENOTUNIQ;
		}

		if (rc != 0) {
			free(dlat_buffer);
			free_disk_segment(mbr);
			LOG_ERROR("cant create MBR, unable to generate disk serial number for DLAT\n");
			LOG_EXIT_INT(rc);
			return rc;
		}


		((SEG_PRIVATE_DATA *)mbr->private_data)->dlat = dlat_buffer;

		if ( EngFncs->register_name( dlat_buffer->Disk_Name) != 0 ) {
			rc = ENOTUNIQ;
			seg_unregister_serial_number( dlat_buffer->Disk_Serial_Number );
			free(dlat_buffer);
			free_disk_segment(mbr);
			LOG_ERROR("cant create MBR, unable to register OS2 disk name\n");
			LOG_EXIT_INT(rc);
			return rc;
		}

		disk_pdata->flags |= DISK_HAS_OS2_DLAT_TABLES;
	}


	/*
	 *  Things look good so before trying to insert the new data segment
	 *  we will need to adjust the freespace segment to leave a 1 track gap at
	 *  the start of the drive for the MBR meta data segment we are about to hang.
	 */
	freeseg->start  += mbr->size;
	freeseg->size   -= mbr->size;

	if ( insert_diskseg_into_list(ld->parent_objects, mbr) == NULL ) {
		rc = EPERM;
		freeseg->start  -= mbr->size;
		freeseg->size   += mbr->size;
		free_disk_segment(mbr);
		free(dlat_buffer);
		LOG_ERROR("cant create MBR, call to insert MBR storage object into disk list failed\n");
		LOG_EXIT_INT(rc);
		return rc;
	}


	// Success ...
	rc = 0;
	mbr->flags |= SOFLAG_DIRTY;

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 *  Returns TRUE if the specified disk segment falls within the extended partition
 *  extent on the drive.
 */
boolean  seg_is_within_the_extended_partition( LOGICALDISK *ld, DISKSEG *seg )
{
	DISK_PRIVATE_DATA *disk_pdata = get_disk_private_data(ld);

	LOG_ENTRY();

	if (disk_pdata) {

		if ( ( seg->start >= disk_pdata->extd_partition_lba )&&
		     ( seg->start <  disk_pdata->extd_partition_end_lba )) {

			LOG_EXIT_BOOL(TRUE);
			return TRUE;

		}

	}

	LOG_EXIT_BOOL(FALSE);
	return FALSE;
}


/*
 * Called to see if a segment can be resized
 */
boolean seg_is_volitile( DISKSEG *seg )
{
	SEG_PRIVATE_DATA  *pdata = (SEG_PRIVATE_DATA *)seg->private_data;
	LOGICALDISK       *ld = get_logical_disk(seg);
	DISK_PRIVATE_DATA *disk_pdata;

	LOG_ENTRY();

	// cant be any kind of embedded segment
	if ( ( pdata->flags & SEG_IS_EMBEDDED ) ||
	     ( pdata->flags & SEG_IS_EMBEDDED_METADATA ) ||
	     ( pdata->flags & SEG_IS_CONSUMED ) ) {

		LOG_EXIT_BOOL(FALSE);
		return FALSE;

	}

	// cant reside on disk with screwball geometry
	if (ld) {

		disk_pdata = get_disk_private_data(ld);

		if (disk_pdata) {

			if ( disk_pdata->flags & DISK_HAS_FORCED_LBA_ADDRESSING ) {
				LOG_EXIT_BOOL(FALSE);
				return FALSE;
			}
			else {
				LOG_EXIT_BOOL(TRUE);
				return TRUE;
			}


		}


	}

	LOG_EXIT_BOOL(FALSE);
	return FALSE;
}


/*
 * Called to make a container from a disk segment. A
 * container segment is one that holds an embedded
 * partitioning scheme, e.g. unixware.
 *
 * Make sure it isnt already in the container_seg list.
 */
int  diskseg_to_container_segment( DISKSEG *seg )
{
	SEG_PRIVATE_DATA  *pdata = (SEG_PRIVATE_DATA *)seg->private_data;
	LOGICALDISK       *ld;
	DISK_PRIVATE_DATA *disk_pdata;
	int                rc=EINVAL;
	list_element_t     e;

	LOG_ENTRY();

	ld = get_logical_disk(seg);
	if (ld) {

		disk_pdata = get_disk_private_data(ld);
		if (disk_pdata) {

			e = EngFncs->insert_thing( disk_pdata->container_segs,
						   seg,
						   INSERT_AFTER|EXCLUSIVE_INSERT,
						   NULL );
			if (e) {
				rc = 0;
				pdata->flags |= SEG_IS_CONSUMED;
			}
			else {
				rc = EPERM;
			}

		}

	}

	LOG_EXIT_INT(rc);
	return rc;
}

/*
 *  Called to release a segment from the container list.
 */
void revert_container_segment( DISKSEG *seg )
{
	SEG_PRIVATE_DATA  *pdata = (SEG_PRIVATE_DATA *)seg->private_data;
	LOGICALDISK       *ld;
	DISK_PRIVATE_DATA *disk_pdata;

	LOG_ENTRY();

	ld = get_logical_disk(seg);
	if (ld) {

		disk_pdata = get_disk_private_data(ld);
		if (disk_pdata) {

			if ( disk_pdata->container_segs ) {

				EngFncs->remove_thing( disk_pdata->container_segs, seg );

				insert_diskseg_into_list( ld->parent_objects, seg);

				pdata->flags &= ~SEG_IS_CONSUMED;

			}

		}

	}

	LOG_EXIT_VOID();
}


/*
 *  Called to see if the specified disk segment resides
 *  entirely within a container segment.
 *
 *  Returns: TRUE if it is contained by a container seg on the disk
 *           FALSE otherwise
 *
 */
boolean seg_is_within_container_segment( DISKSEG *seg ){
	DISKSEG           *container_seg;
	LOGICALDISK       *ld;
	DISK_PRIVATE_DATA *disk_pdata;
	list_element_t     iter;

	LOG_ENTRY();

	ld = get_logical_disk(seg);
	if (ld) {

		disk_pdata = get_disk_private_data(ld);
		if (disk_pdata) {

			if ( disk_pdata->container_segs ) {

				LIST_FOR_EACH( disk_pdata->container_segs, iter, container_seg ) {
					if ( ( seg->start >= container_seg->start ) &&
					     ( (seg->start+seg->size-1) <= (container_seg->start+container_seg->size-1))) {
						LOG_EXIT_BOOL(TRUE);
						return TRUE;
					}
				}

			}

		}

	}

	LOG_EXIT_BOOL(FALSE);
	return FALSE;
}


/*
 *  Called to see if the specified disk segment overlaps
 *  a container segment. It may start before a container and
 *  finish within it ... or else ... it may start within a
 *  container and finish afterwards.
 *
 *  Returns: TRUE if it does overlap some container seg on the disk
 *           FALSE otherwise
 *
 */
boolean seg_overlaps_container_segment( DISKSEG *seg ){
	DISKSEG           *container_seg;
	LOGICALDISK       *ld;
	DISK_PRIVATE_DATA *disk_pdata;
	boolean            overlapping=FALSE;
	list_element_t     iter;

	LOG_ENTRY();

	ld = get_logical_disk(seg);
	if (ld) {

		disk_pdata = get_disk_private_data(ld);
		if (disk_pdata) {

			if ( disk_pdata->container_segs ) {

				LIST_FOR_EACH( disk_pdata->container_segs, iter, container_seg ) {

					// test and set ... overlapping segments flag
					if ( (  seg->start >= container_seg->start )&&
					     (  seg->start <= container_seg->start+container_seg->size-1) ) {
						overlapping = TRUE;
					}
					else if ( ( seg->start  <  container_seg->start ) &&
						  ( container_seg->start <= (seg->start + seg->size - 1)) ) {
						overlapping = TRUE;
					}
					else {
						overlapping = FALSE;
					}

					if ( overlapping == TRUE) {
						LOG_EXIT_BOOL(TRUE);
						return TRUE;
					}

				}

			}

		}

	}

	LOG_EXIT_BOOL(FALSE);
	return FALSE;
}




/*
 *  Called to remove any part of a segment object that overlaps
 *  a container segment. This will happen by either shortening
 *  the end of a segment or by removing sectors from its start.
 *
 *  Returns: 0 if segment was successfully adjusted
 *           -1 otherwise
 *
 */
int  remove_container_seg_overlap( DISKSEG *seg){
	DISKSEG           *container_seg;
	LOGICALDISK       *ld;
	DISK_PRIVATE_DATA *disk_pdata;
	sector_count_t     delta;
	list_element_t     iter;

	LOG_ENTRY();

	ld = get_logical_disk(seg);
	if (ld) {

		disk_pdata = get_disk_private_data(ld);
		if (disk_pdata) {

			if ( disk_pdata->container_segs ) {

				LIST_FOR_EACH( disk_pdata->container_segs, iter, container_seg ) {

					if ( (  seg->start >= container_seg->start )&&
					     (  seg->start <= container_seg->start+container_seg->size-1) ) {

						delta = (container_seg->start + container_seg->size) - seg->start;

						if (delta < seg->size) {
							seg->start  += delta;
							seg->size   -= delta;
							LOG_EXIT_INT(0);
							return 0;
						}
						else {
							LOG_EXIT_INT(-1);
							return -1;
						}


					}
					else if ( ( seg->start  <  container_seg->start ) &&
						  ( container_seg->start <= (seg->start + seg->size - 1)) ) {

						delta = (seg->start + seg->size) - container_seg->start;

						if (delta < seg->size) {
							seg->size   -= delta;
							LOG_EXIT_INT(0);
							return 0;
						}
						else {
							LOG_EXIT_INT(-1);
							return -1;
						}


					}

				}

			}

		}

	}

	LOG_EXIT_INT(-1);
	return -1;
}



/*
 *  Returns a storage object from the child object list
 *  of the specified storage object ... as long as there
 *  is one and only one object in the child object list.
 */
DISKSEG *  only_child( DISKSEG *seg )
{
	storage_object_t *only_child = NULL;

	LOG_ENTRY();

	if (EngFncs->list_count(seg->child_objects)==1) {
		only_child = EngFncs->first_thing(seg->child_objects,NULL);
	}

	LOG_EXIT_PTR(only_child);
	return  only_child;
}

