/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libgpt.so
 *
 *   File: commit.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>

#include "gpt.h"
#include "checks.h"
#include "helpers.h"
#include "commit.h"


/*
 *  Called to convert a GPT header from disk format to the arch specific
 *  format.
 */
static void cpu_gpt_header_to_disk( gpt_header *gh )
{

        gh->signature              = CPU_TO_DISK64(gh->signature);
        gh->version                = CPU_TO_DISK32(gh->version);
        gh->size                   = CPU_TO_DISK32(gh->size);
        gh->crc                    = CPU_TO_DISK32(gh->crc);
        gh->reserve                = CPU_TO_DISK32(gh->reserve);
        gh->my_lba                 = CPU_TO_DISK64(gh->my_lba);
        gh->alternate_lba          = CPU_TO_DISK64(gh->alternate_lba);
        gh->start_useable          = CPU_TO_DISK64(gh->start_useable);
        gh->end_useable            = CPU_TO_DISK64(gh->end_useable);

        gh->disk_id.time_low       = CPU_TO_DISK32(gh->disk_id.time_low);
        gh->disk_id.time_mid       = CPU_TO_DISK16(gh->disk_id.time_mid);
        gh->disk_id.time_high      = CPU_TO_DISK16(gh->disk_id.time_high);

        gh->ptable_lba             = CPU_TO_DISK64(gh->ptable_lba);
        gh->ptable_count           = CPU_TO_DISK32(gh->ptable_count);
        gh->ptable_entry_size      = CPU_TO_DISK32(gh->ptable_entry_size);
        gh->ptable_crc             = CPU_TO_DISK32(gh->ptable_crc);

}


/* UNCOMMENT FOR DEBUGGING
static void Display_GPT_Partition_Record( gpt_partition *part, int i)
{
    if (part) {

        LOG_DEBUG("GPT PARTITION INFO ...Minor= %d\n", i);
        LOG_DEBUG("     Type ... \n");
        LOG_DEBUG("        Time  Low   = 0x%X\n", DISK_TO_CPU32(part->type.time_low) );
        LOG_DEBUG("        Time  Mid   = 0x%X\n", DISK_TO_CPU16(part->type.time_mid) );
        LOG_DEBUG("        Time High   = 0x%X\n", DISK_TO_CPU16(part->type.time_high) );
        LOG_DEBUG("        Clk  High   = 0x%X\n", part->type.clock_seq_high );
        LOG_DEBUG("        Clk   Low   = 0x%X\n", part->type.clock_seq_low );
        LOG_DEBUG("        Node        = %X %X %X %X %X %X\n",
                        part->type.node[0], part->type.node[1], part->type.node[2],
                        part->type.node[3], part->type.node[4], part->type.node[5] );
        LOG_DEBUG("     ID ... \n");
        LOG_DEBUG("        Time  Low   = 0x%X\n", DISK_TO_CPU32(part->id.time_low) );
        LOG_DEBUG("        Time  Mid   = 0x%X\n", DISK_TO_CPU16(part->id.time_mid) );
        LOG_DEBUG("        Time High   = 0x%X\n", DISK_TO_CPU16(part->id.time_high) );
        LOG_DEBUG("        Clk  High   = 0x%X\n", part->id.clock_seq_high );
        LOG_DEBUG("        Clk   Low   = 0x%X\n", part->id.clock_seq_low );
        LOG_DEBUG("        Node        = %X %X %X %X %X %X\n",
                        part->id.node[0], part->id.node[1], part->id.node[2],
                        part->id.node[3], part->id.node[4], part->id.node[5] );
        LOG_DEBUG("     Start LBA         = %"PRIu64"\n", DISK_TO_CPU64(part->start));
        LOG_DEBUG("     End   LBA         = %"PRIu64"\n", DISK_TO_CPU64(part->end));
    }
}

*/

/*
 *  Called to walk a segment storage object list and return
 *  the metadata segment that corresponds to the commit phase.
 *
 *  Returns: metadata segment ptr if successful
 *           null ptr if not successful
 */
static int  build_gpt_partition_table_from_seglist( list_anchor_t   seglist,
                                                    gpt_header     *gh,
                                                    gpt_partition  *ptable )
{
        int                   rc = ENOSYS;
        DISKSEG              *seg=NULL;
        SEG_PRIVATE_DATA *pdata=NULL;
        gpt_partition        *part=(gpt_partition *)malloc(gh->ptable_entry_size);
        char                 *cptr=(char *)ptable;
        list_element_t        iter;

        LOG_ENTRY();

        if (seglist!=NULL && gh!=NULL && ptable !=NULL) {
                rc = 0;
        }

        LIST_FOR_EACH( seglist, iter, seg ) {

                if ( seg->data_type == DATA_TYPE ) {

                        memset(part,0,gh->ptable_entry_size);

                        pdata = (SEG_PRIVATE_DATA *)seg->private_data;

                        memcpy( &part->type, &pdata->guid_type, sizeof(guid_t) );
                        cpu_guid_to_disk( &part->type );

                        memcpy( &part->id, &pdata->guid_id, sizeof(guid_t) );
                        cpu_guid_to_disk( &part->id );

                        memcpy( &part->name, &pdata->name, sizeof(utf16_char_t)*GPT_PNAME_SIZE );

                        part->attributes = CPU_TO_DISK64(pdata->attributes);

                        part->start = CPU_TO_DISK64(seg->start);
                        part->end   = CPU_TO_DISK64(seg->start + seg->size - 1);

                        cptr = ((char *)(ptable)) + (gh->ptable_entry_size * (pdata->minor-1));
                        memcpy(cptr, part, gh->ptable_entry_size);

                }

        }

        /* UNCOMMENT FOR DEBUGGING
        {
            int i;
            for (i=0, part=ptable; i<gh->ptable_count; i++) {
                if (part->end != 0) {
                    Display_GPT_Partition_Record( part, i+1);
                }
                ++part;
            }
        }
        */

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Called to walk a segment storage object list and return
 *  the metadata segment that corresponds to the commit phase.
 *
 *  Returns: metadata segment ptr if successful
 *           null ptr if not successful
 */
static DISKSEG * get_metadata_seg_from_seglist( list_anchor_t seglist, uint phase )
{
        DISKSEG *metadata=NULL;
        DISKSEG *seg;
        char    *mstring;
        list_element_t iter;

        LIST_FOR_EACH( seglist, iter, seg ) {

                if ( seg->data_type == META_DATA_TYPE ) {
                        mstring = strstr( seg->name, "metadata" );  // object called xxx_metadataN ?
                        if (mstring) {
                                if ( atoi(&mstring[8]) == phase) {  // N == commit phase ?
                                        metadata = seg;
                                        break;
                                }
                        }
                }

        }

        return metadata;
}



/*
 *  Called to commit a copy of the gpt header along with
 *  its corresponding partition table.
 *
 *  Returns:  0 if successful
 *            error codes otherwise
 */
static int  write_metadata(LOGICALDISK *ld, DISK_PRIVATE_DATA *disk_pdata,
                           storage_object_t *parent, uint phase, boolean backup)
{
        int rc = EINVAL;
        gpt_header     *gh=NULL;
        gpt_partition  *ptable=NULL;
        struct plugin_functions_s *DevFuncs;
        u_int32_t       crc;
        DISKSEG        *md;
        u_int32_t       gh_size = 0;

        LOG_ENTRY();

        //
        // get metadata segment corresponding to the commit phase
        // and use it to get gpt header
        //
        md = get_metadata_seg_from_seglist( ld->parent_objects, phase);
        if (md) {

                gh = (void *) calloc(1, EVMS_VSECTOR_SIZE );
                if (gh) {
                        memcpy( gh,
                                ((SEG_PRIVATE_DATA *)md->private_data)->gh,
                                ((SEG_PRIVATE_DATA *)md->private_data)->gh->size );
                }

        }

        if (gh) {

                // build the gpt partition table
                ptable = (gpt_partition *) calloc( 1, (md->size-1)*EVMS_VSECTOR_SIZE );

                if (ptable) {
                        rc = build_gpt_partition_table_from_seglist( ld->parent_objects, gh, ptable );
                }
                else {
                        rc = ENOMEM;
                }

                DevFuncs = (struct plugin_functions_s *) ld->plugin->functions.plugin;

                // commit the partition table
                if (rc == 0) {
                        if (backup) {
                                rc = EngFncs->save_metadata( parent->name, ld->name,
                                                             gh->ptable_lba, (md->size - 1), ptable );
                        } else {
                                rc = DevFuncs->write( ld, gh->ptable_lba, (md->size - 1), (void *) ptable );
                        }
                }

                // commit the gpt header
                if (rc == 0) {
                        crc = EngFncs->calculate_CRC(  EVMS_INITIAL_CRC,
                                                             ptable,
                                                             gh->ptable_count * gh->ptable_entry_size);
                        gh->ptable_crc = ~crc;
                        gh_size        = gh->size;
                        cpu_gpt_header_to_disk( gh );
                        gh->crc = 0;
                        crc = EngFncs->calculate_CRC(  EVMS_INITIAL_CRC,
                                                             gh,
                                                             gh_size );
                        gh->crc = CPU_TO_DISK32(~crc);

                        if (backup) {
                                rc = EngFncs->save_metadata( parent->name, ld->name,
                                                             DISK_TO_CPU64(gh->my_lba), 1, gh );
                        } else {
                                rc = DevFuncs->write( ld, DISK_TO_CPU64(gh->my_lba), 1, (void *) gh );
                        }
                }

        }

        if (ptable) free(ptable);
        if (gh) free(gh);

        LOG_EXIT_INT(rc);
        return rc;
}



/*
 *   Called to commit the PMBR on the specified disk.
 *
 *   Returns:  0 if successful
 *             error codes otherwise
 */
static int  write_protective_mbr( LOGICALDISK *ld,
                                  storage_object_t *parent,
                                  boolean backup )
{

        master_boot_record         *boot_sector;
        struct plugin_functions_s  *DevFuncs=NULL;
        int                         rc,i;
        msdos_partition            *part;
        chs_t                       chs;


        LOG_ENTRY();

        boot_sector = (master_boot_record *) malloc(EVMS_VSECTOR_SIZE);

        if (boot_sector) {

                DevFuncs = (struct plugin_functions_s *) ld->plugin->functions.plugin;

                rc = DevFuncs->read( ld, 0, 1, (void *) boot_sector );
                if (rc == 0) {

                        boot_sector->signature = CPU_TO_DISK16(MSDOS_DISKMAGIC);  // just to be sure

                        for (i=0; i<4; i++)  memset(&boot_sector->partition_table[i], 0, sizeof(msdos_partition));

                        part = &boot_sector->partition_table[0];

                        SYS_IND(part)     = GPT_PARTITION;
                        BOOT_IND(part)    = 0;

                        START_LBA(part)   = CPU_TO_DISK32( 1 );
                        NR_SECTS(part)    = CPU_TO_DISK32( ld->size - 1 );

                        LBA_to_Ptable_CHS(ld, 1, &chs);

                        START_CYL(part)   = (unsigned char) ( chs.cylinder  & 0x000000ff);
                        START_HEADS(part) = (unsigned char) ( chs.head      & 0x000000ff);
                        START_SECT(part)  = (unsigned char) ( chs.sector    & 0x0000003f ) |
                                            (( chs.cylinder >> 2 ) & 0xC0) ;

                        LBA_to_Ptable_CHS(ld, (ld->size - 1 ) , &chs);

                        END_CYL(part)     = (unsigned char) ( chs.cylinder  & 0x000000ff);
                        END_HEADS(part)   = (unsigned char) ( chs.head      & 0x000000ff);
                        END_SECT(part)    = (unsigned char) ( chs.sector    & 0x0000003f ) |
                                            (( chs.cylinder >> 2 ) & 0xC0) ;

                        LOG_DEBUG("Committing PMBR to disk ...\n");

                        if (backup) {
                                rc = EngFncs->save_metadata( parent->name, ld->name,
                                                             0, 1, boot_sector );
                        } else {
                                rc = DevFuncs->write( ld, 0, 1, (void *) boot_sector );
                        }

                }

        }
        else {
                rc = ENOMEM;
        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Called by SEG_COMMIT() to write out partition tables and dlat tables for
 *  the specified disk.
 */
int  commit_guid_partition_tables( LOGICALDISK *ld, storage_object_t *parent,
                                   uint commit_phase, boolean backup )
{
        int rc=0;
        DISK_PRIVATE_DATA *disk_pdata;

        LOG_ENTRY();

        disk_pdata = get_gpt_disk_private_data(ld);

        if (disk_pdata) {

                switch (commit_phase) {
                
                case 1:
                        rc = write_protective_mbr( ld, parent, backup );
                        if (rc == 0) {
                                rc = write_metadata(ld, disk_pdata, parent,
                                                    commit_phase, backup);
                        }
                        break;

                case 2:
                        rc = write_metadata(ld, disk_pdata, parent,
                                            commit_phase, backup);
                        break;

                default:
                        LOG_DEBUG("nothing to do, commit phase = %d\n", commit_phase );
                        rc = 0;
                        break;

                }

        }

        LOG_EXIT_INT(rc);
        return rc;
}

