/*
 *   (C) Copyright IBM Corp. 2004
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: LVM2 Plugin
 * File: evms2/engine/plugins/lvm2/move.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#include "lvm2.h"

/**
 * region_is_busy
 *
 * Check if the specified region is open/in-use/mounted. If so, ask the user
 * if they'd like to unmount and retry.
 *
 * Return: TRUE (non-zero): skip processing this region.
 *         FALSE (zero): proceed with this region.
 **/
static boolean region_is_busy(storage_object_t *region, int prompt_user)
{
	char *choices[] = {_("Skip"), _("Retry"), NULL};
	logical_volume_t *evms_vol;
	int offline, answer;
	boolean result = FALSE;

	LOG_ENTRY();
	LOG_DEBUG("Checking if region %s is busy.\n", region->name);

	/* If online move is enabled, we can skip
	 * checking if the region is busy.
	 */
	if (EngFncs->can_online_copy()) {
		goto out;
	}

	/* This loop will not exit unless the region is unmounted,
	 * or the user decides to skip this region.
	 */
	while (!(offline = EngFncs->is_offline(region, &evms_vol))) {
		answer = 0;
		if (prompt_user) {
			QUESTION(&answer, choices,
				 _("Region \"%s\" has mappings scheduled to be "
				   "moved. However, this region is part of "
				   "volume \"%s\", which is mounted at %s. "
				   "Please unmount the volume and choose "
				   "\"Retry\" to continue with the move, or "
				   "choose \"Skip\" to skip the move at this "
				   "time (the move will be attempted again the "
				   "next time changes are saved)."),
				 region->name, evms_vol->name,
				 evms_vol->mount_point);
		}

		if (answer == 0) {
			break;
		}
	}

	result = (!offline && answer == 0);

out:
	LOG_EXIT_BOOL(result);
	return result;
}

/**
 * commit_stripe_move_init_copy_job
 *
 * Setup up the copy job for copying the data for the
 * current "map" to the "new_map".
 **/
static int commit_stripe_move_init_copy_job(logical_extent_map_t *le_map,
					    copy_job_t *copy_job)
{
	physical_extent_t *src_pe = le_map->map[0].pe;
	physical_extent_t *dst_pe = le_map->new_map[0].pe;
	container_data_t *c_data = le_map->r_map->r_data->
				   region->producing_container->private_data;
	u_int64_t extent_count = le_map->r_map->le_count /
				 le_map->r_map->stripe_count;
	u_int64_t pe_size = c_data->pe_size;
	char count1[25], count2[25], count3[25];
	int title_size = 90 + EVMS_NAME_SIZE * 3;
	int rc = 0;

	LOG_ENTRY();

	copy_job->src.obj = src_pe->pv_data->object;
	copy_job->src.start = src_pe->pv_data->pe_start +
			      src_pe->number * pe_size;
	copy_job->src.len = extent_count * pe_size;
	copy_job->trg.obj = dst_pe->pv_data->object;
	copy_job->trg.start = dst_pe->pv_data->pe_start +
			      dst_pe->number * pe_size;
	copy_job->trg.len = extent_count * pe_size;
	copy_job->description = NULL;

	copy_job->title = EngFncs->engine_alloc(title_size);
	if (!copy_job->title) {
		rc = ENOMEM;
		goto out;
	}

	snprintf(count1, 25, "%"PRIu64, extent_count);
	snprintf(count2, 25, "%"PRIu64, src_pe->number);
	snprintf(count3, 25, "%"PRIu64, dst_pe->number);

	snprintf(copy_job->title, title_size,
		 _("LVM2: Moving %s extents for region %s "
		   "from PV %s,PE %s to PV %s,PE %s"),
		 count1, le_map->r_map->r_data->region->name,
		 src_pe->pv_data->object->name, count2,
		 dst_pe->pv_data->object->name, count3);

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * commit_stripe_move_copy_data
 *
 * Copy the data from the source "stripe" to the destination "stripe".
 *
 * Offline implementation:
 * - Submit the copy job to the engine's copy-service.
 *
 * Online implementation:
 * - Call the engine to setup the mirror device.
 * - Load new mapping for the region.
 * - Suspend the region.
 * - Call engine to activate the mirror device.
 * - Resume the region.
 * - Wait for the engine's copy to finish.
 **/
static int commit_stripe_move_copy_data(logical_extent_map_t *le_map,
					copy_job_t *copy_job)
{
	storage_object_t *region = le_map->r_map->r_data->region;
	dm_target_t *target_list;
	int rc;

	LOG_ENTRY();

	if (EngFncs->can_online_copy()) {
		/* Set up the mirror. */
		rc = EngFncs->copy_setup(copy_job);
		if (rc) {
			goto out;
		}

		/* Load the new mapping and suspend. */
		le_map->copy_job = copy_job;

		target_list = build_target_list(region);
		if (!target_list) {
			rc = ENOMEM;
			goto out;
		}

		rc = EngFncs->dm_load_targets(region, target_list);
		EngFncs->dm_deallocate_targets(target_list);
		if (rc) {
			goto out;
		}

		EngFncs->dm_set_suspended_flag(TRUE);
		rc = EngFncs->dm_suspend(region, TRUE);
		if (rc) {
			EngFncs->dm_set_suspended_flag(FALSE);
			EngFncs->dm_clear_targets(region);
			goto out;
		}

		/* Activate the mirror. */
		rc = EngFncs->copy_start(copy_job);
		if (rc) {
			EngFncs->dm_clear_targets(region);
			EngFncs->dm_suspend(region, FALSE);
			EngFncs->dm_set_suspended_flag(FALSE);
			goto out;
		}

		/* Resume and activate the new mapping. */
		EngFncs->dm_suspend(region, FALSE);
		EngFncs->dm_set_suspended_flag(FALSE);

		/* Wait for the copy to complete. */
		rc = EngFncs->copy_wait(copy_job);

	} else {
		rc = EngFncs->offline_copy(copy_job);
	}

out:
	le_map->copy_job = NULL;
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * commit_stripe_move_update_metadata
 *
 * Replace the "current" map with the "new_map". Free up the approriate
 * physical extents for the "current" map, and remove the "current" PV object
 * from the region's child list if necessary. Then write the container
 * metadata to reflect the new mapping.
 **/
static int commit_stripe_move_update_metadata(logical_extent_map_t *le_map)
{
	logical_extent_t *old_map;
	int rc;

	LOG_ENTRY();

	old_map = le_map->map;
	le_map->map = le_map->new_map;
	le_map->new_map = NULL;

	rc = commit_container_metadata(le_map->r_map->r_data->region->
				       producing_container, FALSE);
	if (rc) {
		/* Failed to write the metadata. Fall back to the old map. */
		le_map->new_map = le_map->map;
		le_map->map = old_map;
	} else {
		deconstruct_region_mapping_stripe(old_map);
		deallocate_le_map_stripe(old_map);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * commit_stripe_move_cleanup_copy_job
 *
 * Tell the engine to free up the mirror for the copy-job and free the
 * memory for the title.
 **/
static void commit_stripe_move_cleanup_copy_job(copy_job_t *copy_job)
{
	LOG_ENTRY();

	EngFncs->copy_cleanup(copy_job);
	EngFncs->engine_free(copy_job->title);

	LOG_EXIT_VOID();
}

/**
 * commit_stripe_move
 *
 * Move the data represented by this "stripe".
 **/
static int commit_stripe_move(logical_extent_map_t *le_map)
{
	copy_job_t copy_job;
	int rc;

	LOG_ENTRY();

	memset(&copy_job, 0, sizeof(copy_job));

	rc = commit_stripe_move_init_copy_job(le_map, &copy_job);
	if (rc) {
		goto out;
	}

	rc = commit_stripe_move_copy_data(le_map, &copy_job);
	if (rc) {
		goto out;
	}

	rc = commit_stripe_move_update_metadata(le_map);
	if (rc) {
		goto out;
	}

out:
	/* Re-activate the volume with the new mapping. */
	my_plugin_record->functions.plugin->activate(le_map->r_map->r_data->region);

	commit_stripe_move_cleanup_copy_job(&copy_job);

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * commit_mapping_moves
 *
 * Move each "stripe" in this mapping that needs to be moved. The "stripe"
 * needs to be moved if the "new_map" field in the le_map is set.
 **/
static int commit_mapping_moves(region_mapping_t *r_map)
{
	u_int64_t i;
	int rc = 0;

	LOG_ENTRY();

	for (i = 0; i < r_map->stripe_count; i++) {
		if (r_map->le_maps[i].new_map) {
			rc = commit_stripe_move(r_map->le_maps + i);
			if (rc) {
				break;
			}
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * commit_region_moves
 *
 * Move all mappings that are waiting to be moved in this region.
 **/
static int commit_region_moves(storage_object_t *region)
{
	region_data_t *r_data = region->private_data;
	region_mapping_t *r_map;
	list_element_t iter;
	int rc = 0;

	LOG_ENTRY();

	if (!(r_data->flags & LVM2_REGION_FLAG_MOVE_PENDING)) {
		goto out;
	}

	LOG_DEBUG("Moving mappings for region %s.\n", region->name);

	rc = region_is_busy(region, TRUE);
	if (rc) {
		rc = EBUSY;
		goto out;
	}

	LIST_FOR_EACH(r_data->mappings, iter, r_map) {
		rc = commit_mapping_moves(r_map);
		if (rc) {
			break;
		}
	}

	if (!rc) {
		r_data->flags &= ~LVM2_REGION_FLAG_MOVE_PENDING;
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * commit_container_moves
 *
 * Check each region in this container to see if it has mappings to be moved.
 * Rebuild the freespace mappings when all moves are complete.
 **/
int commit_container_moves(storage_container_t *container)
{
	container_data_t *c_data = container->private_data;
	storage_object_t *region;
	list_element_t iter;
	int rc = 0;

	LOG_ENTRY();
	LOG_DEBUG("Moving mappings for container %s.\n", container->name);

	LIST_FOR_EACH(container->objects_produced, iter, region) {
		rc = commit_region_moves(region);
		if (rc) {
			break;
		}
	}

	/* Recreate the freespace mappings. */
	delete_freespace_mappings(container);
	rc = create_freespace_mappings(container);
	if (rc) {
		/* FIXME: Any cleanup possible? */
		goto out;
	}

	if (!rc) {
		c_data->flags &= ~LVM2_CONTAINER_FLAG_MOVE_PENDING;
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * can_move_stripe
 *
 * Has this mapping stripe already been scheduled for a move?
 **/
int can_move_stripe(logical_extent_map_t *le_maps)
{
	int rc;
	LOG_ENTRY();
	rc = le_maps->new_map ? EBUSY : 0;
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * can_move_a_stripe
 *
 * Does this mapping have at least one stripe that hasn't already been
 * scheduled for a move?
 **/
int can_move_a_stripe(region_mapping_t *r_map)
{
	u_int64_t i;
	int rc = EBUSY;

	LOG_ENTRY();

	for (i = 0; i < r_map->stripe_count; i++) {
		rc = can_move_stripe(r_map->le_maps + i);
		if (!rc) {
			break;
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * can_move_region_mapping
 *
 * Can this region mapping be moved, given the maximum number of consecutive
 * free extents in the container?
 **/
int can_move_region_mapping(region_mapping_t *r_map,
			    u_int64_t max_consecutive_extents)
{
	u_int64_t extents_per_stripe = r_map->le_count / r_map->stripe_count;
	int rc = ENOSPC;

	LOG_ENTRY();

	if (extents_per_stripe <= max_consecutive_extents) {
		rc = can_move_a_stripe(r_map);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * can_move_a_region_mapping
 *
 * Is there at least one mapping in this region that can be moved?
 **/
int can_move_a_region_mapping(storage_object_t *region)
{
	region_data_t *r_data = region->private_data;
	u_int64_t max_con_extents;
	region_mapping_t *r_map;
	list_element_t iter;
	int rc = ENOSPC;

	LOG_ENTRY();
	LOG_DEBUG("Checking if any mappings in region %s can be moved.\n",
		  region->name);

	max_con_extents =
		max_consecutive_extents_in_container(region->producing_container);

	LIST_FOR_EACH(r_data->mappings, iter, r_map) {
		rc = can_move_region_mapping(r_map, max_con_extents);
		if (!rc) {
			break;
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * move_region_mapping
 *
 * Set up the specified region-mapping to be moved from it's current location
 * to the specified destination PV and PE-offset.
 **/
int move_region_mapping(storage_object_t *region, option_array_t *options)
{
	storage_container_t *container = region->producing_container;
	container_data_t *c_data = container->private_data;
	region_data_t *r_data = region->private_data;
	storage_object_t *object;
	region_mapping_t *r_map;
	logical_extent_map_t *le_maps;
	u_int64_t extents_per_stripe;
	u_int64_t stripe_index, extent_index;
	u_int32_t r_map_index;
	pv_data_t *pv_data;
	char *object_name;
	int rc;

	LOG_ENTRY();
	LOG_DEBUG("Moving a mapping in region %s.\n", region->name);

	/* Parse and verify the options. */
	move_mapping_parse_options(options, &r_map_index, &stripe_index,
				   &object_name, &extent_index);
	rc = move_mapping_validate_options(region, r_map_index, stripe_index,
					   object_name, extent_index,
					   &r_map, &object);
	if (rc) {
		LOG_ERROR("Error validating options for move-mapping in "
			  "region %s.\n", region->name);
		goto out;
	}

	/* Create a new LE-map to represent the mapping's destination. */
	extents_per_stripe = r_map->le_count / r_map->stripe_count;
	le_maps = r_map->le_maps + stripe_index;
	le_maps->new_map = allocate_le_map_stripe(le_maps, extents_per_stripe);
	if (!le_maps->new_map) {
		LOG_ERROR("Error allocating new LE-map to move mapping %u in "
			  "region %s.\n", r_map_index, region->name);
		rc = ENOMEM;
		goto out;
	}

	pv_data = object->consuming_private_data;
	construct_region_mapping_stripe(le_maps->new_map,
					pv_data->pe_map, extent_index);

	/* Recreate the freespace mappings. */
	delete_freespace_mappings(container);
	rc = create_freespace_mappings(container);
	if (rc) {
		/* FIXME: Any cleanup possible? */
		goto out;
	}

	container->flags |= SCFLAG_DIRTY;
	c_data->flags |= LVM2_CONTAINER_FLAG_MOVE_PENDING;
	r_data->flags |= LVM2_REGION_FLAG_MOVE_PENDING;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

