/*
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * evms_deactivate
 *
 * Simple command-line utility to issue a REMOVE_ALL command to Device-Mapper,
 * which will deactivate all active devices.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <frontend.h>

#include "../engine/dm-ioctl3.h"	/* Yuck */

static int dm_control_fd = 0;

/**
 * scan_dev_info
 * @buf:	Input buffer.
 * @dev_num:	Returned device number.
 * @dev_name:	Returned device name.
 *
 * Scan a line for device info.
 * This is a simple, job specific function.  Rather than try to use sscanf() or
 * other complex parsers, we do a quick and dirty job here.
 * The input buffer should have the form:
 * | +[0-9]+ +name *[\n]|
 * In English, start with one or more blanks, then a number, one or more blanks,
 * then a name, followed by blanks and/or a newline.
 * Return the number and the name via parameters.
 * The function returns TRUE if it successfully found a number and a name,
 * else FALSE.
 **/
static boolean scan_dev_info(char * buf, int * dev_num, char * dev_name)
{
	boolean result = FALSE;
	char * p = buf;

	/* Skip over white space. */
	while ((*p != '\0') &&
	       ((*p == ' ') || (*p == '\t') || (*p == '\n'))) {
		p++;
	}

	if ((*p != '\0') && (*p >= '0') && (*p <= '9')) {
		*dev_num = atoi(p);

		/* Skip over the number. */
		while ((*p != '\0') && (*p != ' ') &&
		       (*p != '\t') && (*p != '\n')) {
			p++;
		}

		/* Skip over white space. */
		while ((*p != '\0') &&
		       ((*p == ' ') || (*p == '\t') || (*p == '\n'))) {
			p++;
		}

		if (*p != '\0') {
			/* Copy string up to white space. */
			while ((*p != '\0') && (*p != ' ') &&
			       (*p != '\t') && (*p != '\n')) {
				*dev_name++ = *p++;
			}

			result = TRUE;
		}
	}

	/* Terminate the name. If a name was not found, this will set the
	 * returned device name to "".
	 */
	*dev_name = '\0';

	return result;
}

/**
 * make_evms_dir_entry
 *
 * Make a directory in the EVMS dev node tree.
 **/
int make_evms_dir_entry(char * dir_name)
{
	char name_buf[EVMS_VOLUME_NAME_SIZE + 1];
	char * tmp_ptr = name_buf;
	struct stat statbuf;
	int rc = 0;

	/* See if this path-name exists. */
	rc = stat(dir_name, &statbuf);
	if (!rc) {
		/* The file exists. Make sure it is a directory. */
		if (! S_ISDIR(statbuf.st_mode)) {
			fprintf(stderr, "%s is a non-directory file\n",
				dir_name);
			rc = EINVAL;
		}
		goto out;
	}

	if (errno != ENOENT) {
		rc = errno;
		fprintf(stderr, "stat(%s) failed with error code %d: %s.\n",
			dir_name, rc, strerror(rc));
		goto out;
	}

	/* This directory doesn't exist.  Peel off the top-most directory
	 * name, and try again.
	 */
	strcpy(name_buf, dir_name);
	tmp_ptr = strrchr(name_buf, '/');
	if (tmp_ptr == NULL) {
		/* Either we were passed a relative path name, or the root
		 * file system doesn't exist.
		 */
		fprintf(stderr, "Invalid pathname: %s. Must use full "
			"pathnames.\n", dir_name);
		rc = ENODEV;
		goto out;
	}

	*tmp_ptr = '\0';

	/* Call myself recursively to make the parent directory. */
	rc = make_evms_dir_entry(name_buf);
	if (rc) {
		fprintf(stderr, "make_evms_dir_entry(%s) failed with error "
			"code %d.\n", name_buf, rc);
		goto out;
	}

	rc = mkdir(dir_name, (S_IFDIR | S_IRWXU | S_IRGRP |
			      S_IXGRP | S_IROTH | S_IXOTH));
	if (rc) {
		rc = errno;
		fprintf(stderr, "mkdir(%s) failed with error code %d: %s.\n",
			dir_name, rc, strerror(rc));
		goto out;
	}

out:
	return rc;
}

/**
 * open_dm_control_node
 *
 * Setup a dev node for the Device Mapper control node.
 * The Device Mapper control node is a character device.
 * Its major number is the major number of "misc", found in /proc/devices.
 * Its Minor number is listed in /proc/misc as "device"mapper".
 **/
static int open_dm_control_node(void)
{
	int rc = 0;
	FILE * file;
	char work_buf[256];
	char dev_name[64];
	int dev_num;
	int dm_control_major = 0;
	int dm_control_minor = 0;
	boolean found_major = FALSE;
	boolean found_minor = FALSE;
	dev_t devt;
	struct stat stat_buf;

	/* Look for the "misc" entry in /proc/devices. */
	file = fopen("/proc/devices", "r");
	if (!file) {
		rc = errno;
		fprintf(stderr, "Open of /proc/devices failed with error code "
			"%d: %s\n", rc, strerror(rc));
		goto out;
	}

	while (fgets(work_buf, sizeof(work_buf), file) != NULL) {
		if (scan_dev_info(work_buf, &dev_num, dev_name)) {
			if (strcmp(dev_name, "misc") == 0) {
				dm_control_major = dev_num;
				found_major = TRUE;
				break;
			}
		}
	}

	fclose(file);

	if (!found_major) {
		rc = ENOENT;
		fprintf(stderr, "Could not find an entry for \"misc\" in "
			"/proc/devices.\n");
		goto out;
	}

	/* Look for the "device-mapper" entry in /proc/misc. */
	file = fopen("/proc/misc", "r");
	if (!file) {
		rc = errno;
		fprintf(stderr, "Open of /proc/misc failed with error code "
			"%d: %s\n", rc, strerror(rc));
		goto out;
	}

	while (fgets(work_buf, sizeof(work_buf), file) != NULL) {
		if (scan_dev_info(work_buf, &dev_num, dev_name)) {
			if (strcmp(dev_name, "device-mapper") == 0) {
				dm_control_minor = dev_num;
				found_minor = TRUE;
				break;
			}
		}
	}

	fclose(file);

	if (!found_minor) {
		rc = ENOENT;
		fprintf(stderr, "Could not find an entry for \"device-mapper\" "
			"in /proc/misc.\n");
		goto out;
	}

	devt = makedev(dm_control_major, dm_control_minor);

	/* Create the directory if it doesn't already exist. */
	rc = make_evms_dir_entry(EVMS_DM_CONTROL_DIR);
	if (rc) {
		fprintf(stderr, "Could not create directory %s.\n",
			EVMS_DM_CONTROL_DIR);
		goto out;
	}

	/* If the control file doesn't exist, or if it exists but is not
	 * a  character device, or if it has the wrong device number,
	 * create it. */
	rc = stat(EVMS_DM_CONTROL, &stat_buf);
	if (rc) {
		rc = errno;
		if (rc != ENOENT) {
			fprintf(stderr, "stat of %s failed with error code %d: "
				"%s\n", EVMS_DM_CONTROL, rc, strerror(rc));
			goto out;
		}
	} else if (!S_ISCHR(stat_buf.st_mode) ||
		   stat_buf.st_dev != devt) {
		rc = unlink(EVMS_DM_CONTROL);
		if (rc) {
			rc = errno;
			fprintf(stderr, "unlink of %s failed with error code "
				"%d: %s\n", EVMS_DM_CONTROL, rc, strerror(rc));
			goto out;
		}
		rc = ENOENT;
	}

	if (rc) {
		rc = mknod(EVMS_DM_CONTROL, (S_IFCHR | S_IRUSR | S_IWUSR |
					     S_IRGRP | S_IWGRP), devt);
		if (rc) {
			rc = errno;
			fprintf(stderr, "mknod of %s (%d:%d) failed with error "
				"code %d: %s\n", EVMS_DM_CONTROL,
				dm_control_major, dm_control_minor, rc,
				strerror(rc));
			goto out;
		}
	}

	/* Open the control file. */
	rc = open(EVMS_DM_CONTROL, O_RDWR);
	if (rc < 0) {
		rc = errno;
		fprintf(stderr, "Open of %s failed with error code %d: %s\n",
			EVMS_DM_CONTROL, rc, strerror(rc));
		goto out;
	}

	dm_control_fd = rc;
	rc = 0;

out:
	return rc;
}

/**
 * main
 *
 * Open the DM control node, issue the REMOVE_ALL ioctl, and close.
 **/
int main(int argc, char **argv)
{
	dm_ioctl_t dmi;
	int rc;

	rc = open_dm_control_node();
	if (rc) {
		goto out;
	}

	memset(&dmi, 0, sizeof(dmi));
	dmi.version[0] = DM_VERSION_MAJOR;
	dmi.version[1] = DM_VERSION_MINOR;
	dmi.version[2] = DM_VERSION_PATCHLEVEL;
	dmi.data_size = sizeof(dmi);

	rc = ioctl(dm_control_fd, DM_REMOVE_ALL, &dmi);
	if (rc) {
		rc = errno;
		fprintf(stderr, "ioctl failed with error code %d: %s\n", 
			rc, strerror(rc));
		goto out;
	}

out:
	if (dm_control_fd > 0) {
		close(dm_control_fd);
	}
	return rc;
}

