/*
 *
 *   (C) Copyright IBM Corp. 2002, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */ 

#include <stdlib.h>
#include <string.h>
#include <ncurses.h>
#include <panel.h>
#include <glib.h>
#include <frontend.h>

#include "common.h"
#include "window.h"
#include "clist.h"
#include "dlist.h"

/**
 *	refresh_clist - handle refreshing clist and parent window
 *	@clist: the address of the struct clist to refresh
 *
 *	This routine simply makes sure we touch the parent window
 *	before refreshing our choice list. This should not be used
 *	to often.
 */
inline void refresh_clist(struct clist *clist)
{
	touchwin(clist->win->_parent);
	wrefresh(clist->win);
	wrefresh(clist->win->_parent);
}

/**
 *	create_clist - create the choice/selection list
 *	@parent_win: the parent window for out selection list subwindow
 *	@cols: the number of columns in the clist
 *	@height: the number of visible rows in the choice list subwindow
 *	@width: the number of columns in the choice list subwindow
 *	@starty: the row coordinate for the upper left of the subwindow
 *	@startx: the column coordinate for the upper left of the subwindow
 *	@user_data: whatever the caller wants to associate with the clist
 *
 *	This routine allocates the clist structure along with the subwindow
 *	it uses.
 **/
struct clist *create_clist(WINDOW *parent_win, int cols, int height, int width, int starty, int startx,
				void *user_data)
{
	struct clist *clist;

	clist = g_new0(struct clist, 1);
	clist->cols = cols;
	clist->column = g_new0(struct clist_column, cols);
	clist->win = subwin(parent_win, height, width, starty, startx);
	clist->user_data = user_data;
	clist->background = BLUE_BKGD;
	clist->focus_color = COLOR_PAIR(FOCUS_COLOR);
	clist->selected_color = COLOR_PAIR(LISTSEL_COLOR);
	wbkgd(clist->win, clist->background);
	refresh_clist(clist);

	return clist;
}

/**
 *	delete_clist - free resources associated with a choice/selection list
 *	@clist: the address of the choice list structure
 *
 *	This routine clears the clist of all choices, calls the optional
 *	delete callback for the clist, and then frees the clist structure.
 */
void delete_clist(struct clist *clist)
{
	clist->focus_cb = NULL;
	clear_clist(clist);
	if (clist->delete_cb != NULL) {
		clist->delete_cb(clist);
	}
	g_free(clist->column);
	g_free(clist);
}

/**
 *	set_selection_mode - changes the selection mode for a choice list
 *	@clist: the address of the choice list structure
 *	@newmode: the new mode (CLIST_MODE_MULTI or CLIST_MODE_SINGLE)
 *
 *	This routine checks the new mode to set. If one of the valid modes,
 *	it sets the clist mode to the new mode and returns the mode as 
 *	confirmation that it was set.
 */
inline int set_selection_mode(struct clist *clist, int newmode)
{
	int mode;

	if (newmode == CLIST_MODE_MULTI || newmode == CLIST_MODE_SINGLE) {
		mode = clist->mode = newmode;
	} else {
		mode = clist->mode;
	}
	return mode;
}

/**
 *	set_clist_color_attrs - change the background, focus item and selected items colors
 *	@clist: the address of the choice list structure
 *	@background: the color-pair for the clist background
 *	@focus_color: the color-pair index for the row with the current focus
 *	@selected_color: the color-pair index for a row/item that has been selected
 *
 *	This routine sets the color attributes for the window background and for
 *	items that have been selected and for the item that has the current focus.
 */
inline void set_clist_color_attrs(struct clist *clist, chtype background, short focus_color,
				short selected_color)
{
	clist->background = background;
	clist->focus_color = COLOR_PAIR(focus_color);
	clist->selected_color = COLOR_PAIR(selected_color);
	wbkgd(clist->win, background);
	refresh_clist(clist);
}

/**
 *	set_clist_column_count - change the number of columns in the clist
 *	@clist: the address of the choice list structure
 *	@cols: the new number of columns in the list
 *
 *	This routine frees any existing clist_column array and creates a
 *	new one of the given size.
 */
void set_clist_column_count(struct clist *clist, int cols)
{
	g_free(clist->column);
	clist->cols = cols;
	clist->column = g_new0(struct clist_column, cols);
}

/**
 *	set_clist_column_info - sets information for a clist column
 *	@clist: the selection list
 *	@col: the index of the column to set
 *	@width: the width of the column
 *	@startx: the offset in the clist where the column starts
 *	@justification: how to place text within the column
 *
 *	This routine sets the starting offset, width and text justification for
 *	a clist column.
 */
void set_clist_column_info(struct clist *clist, int col, int width, int startx,
				clist_text_justification justification)
{
	clist->column[col].width = width;
	clist->column[col].startx = startx;
	clist->column[col].justification = justification;
}

/*
 *	This following routines are simple set/get functions for setting or getting
 *	the various callout routines registered for various choice list events.
 */
inline void set_clist_delete_cb(struct clist *clist, clist_delete_cb delete_cb)
{
	clist->delete_cb = delete_cb;
}

inline clist_select_item_cb get_clist_select_item_cb(struct clist *clist)
{
	return clist->select_cb;
}

inline clist_unselect_item_cb get_clist_unselect_item_cb(struct clist *clist)
{
	return clist->unselect_cb;
}

inline void set_clist_select_item_cb(struct clist *clist, clist_select_item_cb select_cb)
{
	clist->select_cb = select_cb;
}

inline void set_clist_unselect_item_cb(struct clist *clist, clist_unselect_item_cb unselect_cb)
{
	clist->unselect_cb = unselect_cb;
}

inline void set_clist_activate_item_cb(struct clist *clist, clist_activate_item_cb activate_cb)
{
	clist->activate_cb = activate_cb;
}

inline void set_clist_focus_item_cb(struct clist *clist, clist_focus_item_cb focus_cb)
{
	clist->focus_cb = focus_cb;
}

/**
 *	The following four routines are getter and setter functions for the top_item
 *	and focus_item items in the clist.
 */
inline void set_clist_top_item(struct clist *clist, GList *item)
{
	clist->top_item = item;
}

inline void set_clist_focus_item(struct clist *clist, GList *item)
{
	clist->focus_item = item;
	if (clist->focus_cb != NULL && item != NULL)
		clist->focus_cb(clist, item->data);
}

inline GList *get_clist_top_item(struct clist *clist)
{
	return clist->top_item;
}

inline GList *get_clist_focus_item(struct clist *clist)
{
	return clist->focus_item;
}

/**
 *	get_clist_column_width - return the width of a requested clist column
 *	@clist: the selection list
 *	@col: the requested column index starting at zero
 *
 *	This routine returns the width of the requested column in the clist.
 */
inline int get_clist_column_width(struct clist *clist, int col)
{
	return clist->column[col].width;
}

/**
 *	get_clist_column_start - return the starting offset of a requested clist column
 *	@clist: the selection list
 *	@col: the requested column index starting at zero
 *
 *	This routine returns the starting offset of the requested column in the clist.
 */
inline int get_clist_column_start(struct clist *clist, int col)
{
	return clist->column[col].startx;
}

/**
 *	get_clist_column_end - return the starting offset for a column to follow the given one
 *	@clist: the selection list
 *	@col: the column index starting at zero
 *
 *	This routine is suitable for used in determining the starting column offset for
 *	a new column by starting at the end of the previous column.
 */
inline int get_clist_column_end(struct clist *clist, int col)
{
	return (get_clist_column_start(clist, col) + get_clist_column_width(clist, col) + 1);
}

/**
 *	calc_clist_column_width - calculates the column width based on a percentage of clist width
 *	@clist: the selection list
 *	@width_in_percent: the width as a percent of the width of the clist
 *
 *	This routine returns the width of a column given the percentage of the column
 *	the caller wants the column to take from the clist width.
 */
inline int calc_clist_column_width(struct clist *clist, float width_in_percent) 
{
	return (getmaxx(clist->win) * width_in_percent);
}

/**
 *	print_clist_column_title - prints the column title centered above the given column
 *	@clist: the selection list
 *	@col: the column index
 *	@title: the title string to print above the column
 *
 *	This routine prints the title for the requested column centered above
 *	the column.
 */
void print_clist_column_title(struct clist *clist, int col, char *title)
{
	int x, len, offset = 0;

	len = MIN(clist->column[col].width, strlen(title));
	switch (clist->column[col].justification) {
	case CLIST_TEXT_JUSTIFY_LEFT:
		offset = 0;
		break;
	case CLIST_TEXT_JUSTIFY_CENTER:
		offset = (clist->column[col].width / 2) - (len / 2) - 1;
		break;
	case CLIST_TEXT_JUSTIFY_RIGHT:
		offset = clist->column[col].width - len;
		break;
	}
	x = getparx(clist->win) + clist->column[col].startx + offset;
	mvwaddnstr(clist->win->_parent, getpary(clist->win) - 2, x, title, len);
}

/**
 *	create_item - allocates and initializes a list_item structure
 *	@clist: the columned selection list
 *	@text: the array of text strings to display for this list item row
 *	@user_data: whatever data the caller wants to associate with the item
 *	@delete_cb: the handler to call when this item is to be deleted
 *
 *	This routine simply allocates a list_item structure and initializes it
 *	with the given information. It returns the address of the newly created
 *	list_item.
 */
inline struct clist_item *create_item(struct clist *clist, GPtrArray *text, void *user_data,
					clist_item_delete_cb delete_cb)
{
	int i;
	struct clist_item *item;

	item = g_malloc(sizeof(struct clist_item));
	item->row_text = g_malloc(sizeof(char *) * clist->cols);

	for (i = 0; i < clist->cols; i++) {
		item->row_text[i] = g_strdup(g_ptr_array_index(text,i));
	}
	item->user_data = user_data;
	item->delete_cb = delete_cb;
	item->is_selected = FALSE;
	item->is_selectable = TRUE;

	return item;
}

/**
 *	append_item - creates and appends a list_item to the selection list
 *	@clist: the address of the choice list structure
 *	@text: the array of text strings to display for this list item row
 *	@user_data: whatever data the caller wants to associate with the item
 *	@delete_cb: the handler to call when this item is to be deleted
 *
 *	This routine creates an new list item and appends it to the end
 *	of the linked list of items to choose from.
 */
struct clist_item *append_item(struct clist *clist, GPtrArray *text, void *user_data,
				clist_item_delete_cb delete_cb)
{
	struct clist_item *item;

	item = create_item(clist, text, user_data, delete_cb);
	clist->choices = g_list_append(clist->choices, item);

	if (clist->top_item == NULL) {
		clist->top_item = clist->choices;
		set_clist_focus_item(clist, clist->top_item);
	}
	return item;
}

/**
 *	delete_item - deletes a clist item
 *	@item: the item to delete
 *	@clist: the clist we are deleting this item from
 *
 *	This routine is called as either a delete_all_elements() callback
 *	routine or directly. Either way, it is given the address of the clist
 *	item to delete. It invokes the item's delete callback before actually
 *	deleting the row text and the item structure. If the item being deleted
 *	was the top_item in the view, we determine which item replaces it. If
 *	this item was the focus item we set focus item to something safe like
 *	top_item. Lastly, if the item is a selected item, it is removed from
 *	the selections list as well.
 */
void delete_item(struct clist_item *item, struct clist *clist)
{
	int i;

	for (i = 0; i < clist->cols; i++) {
		g_free(item->row_text[i]);
	}
	if (item == clist->top_item->data) {
		if (g_list_next(clist->top_item) != NULL) {
			clist->top_item = g_list_next(clist->top_item);
		} else {
			clist->top_item = NULL;
		}
	}
	if (item == clist->focus_item->data) {
		set_clist_focus_item(clist, clist->top_item);
	}
	if (item->delete_cb != NULL)
		item->delete_cb(item);
	if (item->is_selected == TRUE) {
		clist->selections = g_list_remove(clist->selections, item);
	}
	g_free(item->row_text);
	g_free(item);
}

/**
 *	delete_clist_item - frees a clist_item and removes the link element for it from choices
 *	@clist: the clist we are deleting this item from
 *	@item: the item to delete
 *
 *	This routine is the public function that is called to delete a clist_item
 *	and remove it from the choices list.
 */
void delete_clist_item(struct clist *clist, struct clist_item *item)
{
	delete_item(item, clist);
	clist->choices = g_list_remove(clist->choices, item);
	draw_clist(clist);
}

/**
 *	unselect_item - marks an item as unselected and removes it from selection list
 *	@clist: the clist the item is being unselected from
 *	@item: the item being unselected
 *
 *	This routine ensures the given item is not selected if not done already and
 *	removes the selected item from the selection linked list. After unselecting
 *	the item we invoke any user unselection callback. If the callback returns a
 *	non-zero return code then we place the item back in the position where it
 *	was in the selection list.
 */
void unselect_item(struct clist *clist, struct clist_item *item)
{
	if (item->is_selected == TRUE && item->is_selectable == TRUE) {
		gint pos;
		GList *element;

		element = g_list_find(clist->selections, item);
		pos = g_list_position(clist->selections, element);

		item->is_selected = FALSE;
		clist->selections = g_list_remove(clist->selections, item);

		if (clist->unselect_cb != NULL) {
			int rc;

			rc = clist->unselect_cb(clist, item);
			if (rc != 0) {
				/*
				 * Callback did not agree with the unselection. Insert
				 * the item back to the selections list in the position
				 * we originally found it. Also, mark the item as selected.
				 */
				item->is_selected = TRUE;
				clist->selections = g_list_insert(clist->selections, item, pos);
			}
		}
	}
}

/**
 *	select_item - marks an item as selected and adds it to selection list
 *	@clist: the clist the item is being selected from
 *	@item: the item being selected
 *
 *	This routine marks the given item as selected if not done already and
 *	appends the selected item to the selection linked list to identify the
 *	order of current selections. After selecting the item we invoke any user
 *	selection callback. If the callback returns a non-zero return code then
 *	we remove the item from the selection list and mark it again unselected.
 */
void select_item(struct clist *clist, struct clist_item *item)
{
	if (item->is_selected == FALSE && item->is_selectable == TRUE) {
		item->is_selected = TRUE;
		clist->selections = g_list_append(clist->selections, item);

		if (clist->mode == CLIST_MODE_SINGLE &&
				g_list_length(clist->selections) > 1) {
			unselect_item(clist, clist->selections->data);
		}
		if (clist->select_cb != NULL) {
			int rc;

			rc = clist->select_cb(clist, item);
			if (rc != 0) {
				/*
				 * The callback did not agree with the selection. Remove
				 * the item from the selections list and mark it unselected.
				 */
				item->is_selected = FALSE;
				clist->selections = g_list_remove(clist->selections, item);
			}
		}
	}
}

/**
 *	compare_item_handles - check if given handle matches one in an clist_item's user data
 *	@item: a clist item
 *	@handle: the handle stored as a void pointer that we will be checking for
 *
 *	This routine is GCompareFunc used by find_item_from_handle() to search for a clist_item
 *	that contains a given handle stored in its user_data field. Zero is returned
 *	when a match is found.
 */
int compare_item_handles(struct clist_item *item, void *handle)
{
	return !(item->user_data == handle);
}

/**
 *	find_item_from_handle - searches the list of choices for an item containing the given handle
 *	@clist: the list of choices
 *	@handle: the object handle to look for
 *
 *	This routine searches the list of choices for the first item that contains the given
 *	handle. It returns the address of the GList link containing the clist_item containing that
 *	handle or NULL if none of the items contain the requested handle.
 */
inline GList *find_item_from_handle(struct clist *clist, engine_handle_t handle)
{
	return g_list_find_custom(clist->choices, GUINT_TO_POINTER(handle),
					(GCompareFunc)compare_item_handles);
}

/**
 *	activate_focus_item - invoke activate callback for focus item
 *	@clist: the choice list
 *
 *	This routine is invoked when we want to invoke the callback
 *	for an activate item event. Basically, this should occur when
 *	the ENTER key is detected by the event handler.
 */
void activate_focus_item(struct clist *clist)
{
	if (clist->activate_cb != NULL && clist->focus_item != NULL) {
		clist->activate_cb(clist, clist->focus_item->data);
	}
}

/**
 *	is_selected_item - returns the selection state
 *	@item: the item to determine if selected
 *
 *	This routine returns TRUE if the item is currently selected.
 */
inline int is_selected_item(struct clist_item *item)
{
	return item->is_selected;
}

/**
 *	set_item_selectable - indicate whether a item is selectable
 *	@item: the clist item to mark a selectable or not
 *	@is_selectable: TRUE if selectable or FALSE if not selectable
 *
 *	This routine allows indicating whether a given clist item is
 *	selectable and therefore processed or not by the select_item()
 *	and unselect_item() routines.
 */
inline void set_item_selectable(struct clist_item *item, int is_selectable)
{
	item->is_selectable = is_selectable;
}

/**
 *	is_clist_empty - returns TRUE if clist contains no items
 *	@clist: the clist to check for items
 *
 *	This routine checks to see if the choices linked list in
 *	the clist structure is empty. It returns TRUE if it is.
 */
inline int is_clist_empty(struct clist *clist)
{
	return (clist->choices == NULL);
}

/**
 *	draw_blank_item - draws a line in a clist full of blanks
 *	@clist: the address of the clist structure
 *	@line: the line in the clist window to clear
 *
 */
inline void draw_blank_item(struct clist *clist, int line)
{
	wmove(clist->win, line, 0);
	wclrtoeol(clist->win);
}

/**
 *	draw_item_text - draws the text for each of the columns in a row
 *	@clist: the address of the clist structure
 *	@item: the item
 *	@line: the line in the clist window to draw it at
 *
 *	This routine draws each of the strings in the row_text array
 *	at their proper column for their maximum with and for the 
 *	proper justification.
 */
inline void draw_item_text(struct clist *clist, struct clist_item *item, int line)
{
	int i, len = 0, col = 0, offset = 0;

	for (i = 0; i < clist->cols; i++) {
		len = MIN(clist->column[i].width, strlen(item->row_text[i]));

		switch (clist->column[i].justification) {
		case CLIST_TEXT_JUSTIFY_LEFT:
			offset = 0;
			break;
		case CLIST_TEXT_JUSTIFY_CENTER:
			offset = (clist->column[i].width / 2) - (len / 2) - 1;
			break;
		case CLIST_TEXT_JUSTIFY_RIGHT:
			offset = clist->column[i].width - len;
			break;
		}
		col = clist->column[i].startx + offset;
		mvwaddnstr(clist->win, line, col, item->row_text[i], len);
	}
}

/**
 *	draw_item - displays a single menu item in a choice list
 *	@element: the address of the linked list item we are in
 *	@index: an index to identify us in a loop
 *	@clist: the address of the clist structure we are in
 *
 *	This routine is meant to be invoked through a for_n_elements()
 *	call. For each iteration we used the index as the row to update.
 *	If the element is NULL or contains no data then we simply end up
 *	drawing a blank line.
 */
void draw_item(GList *element, int index, struct clist *clist)
{
	struct clist_item *item;

	if (element != NULL && element->data != NULL) {
		chtype attr;

		item = element->data;

		if (item == clist->focus_item->data) {
			attr = clist->focus_color;
		} else if (is_selected_item(item) == TRUE) {
			attr = clist->selected_color;
		} else {
			attr = clist->background;
		}
		if (is_selected_item(item))
			attr |= SELECTION_ATTR;

		wbkgdset(clist->win, attr);
		draw_blank_item(clist, index);
		draw_item_text(clist, item, index);
		wbkgdset(clist->win, clist->background);
	} else {
		draw_blank_item(clist, index);
	}
}

/**
 *	draw_clist_full_message - displays a message in parent window if clist has more items than visible
 *	@clist: the choice list
 *
 *	This routine checks the number of items in the clist. If the number exceeds the viewable
 *	lines in the clist window, we place a "More..." message in the parent window just under-
 *	neath the clist window.
 */
void draw_clist_full_message(struct clist *clist)
{
	int x, y, len;

	len = strlen(_("More..."));
	y = getpary(clist->win) + getmaxy(clist->win);
	x = getparx(clist->win) + getmaxx(clist->win) - len;

	if (g_list_nth(clist->top_item, getmaxy(clist->win)) != NULL) {
		mvwaddstr(clist->win->_parent, y, x, _("More..."));
	} else if (g_list_previous(clist->top_item) != NULL) {
		mvwaddstr(clist->win->_parent, y, x, _("Back..."));
	} else {		
		int i;

		/* Make sure any message there is erased */
		for (i = 0; i < len; i++) {
			mvwaddch(clist->win->_parent, y, x + i, ' ');
		}
	}

}

/**
 *	draw_clist - displays the items of a choice list
 *	@clist: the address of the choice list structure
 *
 *	This routine is responsible for	updating the displayed contents
 *	of clist. It first draws the items to be shown in the viewport,
 *	highlights selected items and gives a special highlight for the
 *	current focus item.
 */
void draw_clist(struct clist *clist)
{
	for_n_elements(clist->top_item, getmaxy(clist->win),
			(iterate_func)draw_item, clist);
	draw_clist_full_message(clist);
	refresh_clist(clist);
}

/**
 *	clear_clist - deletes all items in the given choice list
 *	@clist: the address of the choice list structure
 *
 * 	This routine clears all the items/choices out of a choice list.
 */
void clear_clist(struct clist *clist)
{
	delete_all_elements(clist->choices, (GFunc)delete_item, clist);
	clist->choices = clist->selections = clist->top_item = clist->focus_item = NULL;
	draw_clist(clist);
}

/**
 *	is_item_visible - checks to see if the item is within the visible rows
 *	@clist: the choice list
 *	@item: the linked list element containing the choice we want to check
 *
 *	This routine checks up to the max lines in the clist window starting
 *	from the top item looking to see if the choice given is visible. If
 *	it is, we return TRUE.
 */
int is_item_visible(struct clist *clist, GList *item)
{
	int i, result, maxlines;
	GList *element;

	result = FALSE;
	maxlines = getmaxy(clist->win);
	element = clist->top_item;

	for (i = 0; i < maxlines && element != NULL; i++) {
		if (element == item) {
			result = TRUE;
			break;
		}
		element = g_list_next(element);
	}
	return result;
}

/**
 *	scroll_clist_up - scroll the clist up by one item
 *	@clist: the clist that needs scrolling
 *
 *	This routine changes the clist->top_item to allow
 *	a redraw of the clist to make it appear that the
 *	list has scrolled up by one line. 
 */
inline void scroll_clist_up(struct clist *clist)
{
	if (g_list_previous(clist->top_item) != NULL) {
		clist->top_item = g_list_previous(clist->top_item);
	}
}

/**
 *	scroll_clist_down - scroll the clist down by one item
 *	@clist: the clist that needs scrolling
 *
 *	This routine changes the clist->top_item field to allow
 *	a redraw of the clist to make it appear that the list
 *	has scrolled down by one line. This routine also affects
 *	clist->focus_item.
 */
inline void scroll_clist_down(struct clist *clist)
{
	if (g_list_next(clist->top_item) != NULL) {
		clist->top_item = g_list_next(clist->top_item);
	}
}

/**
 *	page_through_clist - advances through a set of clist items my a window full
 *	@clist: the clist that needs scrolling
 *	@page_up: if TRUE we are paging up else paging down
 *
 * 	This routine advances or retreates the visible items by up
 * 	to n items where n is equal to the number of lines in the
 * 	clist window.
 */
inline void page_through_clist(struct clist *clist, int page_up)
{
	int i, maxlines;
	GList *element, *page_to_element = NULL;

	if (clist->top_item != NULL) {
		element = clist->top_item;
		maxlines = getmaxy(clist->win);

		for (i = 0; i <= maxlines && element != NULL; i++) {
			page_to_element = element;
			if (page_up == TRUE) {
				element = g_list_previous(element);
			} else {
				element = g_list_next(element);
			}
		}
		if (is_item_visible(clist, page_to_element) == FALSE) {
			clist->top_item = page_to_element;
		}
		set_clist_focus_item(clist, page_to_element);
	}
}

/**
 *	page_clist_up - jump ahead n items in the clist window
 *	@clist: the clist that needs scrolling
 *
 *	This routine moves into view the next n items that are past
 *	the last visible row. The maximum number of items moved in
 *	is limited to the number of lines available in the clist
 *	window.
 */
inline void page_clist_up(struct clist *clist)
{
	page_through_clist(clist, TRUE);
}

/**
 *	page_clist_down - jump back n items in the clist window
 *	@clist: the clist that needs scrolling
 *
 *	This routine moves into view the previous n items that
 *	are before the first visible row. The maximum number of
 *	items moved in is limited to the number of lines available
 *	in the clist window.
 */
inline void page_clist_down(struct clist *clist)
{
	page_through_clist(clist, FALSE);
}

/**
 *	move_focus_up - move the focus item up a line (scroll clist if necessary)
 *	@clist: the address of the clist structure
 *
 *	This routine manages the upward movement of the focused item. If the
 *	focus has moved to an item just before the top viewable item, the 
 *	clist is scrolled up by one in order to see the new focused item.
 */
inline void move_focus_up(struct clist *clist)
{
	if (g_list_previous(clist->focus_item) != NULL) {
		if (clist->focus_item == clist->top_item) {
			scroll_clist_up(clist);
		}
		set_clist_focus_item(clist, g_list_previous(clist->focus_item));
	}
}

/**
 *	move_focus_down - move the focus item down a line (scroll clist if necessary)
 *	@clist: the address of the clist structure
 *
 *	This routine manages the downward movement of the focused item. If the
 *	focus has moved to an item just below the last viewable item on the last
 *	line of the clist window, the clist is scrolled down in order to see the
 *	new focused item.
 */
inline void move_focus_down(struct clist *clist)
{
	if (g_list_next(clist->focus_item) != NULL) {
		if (is_item_visible(clist, g_list_next(clist->focus_item)) == FALSE) {
			scroll_clist_down(clist);
		}
		set_clist_focus_item(clist, g_list_next(clist->focus_item));
	}
}

/**
 *	invoke_clist_idle_funcs - process any clist idle functions in the idle queue
 *	@clist: the choice list
 *
 *	This routine checks the clist idle queue for entries. If entries exist, it
 *	invokes each item in turn. If the idle function returns a 0 return code, it
 *	is removed from the queue.
 */
inline void invoke_clist_idle_funcs(struct clist *clist)
{
	if (clist->idle_queue != NULL) {
		GList *element;

		element = clist->idle_queue;
		while (element != NULL) {
			int rc;
			clist_idle_func idle_func;

			idle_func = element->data;
			rc = idle_func(clist);

			element = g_list_next(element);
			if (rc == 0)
				clist->idle_queue = g_list_remove(clist->idle_queue, idle_func);
		}
	}
}

/**
 *	process_clist_events - map keyboard event to clist event
 *	@clist: the choice list to process events for
 *	@key: the keyboard input value to process
 *
 *	This routine processes the keyboard events for a
 *	choice list. This means mapping keyboard values
 *	to clist events, such as selection, scrolling, or
 *	activating an item in the choice list. If we process
 *	any events, we redraw the clist and return TRUE
 *	indicating we processed the key. FALSE is returned
 *	if the key value does not correspond to a clist
 *	event.
 */
int process_clist_events(struct clist *clist, int key)
{
	int event_handled = TRUE;
	gboolean redraw_clist = TRUE;
	struct clist_item *focus_item;

	if (clist->choices != NULL) {
		switch (key) {
		case KEY_DOWN:
			move_focus_down(clist);
			break;

		case KEY_UP:
			move_focus_up(clist);
			break;

		case KEY_PAGEDOWN:
			page_clist_down(clist);
			break;

		case KEY_PAGEUP:
			page_clist_up(clist);
			break;
	
		case KEY_ENTER:
			activate_focus_item(clist);
			redraw_clist = FALSE;
			break;

		case KEY_SPACEBAR:
			focus_item = clist->focus_item->data;
			if (focus_item->is_selected == TRUE) {
				unselect_item(clist, focus_item);
			} else {
				select_item(clist, focus_item);
			}
			break;
		default:
			event_handled = FALSE;
			break;
		}
		invoke_clist_idle_funcs(clist);
		if (event_handled == TRUE && redraw_clist == TRUE) {
			draw_clist(clist);
		}
	} else {
		event_handled = FALSE;
	}
	return event_handled;
}

/**
 *	sched_clist_idle_func - adds a clist_idle_func to the clist idle event queue
 *	@clist: the clist to process the idle function
 *	@func: the clist idle function
 *
 *	This routine adds a clist_idle_func to the queue of functions to invoke after
 *	processing regular clist events.
 */
void sched_clist_idle_func(struct clist *clist, clist_idle_func func)
{
	clist->idle_queue = g_list_append(clist->idle_queue, func);
}

/**
 *	clist_allow_all - a clist_filter_func that allows any handle to be processed
 *	@handle: the handle we don't even bother looking at
 *	@user_data: the extra data we care less about
 *
 *	This routine is a standard clist_filter_func function type that can be passed
 *	to clist_populate to avoid the caller providing its own when no filtering of
 *	handles is needed.
 */
int clist_allow_all(object_handle_t handle, void *user_data)
{
	return 0;
}

/**
 *	disallow_select_events - callback that disallows select or unselect clist item events
 *	@clist: the clist for the row item (un)selected
 *	@item: the clist item that was (un)selected
 *
 *	This routine is used to disallow select or unselect row events in a clist. It
 *	essentially just returns an error code so the clist item select/unselect routines
 *	undo the operation.
 */
int disallow_select_events(struct clist *clist, struct clist_item *item)
{
	return EPERM;
}


/**
 *	clist_populate_handle - append one object to a clist
 *	@clist: the choice list to populate
 *	@handle: the handle of the object to place in the clist
 *	@format_func: the user function that produces the text for the clist item
 *	@delete_cb: the callback to invoke when a item is about to be deleted
 *	@user_data: extra data supplied to user functions
 *
 *	This routine formats and appends a row to the given clist using the
 *	information from the given object handle.
 */
inline void clist_populate_handle(struct clist *clist, object_handle_t handle,
				clist_format_func format_func,
				clist_item_delete_cb delete_cb, void *user_data)
{
	GPtrArray *text;

	text = g_ptr_array_new();
	if (format_func(handle, user_data, text) == 0) {
		append_item(clist, text,
				GUINT_TO_POINTER(handle),
				delete_cb);
	}
	g_ptr_array_free(text, TRUE);
}

/**
 *	clist_populate - allows loading a clist with assistance from user functions
 *	@clist: the choice list to populate
 *	@enum_func: the user function that produces the object handle array
 *	@filter_func: the user function that filters out unwanted objects
 *	@format_func: the user function that produces the text for the clist item
 *	@delete_cb: the callback to invoke when a item is about to be deleted
 *	@user_data: extra data to be supplied to user functions
 *
 *	This routine handles the basic steps for loading a clist with items. The
 *	actual work is handled by user supplied functions.
 */
void clist_populate(struct clist *clist, clist_enum_func enum_func, clist_filter_func filter_func,
			clist_format_func format_func, clist_item_delete_cb delete_cb, void *user_data)
{
	int i, rc;
	handle_array_t *handles;

	g_return_if_fail(enum_func != NULL);
	g_return_if_fail(filter_func != NULL);
	g_return_if_fail(format_func != NULL);

	rc = enum_func(user_data, &handles);
	if (rc == 0) {
		for (i = 0; i < handles->count; i++) {
			rc = filter_func(handles->handle[i], user_data);
			if (rc == 0)
				clist_populate_handle(clist, handles->handle[i],
							format_func, delete_cb,
							user_data);
		}
		evms_free(handles);
	}
}
