/*
 *
 *   (C) Copyright IBM Corp. 2002, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#include <sys/types.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <stdio.h>
#include <unistd.h>
#include <getopt.h>
#include <frontend.h>
#include <ncurses.h>
#include <panel.h>
#include <glib.h>
#include <locale.h>

#include "common.h"
#include "window.h"
#include "menu.h"
#include "dialog.h"
#include "views.h"
#include "callbacks.h"

#define REQUIRED_ENGINE_VER_MAJOR	10
#define REQUIRED_ENGINE_VER_MINOR	0
#define REQUIRED_ENGINE_VER_PATCHLEVEL	0

static engine_mode_t open_mode = -1;
static debug_level_t debug_level = -1;
static char *log_name = NULL;
static char *node_name = NULL;
static ui_callbacks_t callbacks;
static WINDOW *splash_win;

/**
 *	show_help - displays version and command line options
 *	@argv: the array of current command line options
 *
 *	This routine simply displays the version, date built,
 *	and command line options and syntax for this program.
 */
static void show_help(char **argv)
{
	printf("\nEnterprise Volume Management System %s\n", VERSION);
	printf("International Business Machines  %s\n\n", DATE);
	printf("Usage: %s [OPTION...] \n\n", argv[0]);
	printf("  -d, --log-level=LEVEL\t\tLogging output level where LEVEL is one of the following:\n");
	printf("  \t\t\t\t\tcritical | serious | error | warning | default |\n");
	printf("  \t\t\t\t\tdetails | debug | extra | entry-exit | everything\n");
	printf("  -?, --help\t\t\tShow this help message\n");
	printf("  -m, --mode=ENGINE-MODE\tEngine mode where ENGINE-MODE is one of the following:\n");
	printf("  \t\t\t\t\treadwrite | readonly\n");
	printf("  -n, --node-name=NODENAME\tAdminister the given node in a cluster\n");
	printf("  -l, --log-name=FILENAME\tFile to send logging output to\n\n");
}

/**
 *	parse_options - parses the command line options
 *	@argc: command line argument count
 *	@argv: command line array of argument strings
 *	
 *	This routine, with the help of getopt_long(), parse the
 *	command line options and sets the globals corresponding
 *	to those options values as necessary.
 */
static int parse_options(int argc, char **argv)
{
	int c;
	int rc = 0;
	char *short_opts = "h?d:m:l:n:";
	struct option long_opts[] = {
		{"help", no_argument, NULL, 'h'},
		{"log-level", required_argument, NULL, 'd'},
		{"mode", required_argument, NULL, 'm'},
		{"log-name", required_argument, NULL, 'l'},
		{"node-name", required_argument, NULL, 'n'},
		{NULL, 0, NULL, 0}
	};

	while (rc == 0 && (c = getopt_long(argc, argv, short_opts, long_opts, NULL)) != EOF) {
		switch (c) {
		case 'm':
			if (g_strcasecmp(optarg, "readwrite") == 0)
				open_mode = ENGINE_READWRITE;
			else if (g_strcasecmp(optarg, "readonly") == 0)
				open_mode = ENGINE_READONLY;
			else {
				printf(_("Unrecognized mode argument -- \"%s\"\n\n"), optarg);
				rc = -1;
			}
			break;
		case 'd':
			if (g_strcasecmp(optarg, "critical") == 0)
				debug_level = CRITICAL;
			else if (g_strcasecmp(optarg, "serious") == 0)
				debug_level = SERIOUS;
			else if (g_strcasecmp(optarg, "error") == 0)
				debug_level = ERROR;
			else if (g_strcasecmp(optarg, "warning") == 0)
				debug_level = WARNING;
			else if (g_strcasecmp(optarg, "default") == 0)
				debug_level = DEFAULT;
			else if (g_strcasecmp(optarg, "details") == 0)
				debug_level = DETAILS;
			else if (g_strcasecmp(optarg, "debug") == 0)
				debug_level = DEBUG;
			else if (g_strcasecmp(optarg, "extra") == 0)
				debug_level = EXTRA;
			else if (g_strcasecmp( optarg, "entry-exit") == 0)
				debug_level = ENTRY_EXIT;
			else if (g_strcasecmp(optarg, "everything") == 0)
				debug_level = EVERYTHING;
			else {
				printf(_("Unrecognized debug level argument -- \"%s\"\n\n"), optarg);
				rc = -1;
			}
			break;
		case 'l':
			log_name = g_strdup(optarg);
			break;
		case 'n':
			node_name = g_strdup(optarg);
			break;
		default:
			printf(_("Unrecognized option -- \"%c\"\n\n"), c);
		case 'h':
		case '?':
			rc = -1;
			break;
		}
	}

	if (rc != 0)
		show_help(argv);

	return rc;			
}

/**
 *	check_engine_version - verify minimum required engine version
 *	@void
 *
 *	This routine verifies that we are running a version of the
 *	engine that we need.
 */
int check_engine_version(void)
{
	int rc;
	evms_version_t version;

	evms_get_api_version(&version);

	if (REQUIRED_ENGINE_VER_MAJOR == version.major &&
		((version.minor > REQUIRED_ENGINE_VER_MINOR) ||
		(REQUIRED_ENGINE_VER_MINOR == version.minor &&
		REQUIRED_ENGINE_VER_PATCHLEVEL <= version.patchlevel))) {
			rc = 0;
	} else {
		printf("Version %d.%d.%d of the engine API required.\n",
			REQUIRED_ENGINE_VER_MAJOR,
			REQUIRED_ENGINE_VER_MINOR,
			REQUIRED_ENGINE_VER_PATCHLEVEL);
		printf("Version %d.%d.%d was detected.\n",
			version.major, version.minor, version.patchlevel);
		rc = EPERM;
	}
	return rc;
}

/**
 *	show_splash_window - displays the startup window
 *	@void
 *
 *	This routine creates a window that displays a message
 *	that notifies the user that EVMS is initializing.
 */
PANEL *show_splash_window(void)
{
	PANEL *panel;

	panel = create_centered_popup_window(5, 60, BLUE_BKGD);
	splash_win = panel_window(panel);
	print_centered(splash_win, 1, _("EVMS is examining your system. Please wait..."));
	show_popup_window(panel);

	return panel;
}

/**
 * 	update_splash_window - update the status text in the startup window
 * 	@status: the new status text
 *
 * 	This routine is invoked as a callback from the engine when the engine
 * 	wants to relay some new status during engine open time.
 */
void update_splash_window(char *status)
{
	werase(splash_win);
	print_centered(splash_win, 1, status);
}

/**
 *	show_main_window - displays the main window
 *	@void
 *
 *	This routine displays the main window and populates
 *	the first view.
 */
void show_main_window(void)
{
	draw_menubar();
	draw_statusbar();
	init_views();
	print_statusbar_text(_("Press 'A' for Actions menu; TAB for next view; ESC for previous view"));
}

/**
 *	open_engine - display a splash screen and open the evms engine on the given node
 *	@node: the node to open the engine on
 *
 *	This routine displays a splash screen, queues information messages during the
 *	engine discovery and
 */
int open_engine(char *node)
{
	int rc = 0;
	PANEL *panel;

	panel = show_splash_window();
	enable_message_queuing();

	rc = evms_open_engine(node, open_mode, &callbacks, debug_level, log_name);
	callbacks.status = print_statusbar_text;
	delete_popup_window(panel);
	disable_message_queuing();
	display_queued_messages();

	return rc;
}

/**
 *	init_evmsn - main initialization routine
 *	@void
 *
 *	This routine controls program initialization. It invokes
 *	the initialization routines for the ncurses screen routines,
 *	creation of the splash window, opening of the engine, and
 *	finally removing the splash window upon return from the
 *	open engine call.
 */
int init_evmsn(void)
{
	int rc = 0;

	callbacks.user_message = show_user_message;
	callbacks.user_communication = NULL;
	callbacks.progress = report_progress;
	callbacks.status = update_splash_window;

	init_window_system();
	
	rc = open_engine(node_name);
	if (rc == 0) {
		show_main_window();
	} else {
		show_message_dialog(NULL, _("Unable to open EVMS engine! %s"), evms_strerror(rc));
	}

	return rc;
}

/**
 *	main_event_loop - handle input event for main window
 *	@void
 *
 *	This routine handle input events for the main window
 *	and returns only when the user has asked to quit.
 */
void main_event_loop(void)
{
	int rc = 0;

	while (rc != ESHUTDOWN) {
		rc = process_current_view_events();
	}
}

/**
 *	shutdown_evmsn - handle program cleanup
 *	@void
 *
 *	This routine is called to handle program cleanup such
 *	as ending the use of ncurses resources.
 */
void shutdown_evmsn(void)
{
	evms_close_engine();
	shutdown_window_system();
}

/**
 *	check_user_privilege - verify that user can open engine
 *	@void
 *
 *	This routine looks at the user privilege. If the engine
 *	mode indicates read/write mode then the effective uid
 *	must be that of root.
 */
int check_user_privilege(void)
{
	int rc = 0;

	if (open_mode & ENGINE_WRITE) {
		if (geteuid() != 0) {
			rc = EPERM;
			printf(_("root user privilege is required for read/write mode.\n"));
		}
	}
	return rc;
}

/**
 *	main - the start of it all baby
 *	@argc: command line argument count
 *	@argv: command line array of argument strings
 *
 *	The whole shebang starts here. We parse command line
 *	options, check the euid, create the startup window,
 *	open the engine, and eventually remove the startup window
 *	and display the main window and then wait on the main
 *	event loop until we are asked to exit.
 **/
int main(int argc, char *argv[])
{
	int rc;

	setlocale(LC_ALL, "");
	bindtextdomain(PACKAGE, LOCALEDIR);
	textdomain(PACKAGE);

	rc = parse_options(argc, argv);
	if (rc == 0) {
		rc = check_user_privilege();
		if (rc == 0) {
			rc = check_engine_version();
			if (rc == 0) {
				rc = init_evmsn();
				if (rc == 0) {
					main_event_loop();
				}
				shutdown_evmsn();
			}
		}
	}
	return rc;
}

