/*
 * libgksuui -- Gtk+ widget and convenience functions for requesting passwords
 * Copyright (C) 2004 Gustavo Noronha Silva
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <string.h>
#include <sys/select.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>

#include <gtk/gtk.h>
#include <gdk/gdkx.h>
#include <X11/Xlib.h>
#include <locale.h>

#include "../config.h"
#include "defines.h"

#include "gksuui-dialog.h"
#include "gksuui-convenience.h"

/* copied from gnome-ssh-askpass */
#define GRAB_TRIES	16
#define GRAB_WAIT	250 /* milliseconds */

static gboolean
gksu_check_init ()
{
  gint fakeargc = 1;
  gchar **fakeargv = g_new (gchar*, 2);
  gint return_value;

  fakeargv[0] = g_strdup ("libgksu");
  fakeargv[1] = NULL;

  return_value = gtk_init_check (&fakeargc, &fakeargv);

  g_free (fakeargv[0]);
  g_free (fakeargv);

  if (!return_value)
    fprintf (stderr, _("libgksu: Failed to init the Gtk+ library, "
		       "GKSu is not able to continue.\n"));

  return return_value;
}

/**
 * gk_dialog:
 * @type: a #GtkMessageType (GTK_MESSAGE_{INFO,ERROR,WARNING})
 * @format: a printf()-like format for the label of the dialog
 * @Varargs: the actual variables.
 *
 * Shows 'msg' in a dialog box with an OK button 
 * This function is to be a helper for functions needing to
 * display some information to the user.
 * Notice that it will not let the program go on if the
 * user does not close the dialog.
*/
static void 
gk_dialog (GtkMessageType type, gchar *format, ...)
{
  GtkWidget *diag_win;

  va_list ap;
  gchar *msg;

  va_start(ap, format);
  msg = g_strdup_vprintf(format, ap);
  va_end(ap);

  diag_win = gtk_message_dialog_new (NULL, GTK_DIALOG_MODAL,
				     type, GTK_BUTTONS_CLOSE,
				     msg);
			     
  gtk_signal_connect_object (GTK_OBJECT(diag_win), "delete_event",
			     GTK_SIGNAL_FUNC(gtk_main_quit), 
			     NULL);
  gtk_window_set_position (GTK_WINDOW(diag_win), GTK_WIN_POS_CENTER);
  gtk_window_set_resizable (GTK_WINDOW(diag_win), FALSE);

  gtk_widget_show_all (diag_win);
  gtk_dialog_run (GTK_DIALOG(diag_win));

  g_free (msg);

  gtk_widget_destroy (diag_win);
}

/**
 * gksu_secure_free:
 * @string: string to be freed securely
 *
 * Clears variables filling them with 0's previously
 */
static void
gksu_secure_free (gchar *string)
{
  if (string != NULL)
    {
      memset (string, 0, strlen(string));
      g_free (string);
    }
}

typedef enum
  {
    FAILED_GRAB_MOUSE,
    FAILED_GRAB_KEYBOARD
  } FailedGrabWhat;

/**
 * report_failed_grab:
 * @what: a #FailedGrabWhat describing what failed to
 * grab (mouse or keyboard)
 *
 * Simple utility function called to report a fail on
 * the process of grabbing the mouse or keyboard when
 * asking for the password through ask_password()
 */
void
report_failed_grab (FailedGrabWhat what)
{
  switch (what)
    {
    case FAILED_GRAB_MOUSE:
      gk_dialog (GTK_MESSAGE_WARNING, 
	     _("Could not grab your mouse.\n"
	       "A malicious client may be eavesdropping\n"
	       "on your session."));
      break;
    case FAILED_GRAB_KEYBOARD:
      gk_dialog (GTK_MESSAGE_WARNING, 
	     _("Could not grab your keyboard.\n"
	       "A malicious client may be eavesdropping\n"
	       "on your session."));
      break;
    }
}

/**
 * gksu_ask_password:
 * @title: the title of the dialog
 * @message: the message to use to request password from user
 * @grab: should the X keyboard and mouse inputs be grabbed
 *
 * This is a convenience function to create a #GksuuiDialog and
 * 
 *
 * Returns: a newly allocated gchar containing the password
 * or NULL if an error happens or the user cancels the action
 */
gchar*
gksu_ask_password (gchar *title, gchar *message, gboolean grab)
{
  gchar *pass = NULL;
	
  GtkWidget *dialog;

  gint response;

  /* 
     make sure we're using UTF-8 and getting our locale files
     from the right place
  */
  bindtextdomain(PACKAGE_NAME, LOCALEDIR);
  bind_textdomain_codeset (PACKAGE_NAME, "UTF-8");

  if (!gksu_check_init ())
    return NULL;

  /* dialog window */
  dialog = gksuui_dialog_new ();

  if(title != NULL)
     gtk_window_set_title(GTK_WINDOW(dialog), title);
  if(message != NULL)
     gksuui_dialog_set_message(GKSUUI_DIALOG(dialog), message);

  gtk_window_set_keep_above(GTK_WINDOW(dialog), TRUE);
  gtk_widget_show_all (dialog);

  while (gtk_events_pending())
    gtk_main_iteration ();

  if (grab)
    {
      GdkGrabStatus status;
      gint grab_tries = 0;

      for(;;) 
	{
	  status = gdk_pointer_grab((GTK_WIDGET(dialog))->window, TRUE, 0, NULL,
				    NULL, GDK_CURRENT_TIME);
	  if (status == GDK_GRAB_SUCCESS)
	    break;
	  usleep(GRAB_WAIT * 1000);
	  if (++grab_tries > GRAB_TRIES) 
	    report_failed_grab (FAILED_GRAB_MOUSE);
	}

	for(;;) 
	  {
	    status = gdk_keyboard_grab((GTK_WIDGET(dialog))->window,
				       FALSE, GDK_CURRENT_TIME);
	    if (status == GDK_GRAB_SUCCESS)
	      break;

	    usleep(GRAB_WAIT * 1000);

	    if (++grab_tries > GRAB_TRIES) 
	      report_failed_grab (FAILED_GRAB_KEYBOARD);
	  }

	gdk_x11_grab_server();
    }

  /* let the magic begin */
  response = gtk_dialog_run (GTK_DIALOG(dialog));
  
  if (grab)
    {
      /* Ungrab */
      XUngrabServer(GDK_DISPLAY());
      gdk_pointer_ungrab(GDK_CURRENT_TIME);
      gdk_keyboard_ungrab(GDK_CURRENT_TIME);
      gdk_flush();
    }

  {
    GtkWidget *entry = GKSUUI_DIALOG(dialog)->entry;

    /* Report passphrase if user selected OK */
    pass = g_strdup (gtk_entry_get_text(GTK_ENTRY(entry)));
    /* Kill the entry's copy of the passphrase. */
    gtk_entry_set_text(GTK_ENTRY(entry), "");
  }
			
  switch (response)
    {
    case GTK_RESPONSE_CANCEL:
    case GTK_RESPONSE_DELETE_EVENT:
    case GTK_RESPONSE_NONE:
      gtk_widget_destroy (dialog);
      gksu_secure_free (pass);
      return 0;
    }

  gtk_widget_destroy (dialog);

  while (gtk_events_pending())
    gtk_main_iteration();

  return pass;
}

