/*
 * Copyright (C) 2006  Sony Computer Entertainment Inc.
 *
 * ps3pf hashtable management.
 *
 * Based upon pSeries_lpar.c which is:
 *    Copyright (C) 2001 Todd Inglett, IBM Corporation
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 */


#undef DEBUG

#include <linux/config.h>
#include <linux/spinlock.h>
#include <linux/bitops.h>
#include <linux/threads.h>
#include <linux/smp.h>

#include <asm/abs_addr.h>
#include <asm/lmb.h>
#include <asm/machdep.h>
#include <asm/mmu.h>
#include <asm/mmu_context.h>
#include <asm/pgtable.h>
#include <asm/tlbflush.h>
#include <asm/tlb.h>
#include <asm/cputable.h>
#include <asm/lv1call.h>

#include "spu_priv.h"

#ifdef DEBUG
#define DBG(fmt...) udbg_printf(fmt)
#define printk	DBG
#else
#define DBG(fmt...)
#endif

extern unsigned long ps3pf_rm_limit, ps3pf_2nd_mem_base, ps3pf_mem_total;

int ps3pf_htab_size;

static hpte_t *htab;
static unsigned long htab_addr;
static unsigned char *bolttab;
static unsigned char *inusetab;

static spinlock_t ps3pf_bolttab_lock = SPIN_LOCK_UNLOCKED;

unsigned long p_to_lp(long pa)
{
	if (pa < ps3pf_rm_limit)
		return pa;
	pa -= ps3pf_rm_limit;
	pa += ps3pf_2nd_mem_base;
	return pa;
}
EXPORT_SYMBOL(p_to_lp);

long ps3pf_hpte_insert(unsigned long hpte_group, unsigned long va,
		       unsigned long pa, unsigned long rflags,
		       unsigned long vflags, int psize)
{
	unsigned long abs;
	unsigned long slot;
	hpte_t lhpte;
	int secondary = 0;
	unsigned long status;
	unsigned long bitmap;
	unsigned long flags;
	unsigned long p_pteg, s_pteg, b_index, b_mask, cb, ci;

	DBG(" -> ps3pf_hpte_insert\n");

	abs = phys_to_abs(pa);
	if (abs >= ps3pf_rm_limit &&
	    abs < ps3pf_mem_total) {
		abs -= ps3pf_rm_limit;
		abs += ps3pf_2nd_mem_base;
	}

	vflags &= ~HPTE_V_SECONDARY; /* this bit is ignored */

	lhpte.v = hpte_encode_v(va, psize) | vflags | HPTE_V_VALID;
	lhpte.r = hpte_encode_r(abs, psize) | rflags;

	p_pteg = hpte_group / HPTES_PER_GROUP;
	s_pteg = ~p_pteg & htab_hash_mask;

	spin_lock_irqsave(&ps3pf_bolttab_lock, flags);

	bitmap = (bolttab[p_pteg] << 8) | bolttab[s_pteg];
	if (bitmap == 0xffff) {
		printk("%s: htab full\n", __FUNCTION__);
		BUG();
	}

	bitmap = (inusetab[p_pteg] << 8) | inusetab[s_pteg];

	if (bitmap == 0xffff) {
		/*
		 * PTEG is full. Search for victim.
		 */
		bitmap &= ~((bolttab[p_pteg] << 8) | bolttab[s_pteg]);
		do {
			ci = mftb() & 15;
			cb = 0x8000UL >> ci;
		} while ((cb & bitmap) == 0);
	} else {
		/*
		 * search free slot in hardware order
		 *	[primary]	0, 2, 4, 6, 1, 3, 5, 7
		 *	[secondary]	0, 2, 4, 6, 1, 3, 5, 7
		 */
		for (ci = 0; ci < HPTES_PER_GROUP; ci += 2) {
			cb = 0x8000UL >> ci;
			if ((cb & bitmap) == 0)
				goto found;
		}
		for (ci = 1; ci < HPTES_PER_GROUP; ci += 2) {
			cb = 0x8000UL >> ci;
			if ((cb & bitmap) == 0)
				goto found;
		}
		for (ci = HPTES_PER_GROUP; ci < HPTES_PER_GROUP*2; ci += 2) {
			cb = 0x8000UL >> ci;
			if ((cb & bitmap) == 0)
				goto found;
		}
		for (ci = HPTES_PER_GROUP+1; ci < HPTES_PER_GROUP*2; ci += 2) {
			cb = 0x8000UL >> ci;
			if ((cb & bitmap) == 0)
				goto found;
		}
	}

 found:

	if (ci < HPTES_PER_GROUP) {
		slot = p_pteg * HPTES_PER_GROUP + ci;
	} else {
		slot = s_pteg * HPTES_PER_GROUP + (ci & 7);
		// lhpte.dw0.dw0.h = 1;
		vflags |= HPTE_V_SECONDARY;
		lhpte.v |= HPTE_V_SECONDARY;
	}

	status = lv1_write_htab_entry(0, slot, lhpte.v, lhpte.r);

	if (status != 0) {
		printk("%s: status = %ld (0x%lx)\n",
		       __FUNCTION__, status, status);
		printk("pa=%lx\n", pa);
		printk("va=%lx hpte_group=%lx\n",
		       va, hpte_group);
		printk("bitmap=%lx hpte.v=%lx hpte.r=%lx psize=%d\n",
		       bitmap, lhpte.v, lhpte.r, psize);
		printk("slot=%lx\n", slot);
		BUG();
	}

	/*
	 * If used slot is not in primary HPTE group,
	 * the slot should be in secondary HPTE group.
	 */
	if ((hpte_group ^ slot) & ~(HPTES_PER_GROUP - 1)) {
		secondary = 1;
		b_index = s_pteg;
	} else {
		secondary = 0;
		b_index = p_pteg;
	}

	b_mask = (lhpte.v & HPTE_V_BOLTED) ? 1 << 7 : 0 << 7;
	bolttab[b_index] |= b_mask >> (slot & 7);
	b_mask = 1 << 7;
	inusetab[b_index] |= b_mask >> (slot & 7);
	spin_unlock_irqrestore(&ps3pf_bolttab_lock, flags);

	DBG(" <- ps3pf_hpte_insert\n");

	return (slot & 7) | (secondary << 3);
}

static long ps3pf_hpte_remove(unsigned long hpte_group)
{
	panic("ps3pf_hpte_remove() not implemented");
	return 0;
}

static long ps3pf_hpte_updatepp(unsigned long slot, unsigned long newpp,
			      unsigned long va, int psize, int local)
{
	unsigned long flags;
	unsigned long status;
	unsigned long pteg, bit;
	unsigned long hpte_v, want_v;

	want_v = hpte_encode_v(va, psize);

	spin_lock_irqsave(&ps3pf_bolttab_lock, flags);

	hpte_v = htab[slot].v;
	if (!HPTE_V_COMPARE(hpte_v, want_v) || !(hpte_v & HPTE_V_VALID)) {
		spin_unlock_irqrestore(&ps3pf_bolttab_lock, flags);
		/*
		 * ps3pf_hpte_insert() will be used to update PTE.
		 */
		return -1;
	}

	status = lv1_write_htab_entry(0, slot, 0, 0);

	if (status != 0) {
		printk("%s: status = %ld (0x%lx)\n",
		       __FUNCTION__, status, status);
		printk("va=%lx slot=%lx psize=%d\n",
		       va, slot, psize);
		BUG();
	}

	pteg = slot / HPTES_PER_GROUP;
	bit = slot % HPTES_PER_GROUP;
	inusetab[pteg] &= ~(0x80 >> bit);
	spin_unlock_irqrestore(&ps3pf_bolttab_lock, flags);
	/*
	 * ps3pf_hpte_insert() will be used to update PTE.
	 */
	return -1;
}

static void ps3pf_hpte_updateboltedpp(unsigned long newpp, unsigned long ea,
				    int psize)
{
	panic("ps3pf_hpte_updateboltedpp() not implemented");
}

static void ps3pf_hpte_invalidate(unsigned long slot, unsigned long va,
					 int psize, int local)
{
	unsigned long flags;
	unsigned long status;
	unsigned long pteg, bit;

	spin_lock_irqsave(&ps3pf_bolttab_lock, flags);
	status = lv1_write_htab_entry(0, slot, 0, 0);

	if (status != 0) {
		printk("%s: status = %ld (0x%lx)\n",
		       __FUNCTION__, status, status);
		printk("va=%lx slot=%lx psize=%d\n",
		       va, slot, psize);
		BUG();
	}

	pteg = slot / HPTES_PER_GROUP;
	bit = slot % HPTES_PER_GROUP;
	inusetab[pteg] &= ~(0x80 >> bit);
	spin_unlock_irqrestore(&ps3pf_bolttab_lock, flags);
}

extern unsigned long ps3pf_vas_id;
extern unsigned long ps3pf_control_outlet;
extern unsigned long ps3pf_io_irq_outlet_id[];

static void ps3pf_hpte_clear(void)
{
	extern void ps3pf_pci_shutdown(void);
	extern void ps3fb_cleanup(void);

	unsigned long pu_id;
	unsigned long i;

	if (ps3pf_2nd_mem_base)
		lv1_release_memory(ps3pf_2nd_mem_base);

	ps3pf_spu_release_all();

#ifdef CONFIG_PPC_PS3PF_SB
	ps3pf_pci_shutdown();
#endif
#ifdef CONFIG_FB_PS3
	ps3fb_cleanup();
#endif

	lv1_unmap_htab(htab_addr);
	lv1_select_virtual_address_space(0);
	lv1_destruct_virtual_address_space(ps3pf_vas_id);

	for (i = 0; i < LV1_RESERVED_PLUG_ID_LIMIT; i++)
		lv1_destruct_io_irq_outlet(ps3pf_io_irq_outlet_id[i]);

	lv1_get_logical_pu_id(&pu_id);
	lv1_configure_irq_state_bitmap(pu_id, 0, 0);
}

void ps3pf_hpte_init(void)
{
	DBG(" -> ps3pf_hpte_init\n");

	ppc_md.hpte_invalidate	= ps3pf_hpte_invalidate;
	ppc_md.hpte_updatepp	= ps3pf_hpte_updatepp;
	ppc_md.hpte_updateboltedpp = ps3pf_hpte_updateboltedpp;
	ppc_md.hpte_insert	= ps3pf_hpte_insert;
	ppc_md.hpte_remove     	= ps3pf_hpte_remove;
	ppc_md.hpte_clear_all   = ps3pf_hpte_clear;

	htab_finish_init();

	DBG(" <- ps3pf_hpte_init\n");
}

void ps3pf_alloc_htab_bitmap(long htab_size, long rm_limit)
{
	long bitmap_size;

	ps3pf_htab_size = htab_size;
	bitmap_size = htab_size / sizeof(hpte_t) / 8;
	bolttab = abs_to_virt(lmb_alloc_base(bitmap_size, 1, rm_limit));
	inusetab = abs_to_virt(lmb_alloc_base(bitmap_size, 1, rm_limit));
	memset(bolttab, 0, bitmap_size);
	memset(inusetab, 0, bitmap_size);
}

void ps3pf_map_htab(void)
{
	long status;

	status = lv1_map_htab(0, &htab_addr);
	DBG("%s: htab_addr=%016lx\n", __FUNCTION__, htab_addr);
	htab = (hpte_t *)__ioremap((unsigned long)__va(htab_addr),
				   ps3pf_htab_size, PAGE_READONLY_X);
	DBG("%s: htab=%016lx\n", __FUNCTION__, (u_long)htab);
}
