/*  Screem:  skel-plugin.c
 *
 *  Copyright (C) 2004 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 */
#include <config.h>

#include <glib/gi18n.h>
#include <gtk/gtk.h>
#include <glade/glade.h>

#include <string.h>

#include "screem-skel-plugin.h"

/* setup your plugin here */

#include "support.h"

/* plugin name should only include a-zA-Z */
static const gchar *plugin_name = "ScreemLinkwizard";

static const gchar *authors[] = {
	"David A Knight <david@screem.org>",
	NULL
};
static const gchar *displayed_name = N_( "Link wizard" );
static const gchar *description = N_( "The link wizard provides a method of inserting HTML links, along with popup window support" );
static const gchar *version = "2.0.0";

/* add any per instance data items here */
struct ScreemSkelPluginPrivate {
	GladeXML *xml;
	GtkWidget *dialog;

	GladeXML *pxml;
	GtkWidget *pdialog;
};

static void link_wizard_display( GtkAction *action, gpointer user_data );
static void popup_response( GtkWidget *dialog, 
		gint button, gpointer data );
static void edit_popup( GtkWidget *widget, gpointer data );
static void link_wizard_clicked( GtkWidget *dialog, 
		gint button, gpointer data );
static void init_dialog( ScreemPlugin *plugin );


/**
 * setup:
 * 
 * this function will be called once for each window,
 * you should add any actions / ui here, eg.
 * 
 * screem_plugin_add_action( plugin, name, label, tip, stock_id,
 * 				callback, error );
 * screem_plugin_add_menu( plugin, path, action, error );
 * screem_plugin_add_toolbar( plugin, path, action, error );
 *
 *
 * to insert text into the current page being edited your callbacks
 * should make use of
 * screem_plugin_get_cursor_position( plugin )
 * screem_plugin_set_cursor_position( plugin, pos )
 * screem_plugin_insert( plugin, pos, text, length, indent )
 * 
 **/
static gboolean setup( ScreemPlugin *plugin )
{
	GError *error;
	gboolean ret;
	
	error = NULL;

	ret = screem_plugin_add_interface( plugin,
				"LinkWizard",
				_( "Link Wizard" ),
				_( "Insert a Link" ),
				"Screem_Link",
				G_CALLBACK( link_wizard_display ),
				&error );
	if( ! ret ) {
		g_print( "Add interface error: %s\n", error->message );
		g_error_free( error );
	}
	return ret;
}

/**
 * cleanup:
 *
 * this function will be called once for each window when
 * it is closed, you should cleanup any dta items you
 * have in ScreemSkelPluginPrivate here
 **/
static void cleanup( ScreemSkelPluginPrivate *priv )
{
	gtk_widget_destroy( priv->dialog );
	gtk_widget_destroy( priv->pdialog );

	g_object_unref( priv->xml );
	g_object_unref( priv->pxml );
}

static void link_wizard_display( GtkAction *action, gpointer user_data )
{
	ScreemPlugin *plugin;
	ScreemSkelPluginPrivate *priv;
	ScreemPage *page;
	GtkWidget *widget;
	
	plugin = SCREEM_PLUGIN( user_data );
	priv = SCREEM_SKEL_PLUGIN( plugin )->priv;
	
	page = screem_plugin_get_current_document( plugin );

	if( page ) {
		init_dialog( plugin );
		widget = glade_xml_get_widget( priv->pxml, "onclick" );
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ), 
				FALSE );
		if( ! GTK_WIDGET_VISIBLE( priv->dialog ) ) {
			screem_plugin_restore_from_session( plugin,
					priv->dialog );
		}
		gtk_widget_show_all( priv->dialog );
		gdk_window_raise( priv->dialog->window );
	}
}

static void popup_response( GtkWidget *dialog, 
		gint button, gpointer data )
{
	ScreemPlugin *plugin;

	plugin = SCREEM_PLUGIN( data );

	screem_plugin_store_in_session( plugin, dialog );
}

static void edit_popup( GtkWidget *widget, gpointer data )
{
	ScreemPlugin *plugin;
	ScreemSkelPluginPrivate *priv;
	
	plugin = SCREEM_PLUGIN( data );
	priv = SCREEM_SKEL_PLUGIN( plugin )->priv;
	
	if( ! GTK_WIDGET_VISIBLE( priv->pdialog ) ) {
		screem_plugin_restore_from_session( plugin,
				priv->pdialog );
	}
	gtk_widget_show_all( priv->pdialog );
	gdk_window_raise( priv->pdialog->window );
}

static void link_wizard_clicked( GtkWidget *dialog, 
		gint button, gpointer data )
{
	ScreemPlugin *plugin;
	ScreemSkelPluginPrivate *priv;
	const gchar *features[] = {
		"directories",
		"fullscreen",
		"location",
		"menubar",
		"resizable",
		"scrollbars",
		"status",
		"toolbars",
		NULL
	};
	gint i;

	plugin = SCREEM_PLUGIN( data );
	priv = SCREEM_SKEL_PLUGIN( plugin )->priv;
	
	if( button == GTK_RESPONSE_OK ) { 
		GtkWidget *widget;
		const gchar *text;
		const gchar *uri;
		const gchar *target;
		gchar *protocol;
		const gchar *name;
		const gchar *relation;
		const gchar *reverse;
		GString *tag;
		gint pos;
		gboolean script;
		gboolean fullscreen;
		
		script = FALSE;
		fullscreen = FALSE;
		
		widget = glade_xml_get_widget( priv->xml, "link_text" );
		widget = GTK_BIN( widget )->child;
		text = gtk_entry_get_text( GTK_ENTRY( widget ) );

		widget = glade_xml_get_widget( priv->xml, "link_to" );
		widget = GTK_BIN( widget )->child;
		uri = gtk_entry_get_text( GTK_ENTRY( widget ) );

		widget = glade_xml_get_widget( priv->xml, "target" );
		widget = GTK_BIN( widget )->child;
		target = gtk_entry_get_text( GTK_ENTRY( widget ) );
		if( ! GTK_WIDGET_SENSITIVE( widget ) ) {
			target = "";
		}

		widget = glade_xml_get_widget( priv->xml, "protocols" );
		protocol = gtk_combo_box_get_active_text( GTK_COMBO_BOX( widget ) );
		widget = glade_xml_get_widget( priv->xml, "protocols" );
		if( ! GTK_WIDGET_IS_SENSITIVE( widget ) ||
		    ! strcmp( _( "Other" ), protocol ) ) {
			g_free( protocol );
			protocol = g_strdup( "" );
		}

		widget = glade_xml_get_widget( priv->xml, "name" );
		widget = GTK_BIN( widget )->child;
		name = gtk_entry_get_text( GTK_ENTRY( widget ) );
	
		widget = glade_xml_get_widget( priv->xml, "relation" );
		widget = GTK_BIN( widget )->child;
		relation = gtk_entry_get_text( GTK_ENTRY( widget ) );
		
		widget = glade_xml_get_widget( priv->xml, "reverse_relation" );
		widget = GTK_BIN( widget )->child;
		reverse = gtk_entry_get_text( GTK_ENTRY( widget ) );
	
		/* FIXME: we should URI encode link_to */

		tag = g_string_new( "<a href=\"" );
		g_string_append_printf( tag, "%s%s\" ", protocol, uri );
		if( *target ) {
			g_string_append_printf( tag, "target=\"%s\" ",
					target );
		}
		if( *name ) {
			g_string_append_printf( tag, "name=\"%s\" ",
					name );
		}
		if( *relation ) {
			g_string_append_printf( tag, "rel=\"%s\" ",
					relation );
		}
		if( *reverse ) {
			g_string_append_printf( tag, "rev=\"%s\" ",
					reverse );
		}
	
		widget = glade_xml_get_widget( priv->xml, "edit_popup" );
		if( GTK_WIDGET_IS_SENSITIVE( widget ) ) {
			/* we may have popup properties to add */

			widget = glade_xml_get_widget( priv->pxml,
					"propbox" );
			script = ( GTK_WIDGET_IS_SENSITIVE( widget ) );
		}
		if( script ) {
			/* add onclick window.open() */
			GString *tmp;
			gboolean set;
			gint x;
			gint y;
			
			g_string_append( tag, " onclick=\"window.open( this.href, this.target" );

			set = FALSE;
			tmp = g_string_new( ", '" );
	
			widget = glade_xml_get_widget( priv->pxml,
					"fullscreen" );
			fullscreen = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );

			widget = glade_xml_get_widget( priv->pxml,
					"dimensions" );
			if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) && ! fullscreen ) {
				set = TRUE;
			
				widget = glade_xml_get_widget( priv->pxml, "width" );
				x = gtk_spin_button_get_value_as_int( GTK_SPIN_BUTTON( widget ) );
				widget = glade_xml_get_widget( priv->pxml, "height" );
				y = gtk_spin_button_get_value_as_int( GTK_SPIN_BUTTON( widget ) );
				g_string_append_printf( tmp, "width=%i,height=%i", x, y );
			}
			widget = glade_xml_get_widget( priv->pxml, 
					"position" );
			if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) && ! fullscreen ) {
				widget = glade_xml_get_widget( priv->pxml, "x" );
				x = gtk_spin_button_get_value_as_int( GTK_SPIN_BUTTON( widget ) );
				widget = glade_xml_get_widget( priv->pxml, "y" );
				y = gtk_spin_button_get_value_as_int( GTK_SPIN_BUTTON( widget ) );
				if( set ) {
					g_string_append_c( tmp, ',' );
				}
				g_string_append_printf( tmp, "left=%i,top=%i", x, y );
				set = TRUE;
			}
			for( i = 0; features[ i ]; ++ i ) {
				widget = glade_xml_get_widget( priv->pxml, features[ i ] );
				if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
					if( set ) {
						g_string_append_c( tmp, ',' );
					}
					g_string_append( tmp, features[ i ] );
					set = TRUE;
				}
			}
			if( set ) {
				g_string_append_c( tmp, '\'' );
				g_string_append( tag, tmp->str );
			}
			g_string_free( tmp, TRUE );
			
			g_string_append( tag, " ); return false;\" " );
		}
		
		
		g_string_append_c( tag, '>' );
			
		g_string_append( tag, text );
		g_string_append( tag, "</a>" );
		
		pos = screem_plugin_get_cursor_position( plugin );
		screem_plugin_insert( plugin, pos, tag->str, tag->len,
				TRUE );
		pos += tag->len;
		screem_plugin_set_cursor_position( plugin, pos );
		g_free( protocol );	
		g_string_free( tag, TRUE );
	}
	screem_plugin_store_in_session( plugin, dialog );
	
	gtk_widget_hide( priv->dialog );
}

static void init_dialog( ScreemPlugin *plugin )
{
	ScreemSkelPluginPrivate *priv;
	GtkWidget *widget;
	
	priv = SCREEM_SKEL_PLUGIN( plugin )->priv;
	if( ! priv->xml ) {
		priv->xml = glade_xml_new( GLADE_PATH"/linkwizard.glade",
				"linkwizard", NULL );
		
		priv->dialog = glade_xml_get_widget( priv->xml, 
				"linkwizard" );
		g_signal_connect( G_OBJECT( priv->dialog ), "response",
				G_CALLBACK( link_wizard_clicked ),
				plugin );

		widget = glade_xml_get_widget( priv->xml, "link_text" );
		screem_gtk_add_history( widget );	
	
		widget = glade_xml_get_widget( priv->xml, "protocols" );
		gtk_combo_box_set_active( GTK_COMBO_BOX( widget ), 0 );

		widget = glade_xml_get_widget( priv->xml, "edit_popup" );
		g_signal_connect( G_OBJECT( widget ), "clicked",
				G_CALLBACK( edit_popup ), plugin );
		
		glade_xml_signal_autoconnect( priv->xml );

		priv->pxml = glade_xml_new( GLADE_PATH"/linkwizard.glade",
				"popup", NULL );
	
		priv->pdialog = glade_xml_get_widget( priv->pxml, "popup" );
		g_signal_connect( G_OBJECT( priv->pdialog ), "response",
				G_CALLBACK( popup_response ), plugin );
		glade_xml_signal_autoconnect( priv->pxml );
	}
}

/* There should be no need to change any code below here */

enum {
	ARG_0
};

static void screem_skel_plugin_class_init( ScreemSkelPluginClass *klass );
static void screem_skel_plugin_init( ScreemSkelPlugin *skel_plugin );
static void screem_skel_plugin_finalize( GObject *object );

/* G Object stuff */
#define PARENT_TYPE SCREEM_TYPE_PLUGIN

static gpointer parent_class;

static void screem_skel_plugin_class_init( ScreemSkelPluginClass *klass )
{
	GObjectClass *object_class;

	object_class = G_OBJECT_CLASS( klass );

	object_class->finalize = screem_skel_plugin_finalize;
	parent_class = g_type_class_peek_parent( klass );
}

static void screem_skel_plugin_init( ScreemSkelPlugin *skel_plugin )
{
	skel_plugin->priv = g_new0( ScreemSkelPluginPrivate, 1 );
	SCREEM_PLUGIN( skel_plugin )->setup = setup;
}

static void screem_skel_plugin_finalize( GObject *object )
{
	ScreemSkelPlugin *skel_plugin;
	ScreemSkelPluginPrivate *priv;
	
	skel_plugin = SCREEM_SKEL_PLUGIN( object );
	priv = skel_plugin->priv;

	cleanup( priv );
	
	g_free( priv );
	
	G_OBJECT_CLASS( parent_class )->finalize( object );
}

static GType screem_skel_plugin_get_type()
{
	static GType type = 0;
	
	if( ! type ) {
		static const GTypeInfo info = {
			sizeof( ScreemSkelPluginClass ),
			NULL, /* base init */
			NULL, /* base finalise */
			(GClassInitFunc)screem_skel_plugin_class_init,
			NULL, /* class finalise */
			NULL, /* class data */
			sizeof( ScreemSkelPlugin ),
			0, /* n_preallocs */
			(GInstanceInitFunc)screem_skel_plugin_init
		};

		type = g_type_register_static( PARENT_TYPE,
					       plugin_name,
					       &info, 0 );
	}

	return type;
}

static ScreemSkelPlugin *screem_skel_plugin_new( void )
{
	ScreemSkelPlugin *skel_plugin;

	skel_plugin = SCREEM_SKEL_PLUGIN( g_object_new( SCREEM_TYPE_SKEL_PLUGIN, 
				"name", plugin_name,
				NULL ) );

	return skel_plugin;
}

G_MODULE_EXPORT void get_details( ScreemPluginDetails **ret )
{
	ScreemPluginDetails *details;

	details = g_new0( ScreemPluginDetails, 1 );
	details->name = plugin_name;
	details->displayed_name = displayed_name;
	details->authors = authors;
	details->description = description;
	details->version = version;
	details->create = screem_skel_plugin_new;
	details->api_version = SCREEM_PLUGIN_REQUIRED_VERSION;

	*ret = details;
}

