/*  Screem:  screem-dbus.c
 *
 *  utility class for handling dbus within screem
 * 
 *  Copyright (C) 2005  David A Knight
 *
 *  NetworkManager code strip from ephy-net-monitor-extension.c in
 *  epiphany,  Copyright (C) 2005 Jean-François Rameau
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>

#ifdef HAVE_DBUS
#define DBUS_API_SUBJECT_TO_CHANGE
#include <dbus/dbus.h>
#include <dbus/dbus-glib-lowlevel.h>

#endif

#include <glib/gi18n.h>

#include "screem-dbus.h"

#include "support.h"

enum {
	PROP_0,
	PROP_APP
};

struct ScreemDbusPrivate {
	ScreemApplication *app;
	gboolean exists;
#ifdef HAVE_DBUS
	DBusConnection *conn;
	DBusConnection *sconn;
#endif

};

#define NM_SERVICE	"org.freedesktop.NetworkManager"
#define NM_OBJECT_PATH	"/org/freedesktop/NetworkManager"
#define NM_INTERFACE	"org.freedesktop.NetworkManager"

#define NM_NO_ACTIVE_DEVICE	"org.freedesktop.NetworkManager.NoActiveDevice"

#ifdef HAVE_DBUS
static void screem_dbus_connect_to_system_bus( ScreemDbus *dbus );
static void screem_dbus_connect_to_user_bus( ScreemDbus *dbus );

static DBusHandlerResult screem_dbus_filter( DBusConnection *conn,
		DBusMessage *message, gpointer data );

static void screem_dbus_check_network( ScreemDbus *dbus );
#endif


/* G Object stuff */

G_DEFINE_TYPE( ScreemDbus, screem_dbus, G_TYPE_OBJECT )

static void screem_dbus_finalize( GObject *dbus );
static void screem_dbus_set_prop( GObject *object, 
		guint prop_id, const GValue *value, GParamSpec *spec );
static void screem_dbus_get_prop( GObject *object, 
		guint prop_id, GValue *value, GParamSpec *spec );
	
static void screem_dbus_class_init( ScreemDbusClass *klass )
{
	GObjectClass *object_class;
	GParamSpec *pspec;

	object_class = G_OBJECT_CLASS( klass );

	object_class->finalize = screem_dbus_finalize;
	object_class->get_property = screem_dbus_get_prop;
	object_class->set_property = screem_dbus_set_prop;

	pspec = g_param_spec_object( "app", "app", "app",
			SCREEM_TYPE_APPLICATION, 
			G_PARAM_READWRITE | G_PARAM_CONSTRUCT );
	g_object_class_install_property( G_OBJECT_CLASS( object_class ),
			PROP_APP, pspec );

}

static void screem_dbus_init( ScreemDbus *dbus )
{
	ScreemDbusPrivate *priv;
	
	dbus->priv = priv = g_new0( ScreemDbusPrivate, 1 );

#ifdef HAVE_DBUS
	dbus_g_thread_init();

	screem_dbus_connect_to_user_bus( dbus );
	screem_dbus_connect_to_system_bus( dbus );

#endif
	
}

static void screem_dbus_finalize( GObject *sdbus )
{
	ScreemDbus *dbus;
	ScreemDbusPrivate *priv;
	
	dbus = SCREEM_DBUS( sdbus );

	priv = dbus->priv;
	
#ifdef HAVE_DBUS
	if( priv->conn ) {
#if DBUS_VERSION > 33000
		dbus_connection_close( priv->conn );
#else
		dbus_connection_disconnect( priv->conn );
#endif
		dbus_connection_unref( priv->conn );
	}
	if( priv->sconn ) {
#if DBUS_VERSION > 33000
		dbus_connection_close( priv->sconn );
#else
		dbus_connection_disconnect( priv->sconn );
#endif
		dbus_connection_unref( priv->sconn );
	}
#endif

	g_free( priv );

	G_OBJECT_CLASS( screem_dbus_parent_class )->finalize( sdbus );
}

static void screem_dbus_set_prop( GObject *object, guint prop_id, 
		const GValue *value, GParamSpec *spec )
{
	ScreemDbus *dbus;
	GObject *obj;

	dbus = SCREEM_DBUS( object );

	switch( prop_id ) {
		case PROP_APP:
			obj = g_value_get_object( value );
			dbus->priv->app = SCREEM_APPLICATION( obj );
			break;
	}
}

static void screem_dbus_get_prop( GObject *object, guint prop_id,
		GValue *value, GParamSpec *spec )
{
	ScreemDbus *dbus;
	GObject *obj;

	dbus = SCREEM_DBUS( object );

	switch( prop_id ) {
		case PROP_APP:
			obj = G_OBJECT( dbus->priv->app );
			g_value_set_object( value, obj );
			break;
	}
}


/* static stuff */
#ifdef HAVE_DBUS
static void screem_dbus_connect_to_system_bus( ScreemDbus *dbus )
{
	ScreemDbusPrivate *priv;
	DBusConnection *conn;
	DBusError error;

	priv = dbus->priv;

	dbus_error_init( &error );
	priv->sconn = conn = dbus_bus_get( DBUS_BUS_SYSTEM, &error );
	if( ! dbus_error_is_set( &error ) ) {
		/* listen for org.freedesktop.NetworkManager signals */
		dbus_bus_add_match( conn,
				"type='signal',interface='org.freedesktop.NetworkManager',sender='org.freedesktop.NetworkManager',path='/org/freedesktop/NetworkManager'", NULL );
		dbus_connection_add_filter( conn,
				screem_dbus_filter,
				dbus, NULL );
	} else {
		g_warning( "Can't connect to system bus\n" );
	}

}

static void screem_dbus_connect_to_user_bus( ScreemDbus *dbus )
{
	ScreemDbusPrivate *priv;
	DBusConnection *conn;
	DBusError error;
	
	priv = dbus->priv;
	
	dbus_error_init( &error );
	priv->conn = conn = dbus_bus_get( DBUS_BUS_SESSION, &error );
	if( conn ) {
		dbus_connection_setup_with_g_main( conn, NULL );
#if DBUS_VERSION >= 33000
		if( dbus_bus_name_has_owner( conn, "org.screem",
					&error ) ) {
#else
		if( dbus_bus_service_exists( conn, "org.screem", 
					&error ) ) {
#endif
			priv->exists = TRUE;
#if DBUS_VERSION >= 33000
		} else if( dbus_bus_request_name( conn, "org.screem",
				0, &error ) != -1 ) {
#else
		} else if( dbus_bus_acquire_service( conn, "org.screem",
				0, &error ) != -1 ) {
#endif
			/* acquired, listen in for 
			 * org.screem stuff */
			dbus_bus_add_match( conn,
					"type='method_call',interface='org.screem.app'", NULL );
			dbus_connection_add_filter( conn,
				screem_dbus_filter,
				dbus, NULL );
		} else {
			g_warning( "Can't acquire service:  %s: %s'", error.name, error.message);
		}
	} else if( dbus_error_is_set( &error ) ) {
		/* failed to connect to session bus */
		gchar *msg;

		msg = g_strdup_printf( _( "%s\n\nDebian users should add \"use-session-bus\" to /etc/X11/Xsession.options\nFor other distributions read 'man dbus-launch'" ), error.message );
		gdk_threads_enter();	
		screem_hig_alert( GTK_MESSAGE_WARNING, 
				GTK_BUTTONS_CLOSE,
			_( "User session bus not available" ), 
			msg, NULL );
		g_free( msg );
		gdk_threads_leave();

		g_print( "DBUS ERROR: %s\n", error.message );
	}	

	dbus_error_free( &error );
}

static DBusHandlerResult screem_dbus_filter( DBusConnection *conn,
		DBusMessage *message, gpointer data )
{
	ScreemDbus *dbus;
	ScreemDbusPrivate *priv;
	DBusMessageIter it;
	gchar *uri;
	DBusHandlerResult res;

	GSList *files;
	
	dbus = SCREEM_DBUS( data );
	priv = dbus->priv;
	res = DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	
	if( dbus_message_is_method_call( message, "org.screem.app",
				"Open" ) ) {
		dbus_message_iter_init( message, &it );
		
		files = NULL;
		
#if DBUS_VERSION >= 33000
		do {
			dbus_message_iter_get_basic( &it, &uri );
			if( uri != NULL ) {
				files = g_slist_prepend( files, 
						g_strdup( uri ) );
			}
		} while( dbus_message_iter_next( &it ) );
#else
		while( ( uri = dbus_message_iter_get_string( &it ) ) ) {
			files = g_slist_prepend( files, uri );	
			if( ! dbus_message_iter_next( &it ) ) {
				break;
			}
		}
#endif
		screem_application_add_files( priv->app, files );
		/* ScreemApplication will now own files, so do not
		 * free */

		res = DBUS_HANDLER_RESULT_HANDLED;
	} else if( dbus_message_is_signal( message,
				NM_INTERFACE,
				"DeviceNoLongerActive" ) ) {
		screem_dbus_check_network( dbus );
		res = DBUS_HANDLER_RESULT_HANDLED;
	} else if( dbus_message_is_signal( message,
				NM_INTERFACE,
				"DeviceNowActive" ) ) {
		screem_dbus_check_network( dbus );
		res = DBUS_HANDLER_RESULT_HANDLED;
	}

	return res;
}

static void screem_dbus_check_network( ScreemDbus *dbus )
{
	ScreemDbusPrivate *priv;
	DBusMessage *message;
	DBusMessage *reply;
	DBusError error;
	gboolean offline;
	GConfClient *client;

	g_return_if_fail( SCREEM_IS_DBUS( dbus ) );
	
	priv = dbus->priv;
	
	offline = FALSE;
	
	message = dbus_message_new_method_call( NM_SERVICE,
			NM_OBJECT_PATH,
			NM_INTERFACE,
			"getActiveDevice" );
	if( message ) {
		dbus_error_init( &error );

		reply = dbus_connection_send_with_reply_and_block( priv->sconn,
				message, -1, &error );

		if( dbus_error_is_set( &error ) ) {
			if( dbus_error_has_name( &error,
						NM_NO_ACTIVE_DEVICE ) ) {
				offline = TRUE;
			}
		}

		if( reply ) {
			dbus_message_unref( reply );
		}

		dbus_message_unref( message );
	}

	client = gconf_client_get_default();

	gconf_client_set_bool( client,
			"/apps/screem/general/work_offline", offline,
			NULL );
	
	g_object_unref( client );
}

#endif


/* public api */
ScreemDbus *screem_dbus_new( ScreemApplication *app )
{
	ScreemDbus *dbus;
	GType type;

	type = screem_dbus_get_type();

	dbus = SCREEM_DBUS( g_object_new( type, 
				"app", app, NULL ) );

	return dbus;
}

gboolean screem_dbus_instance_exists( ScreemDbus *dbus )
{
	g_return_val_if_fail( SCREEM_IS_DBUS( dbus ), FALSE );

	return dbus->priv->exists;
}

void screem_dbus_open_files( ScreemDbus *dbus, const gchar **start_files )
{
#ifdef HAVE_DBUS
	ScreemDbusPrivate *priv;
	DBusMessage *message;
	DBusMessageIter it;

	g_return_if_fail( SCREEM_IS_DBUS( dbus ) );
	
	priv = dbus->priv;
	
	message = dbus_message_new_method_call( "org.screem",
			"/org/screem/app", "org.screem.app", "Open" );
	if( start_files ) {
		while( *start_files ) {
#if DBUS_VERSION >= 33000
			dbus_message_iter_init_append( message, &it );
			dbus_message_iter_append_basic( &it, 
					DBUS_TYPE_STRING, start_files );
#else
			dbus_message_append_iter_init( message, &it );
			dbus_message_iter_append_string( &it,
					*start_files );
#endif
			start_files ++;
		}

		dbus_connection_send( priv->conn, message, 0 );
		dbus_message_unref( message );
	}
#else
	g_warning( "screem_dbus_open_files() called without dbus support\n" );
#endif

}

