/*--- vector.h -----------------------------------------------------------------
Copyright (C) 2004, 2005, 2006 Sylvain Fourmanoit <syfou@users.sourceforge.net>

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.   
------------------------------------------------------------------------------*/
/* This implements a minimalistic vector API with error checking,        
   useful for storing dynamic collections of structures. 
   STL does not exist in pure C! 

   Basically, what we call a vector here is only a dynamic array 
   of references to undefined data structures (therefore cast to 'void')...
   As you see, it is pretty similar to a vector in STL, and you can
   expect the same kind of limitations. Please note that:

   	- vector_push and vector_pop are O(1), while vector_insert 
	and vector_delete are O(n) at best.

	- 'pos' and 'size' should never be modified by external
	routines. 'content' could, but using the limited API
	is better...Of course, you are totally on your own
	for the data itself.

	- Data elements pushed or added into a vector should always 
	be dynamically allocated OR the vector_free_item function
	pointer be redefined.

	- If freeing data elements requires more than calling free(),
	the user should supply its own vector_free_item_func function,
	and store its reference into the vector_free_item field of
	the vector.
	
	- Memory allocation/reallocation of the reference array 
	is automatically taken care of.

	- VECTOR_CHUNK macro should be modified for vectors
	of typically more than a couple references,
	because memory allocation calls would then become
	very expensive. Default value of 5 fits adesklets frugality!
*/

/*----------------------------------------------------------------------------*/
#ifndef HAVE_VECTOR_H
#define HAVE_VECTOR_H

#ifndef HAVE_CONFIG_H
#error Autogenerated config.h should be used.
#endif

/*----------------------------------------------------------------------------*/
#include "config.h"			/* Autoconf header */

#ifdef HAVE_STDLIB_H
#include <stdlib.h>			/* C includes: malloc, free, realloc */
#endif

#include "types.h"			/* Various typedef */
#include "error.h"			/* Error wrapper */

/*----------------------------------------------------------------------------*/
#define VECTOR_CHUNK 5			/* Number of additional elements */
					/* to allocate in one shot when
					   more space is needed */

/*----------------------------------------------------------------------------*/
typedef int (*vector_search_func) (void*,void*);
typedef void * (*vector_free_item_func) (void*);

typedef struct s_vector {               
  void ** content;			/* Ref. to dynamic, unspecified data */
  uint pos,				/* First unused element */
    size; 				/* Number of allocated ref. */ 
  vector_free_item_func                 /* The item freeing function */
    vector_free_item;
} vector;

/*----------------------------------------------------------------------------*/
/* Create a vector. 
   Return the vector address, or NULL in case of failure.
   A default wrapping of free() is put in place as the vector_free_item
   function reference. Users needing different desallocation should 
   overwrite it.
*/
vector * vector_init(void);

/* Push a reference to an untyped element into a vector. 
   Return 1 if successful, 0 otherwise.
*/ 
int      vector_push(vector *, void *);

/* Pop the last reference out of a vector.
   Return 1 if successful, 0 otherwise.
*/
int      vector_pop(vector *);

/* Insert a reference to a untyped element into a vector at a given position. 
   Return 1 if successful, 0 otherwise.
*/ 
int      vector_insert(vector *, uint, void *);

/* Delete a reference to a untyped element from a vector at a given position. 
   Return 1 if successful, 0 otherwise.
*/ 
int      vector_delete(vector *, uint);

/* Go through all the elements of a vector (from the first to the last) 
   until the first that matches a critera according to a search function,
   passing an unspecified callback argument. Second argument will contain
   the index of matching element, if found.
   Return this element's address, or NULL if unfound.
*/ 
void *   vector_find(vector *, uint *, vector_search_func, void*);

/* Destruct a vector.
   Returns NULL if successful, original vector otherwise.
*/ 
vector * vector_free(vector *);

/*----------------------------------------------------------------------------*/
#endif
