/*
   Project: Adun

   Copyright (C) 2005 Michael Johnston & Jordi Villa-Freixa

   Author: Michael Johnston

   This application is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This application is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111 USA.
*/
/** \file **/
#ifndef ADUN_SYSTEM
#define ADUN_SYSTEM

#include "AdunKernel/AdDataSources.h"
#include "AdunKernel/AdUpdating.h"
#include "AdunKernel/AdDelegation.h"
#include "AdunKernel/AdunDefinitions.h"
#include "AdunKernel/AdunObject.h"
#include "AdunKernel/AdunDynamics.h"
#include "AdunKernel/AdunBondedTopology.h"
#include "AdunKernel/AdunNonBondedTopology.h"
#include "AdunKernel/AdunState.h"

/*!
\ingroup Inter
The AdSystem class represents a certain component of a simulation consisting of a set of particles and their
associated topology and state. It is composed of five lower level classes which are all subclasses of 
AdSystemComponent. These classes are AdDyanmics, AdBondedTopology, AdState and two AdNonBondedTopology instances, 
one for normal and one for long range interactions. AdSystem creates and manages this collection of classes providing
a unified interface to the entire system. 

AdSystem uses a dataSource to retrieve the data it manages. The data source object must conform to the 
AdSystemDataSource protocol.
\todo Move removeTranslationalDegreesOfFreedom:velocities to Base
\todo Implement wrappers around all necessary subsystem methods. 
\note Clear up nomenclature of nonbonded interactions - remove shortRange qualifier for explicit interactions since the range
is dependant on the cutoff. Leave long range to imply interactions that are some approximation to explicit interactions for
interactions over a certain threshhold.
\note also need to be able to set separate cutoffs for various nonbondedInteractions. 

Notifications

AdSystemStatusDidChangeNotification - Sent when status is changed. Object is the AdSystem whose status changed.
UserInfo contains two keys @"PreviousStatus" and @"CurrentStatus".

AdSystemContentsDidChangeNotification - Sent when an AdSystem instance reloads its data or when 
certain (yet to be defined) modification methods are used

*/

@interface AdSystem: AdObject <AdUpdating, AdFrameUpdating, AdDataSourceDelegation>
{
	int numberOfAtoms;
	NSMutableArray* systemKeywords;
	NSString* systemName;
	id state;			
	id dynamics;
	id bondedTopology;
	id shortRangeNonbondedTopology;
	id longRangeNonbondedTopology;
	id dataSource;
	NSString* status;		//!< The current status of the system
	NSArray* allowedStates;		//!< Array of the allowed status values
}
- (NSString*) systemName;
- (id) dynamics;
- (id) bondedTopology;
- (id) shortRangeNonbondedTopology;
- (id) longRangeNonbondedTopology;
- (id) state;
- (id) systemKeywords;
- (double) kineticEnergy;
- (double) temperature;
- (double) time;
- (double) potentialEnergy;
- (NSDictionary*) bondedInteractions;
- (NSDictionary*) nonbondedInteractionTypes;
 - (NSValue*) shortRangeNonbondedInteractions;
- (NSValue*) longRangeNonbondedInteractions;
- (NSValue*) coordinates;
- (NSValue*) accelerations;
- (NSValue*) velocities;
/**
The current status of the system. 
*/
- (NSString*) status;
/**
Sets the status of the object. Status can be one of "Active", "Passive" or "Inactive".
The status value is used by AdSystemNode to determine the return values of activeSystems etc.
*/
- (void) setStatus: (NSString*) aString;

- (void) setCurrentForceFieldState: (id) forceFieldState;
- (int) numberOfAtoms;
- (void) zeroAccelerations;
@end

#endif
