/*******************************************************************************
 *
 *  AUTHOR: Jonathon Jongsma
 *
 *  Copyright (c) 2005 Jonathon Jongsma
 *
 *  License:
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the 
 *    Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 *    Boston, MA  02111-1307  USA
 *
 *******************************************************************************/

#include <glib/gstdio.h>
#include "config.h"
#include "gcs-conf.h"
#include "core/gcs-color.h"
#ifdef HAVE_GCONFMM
#include <gconfmm/client.h>
#include <gconfmm/value.h>
#endif // HAVE_GCONFMM

namespace gcs
{
    // gconf recommends storing enumerations as strings rather than integers
    // since it's more robust against changes in enumeration order and is more
    // human-readable.  This is a helper for gconf_string_to_enum and vice versa
    static GConfEnumStringPair schemetype_lookup_table[] = 
    {
        { SCHEME_COMPLEMENTS, "COMPLEMENTS" },
        { SCHEME_SPLIT_COMPLEMENTS, "SPLIT-COMPLEMENTS" },
        { SCHEME_TRIADS, "TRIADS" },
        { SCHEME_TETRADS, "TETRADS" },
        { SCHEME_ANALOGOUS, "ANALOGOUS" },
        { SCHEME_MONOCHROMATIC, "MONOCHROMATIC" },
        { 0, NULL }
    };

    // convenience class to convert between gconf Value classes and basic types
    // more automatically
    class Value : public Gnome::Conf::Value
    {
        public:
            Value(const Gnome::Conf::Value& other) : Gnome::Conf::Value(other) {}
            operator bool() { return get_bool(); }
            operator int() { return get_int(); }
            operator Glib::ustring() { return get_string(); }
            operator double() { return get_float(); }
            operator float() { return get_float(); }
    };

    const Glib::ustring Conf::APP_DIR = "/apps/agave";

    const Glib::ustring Conf::KEY_LAST_SAVE_DIR = Conf::APP_DIR + "/last_save_directory";
    const Glib::ustring Conf::KEY_LAST_COLOR = Conf::APP_DIR + "/last_color";
    const Glib::ustring Conf::KEY_LAST_SCHEME_TYPE = Conf::APP_DIR + "/last_scheme_type";
    const Glib::ustring Conf::KEY_WINDOW_WIDTH = Conf::APP_DIR + "/window_width";
    const Glib::ustring Conf::KEY_WINDOW_HEIGHT = Conf::APP_DIR + "/window_height";
    const Glib::ustring Conf::KEY_FAVORITES_WIDTH = Conf::APP_DIR + "/favorites_width";
    const Glib::ustring Conf::KEY_PALETTE_EXPANDED = Conf::APP_DIR + "/palette_expanded";

    const Glib::ustring Conf::OLD_GCS_CONF_DIR = Glib::build_filename(Glib::get_home_dir(), ".agave");
    const Glib::ustring Conf::GCS_CONF_DIR = Glib::build_filename(g_get_user_data_dir(), "agave");
    const Glib::ustring Conf::FAVORITES_FILE = Conf::GCS_CONF_DIR + "/bookmarks.gpl";
    const Glib::ustring Conf::USER_PALETTE_DIR = Conf::GCS_CONF_DIR + "/palettes";

    const int Conf::WINDOW_BORDER = 12;
    const int Conf::UI_SPACING_SMALL = 6;
    const int Conf::UI_SPACING_MEDIUM = 12;
    const int Conf::UI_SPACING_LARGE = 18;

    void Conf::migrate_old_config_directory()
    {
        // check for an old config directory, and if it exists, rename it to the
        // new directory
        if (Glib::file_test(OLD_GCS_CONF_DIR, Glib::FILE_TEST_EXISTS) &&
                !Glib::file_test(GCS_CONF_DIR, Glib::FILE_TEST_EXISTS))
        {
            g_rename(OLD_GCS_CONF_DIR.c_str(), GCS_CONF_DIR.c_str());
        }
    }


    Glib::ustring Conf::get_last_save_dir(void)
    {
        Glib::ustring dir;
        if (!get_value(KEY_LAST_SAVE_DIR, dir) || dir == "")
        {
            dir = Glib::filename_to_uri(Glib::get_current_dir());
        }
        return dir;
    }


    void Conf::set_last_save_dir(Glib::ustring uri)
    {
        set_value(KEY_LAST_SAVE_DIR, uri);
    }


    ColorPtr Conf::get_last_color(void)
    {
        Glib::ustring clr;
        if (!get_value(KEY_LAST_COLOR, clr))
        {
            clr = "#9DFF00";
        }
        return Color::create(clr);
    }


    void Conf::set_last_color(ColorPtr clr)
    {
        set_value(KEY_LAST_COLOR, clr->get_hexstring());
    }


    void Conf::set_last_scheme_type(tSchemeType t)
    {
        Glib::ustring schemetype_string = gconf_enum_to_string(schemetype_lookup_table, t);
        set_value(KEY_LAST_SCHEME_TYPE, schemetype_string);
    }


    tSchemeType Conf::get_last_scheme_type(void)
    {
        Glib::ustring scheme_type;
        tSchemeType t;
        if (!get_value(KEY_LAST_SCHEME_TYPE, scheme_type) ||
                !gconf_string_to_enum(schemetype_lookup_table,
                    scheme_type.c_str(), reinterpret_cast<int*>(&t)))
        {
            t = SCHEME_TRIADS;
        }
        return t;
    }


    Glib::ustring Conf::get_favorites_file(void)
    {
        return FAVORITES_FILE;
    }


    Glib::ustring Conf::get_conf_dir(void)
    {
        if (!Glib::file_test(GCS_CONF_DIR, Glib::FILE_TEST_EXISTS) ||
                !Glib::file_test(GCS_CONF_DIR, Glib::FILE_TEST_IS_DIR))
        {
            g_mkdir(GCS_CONF_DIR.c_str(), 0755);
        }
        return GCS_CONF_DIR;
    }


    Glib::ustring Conf::get_user_palette_dir(void)
    {
        if (!Glib::file_test(USER_PALETTE_DIR, Glib::FILE_TEST_EXISTS) ||
                !Glib::file_test(USER_PALETTE_DIR, Glib::FILE_TEST_IS_DIR))
        {
            g_mkdir(USER_PALETTE_DIR.c_str(), 0755);
        }
        return USER_PALETTE_DIR;
    }


    int Conf::get_window_width(void)
    {
        int w;
        if (!get_value(KEY_WINDOW_WIDTH, w))
        {
            w = 650;
        }
        return w;
    }


    void Conf::set_window_width(int w)
    {
        set_value(KEY_WINDOW_WIDTH, w);
    }


    int Conf::get_window_height(void)
    {
        int h;
        if (!get_value(KEY_WINDOW_HEIGHT, h))
        {
            h = 400;
        }
        return h;
    }


    void Conf::set_window_height(int h)
    {
        set_value(KEY_WINDOW_HEIGHT, h);
    }


    int Conf::get_favorites_width(void)
    {
        int w;
        if (!get_value(KEY_FAVORITES_WIDTH, w))
        {
            w = 130;    // initial value
        }
        return w;
    }


    void Conf::set_favorites_width(int w)
    {
        set_value(KEY_FAVORITES_WIDTH, w);
    }


    bool Conf::get_palette_expanded(void)
    {
        bool expanded;
        if (!get_value(KEY_PALETTE_EXPANDED, expanded))
        {
            expanded = true;    // initial value
        }
        return expanded;
    }


    void Conf::set_palette_expanded(bool expanded)
    {
        set_value(KEY_PALETTE_EXPANDED, expanded);
    }


    template <typename T>
    bool Conf::get_value(Glib::ustring gconf_key, T& result)
    {
#ifdef HAVE_GCONFMM
        Value val = Gnome::Conf::Client::get_default_client()->get(gconf_key);
        if (val.get_type() != Gnome::Conf::VALUE_INVALID)
        {
            result = static_cast<T>(val);
            return true;
        }
#endif // HAVE_GCONFMM
        return false;
    }


    template <typename T>
    void Conf::set_value(Glib::ustring gconf_key, const T& val)
    {
#ifdef HAVE_GCONFMM
        Gnome::Conf::Client::get_default_client()->set(gconf_key, val);
#endif // HAVE_GCONFMM
    }
} // namespace gcs
