// ---------------------------------------------------------------------------
// - Nameset.hpp                                                            -
// - aleph engine - nameset class definition                                -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  ALEPH_NAMESET_HPP
#define  ALEPH_NAMESET_HPP

#ifndef  ALEPH_OBJECT_HPP
#include "Object.hpp"
#endif

namespace aleph {

  /// The Nameset class defines an environment for a set of objects. Since
  /// a nameset is an object, it is possible to create or evaluate by
  /// name. This class is an abstract class.
  /// @author amaury darsch

  class Nameset : public Object {
  protected:
    Nameset* p_parent;

  public:
    /// create a new nameset
    Nameset (void);
    
    /// destroy this nameset
    ~Nameset (void);

    /// set the parent nameset
    /// @param nset the parent nameset
    void setparent (Nameset* nset);

    /// @return the parent nameset
    Nameset* getparent (void) const {
      return p_parent;
    }

    /// clear this nameset
    virtual void clear (void) =0;

    /// add a new object by quark
    /// @param quark the object quark
    /// @param object the object to bind
    virtual void bind (const long quark, Object* object) =0;

    /// add a new object by name
    /// @param name the object name
    /// @param object the object to bind
    void bind (const String& name, Object* object);

    /// @return true if the quark exists in this nameset
    virtual bool exists (const long quark) const =0;

    /// @return true if the name exists in this nameset
    virtual bool exists (const String& name) const;

    /// @return an object by quark but do not evaluate
    virtual Object* find (const long quark) =0;

    /// @return an object by name but do not evaluate
    Object* find (const String& name);

    /// remove an object by quark in this nameset
    /// @param quark the binding to remove
    virtual void remove (const long quark) =0;

    /// remove an object by name in this nameset
    /// @param name the binding to remove
    void remove (const String& name);

    /// create a new constant symbol by quark
    /// @param quark  the symbol quark to create
    /// @param object the object to bind
    void symcst (const long quark, Object* object);

    /// create a new constant symbol by name
    /// @param name   the symbol name to create
    /// @param object the object to bind
    void symcst (const String& name, Object* object);

    /// create a new symbol by quark
    /// @param quark  the symbol quark to create
    /// @param object the object to bind
    void symdef (const long quark, Object* object);

    /// create a new symbol by name
    /// @param name   the symbol name to create
    /// @param object the object to bind
    void symdef (const String& name, Object* object);

  private:
    // make the copy constructor private
    Nameset (const Nameset&);
    // make the assignment operator private
    Nameset& operator = (const Nameset&);
  };
}

#endif
