// ---------------------------------------------------------------------------
// - Predasio.cpp                                                            -
// - aleph:sio library - predicates implementation                           -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Cons.hpp"
#include "System.hpp"
#include "Boolean.hpp"
#include "Selector.hpp"
#include "Terminal.hpp"
#include "Predasio.hpp"
#include "Directory.hpp"
#include "InputFile.hpp"
#include "Exception.hpp"
#include "OutputFile.hpp"
#include "InputMapped.hpp"
#include "InputString.hpp"
#include "OutputString.hpp"

namespace aleph {

  // this procedure checks that we have one argument only and returns
  // the evaluated object
  static inline Object* get_obj (Runnable* robj, Nameset* nset, Cons* args,
                                 const String& pname) {
    Object* car = nilp;
    if ((args == nilp) || (args->length () != 1))
      throw Exception ("argument-error", "illegal arguments with predicate",
                       pname);
    car = args->getcar ();
    return (car == nilp) ? nilp : car->eval (robj,nset);
  }

  // odirp: directory object predicate

  Object* asio_odirp  (Runnable* robj, Nameset* nset, Cons* args) {
    Object* obj = get_obj (robj, nset, args, "directory-p");
    bool result = (dynamic_cast <Directory*> (obj) == nilp) ? false : true;
    Object::cref (obj);
    return new Boolean (result);
  }

  // inputp: input stream predicate

  Object* asio_inputp  (Runnable* robj, Nameset* nset, Cons* args) {
    Object* obj = get_obj (robj, nset, args, "input-p");
    bool result =  (dynamic_cast <Input*> (obj) == nilp) ? false : true;
    Object::cref (obj);
    return new Boolean (result);
  }

  // ifilep: input file predicate

  Object* asio_ifilep  (Runnable* robj, Nameset* nset, Cons* args) {
    Object* obj = get_obj (robj, nset, args, "input-file-p");
    bool result = (dynamic_cast <InputFile*> (obj) == nilp) ? false : true;
    Object::cref (obj);
    return new Boolean (result);
  }

  // ofilep: output file predicate

  Object* asio_ofilep  (Runnable* robj, Nameset* nset, Cons* args) {
    Object* obj = get_obj (robj, nset, args, "output-file-p");
    bool result = (dynamic_cast <OutputFile*> (obj) == nilp) ? false : true;
    Object::cref (obj);
    return new Boolean (result);
  }

  // outputp: output stream predicate

  Object* asio_outputp  (Runnable* robj, Nameset* nset, Cons* args) {
    Object* obj = get_obj (robj, nset, args, "output-p");
    bool result = (dynamic_cast <Output*> (obj) == nilp) ? false : true;
    Object::cref (obj);
    return new Boolean (result);
  }

  // inpmapp: input mapped predicate

  Object* asio_inpmapp  (Runnable* robj, Nameset* nset, Cons* args) {
    Object* obj = get_obj (robj, nset, args, "input-mapped-p");
    bool result = (dynamic_cast <InputMapped*> (obj) == nilp) ? false : true;
    Object::cref (obj);
    return new Boolean (result);
  }

  // inpstrp: input string predicate

  Object* asio_inpstrp  (Runnable* robj, Nameset* nset, Cons* args) {
    Object* obj = get_obj (robj, nset, args, "input-string-p");
    bool result = (dynamic_cast <InputString*> (obj) == nilp) ? false : true;
    Object::cref (obj);
    return new Boolean (result);
  }

  // outstrp: output string predicate

  Object* asio_outstrp  (Runnable* robj, Nameset* nset, Cons* args) {
    Object* obj = get_obj (robj, nset, args, "output-string-p");
    bool result = (dynamic_cast <OutputString*> (obj) == nilp) ? false : true;
    Object::cref (obj);
    return new Boolean (result);
  }

  // selectp: select stream predicate

  Object* asio_selectp  (Runnable* robj, Nameset* nset, Cons* args) {
    Object* obj = get_obj (robj, nset, args, "selector-p");
    bool result = (dynamic_cast <Selector*> (obj) == nilp) ? false : true;
    Object::cref (obj);
    return new Boolean (result);
  }

  // itermp: input term predicate

  Object* asio_itermp  (Runnable* robj, Nameset* nset, Cons* args) {
    Object* obj = get_obj (robj, nset, args, "input-term-p");
    bool result = (dynamic_cast <InputTerm*> (obj) == nilp) ? false : true;
    Object::cref (obj);
    return new Boolean (result);
  }

  // otermp: output term predicate

  Object* asio_otermp  (Runnable* robj, Nameset* nset, Cons* args) {
    Object* obj = get_obj (robj, nset, args, "output-term-p");
    bool result = (dynamic_cast <OutputTerm*> (obj) == nilp) ? false : true;
    Object::cref (obj);
    return new Boolean (result);
  } 

  // termp: terminal predicate

  Object* asio_termp  (Runnable* robj, Nameset* nset, Cons* args) {
    Object* obj = get_obj (robj, nset, args, "terminal-p");
    bool result = (dynamic_cast <Terminal*> (obj) == nilp) ? false : true;
    Object::cref (obj);
    return new Boolean (result);
  }

  // filep: check for a regular file

  Object* asio_filep  (Runnable* robj, Nameset* nset, Cons* args) {
    Object* obj  = get_obj (robj, nset, args, "file-p");
    String* sval = dynamic_cast <String*> (obj);
    if (sval == nilp) 
      throw Exception ("argument-error", "invalid object with file-p",
		       Object::repr (obj));
    bool result = System::isfile (*sval);
    Object::cref (obj);
    return new Boolean (result);
  }

  // filep: check for a directory

  Object* asio_dirp  (Runnable* robj, Nameset* nset, Cons* args) {
    Object* obj  = get_obj (robj, nset, args, "dir-p");
    String* sval = dynamic_cast <String*> (obj);
    if (sval == nilp) 
      throw Exception ("argument-error", "invalid object with dir-p",
		       Object::repr (obj));
    bool result = System::isdir (*sval);
    Object::cref (obj);
    return new Boolean (result);
  }  
}
