// ---------------------------------------------------------------------------
// - PrintTable.hpp                                                          -
// - aleph:txt library - print table class definition                        -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  ALEPH_PRINTTABLE_HPP
#define  ALEPH_PRINTTABLE_HPP

#ifndef  ALEPH_BUFFER_HPP
#include "Buffer.hpp"
#endif

namespace aleph {

  /// The PrintTable class is a special class that can be used to format
  /// a table for output. At construction the number of columns is given.
  /// The default constructor create a print table of one column. Any
  /// literal object can be added to the table and are formated according
  /// to a format descriptor. Once the table is created, it can be written
  /// to an output stream or a buffer.
  /// @author amaury darsch

  class PrintTable : public virtual Object {
  private:
    /// the table size
    long d_size;
    /// the number of columns
    long d_cols;
    /// the number of rows
    long d_rows;
    /// the table data
    String** p_data;
    /// the column format size
    long* p_csiz;
    /// the column filling character
    char* p_fill;
    /// the column direction
    bool* p_cdir;
    /// the column maximum width
    long* p_cwth;

  public:
    /// create a default print table
    PrintTable (void);

    /// create a print table with a number of columns
    /// @param cols the number of columns
    PrintTable (const long cols);

    /// create a print table with a number of columns and rows
    /// @param cols the number of columns
    PrintTable (const long cols, const long rows);

    /// destroy this print table
    ~PrintTable (void);

    /// @return the class name
    String repr (void) const;

    /// @return the number of rows
    long getrows (void) const;

    /// @return the number of columns
    long getcols (void) const;

    /// set the column size
    /// @param col  the column to set
    /// @param size the column size to set
    void setsize (const long col, const long size);

    /// @return the column size
    long getsize (const long col) const;

    /// set the column filling character
    /// @param col  the column to set
    /// @param fill the character used for filling
    void setfill (const long col, const char fill);

    /// @return the column fill character
    char getfill (const long col) const;

    /// set the column direction flag
    /// @param col  the column to set
    /// @param cdir the column direction 
    void setcdir (const long col, const bool cdir);

    /// @return the column direction flag
    bool getcdir (const long col) const;

    /// add a row to the table and return the row index
    long add (void);
    
    /// set a value by row and column
    /// @param row the table row
    /// @param col the table column
    /// @param val the table value
    void set (const long row, const long col, const String& val);

    /// set a literal by row and column
    /// @param row the table row
    /// @param col the table column
    /// @param obj the literal object to set
    void set (const long row, const long col, Literal* obj);

    /// get a value by row and column
    /// @param row the table row
    /// @param col the table column
    String get (const long row, const long col) const;

    /// @return a formatted row suitable for printing
    String format (const long row) const;

    /// format the table into a buffer
    void format (Buffer& buffer) const;

    /// format the table for an output stream
    void format (Output& os) const;

    /// resize this table
    /// @param size the new table size
    void resize (const long size);

    /// generate a new print table
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);

    /// apply this print table with a set of arguments and a quark
    /// @param robj   robj the current runnable
    /// @param nset   the current nameset    
    /// @param quark  the quark to apply these arguments
    /// @param argv   the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
		   Vector* argv);

  private:
    // make the copy constructor private
    PrintTable (const PrintTable&);
    // make the assignment operator private
    PrintTable& operator = (const PrintTable&);
  };
}

#endif
