# $Id: SMTP.pm,v 1.32 2004/07/27 13:32:52 braeucup Exp $

#
# MTA module for pure SMTP setup
#

package AMAVIS::MTA::SMTP;
use strict;
use vars qw($VERSION);
$VERSION='0.1';

use AMAVIS;
use AMAVIS::Logging;
use IO::File;
use File::Path;

use File::Copy;
use Sys::Hostname;

# For receiving mail
use IO::Socket;
use IO::Socket::INET;

# For sending mail
use Net::SMTP;

use POSIX qw(setsid);

use vars qw(
	    $cfg_x_header
	    $cfg_x_header_tag
	    $cfg_x_header_line

	    $cfg_smtp_in
	    $cfg_smtp_out
	    $cfg_smtp_port_in
	    $cfg_smtp_port_out

	    $cfg_smtp_session_timeout

	    $cfg_daemon
	    $cfg_pidfile
	    $hostname

	    $smtpserver
	    $conn

	    $saved_args
	    $mta_result

	    $server_pid
	    $running
	    $signame

	    $smtpclient
	   );

sub init {
  my $class = shift;
  my $args = shift;

  $cfg_smtp_in = ($AMAVIS::cfg->val('SMTP', 'input address') or '127.0.0.1');
  $cfg_smtp_out = ($AMAVIS::cfg->val('SMTP', 'output address') or '127.0.0.1');
  $cfg_smtp_port_in = ($AMAVIS::cfg->val('SMTP', 'input port') or '10025');
  $cfg_smtp_port_out = ($AMAVIS::cfg->val('SMTP', 'output port') or '10026');

  if ((!defined $cfg_smtp_port_in ) &&
      (!defined $cfg_smtp_port_out )) {
    writelog($args,LOG_CRIT,__PACKAGE__.
	     ": SMTP input and output ports must be specified.");
    return 0;
  }

  $cfg_daemon = ($AMAVIS::cfg->val('SMTP', 'daemon') or 'yes');
  $cfg_pidfile = $AMAVIS::cfg->val('SMTP', 'pidfile')
	  || '/var/run/amavis-ng/amavisd.pid';

  # Session timeout:
  # undef: accept message immediately
  # =0:    no timeout. Response is sent when We Are Done.
  # >0:    timeout is set to n seconds. After n seconds, processing 
  #        is aborted if it has not been finished.
  $cfg_smtp_session_timeout=$AMAVIS::cfg->val('SMTP','session timeout');

  writelog($args,LOG_DEBUG,__PACKAGE__.
	   ": Input  $cfg_smtp_in:$cfg_smtp_port_in");
  writelog($args,LOG_DEBUG,__PACKAGE__.
	   ": Output $cfg_smtp_out:$cfg_smtp_port_out");

  $hostname=hostname();

  my $pid;

  if ($cfg_daemon eq 'yes') {
    (open(STDIN, "< /dev/null") && open(STDOUT, "> /dev/null") && open(STDERR, "> /dev/null")) || do {
      writelog($args,LOG_ERR,__PACKAGE__.
	       ": Error closing stdin, stdout, or stderr: $!");
      return 0;
    };
    if (!defined ($pid = fork())) {
      writelog($args,LOG_ERR, __PACKAGE__.": fork() failed.");
      return 0;
    }
    # If all went well...
    if ($pid) {
      # We are the parent
      writelog($args,LOG_DEBUG, __PACKAGE__.
	       ": fork() successful, child's PID=$pid.");
      exit 0;
    }
  }

  # We are the child.
  # So we become a daemon.

  $smtpserver = IO::Socket::INET->new (
				       LocalAddr=>$cfg_smtp_in,
				       LocalPort=>$cfg_smtp_port_in,
				       Listen=>20,
				       Proto=>'tcp',
				       ReuseAddr=>1
				      )
    || do {
      writelog($args,LOG_ERR, __PACKAGE__.
	       ": Unable to create SMTP server on $cfg_smtp_in:".
	       "$cfg_smtp_port_in: $!");
      return 0;
    };

  # Enter chroot environment if desired
  if ($cfg_chroot) {
    chroot($cfg_chroot)
      or die "Error in chroot($cfg_chroot): $!\n";
  }

  # Drop privileges if desired
  if ($> == 0) {
    if (defined $cfg_gid) {
      writelog($args,LOG_DEBUG,__PACKAGE__.": Dropping GID");
      $)=$cfg_gid;
      if ($) != $cfg_gid) {
	writelog($args,LOG_ERR, __PACKAGE__.": Can't drop GID to $cfg_gid");
	die;
      }
    }
    if (defined $cfg_uid) {
      writelog($args,LOG_DEBUG,__PACKAGE__.": Dropping UID");
      $>=$cfg_uid;
      if ($> != $cfg_uid) {
	writelog($args,LOG_ERR, __PACKAGE__.": Can't drop UID to $cfg_uid");
	die;
      }
    }
  }

  setsid() or do {
    writelog($args,LOG_ERR,__PACKAGE__.": setsid() failed: $!");
    return 0;
  };
  chdir("/");

  $server_pid=$$;

  # crate PID file.
  my $pidfile=IO::File->new(">$cfg_pidfile");
  unless (defined $pidfile) {
    writelog($args,LOG_ERR, __PACKAGE__.
	     ": Unable to create PID file $cfg_pidfile: $!");
    return 0;
  }
  else {
    ($pidfile->print("$$\n") && $pidfile->close()) || do {
      writelog($args,LOG_ERR, __PACKAGE__.
	       ": Unable to write to PID file: $!");
    }
  }
  $0='amavisd';

  if (($AMAVIS::cfg->val('global', 'x-header') || '') eq 'true') {
    $cfg_x_header = 1
  };
  $cfg_x_header_tag = $AMAVIS::cfg->val('global', 'x-header-tag');
  $cfg_x_header_line = $AMAVIS::cfg->val('global', 'x-header-line');

  $SIG{TERM} = \&inthandler;

  writelog($args,LOG_DEBUG,__PACKAGE__." initialized.");
  # Return successfully
  return 1;
}

sub cleanup {
  my $self = shift;
  my $args = shift;
  if ($$ == $server_pid) {
    writelog($args,LOG_DEBUG,__PACKAGE__.
	     ": Received SIG$signame. Cleaning up.");
    unlink $cfg_pidfile;
  }
  return 1;
}

# Create temp dir and write mail
sub get_directory($) {
  my $self = shift;
  my $args = shift;

  my $prefix = "$AMAVIS::cfg_unpack_dir/amavis-unpack-";
  my $i = 0;
  my $message_id;

  # Main loop for accepting SMTP connections
  while(eval{$conn = $smtpserver->accept();}) {
    my $pid;
    writelog($args,LOG_DEBUG, "Accepting connection.");
    if (!defined ($pid = fork)) {
      writelog($args,LOG_ERR, "fork() failed.");
      $conn->print("421 $hostname AMaViS Service not available, closing transmission channel\r\n");
      $conn->close();
      next;
    }
    # If all went well...
    if ($pid) {
      # We are the parent
      # The parent will (should) only have children that have been
      # forked for accepting socket connections.
      $SIG{CHLD} = 'IGNORE';
      writelog($args,LOG_DEBUG, "fork() successful, child's PID=$pid.");
      $conn->close();
      next;
    }
    else {
      # We are the child
      # Our children won't be automatically reaped.
      $SIG{CHLD} = 'DEFAULT';
      $SIG{TERM} = 'IGNORE';

      # Empty in-core log
      $$args{'log'} = '';

      # Make sure that no result has been set for the message.
      undef $mta_result;

      # We have to set up a handler if we
      # * want the client to retransmit the message in case 
      #   of a processing error
      #   AND
      # * set a limit for a timeout.
      if ((defined $cfg_smtp_session_timeout) &&
	  ($cfg_smtp_session_timeout > 0)) {
	# Save $args
	$saved_args=$args;
	$SIG{'ALRM'}=\&alarmhandler;
	# Setting an alarm can result in a period up to one second
	# shorter than $cfg_smtp_session_timeout.
	alarm($cfg_smtp_session_timeout);
      }

      # Create temp directory. Try creating $prefix[date]-[pid] 
      # If unsuccessful after 10 tries, abort
      while (1) {
	$message_id = sprintf("%.8x-%.4x",time,$$);
	unless (defined mkpath ($prefix.$message_id, 0, 0770)) {
	  if (++$i > 10) {
	    writelog($args,LOG_ERR,
		     __PACKAGE__.": Unable to create unpacking directory.");
	    # We "return" to AMAVIS.pm, telling it that message
	    # unpacking failed
	    return 0;
	  }
	  else {
	    next;
	  }
	}
	last;
      }

      $$args{'message_id'}=$message_id;
      my $directory = $prefix.$message_id;
      mkdir "$directory/parts", 0777;
      $$args{'directory'} = $directory;

      # email.txt will be written here.
      if (handle_smtp_connection($conn,$args)) {
      }
      else {
	writelog($args,LOG_ERR,__PACKAGE__.
		 ": Incoming SMTP session failed");
	return 0;
      };

      # Return successfully
      return 1;
    }
  }

  # We reach this point only if there was a problem in the accept loop. 
  $$args{'directory'}='END';
  return 0;
}

# Called from within AMAVIS.pm to continue message delivery
sub accept_message( $ ) {
  my $self = shift;
  my $args = shift;

  writelog($args,LOG_INFO, __PACKAGE__.": Accepting message");
  $mta_result='accept';

  my $result;

  if ( $#{$$args{'recipients'}} < 0) {
    writelog($args, LOG_INFO, __PACKAGE__.
	     ": Back connection SMTP server did not accept any of our recipients. Not sending message.");
    $smtpclient->quit;
    return 1;
  }

  writelog($args,LOG_DEBUG, "Sending DATA");
  $result=$smtpclient->data();
  return 0 unless ($result);
  my $headers=1;
  my $fh = $$args{'filehandle'};
  while (<$fh>) {
    chomp;
    # Insert X- header at the end of the headers.
    if ($headers && /^\s*$/) {
      $headers=0;
      if ($cfg_x_header) {
	$result=$smtpclient->datasend(("$cfg_x_header_tag: $cfg_x_header_line\n"));
	return 0 unless ($result);
      }
    }
    $result=$smtpclient->datasend("$_\n");
    return 0 unless ($result);
  }
  $result=$smtpclient->dataend();
#  return 0 unless ($result);
  writelog($args,LOG_DEBUG, "DATA done.");
  $smtpclient->quit();

  # Tell client that we have taken responsibility for the message
  if (defined $cfg_smtp_session_timeout) {
    my $result=$conn->print("250 I got it. Seems virus-free.\r\n".
			    "421 Closing connection\r\n");
    $conn->close();
    # We won't care if the timeout alarm hits us now.
    $SIG{'ALRM'}='IGNORE';
    # If the client has disconnected, it should try to retransmit.
    unless ($result) {
      writelog($args,LOG_ERR,__PACKAGE__.
	       ": Client has dropped connection on us.");
      $conn->close();
      return 0;
    }
  }

  # Return successfully
  return 1;
}

# Called from within AMAVIS.pm to throw message away
sub drop_message( $ ) {
  my $self = shift;
  my $args = shift;
  writelog($args,LOG_WARNING, __PACKAGE__.": Dropping message (Message-ID: ".$$args{'Message-ID'}.")");

  $mta_result='drop';

  if (defined $cfg_smtp_session_timeout) {
    # We won't care if the timeout alarm hits us now.
    $SIG{'ALRM'}='IGNORE';
    writelog($args,LOG_DEBUG, __PACKAGE__.
	     ": Accepting message at SMTP level, but DROPPING it");
    my $result=$conn->print("250 I WILL DROP THIS MESSAGE.\r\n".
			    "421 Closing connection\r\n");
    # If the client has disconnected, it should try to retransmit the
    # message (RfC1047)
    unless ($result) {
      writelog($args,LOG_ERR,__PACKAGE__.
	       ": Client has dropped connection on us.");
    }
    $conn->close();

    $smtpclient->quit();
  }

  # Return successfully
  return 1;
}

# Called from within AMAVIS.pm to freeze message delivery
sub freeze_message( $ ) {
  my $self = shift;
  my $args = shift;
  writelog($args,LOG_WARNING, __PACKAGE__.": Freezing message");

  $mta_result='freeze';

  # First try to put the message into the problems directory.
  if (AMAVIS->quarantine_problem_message($args)) {
    return 1;
  }
  # If that fails, reject the message
  else {
    writelog($args,LOG_ERR, __PACKAGE__.
	     ': Unable to put message into problem dir ');
    if (defined $cfg_smtp_session_timeout) {
      # Instead of freezing, reject the message at SMTP level.
      writelog($args,LOG_DEBUG, __PACKAGE__.
	       ": Rejecting message at SMTP level with temporary failure");
      my $result=$conn->print("453 Putting message into problems directory failed.\r\n");
      $conn->close();
      # We won't care if the timeout alarm hits us now.
      $SIG{'ALRM'}='IGNORE';
      # If the client has disconnected, it should try to retransmit.
      unless ($result) {
	writelog($args,LOG_ERR,__PACKAGE__.
		 ": Client has dropped connection on us.");
	$conn->close();
	return 0;
      }
    }
    else {
      writelog($args,LOG_ERR,__PACKAGE__.
	       ": Couldn't do anything else.");
      return 0;
    }
  }
  $smtpclient->quit();
}

# Called from Notify::*.pm, i.e. for sending warining messages
sub send_message( $$$ ) {
  my $self = shift;
  my $args = shift;
  my $message = shift;
  my $sender = shift;
  my @recipients = @_;
  writelog($args,LOG_DEBUG, __PACKAGE__.": Sending mail from $sender to ".
	   join(', ',@recipients));

  my $smtpclient = new Net::SMTP("$cfg_smtp_out:$cfg_smtp_port_out",
				 Hello => 'localhost',
				 Timeout => 30);
  unless (defined $smtpclient) {
    writelog($args,LOG_ERR, __PACKAGE__.
	     ": Unable to connect to SMTP server on ".
	     "$cfg_smtp_out:$cfg_smtp_port_out.");
    return 0;
  }


  # FIXME: Error checking
  my $result;

  writelog($args,LOG_DEBUG, "Setting sender: $sender");
  if ($sender eq '<>') {$sender=''}
  $result=$smtpclient->mail($sender);
  unless ($result) {
    writelog($args,LOG_ERR, __PACKAGE__.
	     ": Error while MAIL FROM:");
    return 0;
  }
  writelog($args,LOG_DEBUG, "Setting recipients: ".
	   join(', ',@recipients));
  $result=$smtpclient->recipient(@recipients);
  unless ($result) {
    writelog($args,LOG_ERR, __PACKAGE__.
	     ": Error while RCPT TO:");
    return 0;
  }
  writelog($args,LOG_DEBUG, "Sending DATA");
  $result=$smtpclient->data($message); #split(/\n/m, $message));
  unless ($result) {
    writelog($args,LOG_ERR, __PACKAGE__.
	     ": Error while DATA");
    return 0;
  }
  $smtpclient->quit();

  # Return successfully
  return 1;
}

sub handle_smtp_connection {
  my $conn = shift;
  my $args = shift;

  my $from=undef;
  my @to=();

  # smtpclient result
  my $result;

  # Open back-connection to MTA
  $smtpclient = new Net::SMTP("$cfg_smtp_out:$cfg_smtp_port_out",
			      Hello => 'localhost',
			      Timeout => 30) || do {
    $conn->print("421 $hostname AMaViS/SMTP could not open back connection to MTA, closing transmission channel\r\n");
    writelog($args,LOG_ERR, __PACKAGE__.
	     ": Unable to connect to SMTP server on $cfg_smtp_out:$cfg_smtp_port_out.");
    return 0;
  };

  # Send an SMTP greeting
  $conn->print("220 AMaViS SMTP Ready. Spam me gently.\r\n");

 SMTP:while (<$conn>) {
    # Remove leading, trailing spaces and \r\n
    chomp;
    s/^\s+//;
    s/\s+$//;
    foreach ($_) {
      # RFC 822 4.1.1
      /^HELO (.*)/i && do {
	$conn->print("250 Nice to meet you, $1.\r\n");
	next;
      };
      /^MAIL FROM:(.*)$/i && do {
	$from = p_addr($1);
	@to=();

	writelog($args,LOG_DEBUG, "Setting sender: $from");
	$result=$smtpclient->mail($from);
	if ($result) {
	  $conn->print("250 Ok.\r\n");
	} else {
	  $conn->print("500 Back connection did not accept $from as sender.\r\n");
	  $from=undef;
	}
	next;
      };
      /^RCPT TO:(.*)$/i && do {
	if (defined $from) {
	  $result=$smtpclient->recipient(p_addr($1));
	  if ($result) {
	    push @to, p_addr($1);
	    $conn->print("250 Ok.\r\n");
	  }
	  else {
	    $conn->print("503 Back connection refused recipient.\r\n");
	  }
	}
	else {
	  $conn->print("503 Who are you?\r\n");
	}
	next;
      };
      /^DATA$/i && do {
	# We will not accept data without either sender or recipients.
	unless (defined $from) {
	  $conn->print("503 Who are you?\r\n");
	  next;
	}
	unless ($#to >= 0) {
	  $conn->print("503 Give me at least ONE recipient\r\n");
	  next;
	}
	$conn->print("354 Just go ahead. Don't forget to end with a dot\r\n");
		
	# Open message file that is later going to be disected and scanned
	my $output = IO::File->new("+>$$args{'directory'}/email.txt");

	my $done=undef;
	my $headers=1;
      DATA:while (<$conn>) {
	  s/\r\n$//;
	  if(/^\.$/) {
	    writelog($args,LOG_DEBUG,__PACKAGE__.
		     ": Received end of DATA.");
	    $done = 1;
	    last DATA;
	  }
	  # chomp;
	  # s/\r\n//;
	  # Replace '..' at beginning of line with '.'
	  s/^\.(.+)$/$1/;
	  if (/^$/) {
	    $headers=0;
	  }
	  if (/^Message-ID:\s*(.*)\s*$/i) {
	    $$args{'Message-ID'} = $1;
	  }
	  if ($headers==1) {
	    $$args{'headers'}.="$_\n";
	  }
	  $output->print("$_\n");
	}
	# Message file has been written, reset file pointer and put it into
	# the record.
	$output->seek(0,0);
	$$args{'filehandle'} = $output;

	unless (defined $done) {
	  $conn->print("550 DATA didn't end with a dot\r\n");
	  $conn->close();
	  writelog($args,LOG_ERR,__PACKAGE__.
		   ": DATA didn't end with a dot");
	  return 0;
	}
	# Body and dot have been received.
	# If no timeout is set, the message is acknowledged
	# immediately.
	unless (defined $cfg_smtp_session_timeout) {
	  $conn->print("250 Message received. Processing.\r\n".
		       "421 Closing connection\r\n");
	  $conn->close();
	}
	last SMTP;
      };
      /^RSET/i && do {
	$from=undef;
	@to=();
	$conn->print("250 Okay, back to Square One\r\n");
	next;
      };
      # Do not confirm that recipient exists.
      /^VRFY/i && do {
	$conn->print("252 No.\r\n");
	next;
      };
      # Do not confirm that recipient expands to a mailing list.
      /^EXPN/i && do {
	$conn->print("252 No.\r\n");
	next;
      };
      /^HELP/i && do {
	$conn->print("214 Help yourself\r\n");
	next;
      };
      /^NOOP/i && do {
	$conn->print("250 *Yawn*\r\n");
	next;
      };
      /^QUIT/i && do {
	$conn->print("221 See you later\r\n");
	$conn->close();
	return 0;
      };
      # If none of the above commands was recognized, tell the other
      # side so.
      $conn->print("500 Unknown command?\r\n");
    }
  }

  $$args{'sender'} = $from;
  $$args{'sender'} = "<>" if (!$$args{'sender'});
  push @{$$args{'recipients'}}, @to;
  return 1;
}

sub alarmhandler {
  # We don't do anything if the timeout hits after a decision has been
  # made about the message's fate.
  unless (defined $mta_result) {
    writelog($saved_args,LOG_ERR, __PACKAGE__.": SMTP timeout reached.");
    $conn->print("453 Error in processing the message (timeout reached)\r\n");
    # FIXME How to clean up?
    AMAVIS->cleanup($saved_args);
    die("SMTP timeout reached");
  }
}

sub inthandler {
  $signame = shift;
  $running=0;
  die "Somebody sent me a SIG$signame";
}

sub p_addr {
  (my $addr = shift || "") =~ s/^\s*<?\s*|\s*>?\s*$//sg ;
  "<$addr>";
}

1;
