# $Id: Recipients.pm,v 1.17 2003/03/05 20:21:19 bengen Exp $

package AMAVIS::Notify::Recipients;
use strict;
use vars qw($VERSION);
$VERSION='0.1';
use POSIX qw(strftime setlocale LC_TIME);

use AMAVIS;
use AMAVIS::Logging;

use vars qw(
	    $cfg_sender
	    $cfg_internal_domain
	    $cfg_internal_domain_map
	   );

sub init {
  my $self = shift;
  my $args = shift;
  writelog($args,LOG_DEBUG,__PACKAGE__." initialized.");
  $cfg_sender = ($AMAVIS::cfg->val('Notify', 'mail from') or 'postmaster');
  $cfg_internal_domain = ($AMAVIS::cfg->val('Notify', 'local domain') or '');

  $cfg_internal_domain_map = load_local_domains($args);

  # Return successfully
  return 1;
}

sub notify {
  my $self = shift;
  my $args = shift;
  my @recipients;
  setlocale(LC_TIME, 'C');
  my $date = strftime("%a, %d %b %Y %H:%M:%S %z",localtime);
  setlocale(LC_TIME, '');
  if (!defined($cfg_internal_domain) && !defined($cfg_internal_domain_map)) {
    writelog($args, LOG_WARNING, "Notify for Recipients requested but no local domain defined");
    return 0;
  }

  my $orig_sender = $$args{'sender'};
  if ($orig_sender eq '<>') {
    $orig_sender = '(empty address)';
  }

  foreach my $recipient (@{$$args{'recipients'}}) {

    # Recipient domain matches internal domain regexp ?
    if (defined($cfg_internal_domain)) {
      if ($recipient=~/\@$cfg_internal_domain(?:[> ].*)?/i) {
          writelog($args,LOG_DEBUG, "Warning recipient: $recipient.");
          push @recipients, $recipient;
          next;
      }
    }

    # Recipient domain included in internal domain list ?
    if (defined($cfg_internal_domain_map)) {

      # Extract domain name from recipient
      if ($recipient =~ /@([a-zA-Z0-9\.\-]+)/) {
        my $rcpt_domain = $1;

        # Lookup the domain
        if ($cfg_internal_domain_map->{$rcpt_domain}) {
          writelog($args,LOG_DEBUG, "Warning recipient: $recipient.");
          push @recipients, $recipient;
          next;
        }
      }
      else {
        writelog($args, LOG_WARNING, "Can't extract domain name from recipient: $recipient");
      }
    }

    # If this point is reached, recipient domain is not local
    writelog($args,LOG_DEBUG, "Not warning recipient $recipient: non-local user");
  }

  # Success if nobody has to be warned.
  return 1 if ($#recipients == -1);
  my $recipients=join(',', @recipients);

  my $message = <<"EOF";
From: $cfg_sender
To: $recipients
Date: $date
Subject: Virus in incoming mail for you from $orig_sender.

Our virus checker has found potentially malicious code in a mail from
$orig_sender addressed to you. Delivery has been stopped. For further
questions, please contact $cfg_sender.

EOF
  if (defined $$args{'quarantine_file'}) {
    $message.= << "EOF";

The message has been quarantined as $$args{'quarantine_file'}.

EOF
  };
  foreach my $scanner (@{$$args{'virus_scanners'}}) {
    $message.="\n$scanner found:\n";
    foreach my $virus (@{$$args{'found_viruses'}{$scanner}}) {
      $message.=" $virus\n";
    }
  }

  $message.="\n".('-'x 72)."\nMessage headers follow:\n";
  $message.=$$args{'headers'};

  $AMAVIS::mta->send_message($args, $message, '<>', @recipients)
    or writelog($args,LOG_ERR,"Error notifying recipients: $recipients");;
  return 1;
}

sub notify_unpack {
  return 1;
}

sub load_local_domains {
  my $self = shift;
  my $args = shift;

  my $domain_file = $AMAVIS::cfg->val('Notify', 'local domain-list-file');

  return undef unless defined($domain_file);

  local *IN;
  if (!open(IN, "<$domain_file")) {
    writelog($args,LOG_WARNING, "Can't open local domain list file for reading: $domain_file: $!");
    return undef;
  }

  my %domain_map;

  my $line_number = 0;
  while(<IN>) {
    chomp;
    ++$line_number;

    next if (/^\s*(\#|$)/); # skip #-comments and blank lines

    if (exists $domain_map{$_}) {
      writelog($args,LOG_WARNING, "Redefinition of local domain '$_' in file $domain_file:$line_number");
      next;
    }

    $domain_map{$_} = 1;
  }

  close(IN) || writelog($args,LOG_WARNING, "Failure closing local domain list file: $domain_file: $!");

  \%domain_map;
}

1;

