/* Copyright (C) 2002, 2003, 2004, 2005 Jan Wedekind.
   This file is part of the recipe database application AnyMeal.

   AnyMeal is free software; you can redistribute it and/or modify it under
   the terms of the GNU GENERAL PUBLIC LICENSE as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   AnyMeal is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTIBILITY or FITNESS
   FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
   details.

   You should have received a copy of the GNU General Public License along
   with AnyMeal; if not, contact one of the authors of this software. */
#include "include.hpp"
#include <fstream>
#include <klocale.h>
#include <sstream>
#include "chainedCompiler.hpp"
#include "cookBook.hpp"
#include "cookBookDeprecated.hpp"
#include "utils.hpp"
#include "xmlDocument.hpp"
#include "xsu.hpp"

using namespace std;

CookBook::CookBook( DatabasePtr _database,
                    const string &inputXSLFileName,
                    const string &outputXSLFileName,
                    bool performVersionCheck ) throw (Error):
  database(_database),
  xmlLayer( createXMLLayer( _database, inputXSLFileName, outputXSLFileName ) )
{
  if ( performVersionCheck ) {
    int version = getVersion();
    ERRORMACRO( version >= 13, Error, ,
                i18n( "Update for database versions 0.1 to 0.12 is not "
                      "supported." ) );
    ERRORMACRO( version <= DATABASEVERSION, Error, ,
                i18n( "Recipe database is of version %1. This software can "
                      "only handle databases up to version %2. You need to "
                      "update AnyMeal to access this database." ).
                arg( versionText( version ) ).
                arg( versionText( DATABASEVERSION ) ) );
    ERRORMACRO( version == DATABASEVERSION, CookBookDeprecated, ( version ),
                i18n( "Recipe database appears to be of "
                      "version %1. Client requires database "
                      "to be of version %2." ).
                arg( versionText( version ) ).
                arg( versionText( DATABASEVERSION ) ) );
  };
}

int CookBook::getVersion(void) const throw (Error)
{
  XMLDocument xmlDocument( "" );
  xmlDocument.fromString
    ( xmlLayer->translate
      ( "<?xml version='1.0' encoding='UTF-8'?><version/>" ) );
  int
    version = atoi( xmlDocument.getDocumentElement().selectNode( "version" ).
                    getNodeText().c_str() );
  return version;
}

void CookBook::update(void) throw (Error)
{
  // Read version number.
  int version = getVersion();
  
  // Check for minimum version requirement.
  ERRORMACRO( version >= 13, Error, ,
              i18n( "Update for database versions 0.1 to 0.12 is not "
                    "supported." ) );
  
  // Check against maximum version constraint.
  ERRORMACRO( version <= DATABASEVERSION, Error, ,
              i18n( "Recipe database is of version %1. This software can "
                    "only handle databases up to version %2. You need to "
                    "update AnyMeal to access this database." ).
              arg( versionText( version ) ).
              arg( versionText( DATABASEVERSION ) ) );

  // Update database.
  stringstream query;
  query << "<?xml version='1.0' encoding='UTF-8'?>" << endl
        << "<update><version>" << version << "</version></update>"
        << endl;
  ofstream nullDevice( "/dev/null" );
  xmlLayer->translate( query, nullDevice );
}

CompilerPtr CookBook::createXMLLayer( DatabasePtr _database,
                                      const string &inputXSLFileName,
                                      const string &outputXSLFileName )
{

  XSLCompilerPtr
    sqlIn( new XSLCompiler( inputXSLFileName,
                            FormatterListener::OUTPUT_METHOD_TEXT,
                            SQLENCODING, "" ) ),
    sqlOut( new XSLCompiler( outputXSLFileName,
                             FormatterListener::OUTPUT_METHOD_XML,
                             "UTF-8", "" ) );
  CompilerPtr xmlLayer
    ( new ChainedCompiler( sqlIn,
                           CompilerPtr( new ChainedCompiler
                                        ( CompilerPtr
                                          ( new XSU( _database ) ),
                                          sqlOut ) ) ) );
  return xmlLayer;
}
