/* Copyright (C) 2002, 2003, 2004 Jan Wedekind.
   This file is part of the recipe database application AnyMeal.

   AnyMeal is free software; you can redistribute it and/or modify it under
   the terms of the GNU GENERAL PUBLIC LICENSE as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   AnyMeal is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTIBILITY or FITNESS
   FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
   details.

   You should have received a copy of the GNU General Public License along
   with AnyMeal; if not, contact one of the authors of this software. */
#ifndef MEALMASTERCOMPILER_HPP
#define MEALMASTERCOMPILER_HPP

#include <boost/shared_ptr.hpp>
#include <map>
#include <string>
#include "compiler.hpp"
#include "mealMasterParseErrorHandler.hpp"

/** @addtogroup compilers
    @{ */
/** Scanner for converting Mealmaster recipes to XML.
    Scanning Mealmaster-recipes is awkward, because there is no context-free
    grammar. Exspecially the transition from the ingredients-part to the
    instructions-part of the recipe is difficult to implement.

    Despite of its poor design, the file-format of Mealmaster is widespread.
    
    This class converts the contents of the input to XML and writes it to the
    output-stream.
    
    Here's an example of a Mealmaster file:
    \verbatim
MMMMM----------MEAL-MASTER Format (B&S REZWIN Version 1.0.5.1)--------

     Title: Obst-Baiser-Torte
Categories: Kuchen, Gebaeck
  Servings: 1 Keine Angabe

MMMMM-------------------------Ruehrteig--------------------------------
    150 g  weiche Butter
    150 g  Zucker
      3    Eier
    150 g  Mehl
      1 ts Backpulver

MMMMM--------------------------Baiser---------------------------------
      4    Eiweiss
    150 g  Zucker
     50 g  Mandelblaettchen

MMMMM-------------------------Fuellung---------------------------------
    370 g  Sauerkirschen (Glas) 
           -abgetropft
  1 1/2 tb Speisestaerke
    1/4 l  Kirschsaft
    250 g  Sahne
      1 tb Zucker
      1 pk Sahnesteif

Zwei Springformen (26 cm) mit Butterpapier ausreiben. Ofen auf 200°
vorheizen.

MMMMM-------------------------Ruehrteig--------------------------------
Butter, Zucker und Eier schaumig ruehren, Mehl mit Backpulver daruebersieben
und untermischen. Den Teig in die zwei Springformen fuellen und
glattstreichen. 15 Minuten vorbacken.

MMMMM--------------------------Baiser---------------------------------
Fuer die Baisermasse Eiweiss steif schlagen, Zucker loeffelweise einrieseln
lassen und sehr fest aufschlagen. Die Baisermasse auf die vorgebackenen
Tortenboeden streichen. Mandelblaettchen daraufstreuen. Weitere 10-15
Minuten backen. Auf Gitter auskuehlen lassen.

MMMMM-------------------------Fuellung---------------------------------
Fuer die Fuellung Speisestaerke mit etwas Kirschsaft glattruehren. Den
restlichen Saft zum Kochen bringen, die Speisestaerke einruehren und
aufkochen. Die Sauerkirschen untermischen, etwas abkuehlen lassen, auf
einen Tortenboden streichen.

Sahne mit Zucker und Sahnesteif steif schlagen, auf die Fuellung streichen. Den
zweiten Tortenboden in 12 Stuecke schneiden, auf der Fuellung wieder zur
Tortenscheibe zusammensetzen. Gut durchkuehlen.

Author: Jan Wedekind
:Quelle: Ich helf dir Kochen

MMMMM

Legende:
 tb: Essloeffel.
 ts: Teeloeffel
 pk: Packung
\endverbatim

    The compiler also has to interpret and filter <EM>UTF-8 sequences</EM>.
    The following byte sequences are used to represent a character. The
    sequence to be used depends on the Unicode number of the character:    
    <TABLE BORDER=1>
    <TR><TD>U-00000000 - U-0000007F:
    <TD>0<I>xxxxxxx</I>
    <TR><TD>U-00000080 - U-000007FF:
    <TD>110<I>xxxxx</I> 10<I>xxxxxx</I>
    <TR><TD>U-00000800 - U-0000FFFF:
    <TD>1110<I>xxxx</I> 10<I>xxxxxx</I> 10<I>xxxxxx</I>
    <TR><TD>U-00010000 - U-001FFFFF:
    <TD>11110<I>xxx</I> 10<I>xxxxxx</I> 10<I>xxxxxx</I> 10<I>xxxxxx</I>
    <TR><TD>U-00200000 - U-03FFFFFF:
    <TD>111110<I>xx</I> 10<I>xxxxxx</I> 10<I>xxxxxx</I> 10<I>xxxxxx</I>
    10<I>xxxxxx</I>
    <TR><TD>U-04000000 - U-7FFFFFFF:
    <TD>1111110<I>x</I> 10<I>xxxxxx</I> 10<I>xxxxxx</I> 10<I>xxxxxx</I>
    10<I>xxxxxx</I> 10<I>xxxxxx</I>
    </TABLE>

    For security reasons, a UTF-8 decoder must not accept UTF-8 sequences that
    are longer than necessary to encode a character. All overlong UTF-8
    sequences start with one of the following byte patterns:
    <TABLE BORDER=1>
    <TR><TD>1100000<I>x</I> (10<I>xxxxxx</I>)
    <TR><TD>11100000 100<I>xxxxx</I> (10<I>xxxxxx</I>)
    <TR><TD>11110000 1000<I>xxxx</I> (10<I>xxxxxx</I> 10<I>xxxxxx</I>)
    <TR><TD>11111000 10000<I>xxx</I> (10<I>xxxxxx</I> 10<I>xxxxxx</I>
    10<I>xxxxxx</I>)
    <TR><TD>11111100 100000<I>xx</I> (10<I>xxxxxx</I> 10<I>xxxxxx</I>
    10<I>xxxxxx</I> 10<I>xxxxxx</I>)
    </TABLE>

    Also note that the code positions U+D800 to U+DFFF (UTF-16
    surrogates) as well as U+FFFE and U+FFFF must not occur in normal
    UTF-8 or UCS-4 data. UTF-8 decoders should treat them like malformed
    or overlong sequences for safety reasons.

    <B>At any time there must only be one instance of this class.</B>

    See @ref link "links page" for references on this topics.
    
    @date Thu Mar 28 2005
    @author Jan Wedekind (wedesoft@users.sourceforge.net) */
class MealMasterCompiler: public Compiler
{
public:
  /** Constructor.
      It is possible to limit the number of recipes, which will be parsed
      by a call of \c translate. This is necessary, if this compiler is
      used in a chain with a XSL translator, which will store all data
      in memory before processing it.

      @param _maxRecipes Maximum recipes to parse. Abort afterwards.
      @param _parseErrorHandler Error handler for parse errors.
      @see MealMasterParseErrorHandler
      @see StoreErroneousHandler
      @see XSLCompiler */
  MealMasterCompiler( int _maxRecipes = -1,
                      MealMasterParseErrorHandlerPtr _parseErrorHandler =
                      MealMasterParseErrorHandlerPtr( new MealMasterParseErrorHandler ) );
  /// Destructor.
  virtual ~MealMasterCompiler(void);
  /** Translate recipes.
      If the number of recipes, to be read, was limited by specifying
      \c _maxRecipes with the constructor, the stream will be repositioned
      (because <EM>flex</EM> usually will have read ahead, to fill its buffer).

      The lexer keeps track of its read-position. This was implemented using
      the \c YY_USER_ACTION macro of <EM>flex</EM>. Note that the
      input-position will also change, when \c yymore(), \c yyless(),
      \c unput(), \c input() or \c REJECT are executed.
      
      If no maximum number of recipes was specified in the constructor,
      the \c inputStream will be parsed up to its end.

      @param inputStream Input stream to read mealmaster data from.
      @param outputStream Output stream to write XML data to. */
  virtual void translate( std::istream &inputStream,
                          std::ostream &outputStream ) const
    throw (Error);
  ///
  static MealMasterCompiler *current;
  ///
  std::map< std::string, std::string > inverseUnitMap;
  ///
  void setParseErrorHandler( MealMasterParseErrorHandlerPtr _parseErrorHandler )
    { parseErrorHandler = _parseErrorHandler; }
protected:
  ///
  int maxRecipes;
  ///
  MealMasterParseErrorHandlerPtr parseErrorHandler;
};

///
typedef boost::shared_ptr< MealMasterCompiler > MealMasterCompilerPtr;

///@}

#endif
