/* $Id: ArkImage.h,v 1.16 2003/03/15 00:20:35 zongo Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2002 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef ARK_IMAGE_H
#define ARK_IMAGE_H

#include <Ark/Ark.h>
#include <Ark/ArkMath.h>
#include <Ark/ArkObject.h>

namespace Ark
{

   class Cache;

   //  =======================================================================
   /**
    * An Image is a bitmap which has been loaded in program memory.
    * It can be used directly to create a Texture, before being destroyed, but
    * it  can also be used to generate other bitmaps (semi-procedural texture
    * generation). Unlike textures, images' data can be accessed directly
    * because they are still in program memory; the flip side is that you can't
    * do rendering using Images..
    */
   //  =======================================================================
   class ARK_DLL_API Image : public Object
   {
      public:
	 /// Image format.
	 enum Format
	 {
		 NONE_0 = 0,
		 RGB_888,
		 RGBA_8888,
		 I_8,
		 A_8,
		 IA_88,
		 FORMAT_COUNT
	 };
	 
	  protected:
	 static const int BytesPerPixel[FORMAT_COUNT];
	  public:
	 static int GetBytesPerPixel(Format format) 
	 { return BytesPerPixel[format]; }

	 /// Image repeat mode (used by Ark::Texture).
	 enum RepeatMode
	 {
	    REPEAT,
	    CLAMP,
	 };
	 
	 /// Returns a string describing the object
	 String Dump (bool long_version);
	 
      public:
	 /// Image data. It's a m_Width * m_Height * bpp byte table, with bpp
	 /// corresponding to the number of planes for the format (ie 3 for
	 /// RGB, etc)
	 uchar *m_Data;

	 /// The image width in pixels
	 uint m_Width;
	 
	 /// The image height in pixels
	 uint m_Height;
	 
	 /// The image format. Specifies the number of planes and the number
	 /// of bytes for each of them
	 Format m_Format;
	 
      public:
	 /// Allocate memory for a \c w*h texture, and then copy bitmap
	 /// from \c data  if it isn't null.
	 Image (const String &name, int w, int h, Format format,
		uchar *data = NULL);

	 /// Create a null image.
	 Image ();
	 
	 /// Delete the given image
	 ~Image ();
	 
	 /// Save this image as TGA in a file.
	 ///   \param name is the name of the file
	 void SaveTGA (const String &name);
	 
	 /// Set the image format, and image data if \c data isn't null.
	 bool SetFormat (int w, int h, Format format, uchar *data = NULL);

	/**
	 * Returns the color of a point.
	 */
	 Color GetColor(float u, float v) const;

	 /**
	  * Returns the color of a pixel.
	  *
	  * This only works for RGB or RGBA images.
	  */
	 Color GetPixelColor(int x, int y) const;

	 /**
	  * Returns the intensity of a pixel.
	  */
	 float GetPixelIntensity(int x, int y) const;

	 /**
	  * Returns the alpha value of a pixel.
	  */
	 float GetPixelAlpha(int x, int y) const;

	 /**
	  * Returns the number of bytes per pixel
	  */
	 inline int GetBytesPerPixel() const { return m_BytesPerPixel; }

	  protected:
	 /// Number of bpp
	 int m_BytesPerPixel;
	 
	 /// Gets the byte offset
	 inline int GetPixelOffset(int x, int y) const
	 { return (x + y * m_Width) * m_BytesPerPixel; }

   };

   typedef Ptr<Image> ImagePtr;
   
/* namespace Ark */
}

#endif
