/* $Id: HFAStar.h,v 1.3 2002/10/12 00:58:45 zongo Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2000 The Contributors of the Ark Project
** Copyright (C) 1999 Amit J. Patel
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#ifndef ARK_ASTAR_H
#define ARK_ASTAR_H

#include <Modules/HeightField/ArkHeightField.h>
#include <Modules/HeightField/HFWorld.h>

#include <vector>
#include <algorithm>

#include <math.h>
#include <string.h>


namespace Ark
{

   // Set to true if the propagate_down step is needed (only if best, shortest
   // pathes are needed, which is not really important for the movement of a
   // player)
   extern bool g_PropDown;
   
   struct Coord
   {
	 short X, Y;

	 Coord (short val): X(val), Y(0) {}
	 Coord (): X(0), Y(0) {}
	 Coord (short x, short y): X(x), Y(y) {}
   };
   enum {NONE = -1};

   bool operator == (const Coord& a, const Coord &b);

   class AStar;

// ===========================================================================
// MARK
// ===========================================================================

   struct Marking
   {
	 Coord iPrev;     // !NONE means OPEN || CLOSED
	 short f;         // >= 0 means OPEN
	 short g;         // >= 0 means OPEN || CLOSED
	 Marking(): iPrev(NONE), f(-1), g(-1) {}
   };

// ===========================================================================
// NODES
// ===========================================================================
   struct Node
   {
	 // Location on the map
	 Coord coord;
	 
	 // g represents how far we've gone;
	 // h represents an estimation of how far is left
	 long gval, hval;
	 
      public:
	 Node (): coord (0), gval(0), hval (0) {}
	 ~Node () {}
   };
   
   //bool operator < (const Node& a, const Node& b);
   bool operator > (const Node& a, const Node& b);
   bool operator == (const Node& a, const Node& b);
   typedef std::vector<Node> Container;
   
// ===========================================================================
// MARK ARRAY :  The mark array marks directions on the map.  The direction
//   points to the spot that is the previous spot along the path.  By
//   starting at the end, we can trace our way back to the start, and have a
//   path. It also stores 'f' values for each space on the map.  These are
//   used to determine whether something is in OPEN or not.  It stores 'g'
//   values to determine whether costs need to be propagated down.
// ===========================================================================
   
   class MarkArray
   {
       std::vector <Marking> m_Marks;
	 int m_SizeX;
	 
   public:
	 MarkArray (int sizez);
	 ~MarkArray ();
	 
	 inline Marking &D (const Coord &c)
	 { return m_Marks [c.Y * m_SizeX + c.X]; }
	 
	 // Erase the mark array, for all items in open or visited
	 void Empty (Container &open, Container &visited);
	 void Empty (int n);
   };
   
//  ===========================================================================
// HEURISTIC : The dist virtual function of this class should return an
//   estimation of the cost from a to b.
// ===========================================================================
   
   class ARKHEIGHTFIELD_DLL_API Heuristic
   {
      public:
	 int abort_path;
	 
      public:
	 Heuristic ();
	 virtual int dist(AStar *astar, Coord a, Coord b);
	 virtual int cost(AStar *astar, Coord a, Coord b);
   };
   
//  ==========================================================================
// PATH STATS : Statistics about the path are kept in this structure.
//  ==========================================================================
   struct ARKHEIGHTFIELD_DLL_API PathStats
   {
	 int nodes_searched;
	 int nodes_added;
	 int nodes_removed;
	 int nodes_visited;
	 int nodes_left;
	 int path_length;
	 int path_cost;
	 
	 PathStats()
	    :nodes_searched(0), nodes_added(0), nodes_removed(0),
	     nodes_visited(0), nodes_left(0),
	     path_length(0), path_cost(0)
	 {}
   };
   
//  ==========================================================================
// ASTAR: Here's the class that implements A*.  I take a graph, two nodes of
//   this graph (A and B), and then output the path in the `path' vector,
//   when find_path is called.
//  (A*) code, Copyright (C) 1999 Amit J. Patel
//  ==========================================================================
   class ARKHEIGHTFIELD_DLL_API AStar
   {
	 friend class Heuristic;
	 
      private:
	 // greater<Node> is an STL thing to create a 'comparison' object out
	 // of the greater-than operator.
	 std::greater<Node> m_Comp;
	 
	 MarkArray m_Mark;
	 
	 uchar *m_Costs;
	 int m_SizeX, m_SizeY;
	 
	 Coord m_A, m_B;
	 Heuristic m_Heuristic;
	 
	 // Remember which nodes are in the OPEN set
	 Container m_Open;
	 // Remember which nodes we visited, so that we can clear the mark
	 // array at the end.  This is the 'CLOSED' set plus the 'OPEN' set.
	 Container m_Visited;
	 
      public:
	 PathStats m_Stats;
	 
      public:
	 AStar (uchar *data, int sizex, int sizey);
	 ~AStar ();
	 
	 // Main function:
	 void find_path (const Coord &a, const Coord &b, std::vector<Coord>& path);

	 bool reachable (const Coord &a, const Coord &b);
      
	 // Helpers:
	 void propagate_down( Node H );
	 Container::iterator find_in_open (const Coord &c);
	 
	 inline bool is_valid (const Coord& c)
	 {
	    return ((c.X >= 0) && (c.X < m_SizeX) +
		    (c.Y >= 0) && (c.Y < m_SizeY));
	 }
	 
	 inline bool in_open (const Coord& c)
	 {return m_Mark.D (c).f != -1;}
	 
	 // I'm using a priority queue implemented as a heap.  STL has some
	 // nice heap manipulation functions.  (Look at the source to
	 // `priority_queue' for details.)  I didn't use priority_queue
	 // because later on, I need to traverse the entire data structure to
	 // update certain elements; the abstraction layer on priority_queue
	 // wouldn't let me do that.
	 inline void get_first (Container& v, Node& n)
	 {
	    n = v.front();
		std::pop_heap( v.begin(), v.end(), m_Comp);
	    v.pop_back();
	 }
   };

} // namespace Ark

#endif
