/* $Id: HFWorld.h,v 1.14 2003/03/19 09:25:06 zongo Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2000 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef ARK_HFWORLD_H
#define ARK_HFWORLD_H

#include <Modules/HeightField/ArkHeightField.h>

#include <Ark/ArkMath.h>
#include <Ark/ArkWorld.h>
#include <Ark/ArkConfig.h>

namespace Ark
{

   class AStar;
   class HFRenderer;
   class SkyDome;
   class Quadtree;

   typedef uchar Height;

   /// A specific type of world : an heightfield (terrain)
   class ARKHEIGHTFIELD_DLL_API HeightField : public World
   {
	 friend class Quadtree;
      public:
	 /// The HF offset of the lower corner
	 scalar m_OffsetX;
	 scalar m_OffsetZ;

	 /// The HF X and Z size
	 size_t m_SizeX;
	 size_t m_SizeZ;

	 /// The HF X and Z scale (a point in the HF will represent m_Scale
	 /// meters)
	 scalar m_Scale;

	 /// The HF Y scale
	 scalar m_ScaleY;

      protected:
	 /// Heigh data
	 Height *m_Y;

	 /// Different ground types present in this terrain
	 MaterialList m_Grounds;
     
	 /// Collision list
	 typedef std::vector< Collision > CollisionList;
     
	 /// the GroundMap (as a heightmap, but for the grounds)
	 uchar* m_G;
     
	 /// Pathfinder object
	 AStar *m_Pathfinder;

	 /// Rendering data
	 /// Time of day is between 0.0 and 1.0
	 scalar m_TimeOfDay;
	 /// Horizon colors
	 // scalar* m_HorizonColors
	 /// Fog color
	 Color m_FogColor;
	 /// Fog density
	 scalar m_FogDensity;
	 /// Light ambient color
	 Color m_AmbientColor;
	 
	 /// Quadtree for this heightfield...
	 Quadtree *m_Quadtree;

	 /// Sky box
	 SkyDome* m_SkyDome;

	 /// Configuration read from file
	 Config m_Config;

      public:
	 enum
	 {
	    TDEBUG = 0
	 };

      public:
	 /** Create a new HF :
	  * \param cache the object cache needed to load textures.
	  */
	 HeightField (Cache *cache, WorldUpdater *upd = 0);

	 /** Destroy the given HF */
	 virtual ~HeightField ();

	 /** Create a random heightfield, using the fractal division method */
	 virtual void Random ();

	 /**
	  * Initialize some functions of the world (ie pathfinding,
	  * rendering,  collision)..
	  */
	 virtual bool Init(int init_flags);

	 /// Render the height field (if the rendering system has been
	 /// initialized of course)
	 virtual bool Render (Renderer &renderer, const Camera& camera);

	 /** Read the heightfield from the given directory.
	  * \param path the name of the directory
	  */
	 virtual bool Load (const String &path);

	 /** Write the heightfield to the given directory.
	  * \param path the name of the directory
	  * \param mipmaps true if mipmaps should be written (half size and
	  *  quarter size versions are written, for less details => previews,
	  * etc)
	  */
	 virtual bool Write (const String &path, bool mipmaps = true);

	 /// Init pathfinding stuff (only needed on server side)
	 void InitPathfinder ();

	 /// Destroy the pathfinder (root graph node & astar object)
	 void DestroyPathfinder ();

	 // =============================================================
	 // access functions...

	 /** Get the height in the array */
	 inline Height& ArrayY(unsigned int index)
	 {
	    // should check for boundary here
	    return m_Y[index];
	 }

	 /** Get the ground in the array */
	 inline uchar& ArrayGround(unsigned int index)
	 {
	    return m_G[index];
	 }

	 /** Get the height at (x,z) */
	 inline Height& Y(size_t x, size_t z)
	 {
#ifdef TDEBUG
	       assert (x < m_SizeX && x >= 0);
	       assert (z < m_SizeZ && z >= 0);
#endif // TDEBUG
	    return m_Y[z * m_SizeX + x];
	 }

	 /** Get the height at (x,z) (const version) */
	 inline Height Y (size_t x, size_t z) const
	 {
#ifdef TDEBUG
	       assert (x < m_SizeX && x >= 0);
	       assert (z < m_SizeZ && z >= 0);
#endif // TDEBUG
	    return m_Y[z * m_SizeX + x];
	 }

	 /** Returns the height (y-value) of a point in this heightmap. The
	  * given (x,z) are in world coordinates. Heights outside this
	  * heightmap are  considered to  be 0. Heights between sample points
	  * are bilinearly  interpolated from surrounding points.
	  */
	 scalar GetHeight(scalar x, scalar z) const;

	 /** Finds the rest position for a given point.
	  *
	  * This returns the position with the Y coordinate fixed to be on the 
	  * height field.
	  *   \return The corrected position
	  *   \param position Initial position
	  */
	 virtual Vector3 GetRestPosition(const Vector3& position) const;
	 
	 /** Returns the index of ground type in m_GroundTypes
	  * of a point in this heightmap.
	  */
	 inline uchar &GetGround(int x, int z)
	 {
	    return m_G[z * m_SizeX + x];
	 }

	 /// Returns the material at (x, z)
	 inline Material *GetGrd(int x, int z)
	 {
	    const uchar& grd = GetGround (x, z);
	    return &*m_Grounds[grd];
	 }

	 /// Return the indices for a position
	 inline Vector3 GetVectorInGridSpace(const Vector3& world) const
	 { 
	     return Vector3( (world.X - m_OffsetX) / m_Scale,
		             world.Y / m_ScaleY,
	                     (world.Z - m_OffsetZ) / m_Scale );
	 }

	 /// Return the coordinate at (x,z)
	 inline Vector3 GetCoord(int x, int z) const
	 {
	    return Vector3 (scalar(x) * m_Scale + m_OffsetX,
			    scalar(Y(x, z)) * m_ScaleY,
			    scalar(z) * m_Scale + m_OffsetZ);
	 }

	 /// Return the list of grounds for this terrain.
	 inline const MaterialList& GetGroundTypes () const
	 {
	    return m_Grounds;
	 }

	 /// Return the list of grounds for this terrain.
	 inline MaterialList& GetGroundTypes ()
	 {
	    return m_Grounds;
	 }

      public:
	 /// Functions used mainly by the editor...
	 void DeleteGround (unsigned int idx);
	 void AppendGround (const MaterialPtr& mat);

	 inline Quadtree *GetQuadtree () const {return m_Quadtree;}

	 /// Tells the quadtree the specified area is invalidated.
	 void Invalidate (scalar minx, scalar minz, scalar maxx, scalar maxz);

      public:
	 /// World interface. Have a look at the World class to learn more
	 /// about these functions.

	 // == Pathfinding ==
	 /// Terrain implementation of pathfinding
	 virtual bool FindPath (Path &path);

	 /// Terrain implementation of accessibility test
	 virtual bool IsReachable (const Vector3 &pfrom, const Vector3 &pto, bool gravity);

	 // == Collision detection ==

	 /// Terrain implementation of collision test.
	 virtual bool TestCollision (const BBox &box, int8 colflags, CollisionList &collisions);

	 /// Terrain implementation of collision detection.
	 virtual void DetectCollisions();

	 /// Terrain implementation of raytracing.
	 virtual bool RayTrace (const Ray& ray, int8 colflags, CollisionList& collisions);

	 inline Config& GetConfig() { return m_Config; }

	 // == Entities ==
	 /// Terrain implementation of entity addition
	 virtual void Add (Entity *entity);
	 virtual void Remove (Entity *entity);

	 // == Atmoshere ==
	 void SetTimeOfDay(scalar f);
	 void SetFogColor(const Color& color);
	 void SetFogDensity(scalar density);

	 scalar GetTimeOfDay() const;
	 Color GetFogColor() const;
	 scalar GetFogDensity() const;

     protected:
	 /// Sets the fog for rendering
	 void SetFog();
	    
	 /// Unsets the fog after rendering
	 void UnsetFog();

     private:
	 bool HeightField::LoadLight (int aLightNum, Light &light);
   };

/* namespace Ark */
}

#include <Ark/ArkFactoryDef.h>
ARK_REGISTER_DEF(ARKHEIGHTFIELD_DLL_API);


#endif
