/* $Id: LuaWorld.cpp,v 1.13 2003/03/22 23:51:49 mrq Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2002 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <Modules/Lua/LuaEntity.h>
#include <Modules/Lua/LuaTable.h>
#include <Modules/Lua/LuaWorld.h>

#include <Modules/HeightField/HFWorld.h>

#include <sstream>

namespace Ark
{

  LuaWorld::LuaWorld(lua_State *L)
  {
    m_World = GetEngine()->GetWorld();
    // FIXME refcounting of worlds (ie m_World->Ref();)
  }
  
  LuaWorld::~LuaWorld()
  {}
     
  /// Callbacks
  int
  LuaWorld::insert (lua_State *L)
  {
#ifdef DEBUG
    assert (m_World);
#endif
    
    LuaEntity *entity;
    lua_getobject(L,-1,&entity);
    
    m_World->Add(entity);
    lua_pop (L,1);
    lua_pushbool (L, true);
    return 1;
  }

  int
  LuaWorld::remove (lua_State *L)
  {
#ifdef DEBUG
    assert (m_World);
#endif
    
    LuaEntity *entity;
    lua_getobject(L,-1,&entity);
    
    m_World->Remove(entity);
    lua_pop (L,1);
    lua_pushbool (L, true);
    return 1;
  }
	 
  /****************************************************************************
   * Get all the entities of the world
   ***************************************************************************/
  int
  LuaWorld::get_all_entities(lua_State* L)
  {
#ifdef DEBUG
     assert (m_World);
#endif
     
     Ark::EntityList& entities = m_World->GetEntities();
     
     lua_newtable(L);	// Create a new table on the stack
     for(unsigned i = 0; i < entities.size(); i++)
     {
	LuaEntity* entity = static_cast<LuaEntity*>( entities[i] );
	
	entity->push_object(L);	// Push the entity on the stack.
	lua_rawseti(L, -2, i);	// Put the top of the stack (our entity) in the
				// table at position i.
     }

     return 1;	// One value has been returned on the stack.
  }
    
  /****************************************************************************
   * Find an entity given its id.
   ***************************************************************************/
  int
  LuaWorld::get_entity(lua_State* L)
  {
#ifdef DEBUG
    assert (m_World);
#endif
    
    // Get the parameter which should be an id, let's say a number
    // The function lua_tonumber returns a double
    int id = static_cast<int>( lua_tonumber(L, -1) );
    lua_pop(L, 1);	// Pop the parameter

    // Find the entity given the id
    LuaEntity* entity = static_cast<LuaEntity*>( m_World->Find( id ) );

    if(entity)
    {
      entity->push_object(L);	// Return the found entity on the stack
    }
    else
    {
      lua_pushnil(L);	// Return nil on the stack
    }

    return 1;	// One value has been returned on the stack.
  }
    
  /****************************************************************************
   * Find an entity given its shortname
   ***************************************************************************/
  int
  LuaWorld::get_entity_by_name (lua_State *L)
  {
#ifdef DEBUG
    assert (m_World);
#endif
    
    String name = lua_tostring(L, -1);
    lua_pop (L, 1);

    LuaEntity *entity = static_cast<LuaEntity*>( m_World->FindByName (name) );

    if(entity)
    {
      entity->push_object(L);	// Return the found entity on the stack.
    }
    else
    {
      lua_pushnil(L);	// Return nil on the stack.
    }

    return 1;	// One value has been returned on the stack
  }
  
  /****************************************************************************
   * Find all entities given their class
   ***************************************************************************/
  int
  LuaWorld::find_entities_by_class(lua_State* L)
  {
#ifdef DEBUG
    assert (m_World);
#endif
    
    String className = lua_tostring(L, -1);
    lua_pop(L, 1);

    EntityList* entities = m_World->FindByClass(className);

    if(entities)
    {	// Return the found entities in a table
      lua_newtable(L);	// Create the table on the stack
      for(unsigned i = 0; i < entities->size(); i++)
      {
	LuaEntity* entity = static_cast<LuaEntity*>( (*entities)[i] );

	entity->push_object(L);	// Push the entity on the stack
	lua_rawseti(L, -2, i);	// Put the top of the stack (the entity) in the
				// table at index i.
      }

      delete entities;	// We do not need it anymore
    }
    else
    {	// Return nil
      lua_pushnil(L);	// Return nil on the stack
    }

    return 1;	// One value has been returned on the stack.
  }

  /****************************************************************************
   * Test if collision occurs between a bounding box and entities
   * The parameters given here are those that have to be given in Lua.
   *
   * \paramin box The bounding box which may collide with entities. Give it with
   * two points coordinates, the bottom-left-near corner first and the
   * top-right-far corner after. The two points have to be given in a table.
   * Each point is itself a table, with members x, y and z. For example, I may
   * access the first point coordinates with box[0].x, box[0].y and box[0].z
   *
   * \return nil if something went wrong. Otherwise, returns a table which is a
   * list of the entities collided, which may be empty.
   ****************************************************************************/
  int
  LuaWorld::test_collision_with_entities(lua_State* L)
  {
#ifdef DEBUG
    assert (m_World);
#endif
    
    std::cerr << "plop" << std::endl;
    
    // First create the bounding box
    BBox box;
    LuaTable* boxParameters = new LuaTable(L, -1);
    // BLN = bottom-left-near corner
    LuaTable* boxBLN = boxParameters->getTable(1);
    if( !boxBLN )
    {
      std::cerr << "Bad call to the function need the first point coordinates"
	<< std::endl;
      return 0;
    }
    // TRF = top-right-far corner
    LuaTable* boxTRF = boxParameters->getTable(2);
    if( !boxTRF )
    {
      std::cerr << "Bad call to the function need the second point coordinates"
	<< std::endl;
      return 0;
    }

    std::cerr << "plop" << std::endl;
    
    box.m_Min = Vector3(static_cast<scalar>( boxBLN->getNumber(0) ),
			static_cast<scalar>( boxBLN->getNumber(1) ),
			static_cast<scalar>( boxBLN->getNumber(2) ) );
    box.m_Max = Vector3(static_cast<scalar>( boxTRF->getNumber(0) ),
			static_cast<scalar>( boxTRF->getNumber(1) ),
			static_cast<scalar>( boxTRF->getNumber(2) ) );

    std::cerr << "plop" << std::endl;
    
    delete boxTRF;
    delete boxBLN;
    delete boxParameters;

    std::cerr << "plop" << std::endl;
    
    std::vector<Collision> collisions;

    // Test the collision
    if( m_World->TestCollision( box,
	  			Collision::ENTITY | Collision::POSITION,
				collisions) )
    {	// Return the list of entities colliding the bounding box

      std::cerr << "plop" << std::endl;
    
      LuaEntityList* collidingEntities = LuaEntityList::createTable( L );
      
      std::cerr << "plop" << std::endl;
    
      for( unsigned i = 0; i < collisions.size(); i++ )
      {
	Collision collision = static_cast<Collision>( collisions[i] );
	if( collision.m_Flags & Collision::ENTITY )
	{
	  collidingEntities->addElement( 
	      dynamic_cast<LuaEntity&>( *collision.m_Entity ));
	}
      }
    }
    else
    {	// Return nil
      lua_pushnil(L);	// Return nil on the stack
    }

    return 1;	// One value has been returned on the stack
  }

  /****************************************************************************
   * This is the function to ask for anything about the world. It should at term
   * replace all above functions. The syntax will be developped here later.
   * TODO: Define the syntax for the requests
   * Let's try with a unique argument, a string containing the request
   ***************************************************************************/
  int
  LuaWorld::request (lua_State* L)
  {
    assert (m_World);

    String requestString( lua_tostring(L, -1) );
    std::istringstream requestStream( requestString );
    lua_pop(L, 1);

    // Try to "parse" the string but it isn't really effective...
    String command;
    requestStream >> command;

    /**
     * Find by name
     * \param name The shortname of the entity to find
     */    
    if( command == "find_by_name" )
    {
      String name;
      requestStream >> name;
      
      LuaEntity* entity = static_cast<LuaEntity*>( m_World->FindByName(name) );

      if(entity)
      {
	entity->push_object(L);	// Return the found entity on the stack
      }
      else
      {
	lua_pushnil(L);	// Return nil on the stack
      }
    }
    
    return 1;	// One value has been returned on the stack.
  }

  int
  LuaWorld::set_time_of_day (lua_State* L)
  {
    assert (m_World);
    static_cast<HeightField*>(m_World)->SetTimeOfDay (lua_tonumber(L, -1));
    lua_pop (L,1);
    return 0;
  }

  int
  LuaWorld::set_fog_density (lua_State* L)
  {
    assert (m_World);
    static_cast<HeightField*>(m_World)->SetFogDensity (lua_tonumber(L, -1));
    lua_pop (L,1);
    return 0;
  }

  int
  LuaWorld::set_fog_color (lua_State* L)
  {
    assert (m_World);
    static_cast<HeightField*>(m_World)->SetFogColor
       (Color(scalar(lua_tonumber(L, -3)),
	      scalar(lua_tonumber(L, -2)),
	      scalar(lua_tonumber(L, -1))));
    lua_pop (L,3);
    return 0;
  }

  int
  LuaWorld::get_time_of_day (lua_State* L)
  {
    assert (m_World);
    lua_pushnumber(L, static_cast<HeightField*>(m_World)->GetTimeOfDay ());
    return 1;
  } 

  int
  LuaWorld::get_fog_density (lua_State* L)
  {
    assert (m_World);
    lua_pushnumber(L, static_cast<HeightField*>(m_World)->GetFogDensity ());
    return 1;
  }

  int
  LuaWorld::get_fog_color (lua_State* L)
  {
    assert (m_World);
    Color c = static_cast<HeightField*>(m_World)->GetFogColor ();
    lua_pushnumber(L, c.R);
    lua_pushnumber(L, c.G);
    lua_pushnumber(L, c.B);
    return 3;
  }

   // LIGHTS
   Light &
   LuaWorld::get_light (lua_State *L, int posinstack)
   {
      assert (m_World);
      LightList &lights = m_World->GetLights();

      size_t n = size_t(lua_tonumber(L,posinstack));
      assert(n < 8 && "FIXME no more than eight lights"); 

      return lights[n];
   }

   int
   LuaWorld::set_light_color (lua_State *L)
   {
      get_light(L,-4).SetColor (Color(scalar(lua_tonumber(L, -3)),
				       scalar(lua_tonumber(L, -2)),
				       scalar(lua_tonumber(L, -1))));
      lua_pop (L,4);
      return 0;

   }

   int
   LuaWorld::set_light_position (lua_State *L)
   {
      get_light(L,-4).SetPosition (Vector3(scalar(lua_tonumber(L, -3)),
					    scalar(lua_tonumber(L, -2)),
					    scalar(lua_tonumber(L, -1))));
      lua_pop (L,4);
      return 0;      
   }

   int
   LuaWorld::set_light_type (lua_State *L)
   {
      // FIXME: CHECK TYPE ANYONE ?
      const int lightType = (int)lua_tonumber(L, -1);
      get_light(L,-2).SetLightType(LightType(lightType));
      lua_pop(L,2);
      return 0;
   }

   int
   LuaWorld::set_light_attenuation (lua_State *L)
   {
      get_light(L,-2).SetAttenuation (scalar(lua_tonumber(L, -1)));
      lua_pop (L,2);
      return 0;
   }

   int
   LuaWorld::get_light_color (lua_State *L)
   {
      Color c = get_light(L,-1).GetColor();
      lua_pop (L,1);

      lua_pushnumber(L, c.R);
      lua_pushnumber(L, c.G);
      lua_pushnumber(L, c.B);
      return 3;
   }

   int
   LuaWorld::get_light_position (lua_State *L)
   {
      Vector3 v = get_light(L,-1).GetPosition();
      lua_pop (L,1);

      lua_pushnumber(L, v.X);
      lua_pushnumber(L, v.Y);
      lua_pushnumber(L, v.Z);
      return 3;
   }

   int
   LuaWorld::get_light_type (lua_State *L)
   {
      int l = get_light(L,-1).GetType();
      lua_pop (L,1);

      lua_pushnumber(L, l);
      return 1;
   }

   int
   LuaWorld::get_light_attenuation (lua_State *L)
   {
      scalar l = get_light(L,-1).GetAttenuation();
      lua_pop (L,1);

      lua_pushnumber(L, l);
      return 1;
   }


   /*XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
      LIGHT_AMBIENT, 
      LIGHT_DIRECTIONAL,
      LIGHT_POINT,
      LIGHT_SPOT
   */


  /****************************************************************************
   * Register class name
   ***************************************************************************/
  const char LuaWorld::className[] = "World";

  /****************************************************************************
   * Register functions in Lua
   ***************************************************************************/
  const Luna<LuaWorld>::RegType LuaWorld::Register[] = 
    {
      {"insert",			&LuaWorld::insert},
      {"remove",			&LuaWorld::remove},
      {"get_all_entities",		&LuaWorld::get_all_entities},
      {"get_entity",			&LuaWorld::get_entity},
      {"get_entity_by_name",		&LuaWorld::get_entity_by_name},
      {"find_entities_by_class",	&LuaWorld::find_entities_by_class},
      {"test_collision_with_entities",	&LuaWorld::test_collision_with_entities},
      {"request",			&LuaWorld::request},
      {"set_time_of_day",		&LuaWorld::set_time_of_day},
      {"set_fog_density",		&LuaWorld::set_fog_density},
      {"set_fog_color",		        &LuaWorld::set_fog_color},
      {"get_time_of_day",		&LuaWorld::get_time_of_day},
      {"get_fog_density",		&LuaWorld::get_fog_density},
      {"get_fog_color",		        &LuaWorld::get_fog_color},

      {"set_light_color",		&LuaWorld::set_light_color},
      {"set_light_position",		&LuaWorld::set_light_position},
      {"set_light_type",		&LuaWorld::set_light_type},
      {"set_light_attenuation",		&LuaWorld::set_light_attenuation},

      {"get_light_color",		&LuaWorld::get_light_color},
      {"get_light_position",		&LuaWorld::get_light_position},
      {"get_light_type",		&LuaWorld::get_light_type},
      {"get_light_attenuation",		&LuaWorld::get_light_attenuation},

      {0}
    };

}

