/* $Id: Network.cpp,v 1.15 2003/02/04 20:18:15 zongo Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2002 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <Ark/ArkSystem.h>
#include <Server/Server.h>

namespace Server
{

Network::Network (const Ark::String &port)
{
  Ark::Sys()->Log ("Starting network session...\n");
  m_AcceptSocket = new Ark::Socket (port, "tcp", 5);

  if (m_AcceptSocket->IsDead())
    Ark::Sys()->Fatal ("Cannot open accept socket.");
}

Network::~Network()
{
  Ark::Sys()->Log ("Stopping network services...\n");
  delete m_AcceptSocket;
}

bool Network::Accept ()
{
   
   Ark::Socket *socket;
   while ((socket = m_AcceptSocket->Accept (false)) != NULL)
      m_Clients.push_back (new Client (socket));

   // Delete dead clients
   for (ClientLI it = m_Clients.begin(); it != m_Clients.end(); ++it)
   {
      if ((*it)->m_Dead)
      {
	 delete (*it);
	 m_Clients.erase(it);
	 it--;
      }
   }

   return true;
}


bool Network::Receive ()
{
  for (ClientLI it = m_Clients.begin(); it != m_Clients.end(); ++it)
     (*it)->HandleMessages();

  return true;
}

bool Network::Update ()
{
   Ark::Packet *delta = new Ark::Packet();
   std::stringstream& deltaStream = delta->m_Stream;
   GetServer()->GetEngine()->WriteDelta(deltaStream);
   GetServer()->GetEngine()->ResetDelta();

   for (ClientLI it = m_Clients.begin(); it != m_Clients.end(); ++it)
      (*it)->SendUpdate(delta);

   delta->Unref();

   return true;
}

//////////////////////////////////////////////////////////////////////

Client::Client (Ark::Socket *socket) : 
   m_Socket (socket),
   m_Entity (NULL)
{
    std::cout << "New client.\n";
   m_Socket->SetBlocking (false);
}

Client::~Client ()
{
   std::cout << "Deleted client.\n";
   if(m_Entity) m_Entity->SetDead();
   delete m_Socket;
}

void
Client::HandleMessages ()
{
   m_OutPacket = new Ark::Packet();

   m_Socket->ReceiveData();
   while (Ark::ReadStream *pak = m_Socket->GetPacket())
   {
      Result res;

      while (!pak->eof())
	 HandlePacket (pak, &res);

      if (res.m_Msg != 0)
      {
	  std::stringstream& out = m_OutPacket->m_Stream;
	 Ark::NetWriteByte (out, Ark::NET_UPD_RESULT);
	 Ark::NetWriteByte (out, res.m_Msg);
	 Ark::NetWriteByte (out, (char) res.m_Error);
	 
	 if(res.m_Error)
	    Ark::NetWriteString (out, res.m_Reason);
      }
   }

   m_Dead = m_Socket->IsDead();
}

void
Client::HandlePacket (Ark::ReadStream *packet, Result *result)
{
   unsigned char msg=0;

   Ark::NetReadByte(*packet, msg);

   std::cout << "Received " << packet->eof() << "/" <<(int)msg << "\n";
   switch (msg)
   {
      case Ark::NET_MSG_ENTITY_TELL:
      {
	 int enr; 
	 Ark::String msg;
	 
	 Ark::NetReadInt (*packet, enr);
	 Ark::NetReadString (*packet, msg);

	 Ark::Entity *ent = GetServer()->GetEngine()->GetWorld()->Find(enr);
	 std::cout << m_Entity->m_Name << " says " << msg << " to entity "
	      << ent->m_Name << "\n";

	 m_Entity->AddMessage(ent, msg);
      }
      break;

      case Ark::NET_MSG_ENTITY_SET_NOGOAL:
	 m_Entity->SetGoal(0);
	 break;

      case Ark::NET_MSG_ENTITY_SET_GOAL:
      {
	 Ark::Vector3 v;
	 Ark::NetReadScalar (*packet, v.X);
	 Ark::NetReadScalar (*packet, v.Y);
	 Ark::NetReadScalar (*packet, v.Z);

	 std::cout << v << "\n";
	 m_Entity->SetGoal (v);
      }
      break;

      case Ark::NET_MSG_CONNECT:
      {
	 int proto;
	 Ark::String nick, pass;
	 Ark::NetReadInt (*packet, proto);
	 Ark::NetReadString(*packet, nick);
	 Ark::NetReadString(*packet, pass);

	 if (proto != Ark::NET_PROTOCOL_VERSION)
	 {
	    *result = Result(Ark::NET_MSG_CONNECT, "Bad protocol version");
	    return;
	 }

	 m_Entity = GetServer()->GetEngine()->Login(nick, pass);

	 if (m_Entity == NULL)
	 {
	    *result = Result(Ark::NET_MSG_CONNECT, "Bad login/password");
	    return;
	 }

	 std::cout << nick << " is connected.\n";
	 // Everything seems OK, send an answer.
	 std::stringstream& out = m_OutPacket->m_Stream;
	 Ark::NetWriteByte(out, Ark::NET_UPD_CONNECT_INFO);
	 Ark::NetWriteString(out, GetServer()->GetEngine()->GetWorldName());
	 Ark::NetWriteInt(out, m_Entity->m_ID);

	 // Send a snapshot of the world.
	 GetServer()->GetEngine()->WriteFull(out);
      }
   }
}

void
Client::SendUpdate (Ark::Packet *delta_packet)
{

   if (m_Entity && !m_Entity->m_Messages.empty())
   {
       std::stringstream& out = m_OutPacket->m_Stream;
      Ark::NetWriteByte(out, Ark::NET_UPD_MESSAGES);

      std::vector<Ark::EntityMessage>::iterator it;
      for (it = m_Entity->m_Messages.begin();
	   it != m_Entity->m_Messages.end(); ++it)
      {
	 Ark::NetWriteInt(out, it->m_Entity->m_ID);
	 Ark::NetWriteString (out, it->m_Message);
	 Ark::NetWriteInt (out, it->m_AnswerList.size());
	
	 std::cerr << it->m_Message << "\n";
	 for (std::vector<Ark::String>::iterator jt = it->m_AnswerList.begin();
	      jt != it->m_AnswerList.end(); ++jt)
	    Ark::NetWriteString (out, *jt);
      }

      m_Entity->m_Messages.clear();
      Ark::NetWriteInt (out, 0);
   }
   m_Socket->SendPacket(m_OutPacket);

   if (m_Entity)
      m_Socket->SendPacket(delta_packet);

   m_Socket->SendData();
   
   m_OutPacket->Unref();
   m_OutPacket = NULL;
}


}
