/*
 * Copyright (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*================================================*/
/*	ISO options page.
 *
 *	by Tony Sideris	(10:52AM May 12, 2002)
 *================================================*/
#include "arson.h"

#include <qpushbutton.h>
#include <qtoolbutton.h>
#include <qlistview.h>
#include <qfileinfo.h>
#include <qdir.h>
#include <qxml.h>
#include <qcombobox.h>
#include <qlineedit.h>
#include <qlayout.h>
#include <qlabel.h>

#include <klineeditdlg.h>
#include <kfiledialog.h>
#include <klocale.h>

#include "isopage.h"
#include "isofs.h"

/*========================================================*/
/*	Option list types
 *========================================================*/

iso_option::iso_option (const QXmlAttributes &attr)
	: opt(NULL), desc(QString::null), pItem(NULL)
{
	desc = i18n(attr.value("desc"));	//	Description string
	opt = attr.value("switch");			//	Command line switch

	if (opt.isNull() || desc.isNull())
		throw ArsonError(
			i18n("Required value is missing for switch/description (%1/%2)").
			arg(opt).arg(desc));
}

/*========================================================*/

QListViewItem *iso_option::listViewItem (QListView *plv)
{
	if (plv && !pItem)
	{
		QListViewItem *pt, *pTemp = plv->firstChild();

		/*	Iterate through the contents of the listview window to
		 *	get the last item, this is due to the design of QListView
		 *	(there's no lastChild()), and to keep the list in the
		 *	right order.
		 *
		 *	Yes, this is inefficient as fuck, but oh well, this
		 *	list will remain relatively small anyway...
		 */
		for (pt = pTemp; pt && (pt = pTemp->nextSibling()); pTemp = pt);

		pItem = createItem(plv);
		pItem->moveItem(pTemp);
	}
	return pItem;
}

/*========================================================*/

struct iso_switch : public iso_option
{
	iso_switch (const QXmlAttributes &attr)
		: iso_option(attr)
	{
		//	Nothing...
	}

protected:
	virtual QListViewItem *createItem (QListView *plv) {
		return new QCheckListItem(plv, desc, QCheckListItem::CheckBox);
	}

	virtual void save (ArsonIsoFlags *pFlags) {
		if (((QCheckListItem *) pItem)->isOn())
			pFlags->addFlag(opt);
	}

	virtual void apply (const ArsonIsoFlags *pFlags) {
		((QCheckListItem *) pItem)->setOn(pFlags && pFlags->hasFlag(opt));
	}
};

/*========================================================*/

struct iso_text : public iso_option {
	iso_text (const QXmlAttributes &attr)
		: iso_option(attr), text(QString::null),
		limit(0), pDataItem(NULL)
	{
		limit = attr.value("limit").toInt();
		text = attr.value("value");
	}

	void setValue (const QString &value) {
		pDataItem->setText(0, (text = value));
		if (text.isEmpty())
			text = QString::null;
	}

protected:
	virtual QListViewItem *createItem (QListView *plv) {
		QListViewItem *p = new QListViewItem(plv, desc);
		pDataItem = new QListViewItem(p);

		setValue(text);
		p->setOpen(true);
		return p;
	}

	virtual void takeAction (void)
	{
		bool ok;
		QString inp = KLineEditDlg::getText(desc + i18n(":"),
			text, &ok, (QWidget *) pItem->listView()->parent());

		if (ok)
		{
			if (limit != 0 && inp.length() > limit)
			{
				arsonErrorMsg(
					i18n("This must be %1 characters or less!").arg(limit));
				return;
			}

			setValue(inp);
		}
	}

	virtual void save (ArsonIsoFlags *pFlags) {
		if (text != QString::null)
			pFlags->addFlag(opt, text);
	}

	virtual void apply (const ArsonIsoFlags *pFlags) {
		if (pFlags->hasFlag(opt))
			setValue(pFlags->value(opt));
	}

	QListViewItem *pDataItem;
	QString text;
	int limit;
};

/*========================================================*/

struct iso_file : public iso_text {
	iso_file (const QXmlAttributes &attr)
		: iso_text(attr)
	{
		filter = attr.value("filter");
	}

protected:
	virtual void takeAction (void)
	{
		QFileInfo fi (QFile::encodeName(text));
		const QString filepath = KFileDialog::getOpenFileName(
			fi.dir().canonicalPath(), filter);

		if (filepath != QString::null)
			setValue(filepath);
	}

	QString filter;
};

/*========================================================*/
/*	Dialog class implementation
 *========================================================*/

ArsonIsoPage::ArsonIsoPage (ArsonConfig &config, ArsonConfigDlg *pdlg, QWidget *parent)
	: ArsonIsoPageBase(parent), ArsonConfigPage(config),
	m_pCurFlags(NULL), m_presets (new ArsonIsoPresets)
{
	m_pCurFlags = presets().flags(ArsonIsoPresets::defaultName());

	preset_box->insertStringList(presets().names());
	
	//	Fill the option list
	options->setSorting(-1);

	buildOptionList();
}

ArsonIsoPage::~ArsonIsoPage (void)
{
	//	Free the option list items
	for (int index = 0; index < m_items.count(); ++index)
		delete m_items[index];

	delete m_presets;
}

/*========================================================*/

QString ArsonIsoPage::presetName (void) const
{
	return preset_box->text(
		preset_box->currentItem());
}

/*========================================================*/

void ArsonIsoPage::set_default_clicked (void)
{
	int index;

	if (m_pCurFlags && (index = findComboItem(ArsonIsoPresets::defaultName())) != -1)
	{
		//	This is actually a replace, not an add...
		presets().addPreset(ArsonIsoPresets::defaultName(), *m_pCurFlags);

		preset_box->setCurrentItem(index);
	}
}

/*========================================================*/
/*	When a changable feild (text entry or filename) is
 *	double-clicked take the proper action.
 *========================================================*/

void ArsonIsoPage::on_dblclk (QListViewItem *pItem)
{
	iso_option *popt;
	QListViewItem *ptr;

	if ((ptr = pItem->itemAbove()) && (popt = findItem(ptr)))
		popt->takeAction();
}

/*========================================================*/
/*	Build the option list from the XML data file.
 *========================================================*/

class isoOptionParser : public QXmlDefaultHandler
{
public:
	isoOptionParser (ArsonIsoPage *pd)
		: QXmlDefaultHandler(),
		m_pDlg(pd)
	{}
	
	virtual bool startElement (const QString &ns, const QString &local,
		const QString &name, const QXmlAttributes &attr)
	{
		try
		{
			if (name == "switch")
				m_pDlg->addItem(new iso_switch(attr));

			else if (name == "input")
				m_pDlg->addItem(new iso_text(attr));

			else if (name == "file")
				m_pDlg->addItem(new iso_file(attr));
		}
		catch (ArsonError &err) {
			err.report();
		}

		return true;
	}

private:
	ArsonIsoPage *m_pDlg;
};

void ArsonIsoPage::buildOptionList (void)
{
	isoOptionParser parser (this);
	const KURL url (QString("file:") + arsonDataFile("isoopts.xml").data());

	arsonParseXml(
		url, &parser);
}

/*========================================================*/

void ArsonIsoPage::applyOptions (void)
{
	for (int index = 0; index < m_items.count(); ++index)
		m_items[index]->apply(m_pCurFlags);
}

void ArsonIsoPage::saveOptions (ArsonIsoFlags *pf)
{
	Trace("saving the state of %d options to %p\n", m_items.count(), pf);

	pf->clear();
	
	for (int index = 0; index < m_items.count(); ++index)
		m_items[index]->save(pf);
}

/*========================================================*/

iso_option *ArsonIsoPage::findItem (QListViewItem *ptr)
{
	for (int index = 0; index < m_items.count(); ++index)
		if (m_items[index]->listViewItem(NULL) == ptr)
			return m_items[index];

	return NULL;
}

int ArsonIsoPage::findComboItem (const QString &name)
{
	for (int index = 0; index < preset_box->count(); ++index)
		if (preset_box->text(index) == name)
			return index;

	return -1;
}

/*========================================================*/

void ArsonIsoPage::addItem (iso_option *ptr)
{
	Assert(m_pCurFlags != NULL);
	
	ptr->listViewItem(options);
	ptr->apply(m_pCurFlags);

	m_items.append(ptr);
}

/*========================================================*/

void ArsonIsoPage::on_accept (void)
{
	Trace("Saving presets\n");
	update_preset();
	presets().save();
}

/*========================================================*/

void ArsonIsoPage::add_preset (void)
{
	const QString name = preset_box->currentText();

	if (name.isNull() || name.isEmpty())
	{
		arsonErrorMsg(
			i18n("Invalid preset name."));
		return;
	}

	if (!presets().flags(name))
	{
		ArsonIsoFlags flags;
		const uint count = preset_box->count();

		saveOptions(&flags);
		presets().addPreset(name, flags);

		preset_box->insertItem(name);

		on_preset_change(count);
		preset_box->setCurrentItem(count);
	}
	else
		arsonErrorMsg(
			i18n("A preset with that named already exists, use Update."));
}

void ArsonIsoPage::del_preset (void)
{
	const QString name = preset_box->currentText();

	if (name == ArsonIsoPresets::defaultName())
		arsonErrorMsg(
			i18n("Nope... you can't delete 'Default'."));

	else if (!presets().flags(name))
		arsonErrorMsg(
			i18n("That preset does not exist."));
	else
	{
		int index;
		
		/*	For some reason currentItem() is returning the
		 *	wrong index under certain situations. Manually
		 *	search the contents of the list, and remove
		 *	that index.
		 */
		if ((index = findComboItem(name)) != -1)
		{
			preset_box->removeItem(index);
			presets().remove(name);

			preset_box->setCurrentItem(0);
			on_preset_change(0);
		}
	}
}

/*========================================================*/

void ArsonIsoPage::update_preset (void)
{
	if (m_pCurFlags)
		saveOptions(m_pCurFlags);
}

/*========================================================*/

void ArsonIsoPage::on_preset_change (int index)
{
	const QString current = preset_box->currentText();
	
	if ((m_pCurFlags = presets().flags(current)))
	{
		Trace("Current preset changed to: %s\n",
			(const char *) current);

		applyOptions();
	}
}

/*========================================================*/
