/*
 * Copyright (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*================================================*/
/*	Read wav information from wav files
 *
 *	by Tony Sideris	(08:29AM Sep 12, 2001)
 *================================================*/
#include <string.h>
#include <fstream>

#include "wavinfo.h"

#ifdef HAVE_CONFIG_H
#	include "../config.h"
#	if STDC_HEADERS
#		include <string.h>
#	else
#		if !HAVE_STRCHR
#			define strrchr rindex
#			define strchr index
#		endif
		char *strchr (), *strrchr ();
#		if !HAVE_MEMCPY
#			define memmove(d, s, n) bcopy ((s), (d), (n))
#			define memcpy(d, s, n) bcopy ((s), (d), (n))
#		endif
#	endif
#endif	//	HAVE_CONFIG_H

/*========================================================*/
/*	Chunk header implementation
 *========================================================*/
		
wav::header::header (char *n) : size(0)
{
	if (n)
		memcpy(name, n, chunk_name_len);
	else
		name[0] = 0;
}

/*========================================================*/

wav::header::header (const header &hdr)
	: size(hdr.size)
{
	memcpy(name, hdr.name, chunk_name_len);
}
		
std::istream &wav::header::read (std::istream &str, void **res, int min) const
{
	if (min > 0 && size < min)
		str.setstate(std::ios::failbit);
	else
	{
		unsigned char *ptr = new unsigned char[size];

		str.read((char *) ptr, size);

		if (str) *res = ptr;
		else delete[] ptr;
	}
			
	return str;
}

/*========================================================*/
/*	Utility operator implementations
 *========================================================*/

std::istream &operator>> (std::istream &str, wav::file_header &hdr)
{
	str.read((char *) &hdr, sizeof(wav::file_header));

	if (str && !hdr.valid())
		str.setstate(std::ios::failbit);

	return str;
}

std::ostream &operator<< (std::ostream &str, const wav::fmt &fmt)
{
	str << "Compressed: " << (fmt.compressed() ? "yes" : "no") << '\n'
		<< "Channels: " << fmt.channels << '\n'
		<< "Samples/second: " << fmt.samplesPerSec << '\n'
		<< "Average bytes/second: " << fmt.aveBytesPerSec << '\n'
		<< "Block align: " << fmt.blockAlign << '\n'
		<< "Bits/sample: " << fmt.bitsPerSample << std::endl;

	return str;
}

/*========================================================*/

bool wav::findChunkHeader (std::istream &str, wav::header *phdr, bool rewind)
{
	const header hdr (*phdr);

	if (rewind)
		str.seekg(0);

	if((long) str.tellg() == 0)
	{
		file_header fh;
		
		if (!(str >> fh) || !fh.valid())
		{
			if (!str.eof())
				str.clear();

			return false;
		}
	}

	for (; getChunkHeader(str, phdr); phdr->skip(str))
		if (*phdr == hdr)
			return true;

	return false;
}

/*========================================================*/
/*	A more friendly interface, use this.
 *========================================================*/

wav::stdhdr::stdhdr (void)
	: m_data(NULL),
	m_fmt(NULL),
	m_dataLen(0),
	m_dataPos(0)
{}

wav::stdhdr::stdhdr (std::istream &str)
	: m_data(NULL),
	m_fmt(NULL),
	m_dataLen(0),
	m_dataPos(0)
{
	load(str);
}

wav::stdhdr::~stdhdr (void)
{
	free();
}

/*========================================================*/

bool wav::stdhdr::load (std::istream &str)
{
	header hdr("fmt "), data("data");

	free();
	
	if (findChunkHeader(str, &hdr) &&
		hdr.read(str, (void **) &m_fmt) &&
		findChunkHeader(str, &data))
	{
		m_dataPos = str.tellg();
		m_dataLen = data.size;

		return true;
	}

	return false;
}

/*========================================================*/

void wav::stdhdr::free (void)
{
	m_dataPos = m_dataLen = 0;

	delete[] (uchar *) m_fmt;
	delete[] m_data;
	m_data = NULL;
	m_fmt = NULL;
}

/*========================================================*/
/*	Test code
 *========================================================*/
#ifdef WAVTEST
#include <iomanip>

void error (const char *msg)
{
	std::cerr << "wavinfo: " << msg << std::endl;
	exit(1);
}

int main (int argc, char **argv)
{
	if (argc > 1)
	{
		wav::stdhdr hdr;
		std::ifstream fin (argv[1]);

		if (!fin)
			error("failed to open file");

		if (!hdr.load(fin))
			error("failed to load header");

		std::cout << hdr.format() << std::endl;

		if (wav::ulong secs = wav::seconds(hdr.format(), hdr.dataLen()))
		{
			std::cout << "Time: "
					  << (secs / 60) << ':' << std::setw(2)
					  << std::setfill('0') << (secs % 60)
					  << " (" << secs << ")" << std::endl;
		}
		else if (hdr.format().compressed())
			std::cerr << "Compressed file!" << std::endl;
	
		return 0;
	}

	error("file please");
	return 1;
}
#endif	//	WAVTEST
/*========================================================*/
