/*
 * Copyright (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
#ifndef __WAVINFO_H__
#define __WAVINFO_H__

#include <iostream>

/*========================================================*/

namespace wav
{
	enum { chunk_name_len = 4 };

	typedef unsigned short ushort;
	typedef unsigned long ulong;
	typedef unsigned char uchar;

	inline void free (void *data) { delete[] ((unsigned char *) data); }

	/*========================================================*/
	
	struct header
	{
		header (char *n = NULL);
		header (const header &hdr);

		bool isFmt (void) const { return is("fmt "); }
		bool isData (void) const { return is("data"); }
		bool is (const char *huh) const
		{ return (memcmp(name, huh, chunk_name_len) == 0); }

		bool operator== (const header &hdr) const { return is(hdr.name); }
		bool operator== (const char *n) const { return is(name); }

		std::istream &read (std::istream &str, void **ptr, int min = 0) const;
		std::istream &skip (std::istream &str) const {
			str.seekg(size, std::ios::cur);
			return str;
		}

		char name[chunk_name_len];
		unsigned long size;
	};

	/*========================================================*/
	
	struct file_header
	{
		file_header (void) {
			*wave = 0;
		}

		bool valid (void) const {
			return (!memcmp(hdr.name, "RIFF", chunk_name_len)
				&& !memcmp(wave, "WAVE", chunk_name_len));
		}

		header hdr;
		char wave[chunk_name_len];
	};

	/*========================================================*/

	struct fmt
	{
		bool compressed (void) const { return (formatTag != 1); }
		
		short formatTag;
		ushort channels;
		ulong samplesPerSec;
		ulong aveBytesPerSec;
		ushort blockAlign;
		ushort bitsPerSample;
	};

	/*========================================================*/

	inline bool getChunkHeader (std::istream &str, wav::header *phdr)
	{
		str.read((char *) phdr, sizeof(header));
		return str ? true : false;
	}

	bool findChunkHeader (std::istream &str, wav::header *phdr, bool rewind = false);

	inline ulong seconds (const fmt &format, ulong length)
	{
		return (format.compressed()) ? 0
			: (ulong) ((double) length / (double) format.aveBytesPerSec);
	}

	/*========================================================*/

	class stdhdr
	{
	public:
		stdhdr (void);
		stdhdr (std::istream &str);
		~stdhdr (void);

		bool valid (void) const { return (m_fmt != NULL); }
		ulong dataLen (void) const { return m_dataLen; }
		ulong dataPos (void) const { return m_dataLen; }
		const fmt &format (void) const { return *m_fmt; }
		const uchar *data (void) const { return m_data; }

		bool load (std::istream &str);
		void free (void);

	private:
		ulong m_dataLen;
		ulong m_dataPos;
		uchar *m_data;
		fmt *m_fmt;
	};
};

/*========================================================*/
/*	Utility operators
 *========================================================*/

std::istream &operator>> (std::istream &str, wav::file_header &hdr);
std::ostream &operator<< (std::ostream &str, const wav::fmt &fmt);

inline std::istream &operator>> (std::istream &str, wav::header &hdr) {
	wav::getChunkHeader(str, &hdr);
	return str;
}

/*========================================================*/
#endif	/*	__WAVINFO_H__	*/
