/*
 * Asterisk -- A telephony toolkit for Linux.
 *
 * Trivial application to playback a sound file
 * 
 * Copyright (C) 1999, Mark Spencer
 *
 * Mark Spencer <markster@linux-support.net>
 *
 * This program is free software, distributed under the terms of
 * the GNU General Public License
 */

#include <asterisk/lock.h>
#include <asterisk/file.h>
#include <asterisk/logger.h>
#include <asterisk/channel.h>
#include <asterisk/pbx.h>
#include <asterisk/module.h>
#include <asterisk/translate.h>
#include <asterisk/utils.h>
#include <asterisk/cli.h>

#ifdef ASTERISK_STABLE
#include <asterisk/utils.h>
#else
#include <asterisk/strings.h>
#endif


#include <sys/types.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/wait.h>
#include <signal.h>
#include <string.h>
#include <stdlib.h>
#include <pthread.h>
#include <semaphore.h>
#include <pty.h>
#include <unistd.h>
#include <errno.h>

static char *tdesc = "Pipe Chan Data into forked terminal";

static char *app = "PtyFork";

static char *synopsis = "Pseudo Terminal Fork";

static char *descrip = "";

#define MY_READ_FRAME_SIZE 256


#define PTYWAITER SCRIPTPATH "ptywaiter"


STANDARD_LOCAL_USER;

LOCAL_USER_DECL;

#define PPP_MAX_ARGS	32
//#define PPP_EXEC	"/usr/sbin/pppd"
#define PPP_EXEC	"/tmp/test.sh"


static char flip_table[256];

void init_flip_bits()
{
	int i,k;
	
	for (i = 0 ; i < 256 ; i++) {
		unsigned char sample = 0 ;
		for (k = 0; k<8; k++) {
			if ( i & 1 << k ) sample |= 0x80 >>  k;
		}
		flip_table[i] = sample;
	}
}

static unsigned char * flip_buf_bits ( unsigned char * buf , int len)
{
	int i;
	char * start = buf;
	
	for (i = 0 ; i < len; i++) {
		buf[i] = flip_table[buf[i]];
	}
	
	return start;
}



static pid_t spawn_ras( int pipe[2], char *prog, char *args)
{
  pid_t pid;
  int x;	
  char *c;
  
  char *argv[PPP_MAX_ARGS];
  int argc = 0;
  char *stringp=NULL;
  
  /* Start by forking */
  pid = fork();
  if (pid)
    return pid;
  
  /* Execute RAS on File handles */
	//dup2(chan->fds[0], STDIN_FILENO);
  dup2(pipe[0], STDIN_FILENO);
  dup2(pipe[1], STDOUT_FILENO);
  
  
  /* Close other file descriptors */
  for (x=STDERR_FILENO + 1;x<1024;x++) 
    close(x);
  
  /* Restore original signal handlers */
  for (x=0;x<NSIG;x++)
    signal(x, SIG_DFL);
  
  /* Reset all arguments */
  memset(argv, 0, sizeof(argv));
  
  /* First argument is executable, followed by standard
     arguments for zaptel PPP */
  //argv[argc++] = PPP_EXEC;
  argv[argc++] = prog;
  //argv[argc++] = "nodetach";
  
  /* And all the other arguments */
  stringp=args;
  c = strsep(&stringp, "|");
  while(c && !ast_strlen_zero(c) && (argc < (PPP_MAX_ARGS - 4))) {
    argv[argc++] = c;
    c = strsep(&stringp, "|");
  }
  
  //argv[argc++] = "notty";
  //argv[argc++] = "noauth";
  //argv[argc++] = "sync";
  //argv[argc++] = "192.168.1.1:192.168.1.2";
  
  
#if 0
  for (x=0;x<argc;x++) {
    fprintf(stderr, "Arg %d: %s\n", x, argv[x]);
  }
#endif

  /* Finally launch PPP */
  //execv(PPP_EXEC, argv);
  fprintf(stderr,"Spawning: %s\n",prog);
  execv(prog,argv);
  fprintf(stderr, "Failed to exec %s!\n",prog);
  exit(1);
}


static int ptyfork_exec(struct ast_channel *chan, void *data)
{
  int res = 0;
  struct localuser *u;
  int pid =0 ;
  
  if (!data || ast_strlen_zero((char *)data)) {
    ast_log(LOG_WARNING, "PtyFork Requires arguments\n");
    return -1;
  }
  
  LOCAL_USER_ADD(u);
  
  ast_verbose("Answered now opening Term \n");
  
  /* Answering if not answerd yet */
  if (chan->_state != AST_STATE_UP) {
    res = ast_answer(chan);
    
    if (res <0) return res;
    res=0;
  }
  
  
  {
    int rd_pipe[2], wr_pipe[2], mix_pipe[2];
    int rd_fd,wr_fd;

    pipe (rd_pipe);
    pipe (wr_pipe);

    mix_pipe[0]=wr_pipe[0];
    mix_pipe[1]=rd_pipe[1];    

    rd_fd=rd_pipe[0];
    wr_fd=wr_pipe[1];
    
    {
      char *stringp=data, *prog=strsep(&stringp,"|");
      pid=spawn_ras(mix_pipe, prog, stringp);
    }
    
    ast_verbose("Starting Chan_handler %p \n", chan);
    
    while(1) {
      int ms=1000000000;
      int out;
      struct ast_channel *ast = ast_waitfor_nandfds(&chan,1,&rd_fd,1, NULL, &out, &ms);
      //struct ast_channel *ast=chan;
      //ms = ast_waitfor(chan ,ms);
      
      if (ast) {
	struct ast_frame *frm=ast_read(ast);
	//ast_verbose("Got Frame!\n");
	if (!frm) {
	  ast_verbose("No Frame, closing everything!\n");
	  break;
	}
	
	if (frm && frm->frametype == AST_FRAME_CONTROL) {
	  ast_verbose("Control Frame , closing everything!\n");
 	  break;
 	}
	
	if (frm && frm->frametype == AST_FRAME_VOICE) {
	  int i, k;
	  char *buf=frm->data;
	  //ast_verbose("Read from Chan sending to term \n");
	  flip_buf_bits(buf,frm->samples);
	  i=write(wr_fd, buf, frm->samples);
	  if (i<=0) ast_verbose("written : %d\n",i);
	  for (k=0; k<i; k++)
	    printf("0x%x ",buf[k]);
	  printf("\n");
	  
	  
	}
	
	if (frm) ast_frfree(frm);
	ms=0;
      }
      
      if (out == rd_fd) {
	char rbuf[1024];
 	struct ast_frame frm=
		{ .frametype=AST_FRAME_VOICE,
		  //.subclass=chan->nativeformats,
		  .subclass=AST_FORMAT_ALAW,
		  .mallocd=0, .offset=0, .src=NULL
		};
	
	int i;
	//ast_verbose("Sending frame with chans format: %d (name:%s)\n",chan->nativeformats,chan->name);
	
	i=read(rd_fd,rbuf,1024);
	if (i<=0) {
		ast_verbose("read %d from terminal, so hanging up\n", i);
		close(rd_fd);
	  break;
	}
	
	flip_buf_bits(rbuf,i);
	frm.datalen=frm.samples=i;
	frm.data=rbuf;
	{
	  int k; 
	  for (k=0; k<i; k++)
	    printf("0x%x ",rbuf[k]);
	}
	//printf("\n");
	//ast_verbose("Sending to chan\n");	
	ast_write(chan, &frm);
	
      }
      
      if (!ms) {
	//ast_verbose("Timeout\n");	
      }
      
    }
    
  }
  
  kill(pid, SIGTERM);
  
  LOCAL_USER_REMOVE(u);
  return res;
}
 
int unload_module(void)
{
  STANDARD_HANGUP_LOCALUSERS;
  return ast_unregister_application(app);
}



int load_module(void)
{
	init_flip_bits();
	return ast_register_application(app, ptyfork_exec, synopsis, descrip);
}

char *description(void)
{
  return tdesc;
}

int usecount(void)
{
  int res;
  STANDARD_USECOUNT(res);
  return res;
}

char *key()
{
  return ASTERISK_GPL_KEY;
}
