/*****
 * texfile.cc
 * John Bowman 2003/03/14
 *
 * Encapsulates the writing of commands to a TeX file.
 *****/

#include <ctime>
#include <cfloat>

#include "texfile.h"
#include "errormsg.h"

using std::ofstream;
using settings::getSetting;
  
namespace camp {

std::list<string> TeXpipepreamble, TeXpreamble;
  
texfile::texfile(const string& texname, const bbox& box) : box(box)
{
  texengine=getSetting<mem::string>("tex");
  out=new ofstream(texname.c_str());
  if(!out || !*out) {
    std::cerr << "Cannot write to " << texname << std::endl;
    throw handled_error();
  }
  out->setf(std::ios::fixed);
  out->precision(6);
  texdocumentclass(*out);
  resetpen();
}

texfile::~texfile()
{
  if(out) {
    delete out;  
    out=NULL;
  }
}
  
void texfile::prologue()
{
  texdefines(*out);
  if(settings::latex(texengine)) {
    if(getSetting<bool>("inlinetex"))
      *out << "\\setlength{\\unitlength}{1pt}" << newl;
    else {
      *out << "\\pagestyle{empty}" << newl
	   << "\\textheight=2048pt" << newl
	   << "\\textwidth=\\textheight" << newl;
      if(settings::pdf(texengine))
	*out << "\\oddsidemargin=-17.61pt" << newl
	     << "\\evensidemargin=\\oddsidemargin" << newl
	     << "\\topmargin=-37.01pt" << newl
	     << "\\pdfhorigin=0bp" << newl
	     << "\\pdfvorigin=0bp" << newl
	     << "\\pdfpagewidth=" << box.right-box.left << "bp" << newl
	     << "\\pdfpageheight=" << box.top-box.bottom << "bp" << newl;
      *out << "\\begin{document}" << newl;
    }
  } else {
    *out << "\\hoffset=36.6pt" << newl
	 << "\\voffset=54.0pt" << newl;
    if(settings::pdf(texengine)) {
      *out << "\\hoffset=-20pt" << newl
	   << "\\voffset=0pt" << newl
	   << "\\pdfhorigin=0bp" << newl
	   << "\\pdfvorigin=0bp" << newl
	   << "\\pdfpagewidth=" << box.right-box.left << "bp" << newl
	   << "\\pdfpageheight=" << box.top-box.bottom << "bp" << newl;
    }
  }
}
    
void texfile::beginlayer(const string& psname)
{
  if(box.right > box.left && box.top > box.bottom) {
    *out << "\\includegraphics";
    if(!settings::pdf(texengine))
      *out << "[bb=" << box.left << " " << box.bottom << " "
	   << box.right << " " << box.top << "]";
    *out << "{" << psname << "}%" << newl;
  }
}

void texfile::endlayer()
{
  if(box.right > box.left && box.top > box.bottom)
    *out << "\\kern-" << (box.right-box.left)*ps2tex << "pt%" << newl;
}

void texfile::setlatexcolor(pen p)
{
  if(p.cmyk() && (!lastpen.cmyk() || 
		  (p.cyan() != lastpen.cyan() || 
		   p.magenta() != lastpen.magenta() || 
		   p.yellow() != lastpen.yellow() ||
		   p.black() != lastpen.black()))) {
    *out << "\\definecolor{ASYcolor}{cmyk}{" 
	 << p.cyan() << "," << p.magenta() << "," << p.yellow() << "," 
	 << p.black() << "}\\color{ASYcolor}" << newl;
  } else if(p.rgb() && (!lastpen.rgb() ||
			(p.red() != lastpen.red() ||
			 p.green() != lastpen.green() || 
			 p.blue() != lastpen.blue()))) {
    *out << "\\definecolor{ASYcolor}{rgb}{" 
	 << p.red() << "," << p.green() << "," << p.blue()
	 << "}\\color{ASYcolor}" << newl;
  } else if(p.grayscale() && (!lastpen.grayscale() || 
			      p.gray() != lastpen.gray())) {
    *out << "\\definecolor{ASYcolor}{gray}{" 
	 << p.gray()
	 << "}\\color{ASYcolor}" << newl;
  }
}
  
void texfile::setpen(pen p)
{
  bool latex=settings::latex(texengine);
  
  p.convert();
  if(p == lastpen) return;

  if(latex) setlatexcolor(p);
  else setcolor(p,settings::beginspecial(texengine),settings::endspecial());
  
  if((p.size() != lastpen.size() || p.Lineskip() != lastpen.Lineskip()) &&
     settings::latex(texengine)) {
    *out << "\\fontsize{" << p.size() << "}{" << p.Lineskip()
	 << "}\\selectfont" << newl;
  }

  if(p.Font() != lastpen.Font()) {
    *out << p.Font() << "%" << newl;
  }

  lastpen=p;
}
   
void texfile::gsave()
{
  *out << settings::beginspecial(texengine);
  psfile::gsave(true);
  *out << settings::endspecial() << newl;
}

void texfile::grestore()
{
  *out << settings::beginspecial(texengine);
  psfile::grestore(true);
  *out << settings::endspecial() << newl;
}
  
void texfile::openclip() 
{
  *out << "\\ASYput{" << newl;
}
  
void texfile::closeclip() 
{
  *out << "}%" << newl;
}
  
void texfile::put(const string& label, const transform& T, const pair& z,
		  const pair& align)
{
  double sign=settings::pdf(texengine) ? 1.0 : -1.0;

  if(label.empty()) return;
  
  *out << "\\ASYalign"
       << "(" << (z.getx()-box.right)*ps2tex
       << "," << (z.gety()-box.bottom)*ps2tex
       << ")(" << align.getx()
       << "," << align.gety() 
       << "){" << T.getxx() << " " << sign*T.getyx()
       << " " << sign*T.getxy() << " " << T.getyy()
       << "}{" << label << "}" << newl;
}

void texfile::epilogue()
{
  if(settings::latex(texengine)) {
    if(!getSetting<bool>("inlinetex"))
      *out << "\\end{document}" << newl;
  } else {
      *out << "\\bye" << newl;
  }
  out->flush();
}

} //namespace camp
