/****************************************************************
**
** Attal : Lords of Doom
**
** fight.cpp
** Manages the whole fight
**
** Version : $Id: fight.cpp,v 1.50 2006/03/03 20:36:02 lusum Exp $
**
** Author(s) : Cyrille Verrier, Pascal Audoux - Sardi Carlo
**
** Date : 07/12/2000
**
** Licence :
**  This program is free software; you can redistribute it and/or modify
**      it under the terms of the GNU General Public License as published by
**      the Free Software Foundation; either version 2, or (at your option)
**      any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
****************************************************************/

#include "fight.h"

#include <assert.h>
#include <algorithm>
// include files for QT
#include <QApplication>
#include <QCursor>
#include <QDesktopWidget>
#include <QLabel>
#include <QLayout>
#include <QMessageBox>
#include <QPixmap>
#include <QVBoxLayout>

// application specific include
#include "conf.h"

#include "libCommon/attalCommon.h"
#include "libCommon/attalSocket.h"
#include "libCommon/attalSettings.h"
#include "libCommon/genericFightCell.h"
#include "libCommon/log.h"
#include "libCommon/pathFinder.h"
#include "libCommon/unit.h"

#include "libFight/fightMapView.h"
#include "libFight/fightResult.h"
#include "libFight/fightSettingsDialog.h"

#include "libClient/attalSound.h"
#include "libClient/game.h"
#include "libClient/gui.h"
#include "libClient/imageTheme.h"
#include "libClient/lord.h"
#include "libClient/widget.h"

extern QString DATA_PATH;
extern QString IMAGE_PATH;
extern ImageTheme ImageTheme;


Fight::Fight( QWidget * parent , const char * name )
    : QWidget( parent, name )
{
	_socket = 0;
	_game = 0;
	_popup = 0;
	_isAttack = true;
	_isActive = false;
	_isCreature = false;
	_timer = new QTimer( this );
	_timer->start(100);
	_activeUnit = 0;
	_attUnit = 0;
	_defUnit = 0;
	_lostAttack = new QList<GenericFightUnit *>;
	_lostDefense = new QList<GenericFightUnit *>;
	_attDam = new QList<attalDamages>;

	for( int i = 0; i < MAX_UNIT; i ++ ) {
		_unitsAtt[i] = 0;
		_unitsDef[i] = 0;
	}

	_map = new FightMap( this );
	_pixmap = new QPixmap( IMAGE_PATH + "fight/background_0.png" );
	_map->setBackgroundPixmap( * _pixmap );

	_view = new FightMapView( _map, this );
	_control = new FightControl( this );
	
	_layH = new QHBoxLayout();

	
	_map->resize( _pixmap->width(), _pixmap->height() );
	_view->setMaximumSize( _pixmap->width(), _pixmap->height() );
	_layH->addWidget( _view, 1, Qt::AlignVCenter );

	_layout = new QVBoxLayout( this );
	_layout->addLayout( _layH, 1 );
	_layout->addWidget( _control );
	_layout->activate();

	updateDispositionMode();

	AttalSettings::FightSettings fsettings = AttalSettings::getInstance()->getFightSettings();

	if( fsettings.areCellsVisible ) {
		_map->showCells();
	} else {
		_map->hideCells();
	}
	
	connect( _control, SIGNAL( sig_wait( ) ), SLOT( slot_wait( ) ) );
	connect( _control, SIGNAL( sig_retreat( ) ), SLOT( slot_flee( ) ) );
	connect( _control, SIGNAL( sig_defend() ), SLOT( slot_defend() ) );
	connect( _control, SIGNAL( sig_control() ), SLOT( slot_control() ) );
	connect( _control, SIGNAL( sig_message( QString ) ), SLOT( slot_message( QString ) ) );
	
	connect( _view, SIGNAL( sig_mouseMoved( FightCell * ) ), SLOT( slot_mouseMoved( FightCell * ) ) );
	connect( _view, SIGNAL( sig_mouseRightPressed( FightCell * ) ), SLOT( slot_mouseRightPressed( FightCell * ) ) );
	connect( _view, SIGNAL( sig_mouseLeftPressed( FightCell * , bool) ), SLOT( slot_mouseLeftPressed( FightCell *, bool ) ) );
	connect( _view, SIGNAL( sig_mouseReleased() ), SLOT( slot_mouseReleased() ) );
	connect( _timer, SIGNAL(timeout()),this, SLOT(slot_animateFighting()));
}

Fight::~Fight()
{
	while( ! _lostAttack->isEmpty() ) {
		delete _lostAttack->takeFirst();
	}
	delete _lostAttack;
	
	while( ! _lostDefense->isEmpty() ) {
		delete _lostDefense->takeFirst();
	}
	delete _lostDefense;
	delete _attDam;
}

void Fight::reinit()
{
	for( int i = 0; i < MAX_UNIT; i++ ) {
		if( _unitsAtt[i] ) {
			delete _unitsAtt[i];
			_unitsAtt[i] = 0;
		}
		if( _unitsDef[i] ) {
			delete _unitsDef[i];
			_unitsDef[i] = 0;
		}
	}
	
	while( ! _lostAttack->isEmpty() ) {
		delete _lostAttack->takeFirst();
	}
	
	while( ! _lostDefense->isEmpty() ) {
		delete _lostDefense->takeFirst();
	}
	
	_lordAtt = 0;
	_lordDef = 0;
	_isAttack = true;
	_isActive = false;
	_isCreature = false;
	_activeUnit = 0;
	_attUnit = 0;
	_defUnit = 0;
	_map->reinit();
	_control->clear();
	_result.clear();

	updateDispositionMode();
}

void Fight::updateDispositionMode()
{
	_control->updateResolutionMode();
	update();
}

void Fight::setLord( GenericLord * lord, CLASS_FIGHTER fighter )
{
	if( fighter == FIGHTER_ATTACK ) {
		_isAttack = true;
		_lordAtt = lord;
	} else {
		_isAttack = false;
		_lordDef = lord;
	}
	for( int i = 0; i < MAX_UNIT; i++ ) {
		setUnit( lord->getUnit( i ), fighter, i );
	}
}

void Fight::setOpponent( GenericLord * lord )
{
	if( _isAttack ) {
		_lordDef = lord;
	} else {
		_lordAtt = lord;
	}
}

GenericFightUnit * Fight::getUnit( int num, CLASS_FIGHTER fighter )
{
	if( fighter == FIGHTER_ATTACK ) {
		return _unitsAtt[num];
	} else {
		return _unitsDef[num];
	}
}

void Fight::setUnit( GenericFightUnit * unit, CLASS_FIGHTER fighter, int num )
{
	if( unit ) {
		if( _isCreature ) {
			_creatureRace = unit->getRace();
			_creatureLevel = unit->getLevel();
		}
		FightUnit * currentUnit;
	
		currentUnit = new FightUnit( _map, unit, ( fighter == FIGHTER_ATTACK ) ? true : false );
		currentUnit->setFightMap ( _map );
		currentUnit->setSocket(_socket);
	
		if( fighter == FIGHTER_ATTACK ) {
			if( _unitsAtt[num] ) {
				delete _unitsAtt[num];
			}
			_unitsAtt[num] = currentUnit;
			//currentUnit->goTo( _map->at( num, 1 ) );
			currentUnit->setPosition( _map->at( num, 1 ) );
		} else if( fighter == FIGHTER_DEFENSE ) {
			if( _unitsDef[num] ) {
				delete _unitsDef[num];
			}
			_unitsDef[num] = currentUnit;
			//currentUnit->goTo( _map->at( num, 4 ) );
			currentUnit->setPosition( _map->at( num, 14 ) );
		}
	} else {
		if( ( fighter == FIGHTER_ATTACK ) && ( _unitsAtt[num] ) ) {
			delete _unitsAtt[num];
			_unitsAtt[num] = 0;
		} else if( ( fighter == FIGHTER_DEFENSE ) && _unitsDef[num] ) {
			delete _unitsDef[num];
			_unitsDef[num] = 0;
		}
	}
}


void Fight::slot_mouseMoved( FightCell * cell )
{
	if( cell->getUnit() ) {
		if( cell->getUnit() == _activeUnit ) {
			setCursor( Qt::waitCursor );
		} else if ( !isOpponent(cell->getUnit()) ){
			setCursor( Qt::forbiddenCursor );
		} else {
			if( cell->getAccess() == AttalCommon::FAR_OCCUPIED ) {
				if ( _activeUnit->isDistAttack() ) {
					setCursor( Qt::pointingHandCursor );
				} else {
					setCursor( Qt::forbiddenCursor );
				}
			} else if ( cell->getAccess() == AttalCommon::NEAR_OCCUPIED ) {
				setCursor( Qt::pointingHandCursor );
			} else {
				setCursor( Qt::arrowCursor );
			}
		}
	} else if ( cell->getAccess() == AttalCommon::NEAR_FREE ) {
		setCursor( Qt::arrowCursor );
	} else {
		setCursor( Qt::forbiddenCursor );
	}
}

void Fight::slot_mouseLeftPressed( FightCell * cell, bool isUnit )
{
	GenericFightCell* neib = 0;

	if( _activeUnit && _isActive ) {
		switch( cell->getAccess() ) {
		case AttalCommon::NEAR_FREE :
			moveUnit( cell );
		break;
		case AttalCommon::NEAR_OCCUPIED:
		if( _activeUnit->getDistAttack() > 0 && ( cell->getUnit () ) && ( isOpponent ( cell->getUnit ())) && isUnit) {
			_socket->sendFightDistAttack( giveClass( cell->getUnit() ), giveNum( cell->getUnit() ) );
			//moveUnit( cell );
			_socket->sendFightUnitEndMove();
		} else if ( ( cell->getUnit () ) && ( isOpponent ( cell->getUnit () ) ) && isUnit ) {
			GenericFightCell* startingCell = _activeUnit->getCell();
			moveUnit( cell );
			if( _map->areNeighbours( cell, startingCell )  && _activeUnit->isAnimated() ) {
				_socket->sendFightUnitEndMove();
			}
		} else {
			if( _activeUnit->isLookingToRight() ) {
				neib = _map->getNeighbour5 ( cell );
			} else {
				neib = _map->getNeighbour2 ( cell );
			}
	
			if( neib ) {
				if ( ( neib->getUnit () ) && ( isOpponent ( neib->getUnit () ) ) && isUnit ) {
					moveUnit( cell );
				}
			}
		}
		break;
		case AttalCommon::FAR_OCCUPIED:
		if( ( cell->getUnit () ) && ( _activeUnit->getDistAttack() > 0 ) && ( isOpponent( cell->getUnit() ) ) && isUnit ) {
			_socket->sendFightDistAttack( giveClass( cell->getUnit() ), giveNum( cell->getUnit() ) );
			_socket->sendFightUnitEndMove();
		}
		break;
		default:
		break;
		}
	}
}

void Fight::slot_mouseRightPressed( FightCell * cell )
{
	if( _popup == 0 ) {
		_popup = new PopupUnit( this );
	}
	
	GenericFightUnit * unit = cell->getUnit();
	
	if( unit && ( unit->getNumber() > 0 ) ) {
		_popup->setUnit( unit );
		_popup->show();
		_popup->move( int(cell->x()) + 40, int(cell->y()) + 40 );
	}
}

void Fight::slot_mouseReleased()
{
	if( _popup ) {
		_popup->hide();
	}
}

void Fight::slot_wait( void )
{
	_socket->sendFightWait();
	_activeUnit->setActive( false );
	//_map->clearPath();
}

void Fight::slot_flee( void )
{
	_socket->sendFightFlee();
}

void Fight::slot_defend()
{
	_socket->sendFightDefend();
}

void Fight::slot_animateFighting()
{
	for (int i = 0; i < _attDam->size(); ++i) {
		attalDamages attDam = _attDam->at(i);
		if( !attDam.attUnit->isMoving()) {
			ImageTheme.playSound( AttalSound::SND_HIT );
			//_timer->stop();	
			attDam.attUnit->animate(Fighting);
			handleDamages(attDam);
			if( _popup ) {
				_popup->update();
			}
			_attDam->removeAt(i);
			//_attDam->clear();
		} else {
			//_timer->start(100);
		}
	}
}

void Fight::handleSocket()
{
	_game->handleSocket();
	_socket->reReadData();
	
	switch( _socket->getCla1() ) {
	case SO_MSG:
		socketMsg();
		break;
	case SO_GAME:
		break;
	case SO_TURN:
		break;
	case SO_MODIF:
		break;
	case SO_QR:
		break;
	case SO_MVT:
		break;
	case SO_TECHNIC:
		break;
	case SO_EXCH:
		break;
	case SO_CONNECT:
		break;
	case SO_FIGHT:
		socketFight();
		break;
	default:
		logEE( "Unknown socket_class" );
	}
}

void Fight::socketMsg()
{
	QString msg;
	uchar len = _socket->readChar();
	for( uint i = 0; i < len; i++ ) {
		msg[ i ] = _socket->readChar();
	}
	_control->newMessage( msg );
}




void Fight::socketFight()
{
	switch( _socket->getCla2() ) {
	case C_FIGHT_INIT:
		logEE( "Should not happen (C_FIGHT_INIT in Fight)" );
		break;
	case C_FIGHT_CREATURE:
		_isCreature = true;
		setOpponent( _game->getLord( 0 ) );
		break;
	case C_FIGHT_LORD: {
		GenericLord * lord = _game->getLord( _socket->readChar() );
		lord->setBaseCharac( ATTACK, _socket->readChar() );
		lord->setBaseCharac( DEFENSE, _socket->readChar() );
		setOpponent( lord );
		} break;
	case C_FIGHT_CELL:
		socketFightCell();
		break;
	case C_FIGHT_UNIT: 
		socketFightUnit();
		break;
	case C_FIGHT_MODUNIT:
		socketFightModify();
		break;
	case C_FIGHT_MOVE:
		socketFightMove();
		break;
	case C_FIGHT_ENDMOVE:
		_isActive = false;
		_map->clearPath();
		break;
	case C_FIGHT_ACTIVE:{
		uchar cla = _socket->readChar();
		uchar num = _socket->readChar();
		setActive( (CLASS_FIGHTER)cla, num );
	}
		break;
	case C_FIGHT_DAMAGE:
		socketFightDamage();
		break;
	case C_FIGHT_END: 
		updateUnits();
		_result.fromUChar( _socket->readChar() );
		showFightResults();
		emit sig_quit();
		break;
	}
}

void Fight::updateUnits( void )
{
	GenericLord *oldLord;
	GenericFightUnit * oldUnit, * newUnit;
	uint  i;
	
	if( getAttack() ) {
		oldLord = _game->getLord( getAttack()->getId() );
		for( i = 0; i < MAX_UNIT; i++ ) {
		newUnit = getUnit( i, FIGHTER_ATTACK );
		if(newUnit){
			if(newUnit->getNumber()==0){
			newUnit=0;
			}
		}
		oldUnit = oldLord->getUnit( i );
		if( newUnit && oldUnit ) {
			oldUnit->setNumber( newUnit->getNumber()  );
			oldUnit->setMove( newUnit->getMove() );
			oldUnit->setHealth( newUnit->getHealth() > 0 ? newUnit->getHealth() : 0);
		} else if (oldUnit) {
			oldLord->setUnit(i,0);
		}
		}
	}
	
	if( getDefense() ) {
		oldLord = _game->getLord( getDefense()->getId() );
		for( i = 0; i < MAX_UNIT; i++) {
		newUnit = getUnit( i, FIGHTER_DEFENSE );
		if( newUnit ) {
			if( newUnit->getNumber() == 0 ) {
			newUnit = 0;
			}
		}
		oldUnit = oldLord->getUnit( i );
		if( newUnit && oldUnit ) {
			oldUnit->setNumber( newUnit->getNumber()  );
			oldUnit->setMove( newUnit->getMove() );
			oldUnit->setHealth( newUnit->getHealth() > 0 ? newUnit->getHealth() : 0);
		} else if (oldUnit) {
			oldLord->setUnit(i,0);
		}
		}
	}
	//} else {
		//_crea=false;
	//}
}

void Fight::showFightResults()
{
	FightResultWizard resultDialog( this );
	resultDialog.exec();
}

void Fight::socketFightCell()
{
	int row = _socket->readInt();
	int col = _socket->readInt();
	uchar type = _socket->readChar();
	_map->changeFightCell( row, col, (AttalCommon::TypeFightCell)type );
}

void Fight::socketFightUnit()
{
	uchar cla = _socket->readChar();
	uchar num = _socket->readChar();
	GenericFightUnit * unit = new GenericFightUnit();
	uchar race = _socket->readChar();
	uchar level = _socket->readChar();
	unit->setCreature( race, level );
	unit->setNumber( _socket->readInt() );
	unit->setMove( _socket->readChar() );
	int health = _socket->readInt();
	/*if(health > unit->getMaxHealth()) {
	logEE("bug, health %d, maxhealth %d", health, unit->getMaxHealth());
	}*/
	unit->setHealth( health );
	setUnit( unit, (CLASS_FIGHTER)cla, num );
}

void Fight::socketFightModify()
{
	//logDD("socket modify");
	GenericFightUnit * unit;
	uchar cla = _socket->readChar();
	uchar num = _socket->readChar();
	if( (CLASS_FIGHTER)cla == FIGHTER_ATTACK ) {
		unit = _unitsAtt[ num ];
	} else {
		unit = _unitsDef[ num ];
	}
	uchar race = _socket->readChar();
	uchar level = _socket->readChar();
	unit->setCreature( race, level );
	unit->setNumber( _socket->readInt() );
	unit->setMove( _socket->readChar() );
	unit->setHealth( _socket->readInt() );
}

void Fight::socketFightMove()
{
	uchar cla = _socket->readChar();
	uchar num = _socket->readChar();
	int row = _socket->readInt();
	int col = _socket->readInt();
	
	if( (CLASS_FIGHTER)cla == FIGHTER_ATTACK ) {
		_unitsAtt[ num ]->goTo( _map->at( row, col ) );
	} else {
		_unitsDef[ num ]->goTo( _map->at( row, col ) );
	}
}

void Fight::handleDamages(attalDamages dama)
{
	int numKilled = 0;
	FightUnit* attUnit = 0;
	FightUnit* defUnit = 0;
	
	if ( (CLASS_FIGHTER)dama.claAtt == FIGHTER_ATTACK ) {
		attUnit = _unitsAtt[ dama.numAtt ];
		defUnit = _unitsDef[ dama.numDef ];
	} else {
		attUnit = _unitsDef[ dama.numAtt ];
		defUnit = _unitsAtt[ dama.numDef ];
	}
	defUnit->animate(Defending);
	if( (CLASS_FIGHTER)dama.claDef == FIGHTER_ATTACK ) {
		numKilled = _unitsAtt[ dama.numDef ]->hit( dama.damages );
		//logDD( "DAMAGES %d", numKilled );
		addCasualties( FIGHTER_ATTACK, _unitsAtt[dama.numDef]->getRace(), _unitsAtt[dama.numDef]->getLevel(), numKilled );
	} else {
		numKilled = _unitsDef[ dama.numDef ]->hit( dama.damages );
		addCasualties( FIGHTER_DEFENSE, _unitsDef[dama.numDef]->getRace(), _unitsDef[dama.numDef]->getLevel(), numKilled );
	}
}

void Fight::socketFightDamage()
{
	attalDamages dama; 
	dama.claAtt = _socket->readChar ();
	dama.numAtt = _socket->readChar();
	
	dama.type = (CLASS_ATTACK_TYPE) _socket->readChar ();
	
	dama.claDef = _socket->readChar();
	dama.numDef = _socket->readChar();
	dama.damages = _socket->readInt();

	FightUnit* attUnit = 0;
	FightUnit* defUnit = 0;
	
	if ( (CLASS_FIGHTER)dama.claAtt == FIGHTER_ATTACK ) {
		attUnit = _unitsAtt[ dama.numAtt ];
		defUnit = _unitsDef[ dama.numDef ];
	} else {
		attUnit = _unitsDef[ dama.numAtt ];
		defUnit = _unitsAtt[ dama.numDef ];
	}
	
	_attUnit = attUnit;
	_defUnit = defUnit;
	
	dama.attUnit = attUnit;
	dama.defUnit = defUnit;
	
	_attDam->append(dama);
	
	switch ( dama.type ) {
		case ATTACK_SHOOT: {
			ImageTheme.playSound( AttalSound::SND_ARROW );
			attUnit->animate(Shooting);
			handleDamages(dama);
			_attDam->removeLast();
			if( _popup ) {
				_popup->update();
			}
											 }
				break;
		case ATTACK_FIGHT:
		  slot_animateFighting();	
			break;
		default:
		break;
	};
											 

}

void Fight::setActive( CLASS_FIGHTER cla, int num )
{
	//logDD("set active cla %d, num %d", cla, num);
	if( _activeUnit ) {
		_activeUnit->setActive( false );
	}
	
	if( cla == FIGHTER_ATTACK ) {
		_activeUnit = _unitsAtt[num];
	
		if( _isAttack ) {
			_isActive = true;
		} else {
			_isActive = false;
			_map->clearPath();
		}
	} else {
		_activeUnit = _unitsDef[num];
	
		if( _isAttack ) {
			_isActive = false;
			_map->clearPath();
		} else {
			_isActive = true;
		}
	}
	
	_activeUnit->setActive( true );
	
	if( _isActive ) {
		_map->initPath( _activeUnit );
		///_map->printPath();
	}
}

void Fight::moveUnit( FightCell * cell )
{
	GenericFightCell * current;
	QStack<GenericFightCell *> path = _map->computePath( _activeUnit, cell );
	
	while( ! path.isEmpty() ) {
		current = path.pop();
		_socket->sendFightUnitMove( giveClass( _activeUnit ), giveNum( _activeUnit ), current );
	}
	
	if( !_activeUnit->isAnimated() ) {
		_socket->sendFightUnitEndMove();
		_activeUnit->setActive( false );
	} else {
		_activeUnit->setHaveToStop( true );
	}
	_map->clearPath();
}

int Fight::giveNum( GenericFightUnit * unit )
{
	for( int i = 0; i < MAX_UNIT; i++ ) {
		if( ( _unitsAtt[ i ] == unit ) || ( _unitsDef[ i ] == unit ) ) {
		return i;
		}
	}
	
	logEE( "Should not happen" );
	return -1;
}

CLASS_FIGHTER Fight::giveClass( GenericFightUnit * unit )
{
	for( int i = 0; i < MAX_UNIT; i++ ) {
		if( _unitsAtt[ i ] == unit ) {
		return FIGHTER_ATTACK;
		}
		if( _unitsDef[ i ] == unit ) {
		return FIGHTER_DEFENSE;
		}
	}
	
	logEE( "Should not happen" );
	return FIGHTER_ATTACK;
}

bool Fight::isOpponent( GenericFightUnit * unit )
{
	bool ret;
	
	if( isAttack() ) {
		ret = ( giveClass( unit ) == FIGHTER_DEFENSE );
	} else {
		ret = ( giveClass( unit ) == FIGHTER_ATTACK );
	}
	
	return ret;
}

GenericLord * Fight::getFighterLord( CLASS_FIGHTER fighter )
{
	GenericLord * ret = 0;

	if( fighter == FIGHTER_ATTACK ) {
		ret = _lordAtt;
	} else {
		ret = _lordDef;
	}

	return ret;
}

uint Fight::getCasualtiesNumber( CLASS_FIGHTER fighter )
{
	uint ret = 0;
	
	if( fighter == FIGHTER_ATTACK ) {
		ret =  _lostAttack->count();
	} else {
		ret = _lostDefense->count();
	}
	
	return ret;
}

GenericFightUnit * Fight::getCasualtiesUnit( CLASS_FIGHTER fighter, int numUnit )
{
	GenericFightUnit * ret = 0;
	
	if( fighter == FIGHTER_ATTACK ) {
		if( numUnit < (int)_lostAttack->count() ) {
		ret = _lostAttack->at( numUnit );
		}
	} else {
		if( numUnit < (int)_lostDefense->count() ) {
		ret = _lostDefense->at( numUnit );
		}
	}
	
	return ret;
}

void Fight::addCasualties( CLASS_FIGHTER fighter, uint race, uint level, uint nb )
{
	if( nb == 0 ) {
		return;
	}
	
	bool found = false;
	
	QList<GenericFightUnit *> * list;
	
	if( fighter == FIGHTER_ATTACK ) {
		list = _lostAttack;
	} else {
		list = _lostDefense;
	}
	
	for( int i = 0; i < list->count(); i++ ) {
		GenericFightUnit * unit = list->at( i );
		if( ( unit->getRace() == race ) && ( unit->getLevel() == level ) ) {
		unit->addNumber( nb );
		found = true;
		}
	}
	
	if( !found ) {
		GenericFightUnit * unit = new GenericFightUnit();
		unit->setCreature( race, level );
		unit->setNumber( nb );
		list->append( unit );
	}
}

void Fight::slot_control ( void )
{
	FightSettingsDialog* dlg = new FightSettingsDialog ( this );
	dlg->exec();
	
	if( dlg->result() == QDialog::Accepted ) {
		AttalSettings::FightSettings fsettings = AttalSettings::getInstance()->getFightSettings();
	
		if( fsettings.areCellsVisible ) {
			_map->showCells();
		} else {
			_map->hideCells();
		}
	}
}

void Fight::slot_message( QString msg )
{
	GenericLord * lord;
	
	if( _socket ) {
		if( _isAttack ) {
			lord = _lordAtt;
		} else {
			lord = _lordDef;
		}
		_socket->sendMessage( lord->getOwner()->getName() + " : " + msg );
	} else {
		_control->newMessage( "(Not connected) : " + msg );
	}
}


