/****************************************************************
**
** Attal : Lords of Doom
**
** screen.cpp
** main widget of the editor
**
** Version : $Id: screen.cpp,v 1.38 2006/02/28 14:20:01 lusum Exp $
**
** Author(s) : Pascal Audoux - Sardi Carlo - Forest Darling
**
** Date : 27/08/2000
**
** Licence :    
**	This program is free software; you can redistribute it and/or modify
**   	it under the terms of the GNU General Public License as published by
**     	the Free Software Foundation; either version 2, or (at your option)
**      any later version.
**
**	This program is distributed in the hope that it will be useful,
** 	but WITHOUT ANY WARRANTY; without even the implied warranty of
**	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**	GNU General Public License for more details.
**
****************************************************************/

#include "screen.h"

 
// generic include files
// include files for QT
#include <QCursor>
#include <QFile>
#include <QHBoxLayout>
#include <QInputDialog>
#include <QLabel>
#include <QLayout>
#include <QMenu>
#include <QTextStream>
#include <QVBoxLayout>

// application specific includes
#include "libCommon/genericMapCreature.h"
#include "libCommon/genericPlayer.h"
#include "libCommon/log.h"

#include "libServer/parser.h"

#include "libClient/event.h"
#include "libClient/map.h"
#include "libClient/mapView.h"
#include "libClient/miniMap.h"
#include "libClient/cell.h"
#include "libClient/graphicalArtefact.h"

#include "scenarioEditor/displayQuests.h"
#include "scenarioEditor/itemSelector.h"
#include "scenarioEditor/propertiesBase.h"
#include "scenarioEditor/propertiesBuilding.h"
#include "scenarioEditor/propertiesCreature.h"
#include "scenarioEditor/propertiesEvent.h"
#include "scenarioEditor/propertiesLord.h"
#include "scenarioEditor/propertiesCreature.h"
#include "scenarioEditor/propertiesPlayers.h"
#include "scenarioEditor/scenarioInfo.h"

extern Map * theMap;

Screen::Screen( QWidget * parent, char * name )
	: QWidget( parent, name ), GraphicalGameData()
{
	_leftPressed = false;
	_currentCell = 0;

	theMap = new Map( this );
	_map = theMap;
	_view = new MapView( theMap, this );

	_selector = new ItemSelector( this, this );

	_miniMap = new MiniMap( theMap, this );

	QVBoxLayout * layV1 = new QVBoxLayout();
	layV1->addWidget( _selector, 1 );
	layV1->addWidget( _miniMap );

	QHBoxLayout * layout = new QHBoxLayout( this );
	layout->addLayout( layV1 );
	layout->addWidget( _view, 1 );
	layout->activate();

	_menu = new QMenu( this );
	_menu->insertItem( "&Delete", M_DELETE );
	_menu->insertItem( "&Properties", M_PROPERTIES );

	connect( _menu, SIGNAL( activated( int ) ), SLOT( slot_popup( int ) ) );
	
	connect( _view, SIGNAL( sig_mouseMoved  (Cell *) ), SLOT( slot_mouseMoved  (Cell *) ) );
	connect( _view, SIGNAL( sig_mouseLeftPressed(Cell *) ), SLOT( slot_mouseLeftPressed(Cell *) ) );
	connect( _view, SIGNAL( sig_mouseRightPressed(Cell *) ), SLOT( slot_mouseRightPressed(Cell *) ) );
	connect( _view, SIGNAL( sig_mouseReleased() ), SLOT( slot_mouseReleased() ) );
	connect( _miniMap , SIGNAL( sig_mouseReleasedMinimap( uint, uint ) ), SLOT( slot_centerMinimap( uint, uint ) ) );
	connect( _view, SIGNAL( sig_viewportChanged( int, int , int, int ) ), _miniMap, SLOT( slot_mapviewChanged( int, int , int, int) ) );
	connect( _view, SIGNAL( contentsMoving( int, int ) ), _miniMap, SLOT( slot_mapviewScrolled( int, int) ) );
}

Screen::~Screen()
{
	clearMap();
}


void Screen::slot_mouseLeftPressed( Cell * c )
{
	_selector->handleLeftClick( c );
	if(!_miniMap->isHidden()){
		//XXX : add 3x3 redraw code (doesn't handle well 3x3 brush)
		_miniMap->redrawCell( c );
	}
	_leftPressed = true;
}

void Screen::slot_mouseRightPressed( Cell * c )
{
	_currentCell = c;
	_menu->popup( QCursor::pos() );
}

void Screen::slot_mouseReleased()
{
	_leftPressed = false;
}

void Screen::slot_popup( int num )
{
	switch( num ) {
	case M_DELETE:
		handleDelete();
		break;
	case M_PROPERTIES:
		handleProperties();
		break;
	}
}

void Screen::slot_centerMinimap( uint x, uint y )
{
	_view->center(x, y);
}

void Screen::displayMiniMap( bool state )
{
	if( _miniMap ) {
		if( state ) {
			_miniMap->show();
			_miniMap->redrawMap( theMap );
		} else {
			_miniMap->hide();
		}
	}
}

void Screen::handleDelete()
{
	int it;

	if( _currentCell->getLord() ) {
		GenericLord * lord = _currentCell->getLord();
		if(lord->getOwner()){
			lord->getOwner()->removeLord(lord);
		}
		it = _lords.indexOf( lord );
		if (it != -1) {
			delete _lords.takeAt( it );
		}
		_currentCell->setLord( 0 );
	} else if( _currentCell->getCreature() ) {
		GenericMapCreature * creature = _currentCell->getCreature();
		it = _creatures.indexOf( creature );
		if (it != -1) {
			delete _creatures.takeAt( it );
		}
		_currentCell->setCreature( 0 );
	} else if( _currentCell->getBuilding() ) {
		GenericBuilding * build = _currentCell->getBuilding();
		it = _buildings.indexOf( build );
		if (it != -1) {
			delete _buildings.takeAt( it );
		}
		_currentCell->setBuilding( 0 );
	} else if( _currentCell->getBase() ) {
		GenericBase * base = _currentCell->getBase();
		if(base->getOwner()){
			base->getOwner()->removeBase(base);
		}
		it = _bases.indexOf( base );
		if (it != -1) {
			delete _bases.takeAt( it );
		}
		_currentCell->setBase( 0 );
	} else if( _currentCell->getEvent() ) {
		Event * event = (Event *)( _currentCell->getEvent() );
		it = _events.indexOf( event );
		if (it != -1) {
			delete _events.takeAt( it );
		}
		_currentCell->setEvent( 0 );
	} else if( _currentCell->getDecorationGroup() ) {
		_currentCell->setDecoration( 0, 0 );
	} else if( _currentCell->getTransition() ) {
		_currentCell->setTransition( 0 );
		_currentCell->setTransitionCellType( 0 );
	}
}

void Screen::handleProperties()
{
	if( _currentCell->getLord() ) {
		handlePropertiesLord();
	} else if( _currentCell->getBase() ) {
		handlePropertiesBase();
	} else if( _currentCell->getBuilding() ) {
		handlePropertiesBuilding();
	} else if (_currentCell->getCreature() ) {
		handlePropertiesCreature();
	} else if( _currentCell->getEvent() ) {
		handlePropertiesEvent();
	}
}

void Screen::handlePropertiesLord()
{
	/// XXX: to improve (member for PropeertiesLord... ?? )
	PropertiesLord * propLord = new PropertiesLord( & _players );

	//* Here we'll receive info about all lords and give this info to dialog (for ids...)*/
	int numLords = 0;
	uint* ids = 0;

	for( uint i = 0; i < theMap->getHeight(); i++ ) {
		for( uint j = 0; j < theMap->getWidth(); j++ ) {
			if( theMap->at( i, j )->getLord() ) {
				GenericLord* lord = theMap->at( i, j )->getLord();
				if( lord != _currentCell->getLord() ) {
					numLords++;
				}
			}
		}
	}

	if (numLords > 0) {
		ids = new uint [numLords];
		numLords = 0;
		for( uint i = 0; i < theMap->getHeight(); i++ ) {
			for( uint j = 0; j < theMap->getWidth(); j++ ) {
				if (theMap->at( i, j )->getLord ()) {
					GenericLord* lord = theMap->at( i, j )->getLord ();
					if (lord != _currentCell->getLord ()) {
						ids[numLords] = lord->getId ();
						numLords++;
					}
				}
			}
		}
	}

	propLord->init( _currentCell->getLord(), ids, numLords );
	propLord->exec();

	if (ids) {
		delete[] ids;
	}
	delete propLord;
}

void Screen::handlePropertiesBase()
{
	PropertiesBase * propBase = new PropertiesBase( & _players );
	propBase->init( _currentCell->getBase() );
	propBase->exec();
	
	delete propBase;
}

void Screen::handlePropertiesBuilding()
{
	PropertiesBuilding propBuilding( & _players );
	propBuilding.init( _currentCell->getBuilding() );
	propBuilding.exec();
}

void Screen::handlePropertiesCreature()
{
	PropertiesCreature propCreature;
        propCreature.init( _currentCell->getCreature() );
        propCreature.exec();
}

void Screen::handlePropertiesEvent()
{
	PropertiesEvent propEvent( _currentCell->getEvent() );
	propEvent.exec();
}

void Screen::updateStatusBar( Cell * c )
{
	if( c ) {
		QString msg;

		msg.sprintf( tr( "(Cell %dx%d)" ), c->getRow(), c->getCol() );
		emit sig_status( msg );
	}
}

void Screen::slot_mouseMoved( Cell * c )
{
	if( _currentCell != c ) {
		updateStatusBar( c );
		if( c->getLord() ) {
			setCursor( Qt::waitCursor );
		} else if( c->getBase() ) {
			setCursor( Qt::waitCursor );
		} else if( c->getBuilding() ) {
			setCursor( Qt::waitCursor );
		} else if( c->getEvent() ) {
			setCursor( Qt::waitCursor );
		} else if( c->getCreature() ) {
			setCursor( Qt::waitCursor );
		} else if( c->getDecorationGroup() ) {
			setCursor( Qt::pointingHandCursor );
		} else if( c->getCoeff() < 0  || !c->isStoppable() || !c->isFree() ) {
			setCursor( Qt::forbiddenCursor );
	//	} else if( c->getTransition() ) {
	//		setCursor( Qt::upArrowCursor );
		} else {
			setCursor( Qt::arrowCursor );
		}
		// XXX: _currentCell->setBorder( false );
		_currentCell = c;
		if( _leftPressed ) {
			_selector->handleLeftClick( c );
			if(!_miniMap->isHidden()){
				//XXX : add 3x3 redraw code (doesn't handle well 3x3 brush)
				_miniMap->redrawCell( c );
			}
		}
		// XXX: _currentCell->setBorder( true );
	}
}

void Screen::displayInfo()
{
	ScenarioInfo info( this );

	info.setNbPlayer( getNbPlayer() );
	info.setName( getScenarioName() );
	info.setTheme( getScenarioTheme() );
	info.setDescription( getScenarioDescription() );
	info.setDay(getDateByType( 0 ) );
	info.setWeek(getDateByType( 2 ) );
	info.setMonth(getDateByType( 3 ) );
	info.setYear(getDateByType( 4 ) );
	if( info.exec() ) {
		setNbPlayer( info.getNbPlayer() );
		setScenarioName( info.getName() );
		setScenarioTheme( info.getTheme() );
		setScenarioDescription( info.getDescription() );
		setDateByType( 0, info.getDay() > 0 ? info.getDay() : 1  );
		setDateByType( 1, 1 );
		setDateByType( 2, info.getWeek()  );
		setDateByType( 3, info.getMonth()  );
		setDateByType( 4, info.getYear() );
	}
}

bool Screen::load( const QString filename )
{
	reinit();

	ScenarioParser handler( (GameData*)this );
	QFile file( filename );
	QXmlInputSource source( &file );
	QXmlSimpleReader reader;
	reader.setContentHandler( &handler );
	reader.setErrorHandler( &handler );
	bool ok = reader.parse( source );
	file.close();
	if ( !ok ) {
		logEE( "Parse Error (%s) : %s", filename.toLatin1().constData(), handler.errorProtocol().toLatin1().constData() );
		return false;
	}
	_map->computeStoppable();
	if(!_miniMap->isHidden()){
		_miniMap->redrawMap( theMap );
	}

	return true;
}

void Screen::save( QString filename )
{
	if(!filename.contains(".scn")){
		filename.append(".scn");
	}
	QFile f( filename );

	if (! f.open(QIODevice::WriteOnly) ) {
		logEE("Could not open file %s for writing\n", filename.toLatin1().constData() );
		return;
	}

	QTextStream ts( &f );

	GraphicalGameData::save( &ts );

	f.close();
}

bool Screen::newScenario()
{
	bool ret = false;

	ScenarioInfo info( this );

	if( info.exec() ) {
		ret = true;

		reinit();
		theMap->newMap( info.getMapHeight(), info.getMapWidth() );
		_players.clear();
		setScenarioName( info.getName() );
		setScenarioTheme( info.getTheme() );
		setScenarioDescription( info.getDescription() );
		setNbPlayer( info.getNbPlayer() );
		setDateByType( 0, info.getDay() > 0 ? info.getDay() : 1  );
		setDateByType( 1, 1 );
		setDateByType( 2, info.getWeek() > 0 ? info.getWeek() : 1 );
		setDateByType( 3, info.getMonth()  > 0 ? info.getMonth() : 1);
		setDateByType( 4, info.getYear() > 0 ? info.getYear() : 1044);
		for( uint i = 0; i < info.getNbPlayer(); i++ ) {
			GenericPlayer * player = new GenericPlayer( theMap );
			player->setNum( i );
			player->setName( QString("Player %1").arg(i) );
			_players.append( player );
		}

		if( ! _miniMap->isHidden() ) {
			_miniMap->redrawMap( theMap );
		}
	}
	
	return ret;
}

void Screen::clearMap()
{
	int it;

	if( theMap ) {
		for( uint i = 0; i < theMap->getHeight(); i++ ) {
			for( uint j = 0; j < theMap->getWidth(); j++ ) {
				if( theMap->at( i, j )->getLord() ) {
					GenericLord * lord = theMap->at( i, j )->getLord();
					if(lord->getOwner()){
						lord->getOwner()->removeLord(lord);
					}
					it = _lords.indexOf( lord );
					if (it != -1) {
						delete _lords.takeAt( it );
					}
					theMap->at( i, j )->setLord( 0 );
				}
				if( theMap->at( i, j )->getBuilding() ) {
					GenericBuilding * build = theMap->at( i, j )->getBuilding();
					it = _buildings.indexOf( build );
					if (it != -1) {
						delete _buildings.takeAt( it );
					}
					theMap->at( i, j )->setBuilding( 0 );
				}
				if( theMap->at( i, j )->getBase() ) {
					GenericBase * base = theMap->at( i, j )->getBase();
					if(base->getOwner()){
						base->getOwner()->removeBase(base);
					}
					it = _bases.indexOf( base );
					if (it != -1) {
						delete _bases.takeAt( it );
					}
					theMap->at( i, j )->setBase( 0 );
				}
				if( theMap->at( i, j )->getEvent() ) {
					Event * event = (Event *)( theMap->at( i, j )->getEvent() );
					it = _events.indexOf( event );
					if (it != -1) {
						delete _events.takeAt( it );
					}
					theMap->at( i, j )->setEvent( 0 );
				}
				if( theMap->at( i, j )->getCreature() ) {
					GenericMapCreature * creature = theMap->at( i, j )->getCreature();
					it = _creatures.indexOf( creature );
					if (it != -1) {
						delete _creatures.takeAt( it );
					}
					theMap->at( i, j )->setCreature( 0 );
				}
				if( theMap->at( i, j )->getDecorationGroup() ) {
					theMap->at( i, j )->setDecoration( 0, 0 );
				}
				if( theMap->at( i, j )->getTransition() ) {
					theMap->at( i, j )->setTransitionCellType( 0 );
					theMap->at( i, j )->setTransition( 0 );
				}
			}
		}
		_players.clear();
		theMap->newMap( theMap->getHeight(), theMap->getWidth() );
		if(!_miniMap->isHidden()){
			_miniMap->redrawMap( theMap );
		}
		_map->computeStoppable();
	}
}

void Screen::fillMap( uint tile, uint divers )
{
	if( theMap ) {
		for( uint i = 0; i < theMap->getHeight(); i++ ) {
			for( uint j = 0; j < theMap->getWidth(); j++ ) {
				theMap->at( i, j )->setType( tile );
				if( ( divers == 0 ) && ( tile != 0 ) ) {
					theMap->at( i, j )->setRandomDiversification();
				} else {
					theMap->at( i, j )->setDiversification( divers - 1 );
				}
			}
		}
	}		
	if(!_miniMap->isHidden()){
		_miniMap->redrawMap( theMap );
	}
}

void Screen::displayPlayersProperties ()
{
	PropertiesPlayers propertiesPlayerDialog (this, &_players);

	propertiesPlayerDialog.exec ();
}

void Screen::manageQuests()
{
	DisplayQuests * quests = new DisplayQuests( this );
	quests->reinit( _quests );
	quests->exec();
	delete quests;
}

void Screen::gridMap()
{
	if(theMap) {
		theMap->activateGrid();
	}
}
