/****************************************************************
**
** Attal : Lords of Doom
**
** serverInterface.cpp
** interface for the server
**
** Version : $Id: serverInterface.cpp,v 1.59 2006/08/09 11:36:00 lusum Exp $
**
** Author(s) : Pascal Audoux - Sardi Carlo
**
** Date : 01/11/2000
**
** Licence :    
**	This program is free software; you can redistribute it and/or modify
**   	it under the terms of the GNU General Public License as published by
**     	the Free Software Foundation; either version 2, or (at your option)
**      any later version.
**
**	This program is distributed in the hope that it will be useful,
** 	but WITHOUT ANY WARRANTY; without even the implied warranty of
**	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**	GNU General Public License for more details.
**
****************************************************************/


#include "serverInterface.h"

// generic include files
#include <unistd.h>
#include <stdlib.h>
#include <time.h>

/// XXX: usefull ?
#ifndef WIN32
#include <sys/types.h>
#include <sys/wait.h>
#endif

#include <signal.h>
// include files for QT
#include <QAction>
#include <QApplication>
#include <QButtonGroup>
#include <QCloseEvent>
#include <QFileDialog>
#include <QGroupBox>
#include <QLabel>
#include <QLayout>
#include <QLineEdit>
#include <QMenu>
#include <QMenuBar>
#include <QMessageBox>
#include <QPushButton>
#include <QStatusBar>
#include <QStringList>
#include <QTimer>

// application specific includes
#include "conf.h"
#include "libCommon/dataTheme.h"
#include "libCommon/log.h"

#include "libClient/gui.h"

#include "libServer/campaign.h"

#include "displayScenarii.h"

extern DataTheme DataTheme;
extern QString DATA_PATH;
extern QString SCENARIO_PATH;
extern QString CAMPAIGN_PATH;
extern QString SAVE_PATH;
extern QString THEME;
extern QString PORT;
extern bool FAST;

#define CLIENT

ServerInterface::ServerInterface()
	:QMainWindow()
{
	initActions();
	initMenuBar();
	initStatusBar();
	if( ! DataTheme.init() ) {
		/// XXX: we could manage this better (later :) )
		QMessageBox::critical( this, tr( "Can't load theme" ), tr( "Theme " ) + THEME + tr( " has not been loaded successfully" ) );
	}

	_widget = new ServerWidget( this );
	setCentralWidget( _widget );

	connect( _widget, SIGNAL( sig_stop() ), SLOT( slot_stop() ) );
	connect( _widget, SIGNAL( sig_load( QString ) ), SLOT( slot_load( QString ) ) );
	connect( _widget, SIGNAL( sig_save() ), SLOT( slot_save() ) );
	connect( _widget, SIGNAL( sig_banned( QString ) ), SLOT( slot_banned( QString ) ) );

	_config = 0;
	_actScen = 0;
	_aiDbg = false;
	_hide = false;
	_inLoad = false;
	_winner = true;
	_ready = true;
	_server = 0;
	_engine = 0;

	setMinimumSize( 350, 200 );
	srand( time( NULL ) );
	connect( qApp, SIGNAL( aboutToQuit() ), this, SLOT( slot_stop() ) );
	//connect( qApp, SIGNAL( destroyed() ), this, SLOT( slot_stop() ) );
}

ServerInterface::~ServerInterface()
{
	killAI();
	DataTheme.clear();
}
void ServerInterface::changeEvent ( QEvent * e )
{
	//used to replace setCaption
	switch (e->type()){
	case QEvent::WindowTitleChange:
	{
		setWindowTitle( tr( "Attal - Lords of Doom (Server)" ) );
	}
	break;
	default:
	QWidget::changeEvent ( e );
	break;
	}
}



void ServerInterface::closeEvent( QCloseEvent* ce )
{
	_engine->setFinished(true);
	ce->accept();
}

bool ServerInterface::initServer()
{
	bool ret = false;
	_config = new ConfigConnection( this );
	_config->setHost( "localhost" );
	_config->setPort( PORT.toInt() );
	if(FAST){
		_config->accept();
	} else { 
		_config->exec();
	}
	if( _config->result() == QDialog::Accepted ) {
		_server = new AttalServer( _config->getPort() );

		if( _server->isListening() ) {

			ret = true;
			_engine = new Engine( _server );
			connect( _engine, SIGNAL( sig_newPlayer( AttalPlayerSocket * ) ), _widget, SLOT( slot_newPlayer( AttalPlayerSocket * ) ) );
			connect( _engine, SIGNAL( sig_newPlayer( AttalPlayerSocket * ) ), this, SLOT( slot_ready() ) );
			connect( _engine, SIGNAL( sig_result(bool) ), this, SLOT( slot_result(bool)) );
			connect( _engine, SIGNAL( sig_endGame() ), this, SLOT( slot_endGame()) );
			connect( _server, SIGNAL( sig_endConnection( QString ) ), _widget, SLOT( slot_endConnection( QString ) ) );

		} else {
			delete _server;
			_server = 0;
			if( QMessageBox::warning( this,
						tr( "Server error" ),
						tr( "Could not listen for sockets." ),
						tr( "Try again" ),
						tr( "Quit" ),
						0, 0, 1 )  == 0 ) {
				ret = initServer();
			} else {
				ret = false;
				qApp->quit();
			}
		}
	} else {
		ret = false;
		qApp->quit();
	}

	if (_config){
		delete _config;
	}
	_config = 0;

	return ret;
}

void ServerInterface::initStatusBar()
{
	statusBar()->showMessage( tr( "Status Bar" ), 0 );
}

QAction * ServerInterface::addAction( const QString & label, MENU_ACTIONS id, QSignalMapper * sigmap )
{
	QAction * action;

	action = new QAction( label, this );
	_actions.insert( id, action );
	sigmap->setMapping( action, id );
	connect( action, SIGNAL( activated() ), sigmap, SLOT( map() ) );

	return action;
}

void ServerInterface::initActions()
{
	_actions.resize( NB_ACTIONS );
	
	QAction * action;
	QSignalMapper * sigmap = new QSignalMapper( this );

	addAction( tr( "Load scenario" ), ACTION_LOADSCENARIO, sigmap );
	addAction( tr( "Load campaign" ), ACTION_LOADCAMPAIGN, sigmap );
	addAction( tr( "Continue campaign" ), ACTION_CONTCAMPAIGN, sigmap );
	addAction( tr( "Load game" ), ACTION_LOADGAME, sigmap );
	addAction( tr( "Save game" ), ACTION_SAVE, sigmap );
	addAction( tr( "End game" ), ACTION_END, sigmap );
	addAction( tr( "Quit" ), ACTION_QUIT, sigmap );
	addAction( tr( "Fill with AI" ), ACTION_FILLAI, sigmap );
	addAction( tr( "Fill with External AI" ), ACTION_FILL_EXTERNAL_AI, sigmap );
	addAction( tr( "Add external AI player" ), ACTION_ADDAI, sigmap );
	action = addAction( tr( "Debug AI" ), ACTION_AIDBG, sigmap );
	action->setCheckable( true );
	action->setChecked( _aiDbg );
	
	action = addAction( tr( "Hide AI" ), ACTION_HIDE, sigmap );
	action->setCheckable( true );
	action->setChecked( _hide );
			
	connect( sigmap, SIGNAL( mapped( int ) ), SLOT( slot_action( int ) ) );
}

void ServerInterface::initMenuBar()
{
	QMenu * menuFile = menuBar()->addMenu( tr( "&File" ) );
	QMenu * menuGame = menuBar()->addMenu( tr( "&Game" ) );

	menuFile->addAction( _actions[ ACTION_LOADSCENARIO ] );
	menuFile->addAction( _actions[ ACTION_LOADCAMPAIGN ] );
	menuFile->insertSeparator(_actions[ ACTION_LOADCAMPAIGN ]);
	menuFile->addAction( _actions[ ACTION_CONTCAMPAIGN ] );
	menuFile->addAction( _actions[ ACTION_LOADGAME ] );
	menuFile->addAction( _actions[ ACTION_SAVE ] );
	menuFile->addAction( _actions[ ACTION_END ] );
	menuFile->addAction( _actions[ ACTION_QUIT ] );

	menuGame->addAction( _actions[ ACTION_FILLAI ] );
	menuGame->addAction( _actions[ ACTION_FILL_EXTERNAL_AI ] );
	menuGame->addAction( _actions[ ACTION_ADDAI ] );
	menuGame->addAction( _actions[ ACTION_AIDBG ] );
	menuGame->addAction( _actions[ ACTION_HIDE ] );
	menuGame->addAction( _actions[ ACTION_HIDE ] );

}

void ServerInterface::slot_action( int num )
{
	switch( num ) {
	case ACTION_LOADSCENARIO:{
		DisplayScenariiDialog * scen = new DisplayScenariiDialog( this );
		if( scen->exec() ) {
			slot_load( scen->getFileName() );
		}
		} break;
	case ACTION_LOADCAMPAIGN:{
		QString filename;
		filename = QFileDialog::getOpenFileName( this, tr( "Load campaign" ), CAMPAIGN_PATH, "*.cmp" );
		loadCampaign( filename );
		} break;
	case ACTION_CONTCAMPAIGN:{
		QString filename;
		filename = QFileDialog::getOpenFileName( this, tr( "Load campaign" ), SAVE_PATH, "*.cms" );
		loadCampaign( filename );
		} break;
	case ACTION_LOADGAME:{
		QString filename;
		filename = QFileDialog::getOpenFileName( this, tr( "Load game" ), SAVE_PATH, "*.scn *.gam" );
		slot_load( filename );
		} break;
	case ACTION_SAVE:
		slot_save();
		break;
	case ACTION_END:
		slot_stop();
		break;
	case ACTION_QUIT:
		_engine->setFinished(true);
		qApp->quit();
		break;
	case ACTION_FILLAI:{
		QString filename = _widget->getFilename();
		fillWithAI(filename);
	  } break;
	case ACTION_FILL_EXTERNAL_AI:{
		QString filename = _widget->getFilename();
		fillWithExternalAI(filename);
	  } break;
	case ACTION_ADDAI:
		addAI(false);
		break;
	case ACTION_AIDBG:{
		_aiDbg = !_aiDbg;
	  _actions[ ACTION_AIDBG ]->setChecked( _aiDbg );
		}
		break;
	case ACTION_HIDE:{
			_hide = !_hide;
			_actions[ ACTION_HIDE ]->setChecked( _hide );
		}
		break;
	}
}

bool ServerInterface::killAI()
{
	int count;
	Analyst * ai;
		
	count = _aiList.count();
	if( _aiList.count() > 0 ) {
		for( int i = 0; i < count; i++ ) {
				ai = _aiList.takeFirst();
				delete ai->getSocket();
				delete ai;
		}
	}

	count = _proclist.count();
	if( _proclist.count() > 0 && !_aiDbg) {
		for( int i = 0; i < count; i++ ) {
			_proclist.at(0)->terminate();
			_proclist.at(0)->waitForFinished();
			//QTimer::singleShot( 5000, _proclist.at(0), SLOT( kill() ) );
			delete _proclist.takeFirst();
		}
		return true;	
	}
	return false;
}

bool ServerInterface::fillWithAI(QString filename)
{
	//logDD("filename %s",filename.toLatin1().constData());

	if ( _engine->loadGame(filename, true ) == false ) {
		int nplay = _engine->getNumFillPlayers();
		//logDD("nplay %d",nplay);
		if( nplay > 0 ) {
			//QTimer::singleShot( (nplay+2)* 1000, this, SLOT( slot_ready() ) );
			for( int i = 0; i < nplay; i++ ) {
				_ready = false;
				//logDD("ai %d",i);
				addInternalAI();	
				while( !_ready){
					qApp->processEvents();
				}
			}	
			return true;
		}
	}
	return false;
}

bool ServerInterface::fillWithExternalAI(QString filename)
{
	//logDD("filename %s",filename.toLatin1().constData());

	if ( _engine->loadGame(filename, true ) == false ) {
		int nplay = _engine->getNumFillPlayers();
		if( nplay > 0 ) {
			QTimer::singleShot( (nplay+2)* 1000, this, SLOT( slot_ready() ) );
			for( int i = 0; i < nplay; i++ ) {
				_ready = false;
				//logDD("ai %d",i);
				addAI(_hide);	
				while( !_ready){
					qApp->processEvents();
				}
			}	
			return true;
		}
	}
	return false;
}

void ServerInterface::addInternalAI()
{
	QStringList arglist;
	Analyst * ai = new Analyst();
	AttalSocket * socket = new AttalSocket;
	socket->connectToHost( "localhost", PORT.toInt() );
	ai->aiConnect( socket );
	_aiList.append( ai );
}
	
void ServerInterface::addAI(bool hide)
{
	QProcess * proc = new QProcess(this);
	QStringList arglist;


	connect(proc, SIGNAL(started()), SLOT(slot_started())); 
	_proclist.append( proc );
  
	if( hide ) {
		arglist.append( "--hide" );
	}
	arglist.append( "--fast" );

	_ready = false;
#ifdef WIN32
	proc->start( "attal-ai.exe", arglist );
#else
	QFile file("./attal-ai");

	if (file.exists()) {
		proc->start( "./attal-ai", arglist );
	} else {
		proc->start( "attal-ai", arglist );
	}
#endif
}

void ServerInterface::slot_started()
{
	qApp->processEvents();
}

void ServerInterface::slot_banned( QString name)
{
	_server->closeConnectionPlayer( name );
}

void ServerInterface::slot_ready()
{
	_ready =  true;
}

void ServerInterface::slot_status( QString text )
{
	statusBar()->showMessage( text, 0 );
}

void ServerInterface::slot_stop()
{
	killAI();

	if( _engine ) {
		_engine->endGame();
	}
	_actScen = 0;
	_widget->setGameLoaded( false );
	_winner = false;
}

void ServerInterface::loadCampaign( const QString & fileName )
{
	QString currentPath;
		
	if( ( _server->getNbSocket() == 1 ) && ( ! fileName.isNull() ) ) {
		if( _engine->loadCampaign( fileName ) ) {
			if( _engine->getCampaign() ) {
				QString filename;
				uint nbScen = _engine->getCampaign()->getScenarioNumber();
				uint i;
				for( i = _engine->getCampaign()->getCurrentScenario() ; i < nbScen; i++ ) {
					currentPath = CAMPAIGN_PATH;
					_actScen = i;
					killAI();
					filename = _engine->getCampaign()->getScenario( i );
					if(filename.contains(".gam")){
						currentPath = SAVE_PATH;
					}

					if(_winner) {
						fillWithAI( currentPath + filename );
restart:
						if( _engine->loadGame( currentPath + filename , false) ) {
							//keep these instruction in right order (otherwise, various bugs)
							_winner = false;
							_widget->setGameLoaded( true );
							_engine->startGame();
						} else {
							QMessageBox msb( tr( "Problem" ), tr( "Do you want to continue campaign (control right number of AI)?" ), QMessageBox::Warning, QMessageBox::Yes | 			QMessageBox::Default, QMessageBox::No | QMessageBox::Escape, 0, this );
							if( msb.exec() == QMessageBox::Yes){
								goto restart;
							}
						}
					} else {
						_winner = true;
						break;
					}
				}
				_actScen = 0;
				_engine->cancelCampaign();
			}
		}
	}
}

void ServerInterface::slot_load( QString filename )
{
	if( ( _server->getNbSocket() > 0 ) && ( !filename.isNull() ) ) {
		/* semaphore to avoid process 2 signal at same time */
		if(!_inLoad) {
			_inLoad = true;
			fillWithAI(filename);
			if( _engine->loadGame( filename  , false) ) {
				_widget->setGameLoaded( true );
				_engine->startGame();
			}
			_inLoad = false;
		}
	}
}

void ServerInterface::slot_save()
{
	QString filen;
	filen = QFileDialog::getSaveFileName( this ,"", SAVE_PATH , "*.gam" );
	
	if(!filen.contains(".gam")){
		filen.append(".gam");
	}
	if (!filen.isNull()) {
		_engine->saveGame( filen );
	}

	if(_engine->getCampaign()) {
		QString filenamecmp = filen;
		QString filename;

		filenamecmp.remove(".gam");
		filenamecmp.append(".cms");	
		//filenamecmp = QFileDialog::getSaveFileName( this, "",CAMPAIGN_PATH ,  "*.cms" );

#ifdef WIN32
		filename = filen.section("\\",-1,-1);
#else
		filename = filen.section("/",-1,-1);
#endif

		QFile f( filenamecmp );

		if (! f.open(QIODevice::WriteOnly) ) {
			logEE( "Could not open file %s for writing\n", filename.toLatin1().constData() );
			return;
		}
		Campaign  * campaign = new Campaign;

		QTextStream ts( &f );

		uint nbScen = _engine->getCampaign()->getScenarioNumber();

		campaign->setCurrentScenario(_actScen);
			
		for( uint i = 0; i < nbScen; i++ ) {
			if(i == _actScen){
				campaign->addScenario( filename );
			} else {
				campaign->addScenario( _engine->getCampaign()->getScenario(i) );
			}
		}

		campaign->setTheme( _engine->getCampaign()->getTheme() );
		campaign->setName( _engine->getCampaign()->getName() );
		campaign->setDescription( _engine->getCampaign()->getDescription() );
		campaign->save( & ts );

		f.close();
	}
}

void ServerInterface::slot_result( bool result )
{
	_winner = result;
}

void ServerInterface::slot_endGame()
{
	if(!_aiDbg) {
		killAI();

		_actScen = 0;
		_widget->setGameLoaded( false );
		_winner = false;
	}
}


//
// ----- ChooseFileRadioButton -----
//

ChooseFileWidget::ChooseFileWidget( QWidget * parent, const char * /* name */)
	:QWidget( parent )
{
	QHBoxLayout * layout = new QHBoxLayout( this );
	//layout->addSpacing( 20 );
	
	_edit = new QLineEdit( this );
	layout->addWidget( _edit, 1 );
	layout->addSpacing( 10 );
	
	_choose = new QPushButton( this );
	_choose->setText( tr( "Choose" ) );
	FIXEDSIZE( _choose );
	layout->addWidget( _choose );
	
	layout->activate();
	
	connect( _choose, SIGNAL( clicked() ), SLOT( slot_choose() ) );
	setFixedHeight( _choose->size().height() + 12 );
}

void ChooseFileWidget::slot_choose()
{
	QString filename = QFileDialog::getOpenFileName( this, tr( "Choose scenario" ), "", "*.scn *.gam" ); 
	if( ! filename.isNull() ) {
		_edit->setText( filename );
	}
}

void ChooseFileWidget::slot_toggle( bool st )
{
	_choose->setEnabled( st );
	_edit->setEnabled( st );
}

//
// ----- ServerWidget -----
//

ServerWidget::ServerWidget( QWidget * parent, const char * /* name */)
	: QWidget( parent )
{	
	_groupWidget = new QGroupBox( this );

	_radio1 = new QRadioButton( _groupWidget );
	_radio1->setText( "Demo 1 player" );

	_radio2 = new QRadioButton( _groupWidget );
	_radio2->setText( "Demo 2 players" );

	_radio3 = new QRadioButton( _groupWidget );
	_radio3->setText( "Load game" );

	_choose = new ChooseFileWidget( _groupWidget );
	_choose->slot_toggle( false );

	_playerList = new QTreeWidget( this );
	_playerList->setColumnCount( 2 );
	QStringList labels;
	labels << tr( "Name" );
	labels << tr( "Address" ); 
	_playerList->setHeaderLabels( labels );
	_playerList->setMinimumHeight( 50 );

	QHBoxLayout * layH2 = new QHBoxLayout();
	layH2->setSpacing( 5 );
	layH2->addWidget( _radio3 );
	layH2->addWidget( _choose, 1 );

	QVBoxLayout * layV1 = new QVBoxLayout( _groupWidget );
	layV1->setMargin( 5 );
	layV1->addWidget( _radio1 );
	layV1->addSpacing( 5 );
	layV1->addWidget( _radio2 );
	layV1->addLayout( layH2 );
	layV1->addStretch( 1 );
	layV1->activate();
	
	_butStart = new QPushButton( this );
	_butStart->setText( tr( "Start" ) );
	FIXEDSIZE( _butStart );

	_butSave = new QPushButton( this );
	_butSave->setText( tr( "Save" ) );
	FIXEDSIZE( _butSave );

	_butStop = new QPushButton( this );
	_butStop->setText( tr( "End" ) );
	FIXEDSIZE( _butStop );
	
	QHBoxLayout * layH1 = new QHBoxLayout();
	layH1->setMargin( 5 );
	layH1->setSpacing( 5 );
	layH1->addStretch( 1 );
	layH1->addWidget( _butStart );
	layH1->addStretch( 1 );
	layH1->addWidget( _butSave );
	layH1->addStretch( 1 );
	layH1->addWidget( _butStop );
	layH1->addStretch( 1 );
	
	_butStart->setEnabled( true );
	_butStop->setEnabled( false );
	_butSave->setEnabled( false );
	_groupWidget->setEnabled( true );
	
	QVBoxLayout * layout = new QVBoxLayout( this );
	layout->addWidget( _groupWidget );
	layout->addSpacing( 5 );
	layout->addWidget( _playerList, 1 );
	layout->addSpacing( 5 );
	layout->addLayout( layH1 );
	layout->activate();

	_group = new QButtonGroup( this );
	_group->setExclusive( true );
	_group->addButton( _radio1, 0 );
	_group->addButton( _radio2, 1 );
	_group->addButton( _radio3, 2 );
	_radio1->setChecked( true );
		
	connect( _butStart, SIGNAL( clicked() ), SLOT( slot_start() ) );
	connect( _butSave, SIGNAL( clicked() ), SIGNAL( sig_save() ) );
	connect( _butStop, SIGNAL( clicked() ), SIGNAL( sig_stop() ) );
	connect( _radio3, SIGNAL( toggled( bool ) ), _choose, SLOT( slot_toggle( bool ) ) );
	connect( _playerList, SIGNAL( itemClicked ( QTreeWidgetItem *, int ) ),  SLOT( slot_clicked( QTreeWidgetItem *, int ) ) );
}

void ServerWidget::slot_newPlayer( AttalPlayerSocket * player )
{
	//_playerList->addTopLevelItem( new QTreeWidgetItem( _playerList, player->getPlayer()->getName(), player->peerAddress().toString() )  );
	QStringList entry;
	entry << player->getPlayer()->getName();
	entry << player->peerAddress().toString();
	new QTreeWidgetItem( _playerList, entry );
	/// XXX: to check localAddress or peerAddress ?
}

void ServerWidget::slot_endConnection( QString name )
{
	bool found = false;
	QTreeWidgetItem * item;
	int i = 0;

	while( i < _playerList->topLevelItemCount() ) {
		item = _playerList->topLevelItem( i );

		if( item ) {
			if( item->text( 0 ) == name ) {
				found = true;
				_playerList->takeTopLevelItem( i );
				delete item;
				break;
			} else {
				++i;
			}
		} else {
			found = true;
			break;
		}
	}

	/*
	QTreeWidgetItem * item = _playerList->firstChild();
	do {
		if( item ) {
			if( item->text( 0 ) == name ) {
				found = true;
				_playerList->takeItem( item );
				delete item;
			} else {
				item = item->nextSibling();
			}
		} else {
			found = true;
		}
	} while( !found );*/
}

void ServerWidget::setGameLoaded( bool b )
{
	_loaded = b;
	if( _loaded ) {
		_butStart->setEnabled( false );
		_butStop->setEnabled( true );
		_butSave->setEnabled( true );
		_groupWidget->setEnabled( false );
	} else {
		_butStart->setEnabled( true );
		_butStop->setEnabled( false );
		_butSave->setEnabled( false );
		_groupWidget->setEnabled( true );
	}
}

QString ServerWidget::getFilename()
{
	QString filename;
	logDD(" %d", _group->checkedId() );
	switch( _group->checkedId() ) {
		case 0:
			filename = SCENARIO_PATH + "demo_1player.scn";
			break;
		case 1:
			filename = SCENARIO_PATH + "demo_2players.scn";
			break;
		case 2:
			filename = _choose->getText();
			break;
	}
	return filename;

}

void ServerWidget::slot_start()
{
	QString filename = getFilename();

	emit sig_load( filename );
}

void ServerWidget::slot_clicked(QTreeWidgetItem * item, int column)
{
	//logDD("clicked %s, column %d",item->text(column).toLatin1().constData(), column);
	if(column == 0) {
		QMessageBox msb( "Player", "Do you want kick this player ?", QMessageBox::Warning, QMessageBox::Yes | QMessageBox::Default, QMessageBox::No | QMessageBox::Escape, 0, this );
		if ( msb.exec() == QMessageBox::Yes ) {
			emit sig_banned( item->text(0) );
		}
	}
}

//
// ----- ConfigConnection -----
//

ConfigConnection::ConfigConnection( QWidget * parent )
	: QDialog( parent, Qt::Dialog )
{
	setWindowTitle( tr( "Start server" ) );
	
	QLabel * labHost = new QLabel( tr( "Host : " ), this );
	FIXEDSIZE( labHost );
	
	_host = new QLineEdit( this );
	_host->setFixedSize( 160, 20 );
	_host->setEnabled(false);
		
	QHBoxLayout * layH1 = new QHBoxLayout();
	layH1->addSpacing( 5 );
	layH1->addWidget( labHost );
	layH1->addSpacing( 5 );
	layH1->addWidget( _host );
	layH1->addStretch( 1 );
		
	QLabel * labPort = new QLabel( tr( "Port : " ), this );
	FIXEDSIZE( labPort );
	
	_port = new QLineEdit( this );
	_port->setFixedSize( 80, 20 );
	
	QHBoxLayout * layH2 = new QHBoxLayout();
	layH2->addSpacing( 5 );
	layH2->addWidget( labPort );
	layH2->addSpacing( 5 );
	layH2->addWidget( _port );
	layH2->addStretch( 1 );
		
	QPushButton * pbOk = new QPushButton( this );
	pbOk->setText( tr( "Start" ) );
	pbOk->setFixedSize( 60, 60 );
	FIXEDSIZE( pbOk );
	
	QPushButton * pbCan = new QPushButton( this );
	pbCan->setText( tr( "Quit" ) );
	FIXEDSIZE( pbCan );
	
	QHBoxLayout * layH3 = new QHBoxLayout();
	layH3->addStretch( 1 );
	layH3->addWidget( pbOk );
	layH3->addStretch( 1 );
	layH3->addWidget( pbCan );
	layH3->addStretch( 1 );
	
	QVBoxLayout * layout = new QVBoxLayout( this );
	layout->addLayout( layH1, 1 );
	layout->addLayout( layH2, 1 );
	layout->addLayout( layH3, 2 );
	layout->activate();

	connect( pbOk, SIGNAL( clicked() ), SLOT( accept() ) );
	connect( pbCan, SIGNAL( clicked() ), SLOT( reject() ) );

	setFixedSize( 250, 150 );
}
