#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <sys/stat.h>
#include <gtk/gtk.h>

#include "../include/string.h"

#include "cdialog.h"
#include "obj.h"
#include "win.h"
#include "winlist.h"
#include "winresultsfio.h"
#include "core.h"
#include "config.h"

/* Utilities */
static gchar *CLIST_GET_CELL_TEXT(GtkCList *clist, gint row, gint column);

void WinResultsOpen(
	win_struct *win, const gchar *path,
	const gboolean verbose
);
void WinResultsSave(
	win_struct *win, const gchar *path,
	const gboolean verbose
);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


static gchar *CLIST_GET_CELL_TEXT(GtkCList *clist, gint row, gint column)
{
	gchar *text = NULL;
	guint8 spacing;
	GdkPixmap *pixmap;
	GdkBitmap *mask;

	/* Handle by cell type */
	switch(gtk_clist_get_cell_type(clist, row, column))
	{
	  case GTK_CELL_TEXT:
	    gtk_clist_get_text(
		clist, row, column, &text
	    );
	    break;
	  case GTK_CELL_PIXTEXT:
	    gtk_clist_get_pixtext(
		clist, row, column, &text,
		&spacing, &pixmap, &mask
	    );
	    break;
	  case GTK_CELL_PIXMAP:
	  case GTK_CELL_WIDGET:
	  case GTK_CELL_EMPTY:
	    break;
	}

	return(text);
}


/*
 *	Opens the results from the specified results log file.
 */
void WinResultsOpen(
	win_struct *win, const gchar *path,
	const gboolean verbose
)
{
	struct stat stat_buf;
	gulong file_size;
	gchar buf[1024];
	FILE *fp;
	GtkWidget *toplevel;
	GtkCList *clist;
	win_scan_context_struct *scan_ctx;
	core_struct *core;

	if((win == NULL) || STRISEMPTY(path))
	    return;

	toplevel = win->toplevel;
	clist = GTK_CLIST(win->results_clist);
	core = win->core;
	scan_ctx = win->scan_context;

	/* Open the results file for reading */
	fp = fopen((const char *)path, "rb");
	if(fp == NULL)
	{
	    const gint error_code = (gint)errno;
	    gchar *s, *error_msg = STRDUP(g_strerror(error_code));
	    if(error_msg != NULL)
	    {
		*error_msg = (gchar)toupper((char)*error_msg);
		s = g_strdup_printf(
		    "%s:\n\n    %s",
		    error_msg,
		    path
		);
#ifdef HAVE_EDV2
                EDVPlaySoundWarning(core->edv_ctx);
#endif
                CDialogSetTransientFor(toplevel);
                CDialogGetResponse(
                    "Open Results Log Failed",
		    s,
                    NULL,   
                    CDIALOG_ICON_WARNING,
                    CDIALOG_BTNFLAG_OK,
                    CDIALOG_BTNFLAG_OK
                );
                CDialogSetTransientFor(NULL);
		g_free(s);
		g_free(error_msg);
	    }
	    return;
	}

	if(verbose)
	{
	    gchar *msg = g_strdup_printf(
		"Opening scan results log \"%s\"...",
		g_basename(path)
	    );
	    WinStatusMessage(win, msg, FALSE);
	    g_free(msg);
	    WinStatusProgress(win, 0.0f, TRUE);
	}

	WinResultsLocationUpdate(win, NULL);

	/* Get the file's stats and update the results status */
	if(fstat(fileno(fp), &stat_buf) == 0)
	{
	    file_size = (gulong)stat_buf.st_size;

	    if(stat_buf.st_mtime > 0l)
	    {
		time_t t = (gulong)stat_buf.st_mtime;
		gchar *ts, *msg;

#ifdef HAVE_EDV2
		ts = STRDUP(EDVDateString(core->edv_ctx, (gulong)t));
#else
		ts = STRDUP(ctime(&t));
		if(ts != NULL)
		{
		    gchar *s = (gchar *)strchr((char *)ts, '\n');
		    if(s != NULL)
			*s = '\0';
		}
#endif

		msg = g_strdup_printf(
		    "Log of scan performed on %s",
		    ts
		);
		WinResultsStatusUpdate(win, msg);
		g_free(msg);
		g_free(ts);
	    }
	}
	else
	{
	    file_size = 0l;
	}

	gtk_clist_freeze(clist);

	/* Clear the results list */
	WinResultsListClear(win);

	/* Begin reading the results file */
	while(fgets((char *)buf, sizeof(buf), fp) != NULL)
	{
	    gboolean infected;
	    gint len;
	    gchar	*s,
			*path,
			*virus_name;

	    if(verbose && (file_size > 0l))
		WinStatusProgress(
		    win,
		    (gfloat)ftell(fp) / (gfloat)file_size,
		    TRUE
		);

	    buf[sizeof(buf) - 1] = '\0';

	    /* Remove the first newline character */
	    s = (gchar *)strpbrk((char *)buf, "\n\r");
	    if(s != NULL)
		*s = '\0';

	    /* Set s to the start of the line's contents and seek
	     * past any initial spaces
	     */
	    s = buf;
	    while(ISBLANK(*s))
		s++;

	    /* Skip empty lines */
	    if(*s == '\0')
		continue;

	    /* Is this a comment? */
	    if(*s == CFG_COMMENT_CHAR)
	    {
		s++;

		/* Check if this is a special token
		 *
		 * StartTime
		 */
		if(strcasepfx((const char *)s, "StartTime"))
		{
		    /* Format: "<t>" */
		    const gchar *v = (gchar *)strchr((char *)s, '=');
		    if(v != NULL)
		    {
			v++;
			while(ISBLANK(*v))
			    v++;

			scan_ctx->start_time = (gulong)ATOL(v);
		    }
		}
		/* Duration */
		else if(strcasepfx((const char *)s, "Duration"))
		{
		    /* Format: "<t>" */
		    const gchar *v = (gchar *)strchr((char *)s, '=');
		    if(v != NULL)
		    {
			v++;
			while(ISBLANK(*v))
			    v++;

			scan_ctx->duration = (gulong)ATOL(v);
		    }
		}
		/* NScanned */
		else if(strcasepfx((const char *)s, "NScanned"))
		{
		    /* Format: "<t>" */
		    const gchar *v = (gchar *)strchr((char *)s, '=');
		    if(v != NULL)
		    {
			v++;
			while(ISBLANK(*v))
			    v++;

			scan_ctx->nscanned = (gulong)ATOL(v);
		    }
		}
		/* NInfected */
		else if(strcasepfx((const char *)s, "NInfected"))
		{
		    /* Format: "<t>" */
		    const gchar *v = (gchar *)strchr((char *)s, '=');
		    if(v != NULL)
		    {
			v++;
			while(ISBLANK(*v))
			    v++;

			scan_ctx->ninfected = (gulong)ATOL(v);
		    }
		}
		/* NProblems */
		else if(strcasepfx((const char *)s, "NProblems"))
		{
		    /* Format: "<t>" */
		    const gchar *v = (gchar *)strchr((char *)s, '=');
		    if(v != NULL)
		    {
			v++;
			while(ISBLANK(*v))
			    v++;

			scan_ctx->nproblems = (gulong)ATOL(v);
		    }
		}
		/* Total */
		else if(strcasepfx((const char *)s, "Total"))
		{
		    /* Format: "<t>" */
		    const gchar *v = (gchar *)strchr((char *)s, '=');
		    if(v != NULL)
		    {
			v++;
			while(ISBLANK(*v))
			    v++;

			scan_ctx->total = (gulong)ATOL(v);
		    }
		}
		/* BlocksScanned */
		else if(strcasepfx((const char *)s, "BlocksScanned"))
		{
		    /* Format: "<t>" */
		    const gchar *v = (gchar *)strchr((char *)s, '=');
		    if(v != NULL)
		    {
			v++;
			while(ISBLANK(*v))
			    v++;

			scan_ctx->blocks_scanned = (gulong)ATOL(v);
		    }
		}
		/* Location */
		else if(strcasepfx((const char *)s, "Location"))
		{
		    /* Format: "<path1>,<path2>,<pathn>" */
		    const gchar *v = (gchar *)strchr((char *)s, '=');
		    if(v != NULL)
		    {
			v++;
			while(ISBLANK(*v))
			    v++;

			gtk_label_set_text(
			    GTK_LABEL(win->results_location_label),
			    v
			);
		    }
		}
		/* Status */
		else if(strcasepfx((const char *)s, "Status"))
		{
		    /* Format: "<msg>" */
		    const gchar *v = (gchar *)strchr((char *)s, '=');
		    if(v != NULL)
		    {
			v++;
			while(ISBLANK(*v))
			    v++;

			gtk_label_set_text(
			    GTK_LABEL(win->results_status_label),
			    v
			);
		    }
		}

		continue;
	    }

	    /* Infected */
	    if(*s == '!')
	    {
		infected = TRUE;
		s++;
	    }
	    else
	    {
		infected = FALSE;
	    }

	    /* Path and virus|problem format? */
	    if(*s == '"')
	    {
		gchar *s2;

		s++;		/* Seek past initial '"' character */

		/* Seek s2 to next '"' character */
		for(s2 = s; *s2 != '\0'; s2++)
		{
		    if(*s2 == '"')
			break;
		}
		/* Calculate the length and copy the path */
		len = (gint)(s2 - s);
		path = (gchar *)g_malloc((len + 1) * sizeof(gchar));
		if(path != NULL)
		{
		    memcpy(path, s, len * sizeof(gchar));
		    path[len] = '\0';
		}

		/* Seek s to start of virus|problem */
		s = (*s2 == '"') ? (s2 + 1) : s2; 
		while(ISBLANK(*s))
		    s++;
		virus_name = STRDUP(s);
	    }
	    else
	    {
		path = NULL;
		virus_name = STRDUP(s);
	    }

	    /* Append this row to the results list */
	    WinResultsListAppend(win, path, virus_name, infected);
		 
	    g_free(path);
	    g_free(virus_name);
	}

	gtk_clist_thaw(clist);

	/* Close the results file */
	fclose(fp);

	if(verbose)
	{
	    gchar *msg = g_strdup_printf(
		"Scan results log \"%s\" opened",
		g_basename(path)
	    );
	    WinStatusMessage(win, msg, FALSE);
	    g_free(msg);
	    WinStatusProgress(win, 0.0f, TRUE);
            WinUpdate(win);
	}
}

/*
 *	Saves the results from the Win's Results List to the
 *	specified results log file.
 */
void WinResultsSave(
	win_struct *win, const gchar *path,
	const gboolean verbose
)
{
	gint i, n;
	FILE *fp;
	gchar *virus_name;
	const obj_struct *obj;
	GtkWidget *toplevel;
	GtkCList *clist;
	win_scan_context_struct *scan_ctx;
	core_struct *core;

	if((win == NULL) || STRISEMPTY(path))
	    return;

	toplevel = win->toplevel;
	clist = GTK_CLIST(win->results_clist);
	core = win->core;
	scan_ctx = win->scan_context;

	/* Open the results file for writing */
	fp = fopen((const char *)path, "wb");
	if(fp == NULL)
	{
	    const gint error_code = (gint)errno;
	    gchar *s, *error_msg = STRDUP(g_strerror(error_code));
	    if(error_msg != NULL)
	    {
		*error_msg = (gchar)toupper((char)*error_msg);
		s = g_strdup_printf(
		    "%s:\n\n    %s",
		    error_msg,
		    path
		);
#ifdef HAVE_EDV2
                EDVPlaySoundWarning(core->edv_ctx);
#endif
                CDialogSetTransientFor(toplevel);
                CDialogGetResponse(
                    "Save Results Log Failed",
		    s,
                    NULL,   
                    CDIALOG_ICON_WARNING,
                    CDIALOG_BTNFLAG_OK,
                    CDIALOG_BTNFLAG_OK
                );
                CDialogSetTransientFor(NULL);
		g_free(s);
		g_free(error_msg);
	    }
	    return;
	}

	if(verbose)
	{
	    gchar *msg = g_strdup_printf(
		"Saving scan results log \"%s\"...",
		g_basename(path)
	    );
	    WinStatusMessage(win, msg, FALSE);
	    g_free(msg);
	    WinStatusProgress(win, 0.0f, TRUE);
	}

	/* Write the header */
	if(TRUE)
	{
	    const gchar *s;

	    fprintf(
		fp,
"%c %s - Version %s - Scan Results Log\n\
%c\n\
%c Run view this log verbosely; run %s, click on the Results\n\
%c tab, go to Results->Open..., and open this file.\n\
%c\n",
		CFG_COMMENT_CHAR,
		PROG_NAME_FULL,
		PROG_VERSION,
		CFG_COMMENT_CHAR,
		CFG_COMMENT_CHAR, PROG_NAME,
		CFG_COMMENT_CHAR,
		CFG_COMMENT_CHAR
	    );

	    /* StartTime */
	    fprintf(
		fp,
		"%cStartTime=%ld\n",
		CFG_COMMENT_CHAR,
		scan_ctx->start_time
	    );
	    /* Duration */
	    fprintf(
		fp,
		"%cDuration=%ld\n",
		CFG_COMMENT_CHAR,
		scan_ctx->duration
	    );
	    /* NScanned */
	    fprintf(
		fp,
		"%cNScanned=%ld\n",
		CFG_COMMENT_CHAR,
		scan_ctx->nscanned
	    );
	    /* NInfected */
	    fprintf(
		fp,
		"%cNInfected=%ld\n",
		CFG_COMMENT_CHAR,
		scan_ctx->ninfected
	    );
	    /* NProblems */
	    fprintf(
		fp,
		"%cNProblems=%ld\n",
		CFG_COMMENT_CHAR,
		scan_ctx->nproblems
	    );
	    /* Total */
	    fprintf(
		fp,
		"%cTotal=%ld\n",
		CFG_COMMENT_CHAR,
		scan_ctx->total
	    );
	    /* BlocksScanned */
	    fprintf(
		fp,
		"%cBlocksScanned=%ld\n",
		CFG_COMMENT_CHAR,
		scan_ctx->blocks_scanned
	    );

	    /* Location */
	    s = GTK_LABEL(win->results_location_label)->label;
	    if(!STRISEMPTY(s))
		fprintf(
		    fp,
		    "%cLocation=%s\n",
		    CFG_COMMENT_CHAR,
		    s
		);

	    /* Status */
	    s = GTK_LABEL(win->results_status_label)->label;
	    if(!STRISEMPTY(s))
		fprintf(
		    fp,
		    "%cStatus=%s\n",
		    CFG_COMMENT_CHAR,
		    s
		);
	}

	/* Begin writing each result */
	n = clist->rows;
	for(i = 0; i < n; i++)
	{
	    obj = OBJ(gtk_clist_get_row_data(clist, i));
	    if(obj == NULL)
		continue;

	    /* Get virus|problem from the cell (it is not specified
	     * in the Object
	     */
	    virus_name = CLIST_GET_CELL_TEXT(clist, i, 1);

	    if(STRISEMPTY(obj->path))
	    {
		if(!STRISEMPTY(virus_name))
		    fprintf(fp, "%s\n", virus_name);
	    }
	    else
	    {
		fprintf(
		    fp, "%s\"%s\"",
		    (obj->options & AVSCAN_OPT_INFECTED) ? "!" : "",
		    obj->path
		);
		if(!STRISEMPTY(virus_name))
		    fprintf(fp, " %s\n", virus_name);
	    }

	    if(verbose)
		WinStatusProgress(
		    win,
		    (gfloat)i / (gfloat)n,
		    TRUE
		);
	}

	/* Close the results file */
	fclose(fp);

	if(verbose)
	{
	    gchar *msg = g_strdup_printf(
		"Scan results log \"%s\" saved",
		g_basename(path)
	    );
	    WinStatusMessage(win, msg, FALSE);
	    g_free(msg);
	    WinStatusProgress(win, 0.0f, TRUE);
	    WinUpdate(win);
	}
}
