# Arch Perl library, Copyright (C) 2005 Mikhael Goikhman
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

use 5.005;
use strict;

package AXP::Command::cset::dump;
use base 'AXP::Command::cset';

use Arch::Session;
use Arch::Name;

sub infoline {
	"show all changeset diffs"
}

sub optusage {
	"[options] revision-or-dir-or-tarball"
}

sub options {
	(
		show_changes  => { sh => 'c', desc => "show listing of changes" },
		show_log      => { sh => 'l', desc => "show revision log before the diffs" },
		no_new_files  => { sh => 'N', desc => "don't show added or deleted file diffs" },
		no_arch_files => { sh => 'A', desc => "don't show arch specific file diffs" },
	)
}

sub helptext {
	q{
		Given the revision name or changeset directory or project tree
		directory or changeset/tree tarball (including http:// ones),
		show the content of all diffs in this changeset, optionally
		prefixed by listing of changes or revision log.

		The added and deleted files, as well as arch specific files,
		may be optionally excluded from this combined diff.

		Somewhat similar to "tla show-changeset --diffs", but works on
		more input types.
	}
}

sub execute {
	my $self = shift;
	my %opt = %{$self->{options}};

	my $cset = $self->changeset;
	my $output = "";

	if (delete $opt{show_changes}) {
		$output .= $cset->get_changes->get_listing;
		$output .= "\n" if $opt{show_log};
	}

	if (delete $opt{show_log}) {
		my $revision = $cset->{revision};
		my $session = Arch::Session->instance;
		$output .= $session->get_revision_log($revision)->get_message;
	}

	$output .= $cset->join_all_diffs(%opt);

	print $output;
}

1;
