#!/usr/bin/php4 -q
<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    dbdiff.php                                              */
/* Author:      Paul Waite                                              */
/* Description: Attempt to produce a diff output for differences in     */
/*              two databases. Our output is the SQL required to turn   */
/*              the schema of the target database (dbtarg) into a copy  */
/*              of the reference database (dbref).                      */
/*                                                                      */
/*              NB: In the case of missing columns, we use the special  */
/*              method 'capable_of()' to check whether the database can */
/*              do ALTER <table> DROP <column> and use this if so, else */
/*              we will issue commands to recreate the table.           */
/*                                                                      */
/*              Username and password are optional. If neither are      */
/*              provided, on Unix systems a 'who' command will get your */
/*              currently logged-on username and use that, with no      */
/*              password on the database connection.                    */
/*                                                                      */
/*              Reference and target host/port settings are also        */
/*              optional. These allow you to diff remote databases.     */
/*                                                                      */
/*              The option --no-drops will supress drop statements and  */
/*              you will only get the SQL required to create entitites  */
/*              which are missing. This is useful if you have, for      */
/*              example, a core database schema which is included in    */
/*              a bigger schema. Changes to the core schema can then be */
/*              applied to the larger schema, without dropping all of   */
/*              the additional tables etc.                              */
/*                                                                      */
/*              The --db option is for specifying the database type     */
/*              you are dealing with. It defaults to 'postgres'.        */
/*                                                                      */
/*       usage: ./dbdiff --target=dbtarg --ref=dbref                    */
/*                       [--db=postgres|mysql|oracle|odbc|mssql_server] */
/*                       [--user=username] [--password=passwd]          */
/*                       [--no-drops]                                   */
/*                       [--refver=n.n]                                 */
/*                       [--targver=n.n]                                */
/*                       [--refhost=somehost]                           */
/*                       [--refport=nnnn]                               */
/*                       [--targhost=somehost]                          */
/*                       [--targport=nnnn]                              */
/*                                                                      */
/* ******************************************************************** */
// Find axyl configuration or die..
$AXCONF = "/etc/axyl/axyl.conf";
$fp = fopen($AXCONF, "r");
if ($fp === false) {
  echo "Axyl configuration file $AXCONF not found.\n";
  echo "Please install Axyl first.\n";
  exit;
}
else {
  $config = fread($fp, filesize($AXCONF));
  if (preg_match("/AXYL_HOME=([\S]+)/", $config, $matches)) {
    $AXYL_HOME = $matches[1];
    ini_set("include_path", "$AXYL_HOME/lib");
  }
  else {
    echo "error: failed to parse Axyl configuration file $AXCONF\n";
    echo "Please check contents of this file for AXYL_HOME setting.\n";
    exit;
  }
}

// ----------------------------------------------------------------------
// INCLUDES
include_once("optlist-defs.php");
include_once("response-defs.php");

// ----------------------------------------------------------------------
// PROCESS OPTIONS
$PROGNAME = "dbdiff.php";
$VERSION = "2.1.0";

// Only do anything if there are args..
$dbtarg = "";
$dbref = "";
$user = "";
$password = "";
$nodrops = false;
$dbtype = "postgres";
$DBTYPENAME = "PostgreSQL";

$opts = new optlist();
if ($opts->optcount > 1) {
  // Program name..
  $PROGNAME = $opts->progname;

  // Debugging..
  if ($opts->opt_exists("verbose")) {
    debug_on(DBG_DIAGNOSTIC);
  }

  // Database type
  if ($opts->opt_exists("db")) {
    $dbtype = $opts->opt_value("db");
  }
  switch ($dbtype) {
    case "postgres":
      $DBTYPENAME = "PostgreSQL";
      include_once("pg-schema-defs.php");
      break;
    case "mysql":
      $DBTYPENAME = "MySQL";
      //include_once("my-schema-defs.php");
      die("sorry, schema diff for $DBTYPENAME is not implemented yet.");
      break;
    case "oracle":
      $DBTYPENAME = "ORACLE";
      //include_once("or-schema-defs.php");
      die("sorry, schema diff for $DBTYPENAME is not implemented yet.");
      break;
    case "odbc":
      $DBTYPENAME = "ODBC";
      //include_once("od-schema-defs.php");
      die("sorry, schema diff for $DBTYPENAME is not implemented yet.");
      break;
    case "mssql_server":
      $DBTYPENAME = "SQLServer";
      //include_once("ss-schema-defs.php");
      die("sorry, schema diff for $DBTYPENAME is not implemented yet.");
      break;
    default:
      die("unknown database type '$dbtype'.");
  } // switch

  // Option setting(s)
  $nodrops = $opts->opt_exists("no-drops");

  // Database names
  $dbref  = $opts->opt_value("ref");
  $dbtarg = $opts->opt_value("target");

  // Username
  if ($opts->opt_exists("user")) {
    $user = $opts->opt_value("user");
  }
  else {
    $ubits = explode(" ", exec("who -m"));
    if ($ubits[0] != "") $user = $ubits[0];
    else $user = "postgres";
  }

  // Password..
  if ($opts->opt_exists("password")) {
    $password = $opts->opt_value("password");
  }

  // Reference TCP settings..
  $refhost = "";
  $refport = "";
  if ($opts->opt_exists("refhost")) {
    $refhost = $opts->opt_value("refhost");
    $refport = 5432;
  }
  if ($opts->opt_exists("refport")) {
    $refport = $opts->opt_value("refport");
    if ($refhost == "") $refhost = "localhost";
  }

  // Target TCP settings..
  $targhost = "";
  $targport = "";
  if ($opts->opt_exists("targhost")) {
    $targhost = $opts->opt_value("targhost");
    $targport = 5432;
  }
  if ($opts->opt_exists("targport")) {
    $targport = $opts->opt_value("targport");
    if ($targhost == "") $targhost = "localhost";
  }
}
else {
  echo "usage: $opts->progname --target=dbname --ref=dbname\n";
  echo "              [--db=postgres|mysql|oracle|odbc|mssql_server]\n";
  echo "              [--user=user] [--password=passwd]\n";
  echo "              [--refver=n.nn]\n";
  echo "              [--targver=n.nn]\n";
  echo "              [--no-drops]\n";
  echo "              [--refhost=somehost]\n";
  echo "              [--refport=nnnn]\n";
  echo "              [--targhost=somehost]\n";
  echo "              [--targport=nnnn]\n";
  echo "  Returns SQL which will turn the database schema of --target into --ref.\n";
  echo "  --no-drops returns SQL to create only entities missing from --ref db.\n";
  echo "  --refver forces the Postgres database version.\n";
  echo "  --targver specifies a different Postgres version for generating output.\n";
  echo "  --refhost/refport specifies reference database is on remote host/port.\n";
  echo "  --targhost/targport specifies target on remote host/port.\n";
}

// Debugging..
debugbr("dbtarg = $dbtarg");
debugbr("dbref = $dbref");
debugbr("user = $user");
debugbr("password = $password");
debugbr("nodrops = $nodrops");
debugbr("dbtype = $dbtype");
debugbr("DBTYPENAME = $DBTYPENAME");

// ----------------------------------------------------------------------
// DIFF SCHEMA CLASS
// Supports mechanisms for generating the diff..

class diff_schema extends DB_schema {

  function diff_schema($dbname) {
    $this->DB_schema($dbname);
  }

  // ....................................................................
  /**
  * Produce the SQL required to morph the schema described in the passed
  * dbschema object $db, into the schema we have in this current object.
  * The resulting SQL is commented. This virtual function is database
  * specific.
  * @param object $schema The schema to morph into the current schema
  * @Return string The SQL required to make passed-in schema same as current
  */
  function diff($db, $nodrops=false) {

    // Triggers to drop..
    if ($db->capable_of("triggers")) {
      if (!$nodrops) {
        $dropped_trigs = array();
        foreach ($db->triggers as $dbtrig) {
          if (!isset($this->triggers[$dbtrig->name])) {
            $dropped_trigs[$dbtrig->name] = $dbtrig;
            $diff .= "\n-- DROPPING NON-EXISTENT TRIGGER $dbtrig->name\n";
            $diff .= $dbtrig->drop();
          }
        }
      }
    }

    // Functions to drop
    if ($db->capable_of("stored_procedures")) {
      if (!$nodrops) {
        foreach ($db->functions as $dbfunc) {
          if (!isset($this->functions[$dbfunc->name])) {
            $diff .= "\n-- DROPPING NON-EXISTENT FUNCTION $dbfunc->name\n";
            $diff .= $dbfunc->drop();
          }
        }
      }
    }

    // Check for entities we need to drop or change..
    $tables_dropped = array();
    foreach ($db->tables as $dbtable) {
      if (!isset($this->tables[$dbtable->name])) {
        if (!$nodrops) {
          $diff .= "\n-- DROPPING NON-EXISTENT TABLE $dbtable->name\n";
          $diff .= $dbtable->drop();
          $tables_dropped[] = $dbtable->name;
        }
      }
      else {
        $table = $this->gettable($dbtable->name);
        $dropfields = array();
        $dropdiff = "";
        $alterdiff = "";
        foreach ($dbtable->fields as $dbfield) {
          if (!isset($table->fields[$dbfield->name])) {
            if (!$nodrops) {
              // Field needs to be dropped..
              if ($db->capable_of("alter_table_drop_column")) {
                $dropdiff .= $table->dropcolumn($dbfield);
              }
              else {
                $dropfields[] = $dbfield->name;
              }
            }
          }
          else {
            // Field exists but might have been changed..
            $field = $table->getfield($dbfield->name);
            if ($field->create() != $dbfield->create()) {
              if (!$nodrops && $field->type != $dbfield->type) {
                // Must always recreate to change the type..
                if ($db->capable_of("alter_table_drop_column")) {
                  $dropdiff .= $dbtable->dropcolumn($dbfield);
                  $dropdiff .= $table->addcolumn($field);
                }
                else {
                  $dropfields[] = $dbfield->name;
                }
              }
              else {
                // Changed constraints..
                if ($db->capable_of("check_constraints")) {
                  if (!$dbfield->constraints_match($field)) {
                    foreach ($dbfield->constraints as $dbcon) {
                      if (!isset($field->constraints[$dbcon->name])) {
                        $alterdiff .= $dbcon->drop();
                      }
                      else {
                        $con = $field->constraints[$dbcon->name];
                        if ($con->create_inline() != $dbcon->create_inline()) {
                          $alterdiff .= $dbcon->drop();
                          $alterdiff .= $con->create();
                        }
                      }
                    }
                    foreach ($field->constraints as $con) {
                      if (!isset($dbfield->constraints[$con->name])) {
                        $alterdiff .= $con->create();
                      }
                    }
                  }
                }
                // Changed default or NULL setting..
                if (preg_replace("/[()]/", "", $field->default) != preg_replace("/[()]/", "", $dbfield->default)) {
                  $alterdiff .= $table->setdefault($field);
                }
                if ($field->notnull !== $dbfield->notnull) {
                  $alterdiff .= $table->setnullconstraint($field);
                }
              }
            }
          }
        } // foreach field

        if ($db->capable_of("alter_table_drop_column")) {
          if ($dropdiff != "") {
            $diff .= "\n-- COLUMNS to DROP/RECREATE on TABLE $table->name\n";
            $diff .= $dropdiff;
          }
          if ($alterdiff != "") {
            $diff .= "\n-- COLUMNS to ALTER on TABLE $table->name\n";
            $diff .= $alterdiff;
          }
        }
        else {
          if (!$nodrops && count($dropfields) > 0) {
            $diff .= "\n-- RE-CREATING CHANGED TABLE $table->name\n";
            $diff .= "-- Dropped (altered) fields: " . implode(" ", $dropfields) . "\n";
            $diff .= $dbtable->drop();
            $diff .= $table->create();
          }
          elseif ($alterdiff != "") {
            $diff .= "\n-- COLUMNS to ALTER on TABLE $table->name\n";
            $diff .= $alterdiff;
          }
        }
      } // table exists
    } // foreach $db->tables

    // Check for entities we need to create..
    foreach ($this->tables as $table) {
      if (!isset($db->tables[$table->name])) {
        $diff .= "\n-- CREATING NEW TABLE $table->name\n";
        $diff .= $table->create();
      }
      else {
        $dbtable = $db->gettable($table->name);
        foreach ($table->fields as $field) {
          if (!isset($dbtable->fields[$field->name])) {
            $diff .= "\n-- ADDING NEW COLUMN $field->name\n";
            $diff .= $dbtable->addcolumn($field);
          }
        }
      }
    }

    // Constraints to drop. We always drop these even if the user
    // has specified --no-drops, which only controls entities, not
    // relationship constraints..
    if ($db->capable_of("RI_constraints")) {

      $pkcons_dropped = array();
      foreach ($db->tables as $dbtable) {
        if (!in_array($dbtable->name, $tables_dropped)) {
          foreach ($dbtable->constraints as $dbcon) {
            if (!$this->constraint_exists($dbcon->name)) {
              if (!$nodrops) {
                $diff .= "\n-- DROPPING NON-EXISTENT CONSTRAINT $dbcon->name\n";
                $diff .= $dbcon->drop();
                if ($dbcon->type == "p") $pkcons_dropped[] = $dbcon->name;
              }
            }
          }
        }
      }

      // Constraints to create, or re-create..
      $pkcons_created = array();
      foreach ($this->tables as $table) {
        foreach ($table->constraints as $con) {
          if (!$db->constraint_exists($con->name)) {
            // Constraint not present in target DB..
            if ($con->type == "p") {
              $tablename = $table->name;
              $dbpkexists = false;
              foreach ($db->tables as $dbtable) {
                $dbtable = $db->gettable($tablename);
                if (is_object($dbtable)) {
                  foreach ($dbtable->constraints as $dbcon) {
                    if ($dbcon->type == "p") {
                      $dbpkexists = true;
                      break;
                    }
                  }
                }
              }
              if (!$dbpkexists || in_array($dbcon->name, $pkcons_dropped)) {
                $diff .= "\n-- CREATING NEW PRIMARY KEY CONSTRAINT $con->name\n";
                $diff .= $con->create();
                $pkcons_created[] = $con->name;
              }
              else {
                $diff .= "\n-- NOTICE: Primary key constraint $dbcon->name on table $tablename ";
                $diff .= "is named $con->name in the reference database.\n";
              }
            }
            else {
              $diff .= "\n-- CREATING NEW CONSTRAINT $con->name\n";
              $diff .= $con->create();
            }
          }
          else {
            // Constraint exists in both Dbs..
            $dbcon = $db->getconstraint($con->name);
            if (!$con->matches($dbcon)) {
              $diff .= "\n-- RE-CREATING CHANGED CONSTRAINT $dbcon->name\n";
              $diff .= $dbcon->drop();
              $diff .= $con->create();
              if ($con->type == "p") $pkcons_created[] = $con->name;
            }
            else {
              if ($con->name != $dbcon->name) {
                $diff .= "\n-- NOTICE: Foreign key constraint $dbcon->name on table $tablename ";
                $diff .= "is named $con->name in the reference database.\n";
              }
            }
          }
        }
      }

    } // RI_constaint capable

    // Indexes to drop
    if ($db->capable_of("indexes")) {
      foreach ($db->tables as $dbtable) {
        foreach ($dbtable->indexes as $dbindex) {
          if (!$this->index_exists($dbindex->name)) {
            if (!$nodrops) {
              if (!in_array($dbtable->name, $tables_dropped)) {
                $diff .= "\n-- DROPPING NON-EXISTENT INDEX $dbindex->name\n";
                if (in_array($dbindex->name, $pkcons_dropped)) {
                  $diff .= "-- dropped already via constraint\n";
                }
                elseif ($db->capable_of("unique_index_with_constraint") && $dbindex->unique) {
                  $diff .= "-- cannot drop - index is tied to unique constraint\n";
                }
                $diff .= $dbindex->drop();
              }
            }
          }
        }
      }
      // Indexes to create, or re-create..
      foreach ($this->tables as $table) {
        foreach ($table->indexes as $index) {
          if (!$db->index_exists($index->name)) {
            $diff .= "\n-- CREATING NEW INDEX $index->name\n";
            if ($db->capable_of("unique_index_with_constraint") && ($index->unique || $index->primary)) {
              if (in_array($index->name, $pkcons_created)) {
                $diff .= "-- primary key index already implicitly created via constraint\n";
              }
              else {
                $diff .= "-- cannot create unique index - must be done via a constraint\n";
              }
            }
            else {
              $diff .= $index->create();
            }
          }
          else {
            $dbindex = $db->getindex($index->name);
            if (strcasecmp($index->create(), $dbindex->create()) != 0) {
              $diff .= "\n-- RE-CREATING MODIFIED INDEX $dbindex->name\n";
              if ($this->constraint_exists($dbindex->name)) {
                $diff .= "-- index is part of a constraint - cannot drop/recreate\n";
              }
              else {
                $diff .= $dbindex->drop();
                $diff .= $index->create();
              }
            }
          }
        }
      }
    } // index capable

    // Triggers to create, or re-create. New triggers to be
    // created are deferred until after we have dealt to
    // functions (see below)..
    if ($db->capable_of("triggers")) {
      $trigs_tocreate = array();
      foreach ($this->triggers as $trig) {
        if (!isset($db->triggers[$trig->name])) {
          $trigs_tocreate[$trig->name] = $trig;
        }
        else {
          $dbtrig = $db->gettrigger($trig->name);
          if (strcasecmp($trig->create(), $dbtrig->create()) != 0) {
            $diff .= "\n-- RE-CREATING TRIGGER $dbtrig->name\n";
            $diff .= $dbtrig->drop();
            $diff .= $trig->create();
          }
        }
      }
    } // triggers capable

    // Functions to create, or re-create..
    if ($db->capable_of("stored_procedures")) {
      foreach ($this->functions as $func) {
        if (!isset($db->functions[$func->name])) {
          $diff .= "\n-- CREATING NEW FUNCTION $func->name\n";
          $diff .= $func->create();
        }
        else {
          // This is complicated by the fact that the function is most
          // probably referenced by one or more triggers. We must drop
          // these first, and then recreate them afterward..
          $dbfunc = $db->getfunction($func->name);
          if (strcasecmp($func->create(), $dbfunc->create()) != 0) {
            $diff .= "\n-- RE-CREATING FUNCTION $dbfunc->name\n";

            // Drop triggers referencing our function..
            $diff .= "\n-- Dropping triggers whch reference this function..\n";
            $trigs_to_recreate = array();
            foreach ($this->triggers as $trig) {
              if (!isset($trigs_tocreate[$trig->name])
              && !isset($dropped_trigs[$trig->name])) {
                if ($trig->funcname == $dbfunc->name) {
                    $trigs_to_recreate[] = $trig;
                    $diff .= $trig->drop();
                }
              }
            } // foreach

            // Recreate the function..
            $diff .= "\n-- Recreating function..\n";
            $diff .= $dbfunc->drop();
            $diff .= $func->create();

            // Recreate dropped triggers..
            $diff .= "\n-- Re-establishing dropped triggers..\n";
            foreach ($trigs_to_recreate as $trig) {
              $diff .= $trig->create();
            }
          }
        }
      }
    } // stored_procs capable

    // Create new triggers deferred 'til we'd done functions..
    if ($db->capable_of("triggers")) {
      foreach ($trigs_tocreate as $trigname => $trig) {
        $diff .= "\n-- CREATING NEW TRIGGER $trig->name\n";
        $diff .= $trig->create();
      }
    } // triggers capable

    if ($db->capable_of("named_sequences")) {
      // Sequences to drop
      if (!$nodrops) {
        foreach ($db->sequences as $dbseq) {
          if (!isset($this->sequences[$dbseq->name])) {
            $diff .= "\n-- DROPPING NON-EXISTENT SEQUENCE $dbseq->name\n";
            $diff .= $dbseq->drop();
          }
        }
      }
      // Sequences to create..
      foreach ($this->sequences as $seq) {
        if (!isset($db->sequences[$seq->name])) {
          $diff .= "\n-- CREATING NEW SEQUENCE $seq->name\n";
          $diff .= $seq->create();
        }
      }
    } // sequences capable

    // Return the diffs SQL..
    return $diff;
  } // sqldiff

} // class diff_schema

// ######################################################################
// MAIN PROGRAM
// ######################################################################
//debug_on(DBG_DIAGNOSTIC|DBG_SQL);
//debug_output(DBG_O_CLI);

$RESPONSE = new response();
$RESPONSE->datasource = new datasources();

$RESPONSE->datasource->add_database(
  $dbtype,               // Database type eg: postgres, mssql_server, oracle etc.
  $dbtarg,               // Database name
  $user,                 // Name of user with access to the database
  $password,             // Password for this user
  $targhost,             // Host machine name
  $targport,             // Port number
  "","ISO",              // Char encoding and datestyle
  DEFAULT_DATASOURCE     // DEFAULT_DATASOURCE, or omit for other databases
  );

$RESPONSE->datasource->add_database(
  $dbtype,               // Database type eg: postgres, mssql_server, oracle etc.
  $dbref,                // Database name
  $user,                 // Name of user with access to the database
  $password,             // Password for this user
  $refhost,              // Host machine name
  $refport,              // Port number
  "","ISO"               // Char encoding and datestyle
  );

// Check db's connected ok..
$RESPONSE->datasource->select($dbref);
$RESPONSE->datasource->connect();

$stat1 = $RESPONSE->datasource->connected($dbtarg);
$stat2 = $RESPONSE->datasource->connected($dbref);
if (!$stat1 || !$stat2) {
  if (!$stat1) echo "connect to $dbtarg failed.\n";
  if (!$stat2) echo "connect to $dbref failed.\n";
  exit;
}

// Define both schema holders..
$DBRef  = new diff_schema($dbref);
$DBTarg = new diff_schema($dbtarg);

// User schema versions override..
if ($opts->opt_exists("refver")) {
  $DBRef->database_version = (float) $opts->opt_value("refver");
  $DBTarg->database_version = $DBRef->database_version;
}
if ($opts->opt_exists("targver")) {
  $DBTarg->database_version = (float) $opts->opt_value("targver");
}

// Get schemas..
$DBRef->getschema();
$DBTarg->getschema();

// Announce
$tstamp = date("d/m/Y H:i:s");

echo "----------------------------------------------------------------------------\n";
echo "-- $PROGNAME $VERSION $DBTYPENAME v$DBRef->database_version";
if ($DBRef->database_version != $DBTarg->database_version) {
  echo " (script target: $DBTYPENAME v$DBTarg->database_version)\n";
}
echo "\n";
echo "-- connecting as user $user";
if ($password != "") echo " (password supplied)";
echo " at $tstamp\n";
if ($refhost != "") {
  echo "-- tcp connect $dbref to $refhost:$refport\n";
}
if ($targhost != "") {
  echo "-- tcp connect $dbtarg to $targhost:$targport\n";
}
echo "--\n";
if ($nodrops) {
  echo "-- Option override: elements absent in database $dbref will NOT be\n";
  echo "-- dropped from the target $dbtarg (--no-drops).\n";
  echo "--\n";
  echo "-- Instructions:\n";
  echo "-- apply this script to target database $dbtarg, This will then add\n";
  echo "-- any new elements found in reference database, $dbref.\n";
}
else {
  echo "-- Instructions:\n";
  echo "-- apply this script to target database $dbtarg, This will then\n";
  echo "-- make it identical to the reference database, $dbref.\n";
}
echo "--\n";
echo "----------------------------------------------------------------------------\n";

// Output diff SQL..
echo $DBRef->diff($DBTarg, $nodrops);

// ----------------------------------------------------------------------
?>