#!/usr/bin/php4 -q
<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* NOTE: THIS UTILITY IS NOW DEPRECATED IN FAVOUR OF dbdiff.php         */
/*                                                                      */
/* Filename:    pgdiff.php                                              */
/* Author:      Paul Waite                                              */
/* Description: Attempt to produce a diff output for differences in     */
/*              two databases. Our output is the SQL required to turn   */
/*              the schema of the first named database (dbtarg) into a  */
/*              copy of that of the second named database (dbref).      */
/*                                                                      */
/*              NB: In the case of missing columns, if you are unlucky  */
/*              enough to have Postgres < v7.3 we output a DROP TABLE,  */
/*              followed by a CREATE TABLE with the new schema  which   */
/*              will obviously lose data. Beware! For v7.3+ we use the  */
/*              ALTER <table> DROP <column> command.                    */
/*                                                                      */
/*              Username and password are optional. If neither are      */
/*              provided, on Unix systems a 'who' command will get your */
/*              currently logged-on username and use that, with no      */
/*              password on the database connection.                    */
/*                                                                      */
/*              The option --no-drops will supress drop statements and  */
/*              you will only get the SQL required to create entitites  */
/*              which are missing. This is useful if you have, for      */
/*              example, a core database schema which is included in    */
/*              a bigger schema. Changes to the core schema can then be */
/*              applied to the larger schema, without dropping all of   */
/*              the additional tables etc.                              */
/*                                                                      */
/*       usage: ./pgdiff --target=dbtarg --ref=dbref                    */
/*                       [--user=username] [--password=passwd]          */
/*                       [--no-drops]                                   */
/*                       [--pgversion=7.3]                              */
/*                       [--opversion=7.2]                              */
/*                                                                      */
/* ******************************************************************** */
// CLASS DEFINITIONS
// ----------------------------------------------------------------------
class optlist {
  var $opts = array();
  var $optcount = 0;
  var $progname = "";
  // .....................................................................
  /*
  * Instantiate the optlist object. This does all the work since we expect
  * that this is being done to process command line arguments/options.
  */
  function optlist() {
    global $argc, $argv;
    $this->progname = basename(array_shift($argv));
    $argc -= 1;
    if (isset($argc) && $argc > 0) {
      $optstr = implode(" ", $argv);
      $optstr = str_replace("--", "~", $optstr);
      $optstr = str_replace(" -", "~", $optstr);
      $opts = explode("~", $optstr);
      foreach ($opts as $opt) {
        if ($opt != "") {
          $pos = strpos($opt, "=");
          if ($pos > -1) {
            $optname = trim(substr($opt, 0, $pos));
            $optval  = trim(substr($opt, $pos + 1));
          }
          else {
            $pos = strpos($opt, " ");
            if ($pos > -1) {
              $optname = trim(substr($opt, 0, $pos));
              $optval  = trim(substr($opt, $pos + 1));
            }
            else {
              $optname = trim($opt);
              $optval = "";
            }
          }
          $optval = str_replace("\"", "", $optval);
          $optval = str_replace("'", "", $optval);
          $this->opts[$optname] = $optval;
        }
      }
      $this->optcount = count($this->opts);
    }
  } // optlist

  // .....................................................................
  /**
  * Return the value of the named option. Returns the string associated
  * with this option, or false if it does not exist.
  * @param string $optname Name of the option
  * @return mixed The string value of the option, or false if not valid
  */
  function opt_value($optname) {
    if (isset($this->opts[$optname])) {
      return $this->opts[$optname];
    }
    else {
      return false;
    }
  }

  // .....................................................................
  /**
  * Return status of the named option. Returns true if the option exists
  * or false if it does not exist.
  * @param string $optname Name of the option
  * @return boolean True if it exists or false if it does not.
  */
  function opt_exists($optname) {
    return (isset($this->opts[$optname]));
  }

} // optlist class

// ----------------------------------------------------------------------
/**
* List of things class
* Encapsulates lists of items. A general-purpose class for containing
* lists of things. A utility class to hold lists of things like field
* lists, lists of tablenbames, orderby lists, etc.
* @package query
*/
class listofthings {
  /** The list of things being held */
  var $things;
  /** Total things we have */
  var $total = 0;
  // ....................................................................
  /**
  * Constructor
  * Create a new listofthings object.
  */
  function listofthings() {
    $this->clear();
  }
  // ....................................................................
  /**
  * Add thing
  * @param string $thing The identifier of the thing we are adding
  * @param mixed  $val   The value of the thing we are adding
  */
  function add($thing, $val="") {
    $this->things[$thing] = $val;
    $this->total += 1;
  }
  // ....................................................................
  /**
  * Clear things
  * Clears all things from the list.
  */
  function clear() {
    if (isset($this->things)) unset($this->things);
    $this->total = 0;
  }
  // ....................................................................
  /**
  * Return list
  * @param  string $delim The delimiter for the returned list
  * @return string The delimited list of names of things
  */
  function listed($delim=",") {
    $list = "";
    if (isset($this->things)) {
      reset($this->things);
      while (list($thing, $val) = each($this->things)) {
        $list .= "$thing~^";
      }
      $list = str_replace("~^", $delim, trim($list));
      if (substr($list, -1) == $delim) {
        $list = substr($list, 0, strlen($list) - 1);
      }
    }
    return $list;
  }
  // ....................................................................
  /**
  * Return values
  * @param  string $delim The delimiter for the returned list
  * @return string The delimited list of values of things
  */
  function values($delim=",") {
    $list = "";
    if (isset($this->things)) {
      reset($this->things);
      while (list($thing, $value) = each($this->things)) {
        if ($value === "") $value = "''";
        $list .= "$value~^";
      }
      $list = str_replace("~^", $delim, trim($list));
      if (substr($list, -1) == $delim) {
        $list = substr($list, 0, strlen($list) - 1);
      }
    }
    return $list;
  }
  // ....................................................................
  /**
  * Return equates
  * Returns the things we contain in key=value format, and all
  * as a string delimited by the given character.
  * @param  string $delim The delimiter for the returned list
  * @return string The delimited list of equated things
  */
  function equated($delim=",") {
    $list = "";
    if (isset($this->things)) {
      reset($this->things);
      while (list($thing, $value) = each($this->things)) {
        if ($value === "") $value = "''";
        $list .= "$thing=$value~^";
      }
      $list = str_replace("~^", $delim, trim($list));
      if (substr($list, -1) == $delim) {
        $list = substr($list, 0, strlen($list) - 1);
      }
    }
    return $list;
  }
} // listofthings class

// ----------------------------------------------------------------------
/**
* SQLquery class
* An SQL Statement Text Container.
* This class is the parent of the main dbquery class which directs the
* query to the database. It is mainly a container of SQL query text, in
* the variable 'sql', but also offers a few basic methods for building
* queries. For complex queries however, build your own in a string and
* then just set the 'sql' variable.
* @package query
*/
class sqlquery {
  /** The formatted SQL query itself @see build() */
  var $sql = "";
  /** Type of query 'SELECT', 'INSERT' or 'UPDATE' */
  var $type = "";
  /** List of fields in the query */
  var $fields;
  /** List of tables in the query */
  var $tables;
  /** The query WHERE clause components */
  var $where;
  /** The GROUP BY clause */
  var $groupby;
  /** The ORDER BY clause */
  var $orderby;
  /** The LIMIT clause */
  var $limit;
  /** The OFFSET clause */
  var $offset;
  // ....................................................................
  /**
  * Constructor
  * Create a new SQL Query object.
  * @param string $sql The SQL statement in full
  */
  function sqlquery($sql="") {
    $this->clear();
    $this->sql = $sql;
  }
  // ....................................................................
  /**
  * Clear query
  * Wipe the current sql definition.
  * @param string $sql The SQL statement in full
  */
  function clear() {
    if (isset($this->fields))  unset($this->fields);
    if (isset($this->tables))  unset($this->tables);
    if (isset($this->where))   unset($this->where);
    if (isset($this->groupby)) unset($this->groupby);
    if (isset($this->orderby)) unset($this->orderby);
    $this->fields  = new listofthings();
    $this->tables  = new listofthings();
    $this->where   = new listofthings();
    $this->groupby = new listofthings();
    $this->orderby = new listofthings();
    $this->sql = "";
    $this->limit = 0;
    $this->offset = 0;
  }
  // ....................................................................
  /**
  * Utility function to help building list of things
  * @param listofthings  $list_of_things  listofthings to add to
  * @param mixed         $list            A simple array or a delimited list
  * @param string        $delim           Delimiter, "," default
  */
  function addlist(&$list_of_things, $list, $delim=",") {
    if (is_array($list)) {
      $items = $list;
    }
    else {
      $items = explode($delim, $list);
    }
    // Add to our existing list..
    foreach ($items as $item) {
      if ($item != "") $list_of_things->add($item);
    }
  }
  // ....................................................................
  /**
  * Define field list
  * Add a list of fields to return in query. This is a cumulative function
  * which may be called more than once to add fields. You can specify the
  * list of fields either as an array, or as a delimited list. If the latter,
  * then default delimiter is a comma, unless you specify your own.
  * Applicable to SELECT, DELETE and UPDATE.
  * @param string $field_spec The field list to add to the query
  * @param string $delim The delimter you want to separate fields with
  */
  function fieldlist($field_spec="*", $delim=",") {
    $this->addlist($this->fields, $field_spec, $delim);
  }
  // ....................................................................
  /**
  * Define table list
  * Add the table specification to our list. This is a cumulative function
  * which may be called more than once to add tables. You can specify the
  * list of tables either as an array, or as a delimited list. If the latter,
  * then default delimiter is a comma, unless you specify your own.
  * @param string $table_spec The table list to add to the query
  * @param string $delim The delimiter you want to separate tables with
  */
  function tables($table_spec, $delim=",") {
    $this->addlist($this->tables, $table_spec, $delim);
  }
  // ....................................................................
  /**
  * Define table FROM list
  * A nicer synonym for "tables()" for SELECT
  * @param string $table_spec The table list to add to the query
  * @param string $delim The delimiter you want to separate tables with
  */
  function from($table_spec, $delim=",") {
    $this->tables($table_spec, $delim);
  }
  // ....................................................................
  /**
  * Define table INSERT INTO list
  * A nicer synonym for "tables()" for INSERT
  * @param string $table_spec The table list to add to the query
  * @param string $delim The delimiter you want to separate tables with
  */
  function into($table_spec, $delim=",") {
    $this->tables($table_spec, $delim);
  }
  // ....................................................................
  /**
  * Define group by field list
  * The fields can be an array, or a delimited list. If the latter, then default delimiter is a comma,
  * unless you specify your own.
  * @param string $field_spec The field list to add to the GROUP BY. Do not include words "GROUP BY".
  * @param string $delim The delimiter you want to separate the fields with
  */
  function groupby($field_spec="", $delim=",") {
    $this->addlist($this->groupby, $field_spec, $delim);
  }
  // ....................................................................
  /**
  * Define order field list
  * Defines the Sort order field list. The fields can be an array, or a
  * delimited list. If the latter, then default delimiter is a comma,
  * unless you specify your own.
  * @param string $field_spec The field list to add to the ORDER BY. Do not include words "ORDER BY".
  * @param string $delim The delimiter you want to separate the fields with
  */
  function orderby($field_spec="", $delim=",") {
    $this->addlist($this->orderby, $field_spec, $delim);
  }
  // ....................................................................
  /**
  * Define query LIMIT
  * @param integer $limit Numeric value for limit rows to return. Do not include the word "LIMIT".
  */
  function limit($limit) {
    $this->limit = $limit;
  }
  // ....................................................................
  /**
  * Define query OFFSET
  * @param integer $offset Numeric value for start row. Do not include the word "OFFSET".
  */
  function offset($offset) {
    $this->offset = $offset;
  }
  // ....................................................................
  /**
  * Define field assignments
  * Defines the field assignment clauses for UPDATE and INSERT queries.
  * @param string $field The name of the field to assign a value to
  * @param mixed  $val   The value to assign to the field. Processed according to type.
  */
  function set($field, $val) {
    global $RESPONSE;
    // Numerics are done without quotes
    if (is_int($val) || is_float($val)) {
      $this->fields->add($field, $val);
    }
    // Boolean formats dependent on database type..
    elseif (is_bool($val)) {
      $dbtype = (isset($RESPONSE) ? $RESPONSE->datasource->dbtype() : "postgres");
      switch ($dbtype) {
        case "postgres":
          $this->fields->add($field, ($val ? "'t'" : "'f'"));
          break;
        default:
          $this->fields->add($field, ($val ? "1" : "0"));
          break;
      } // switch
    }
    // Everything else is a quoted, escaped string..
    else {
      $this->fields->add($field, "'" . escape_string(trim($val)) . "'");
    }
  }
  // ....................................................................
  /**
  * Add WHERE clause component
  *
  * This function allows you to add a WHERE clause component. An example might
  * be something like: "AND c.foo='myval'". Either call this once with the whole
  * WHERE cluase string (minus the word "WHERE"), or multiple times with
  * parts of the where clause as in the example above.
  *
  * @param string $where_clause A WHERE clause component, without the "WHERE".
  */
  function where($where_clause) {
    if ($where_clause != "") {
      $this->where->add($where_clause);
    }
  }
  // ....................................................................
  /**
  * Build the SQL query
  * This takes the various components which have been added to the object
  * and parses them to build the full SQL statement which will be sent
  * to the server. The result is stored in $this->sql.
  */
  function build() {
    $sql = "";
    switch (strtoupper($this->type)) {
      case "SELECT":
        $sql .= "SELECT ";
        if ($this->fields->total == 0) $sql .= "*";
        else $sql .= $this->fields->listed();
        $sql .= " FROM ";
        $sql .= $this->tables->listed();
        if ($this->where->total > 0) {
          $sql .= " WHERE ";
          $sql .= $this->where->listed(" ");
        }
        if ($this->groupby->total > 0) {
          $sql .= " GROUP BY ";
          $sql .= $this->groupby->listed();
        }
        if ($this->orderby->total > 0) {
          $sql .= " ORDER BY ";
          $sql .= $this->orderby->listed();
        }
        if ($this->limit > 0) {
          $sql .= " LIMIT $this->limit";
        }
        if ($this->offset > 0) {
          $sql .= " OFFSET $this->offset";
        }
        break;

      case "INSERT":
        $sql .= "INSERT INTO ";
        $sql .= $this->tables->listed();
        if ($this->fields->total > 0) {
          $sql .= " (" . $this->fields->listed() . ")";
        }
        $sql .= " VALUES ";
        $sql .= "(" . $this->fields->values() . ")";
        break;

      case "DELETE":
        $sql .= "DELETE FROM ";
        $sql .= $this->tables->listed();
        if ($this->where->total > 0) {
          $sql .= " WHERE ";
          $sql .= $this->where->listed(" ");
        }
        break;

      case "UPDATE":
        $sql .= "UPDATE ";
        $sql .= $this->tables->listed();
        $sql .= " SET ";
        $sql .= $this->fields->equated();
        if ($this->where->total > 0) {
          $sql .= " WHERE ";
          $sql .= $this->where->listed(" ");
        }
        break;
    }
    // Render any NULL values..
    $this->sql = str_replace("'".NULLVALUE."'", "NULL", $sql);
    return $this->sql;
  }
} // sqlquery class

// ----------------------------------------------------------------------
/**
* DB Query class
* This class is the one which executes queries against the
* connected database.
* @package query
*/
class dbquery extends sqlquery {
  /** Number of rows returned after execute */
  var $rowcount = 0;
  /** Current row in the query */
  var $rowno = 0;
  /** Current row resource ID */
  var $rid = false;
  /** True if query is valid, post execution */
  var $valid = false;
  /** True if data was returned, after execute */
  var $hasdata = false;
  // ....................................................................
  /**
  * Constructor
  * Create a new DB Query object.
  * @param string $sql An SQL statement in full
  */
  function dbquery($sql="") {
    $this->sqlquery($sql);
    return $this;
  }
  // ....................................................................
  /**
  * Exceute the query
  * If we have an SQL phrase, execute it now. We store
  * the result in this->valid, and also return it. If
  * a transaction is open, update the status.
  * @return bool True if query was executed successfully
  */
  function execute() {
    global $RESPONSE;

    $this->rid = false;
    if ($this->sql == "") {
      $this->build();
    }
    if ($this->sql != "") {
      $this->rid = $RESPONSE->datasource->query($this->sql);
      // Now examine the result..
      if ($this->rid != false) {
        $this->rowcount = $RESPONSE->datasource->numrows($this->rid);
        $this->rowno = 0;
        $this->hasdata = ($this->rowcount > 0);
      }
      else {
        // Log the failed query..
        $errstr = "QFAIL: " . APP_NAME . ": " . $this->sql;
        error_log($errstr, 0);
      }
    }
    $this->valid = ($this->rid != false);
    return $this->valid;
  }
  // ....................................................................
  /**
  * Set the SQL statement
  * @param string $sql An SQL statement in full
  */
  function set_sql($sql) {
    $this->tidyup();
    $this->sql = $sql;
    return $this;
  }
  // ....................................................................
  /**
  * Free resources
  * Not really necessary, but you might be that
  * fastidious kind of person.
  */
  function tidyup() {
    global $RESPONSE;
    if ($this->rid) {
      $RESPONSE->datasource->freeresult($this->rid);
      $this->clear();
      $this->rowcount = 0;
      $this->rid = false;
      $this->valid = false;
      $this->hasdata = false;
    }
  }
} // dbquery class

// ----------------------------------------------------------------------
/**
* DB Rows class
* Renders a query into data and allows access to the data either
* directly or via the usual get first,last,next,previous cursor
* navigation.
* This class returns data as "rows" which is to say a standard
* array of data. For the associative array version then please
* @see dbrecords
* NOTE: On creation, it executes the query and positions to the
* initial record (defaulted to the first).
* @package query
*/
class dbrows extends dbquery {
  /** An array containing the current DB row */
  var $current_row;
  // ....................................................................
  /**
  * Constructor
  * Create a new DB Rows object.
  * @param string $sql An SQL statement in full
  */
  function dbrows($sql="") {
    $this->dbquery($sql);
    if ($sql != "") {
      $this->execute();
    }
  }
  // ....................................................................
  /**
  * Execute query
  * Execute this query. We override the parent method here
  * simply to ensure we are positioned at the first row.
  * @return bool True if query was executed successfully
  */
  function execute() {
    dbquery::execute();
    if ($this->valid) {
      $this->get_first();
    }
    return $this->valid;
  }
  // ....................................................................
  /**
  * Set the SQL statement
  * In this case we re-execute the SQL automatically.
  * @param string $sql An SQL statement in full
  * @return bool True if query was executed successfully
  */
  function set_sql($sql) {
    $this->tidyup();
    $this->sql = $sql;
    return $this->execute();
  }
  // ....................................................................
  /**
  * Get row raw
  * Return the given database row from the resultset. This method may
  * be over-ridden in subsequent child classes.
  * @param integer $rowno The row number to return
  * @return array True if row was available
  */
  function get_row_raw($rowno) {
    global $RESPONSE;
    if ($this->rid != false) {
      return $RESPONSE->datasource->fetch_row($this->rid, $rowno);
    }
    else return false;
  }
  // ....................................................................
  /**
  * Get row
  * Return the given database row from the resultset. Uses the
  * get_row_raw() method applicable to this class.
  * @see get_row_raw()
  * @param integer $rowno The row number to return
  * @return mixed The row if it is available, else returns FALSE.
  */
  function get_row($rowno) {
    if ($this->valid && ($this->rowcount > 0)) {
      if ($rowno > ($this->rowcount - 1)) $rowno = $this->rowcount - 1;
      elseif ($rowno < 0) $rowno = 0;
      $this->current_row = $this->get_row_raw($rowno);
      if ($this->current_row) {
        $this->rowno = $rowno;
      }
    }
    else {
      if (isset($this->current_row)) unset($this->current_row);
      $this->current_row = false;
    }
    return $this->current_row;
  }
  // ....................................................................
  /**
  * Refresh the query
  * Re-run the current SQL query. If successful the row will be stored
  * in $this->current_row.
  */
  function refresh() {
    $rowno = $this->rowno;
    $this->execute();
    $this->get_row($rowno);
  }
  // ....................................................................
  /**
  * Get current row
  * If current query is invalid, try to execute it first, then do a
  * get_first(). If query is then valid, return the current row.
  * @see get_first()
  * @return mixed The row if it is available, else returns FALSE.
  */
  function get_current() {
    if (!$this->valid) {
      $this->execute();
      $this->get_first();
    }
    return $this->current_row;
  }
  // ....................................................................
  /**
  * Get current row
  * If current query is invalid, try to execute it first, then do a
  * get_first(). If query is then valid, return the current row.
  * @see get_first()
  * @return mixed The row if it is available, else returns FALSE.
  */
  function get_first() {
    if (!$this->valid) $this->execute();
    return $this->get_row(0);
  }
  // ....................................................................
  /**
  * Get next row
  * If current query is invalid, try to execute it first, then get
  * the next row from the resultset.
  * @return mixed The row if it is available, else returns FALSE.
  */
  function get_next() {
    if (!$this->valid) $this->execute();
    if ($this->rowno < ($this->rowcount - 1)) {
      return $this->get_row($this->rowno + 1);
    }
    else return false;
  }
} // dbrows class

// ----------------------------------------------------------------------
/**
* DB Record class
* Renders a query into data and allows access to the data either
* directly or via the usual get first,last,next,previous cursor
* navigation.
* This class returns data as an associative array and is thus
* the most useful of all the data access methods. It extends the
* dbrows class, and over-rides the get_row_raw method to retrieve
* data.
* @see dbrows.
* @package query
*/
class dbrecords extends dbrows {
  /**
  * Constructor
  * Create a new DB Records object.
  * @param string $sql An SQL statement in full
  */
  function dbrecords($sql="") {
    $this->dbrows($sql);
  }
  // ....................................................................
  /**
  * Get row raw
  * Return the given database row from the resultset. This over-rides
  * the parent method of the same name and returns an array.
  * @param integer $rowno The row number to return
  * @return array True if row was available
  */
  function get_row_raw($rowno) {
    global $RESPONSE;
    if ($this->rid) {
      return $RESPONSE->datasource->fetch_array($this->rid, $rowno);
    }
    else return false;
  }
  // ....................................................................
  /**
  * Get field content
  * Return the field content from the current database array (row).
  * Does not provide ANY pre/post-processing.
  * @param string $fieldname The name of the field to return value of
  * @return mixed Value of the named field
  */
  function rawfield($fieldname) {
    global $RESPONSE;
    if ($this->rid) {
      $value = $this->current_row[$fieldname];
      return $value;
    }
    else return false;
  }
  // ....................................................................
  /**
  * Get field content
  * Return the field content from the current database array (row).
  * We provide some extra functionality here too:
  *  > If the fieldname startes with "^", then the field is expected
  *    to be a datetime and is returned as such.
  *  > If fieldname is formatted: fkname@tablename%keyfield%dispfield
  *    then 'fkname' is the field name to look up from the current
  *    record. We then use the value to lookup in table 'tablename'.
  *    The keyfield is 'keyfield', and the value to return is to
  *    be found in field 'dispfield'. This is foreign key lookup.
  *  > Finally, if the value is a string, then unescape_string is
  *    done automatically for you.
  *
  * @param string $fieldname The name of the field to return value of
  * @return mixed Value of the named field
  */
  function field($fieldname) {
    global $RESPONSE;
    if ($this->rid) {
      $value = $this->rawfield($fieldname);
      if (is_string($value)) {
        $value = unescape_string($value);
      }
      return $value;
    }
    else return false;
  }
  // ....................................................................
  /** Database independent boolean handling. Returns TRUE if the named
  * field in the current row is boolean true, else false.
  * @param string $fieldname The name of the field to return boolean value of
  * @return boolean True if field contains database-dependent true value
  */
  function istrue($fieldname) {
    global $RESPONSE;
    $boolvalue = $this->field($fieldname);
    $dbtype = (isset($RESPONSE) ? $RESPONSE->datasource->dbtype() : "postgres");
    switch ($dbtype) {
      case "postgres":
        $istrue = ($boolvalue == "t");
        break;
      default:
        $istrue = ($boolvalue == 1);
        break;
    }
    // Return result..
    return $istrue;
  }
} // dbrecords class

// ----------------------------------------------------------------------
/** Connect persistent to DB */
define("PERSISTENT",         true);

/** Connect non-persistent to DB */
define("NOT_PERSISTENT",     false);

/** Default datasource for queries @see add_database() */
define("DEFAULT_DATASOURCE", true);

// ----------------------------------------------------------------------
/**
* Datasources
*
* A datasources class is just a bunch of databases. If you want
* to access a database, register it in here first, then you
* can select it to perform queries on later.
* @package database
*/
class datasources {
  /** An array of database objects. All databases we can use as datasources */
  var $database;
  /** Default database name */
  var $db_name_default = "";
  /** Name of currently selected database */
  var $db_name_selected = "";
  /**
  * Constructor
  */
  function datasources() {
  }
  // ....................................................................
  /**
  * Constructor
  * Add a new base to our list of datasources. The dbtype and the name
  * are the only mandatory parameters.
  * @param string  $dbtype    The type of database eg: 'postgres', 'mssql' etc.
  * @param string  $name      The name of the database
  * @param string  $user      Name of a user who can access the database
  * @param string  $passwd    The password the user can access the database with
  * @param string  $host      The hostname of the machine running the database (TCP/IP)
  * @param integer $port      The port number of the database server
  * @param boolean $default   True if the database is the default database
  */
  function add_database($dbtype, $name, $user="", $passwd="", $host="", $port=0, $default=false) {
    global $LIBDIR;
    switch ($dbtype) {
      case "postgres":
        $this->database[$name] = new db_postgres($name, $user, $passwd, $host, $port);
        break;
    }
    // Make sure the default database is selected..
    if ($default) {
      // Select the default DB. This tries to
      // connect to it..
      $this->set_default($name);
      $this->select($name);

      // It is a fatal application error if the default
      // database cannot be connected..
      if (!$this->connected($name)) {
        die();
      }
    }
    return $this;
  }
  // ....................................................................
  /**
  * Selects a database to use
  *
  * This will connect it if it isn't already connected. Calling this
  * with no database name will select the default one. Returns the
  * database unique identifier, or false if none was selected.
  *
  * @param string $dbname  The name of the database to select
  * @return resource The database resource ID
  */
  function select($db_name="") {
    $dbid = false;
    if ($db_name == "") {
      $db_name = $this->db_name_default;
    }
    if (isset($this->database[$db_name])) {
      $db = $this->database[$db_name];
      $this->db_name_selected = $db_name;
      if (!$db->connected) {
        $db->connect(NOT_PERSISTENT);
        $this->database[$db_name] = $db;
      }
      if ($db->connected) {
        $dbid = $db->dbid;
      }
    }
    return $dbid;
  }
  // ....................................................................
  /**
  * Sets default database
  *
  * Internal function to set the name of the default database.
  * The database must exist as a defined database already.
  * @access private
  * @param string $dbname  The name of the database
  */
  // Private method:
  // Sets the database to be used as the default.
  function set_default($db_name) {
    if (isset($this->database[$db_name])) {
      $this->db_name_default = $db_name;
      return $this;
    }
  }
  // ....................................................................
  /**
  * Database resource ID
  *
  * Returns the database resource ID of the given database name.
  * If dbname is not given, returns ID of currently selected DB.
  * @access private
  * @param string $dbname  The name of the database
  * @return resource Database resource ID
  */
  function dbid($db_name="") {
    $res = false;
    if ($db_name == "") {
      $db_name = $this->db_name_selected;
    }
    if (isset($this->database[$db_name])) {
      $db = $this->database[$db_name];
      $res = $db->dbid;
    }
    return $res;
  }
  // ....................................................................
  /**
  * Database type
  *
  * Returns the database type of the given database name.
  * If dbname is not given, returns type of DB currently selected.
  * @access private
  * @param string $dbname  The name of the database
  * @return string Database type string
  */
  // Returns the database type of the selected database.
  function dbtype($db_name="") {
    $res = false;
    if ($db_name == "") {
      $db_name = $this->db_name_selected;
    }
    if (isset($this->database[$db_name])) {
      $db = $this->database[$db_name];
      $res = $db->type;
    }
    return $res;
  }
  // ....................................................................
  /**
  * Connected status
  *
  * Returns connected status of named database, or the currently
  * selected one if no name given.
  * @access private
  * @param string $dbname  The name of the database
  * @return boolean Database connection status true or false
  */
  function connected($db_name="") {
    $res = false;
    if ($db_name == "") {
      $db_name = $this->db_name_selected;
    }
    if (isset($this->database[$db_name])) {
      $db = $this->database[$db_name];
      $res = $db->connected;
    }
    return $res;
  }
  // ....................................................................
  /**
  * Connect
  *
  * Connects to the database which has been selected in the mode
  * specified, or non-peristent otherwise.
  * @access private
  * @param boolean $persistent  Whether to connect persistently or not
  * @return boolean Whether database connection was successful
  */
  function connect($persistent=NOT_PERSISTENT) {
    $connected = false;
    if (isset($this->database[$this->db_name_selected])) {
      $db = $this->database[$this->db_name_selected];

      $db->connect($persistent);
      if ($db->connected) {
        $connected = true;
      }
      else {
        $errmsg  = "Failed to connect to database '" . $this->name . "' ";
        $errmsg .= "type='"   . $this->type . "' ";
        $errmsg .= "host='"   . $this->host . "' ";
        $errmsg .= "port='"   . $this->port . "' ";
        $errmsg .= "user='"   . $this->user . "' ";
        $errmsg .= "passwd='" . $this->passwd . "' ";
        if ($persistent) $errmsg .= "persistent=yes";
        else $errmsg .= "persistent=no";
        error_log("CONNFAIL: $errmsg");
      }
    }
    return $connected;
  }
  // ....................................................................
  /**
  * Disconnect
  *
  * Disconnect the currently selected database.
  * @access private
  */
  function disconnect() {
    if (isset($this->database[$this->db_name_selected])) {
      $db = $this->database[$this->db_name_selected];
      $db->disconnect();
    }
  }
  // ....................................................................
  /**
  * Query
  *
  * The main query interface. Send a query to the database.
  * @access private
  * @param string $sql The query string to be executed
  */
  function query($sql) {
    $rid = false;
    if (isset($this->database[$this->db_name_selected])) {
      $db = $this->database[$this->db_name_selected];
      $rid = $db->query($sql);
    }
    return $rid;
  }
  // ....................................................................
  /**
  * Return number of rows
  *
  * Returns the number of rows in the current data set.
  * @access private
  * @param resource $rid The query resource ID
  */
  function numrows($rid) {
    $rows = 0;
    if (isset($this->database[$this->db_name_selected])) {
      $db = $this->database[$this->db_name_selected];
      $rows = $db->numrows($rid);
    }
    return $rows;
  }
  // ....................................................................
  /**
  * Free result
  *
  * Free the result of a query
  * @access private
  * @param resource $rid The query resource ID
  */
  function freeresult($rid) {
    if (isset($this->database[$this->db_name_selected])) {
      $db = $this->database[$this->db_name_selected];
      $db->freeresult($rid);
    }
  }
  // ....................................................................
  /**
  * Get DB error message
  *
  * Return the last error message reported by the database
  * @access private
  * @return string The error message, if any
  */
  function errormessage() {
    $errmsg = "";
    if (isset($this->database[$this->db_name_selected])) {
      $db = $this->database[$this->db_name_selected];
      $errmsg = $db->errormessage();
    }
    return $errmsg;
  }
  // ....................................................................
  /**
  * Fetch result row
  *
  * Get the given row of the result set as an enumerated array. Note
  * that fetch_array is more useful, as it returns the data in an
  * associative array.
  * @access private
  * @see fetch_array()
  * @param resource $rid   The query resource ID
  * @param integer  $rowno The number of the row to fetch
  * @return array  The row of fields as an enumerated array
  */
  function fetch_row($rid, $rowno) {
    $rows = false;
    if (isset($this->database[$this->db_name_selected])) {
      $db = $this->database[$this->db_name_selected];
      $rows = $db->fetch_row($rid, $rowno);
    }
    return $rows;
  }
  // ....................................................................
  /**
  * Fetch array
  *
  * Get the given row of the result set as an associative array. Each
  * element is stored in the form fieldname=>value.
  * @access private
  * @see fetch_row()
  * @param resource $rid   The query resource ID
  * @param integer  $rowno The number of the row to fetch
  * @return array  The row of fields as an associative array
  */
  // Get the given row of the result set as an array..
  function fetch_array($rid, $rowno) {
    $arr = false;
    if (isset($this->database[$this->db_name_selected])) {
      $db = $this->database[$this->db_name_selected];
      $arr = $db->fetch_array($rid, $rowno);
    }
    return $arr;
  }
} // datasources class

// ----------------------------------------------------------------------
/**
* Database
*
* Define a database. This is a parent class to all of the supported
* database flavours. It holds the main data describing a database
* and it's connection. The actual functionality to connect to a
* physical database and access its data is defined in the child
* classes of this one. For example, see file 'db-postgres.php'.
* NB: Normal users of the system should not have to deal with this
* class. This class is used by the datasources class.
* @package database
* @access private
*/
class database {
  /** Type of database eg: "postgres", "mssql_server".. */
  var $type = "";
  /** Name of this database */
  var $name = "";
  /** Host server of this database */
  var $host = "";
  /** Port to access it via TCP */
  var $port = 0;
  /** Default user to connect as */
  var $user = "";
  /** Default password to connect as */
  var $passwd = "";
  /** True if we want a persistent connection */
  var $persistent = false;
  /** Flag true if database was connected ok */
  var $connected = false;
  /** Unique identifier for database access */
  var $dbid = false;
  /**
  * Constructor
  * @param string  $name    The database name
  * @param string  $user    The username of user who can access the database
  * @param string  $passwd  The user password which can access the database
  * @param string  $host    The hostname of the machine running the database
  * @param integer $port    The port number of the database server
  */
  function database($name="", $user="", $passwd="", $host="", $port=0) {
    $this->name = $name;

    // If host and port ar not specified, then
    // we assume the database is local..
    $this->host = $host;
    $this->port = $port;

    // These can be used as defaults..
    $this->user = $user;
    $this->passwd = $passwd;

  } // database

} // database class

// ----------------------------------------------------------------------
/**
* POSTGRES database interface
*
* This is a database interface class. It is an impedance-matcher
* between the high-level Phplib functions for accessing data, and
* the specific functions suplpied by Php to access a particular
* flavour of databse such as Postgres, MS-SQL Server, Sybase etc.
* @package database
* @access private
*/
class db_postgres extends database {

  function db_postgres($name="", $user="", $passwd="", $host="", $port=0) {
    $this->database($name, $user, $passwd, $host, $port);
    $this->type = "postgres";
  }

  function connect($persistent) {
    if (!$this->connected) {
      $connstr = "";
      if ($this->host != "") $connstr .= " host=" . $this->host;
      if ($this->port != 0 ) $connstr .= " port=" . $this->port;
      $connstr .= " dbname=" . $this->name;
      $connstr .= " user=" . $this->user;
      if ($this->passwd != "") $connstr .= " password=" . $this->passwd;
      $connstr = trim($connstr);
      if ($persistent)
        $this->dbid = pg_pconnect("$connstr");
      else
        $this->dbid = pg_connect("$connstr");
      if ($this->dbid) {
        $this->connected = TRUE;
      }
    }
    return $this->connected;
  }
  function disconnect() {
    if (pg_close($this->dbid))
      $this->connected = FALSE;
  }
  function query($sql) {
    return pg_exec($this->dbid, $sql);
  }
  function numrows($rid) {
    return pg_numrows($rid);
  }
  function freeresult($rid) {
    pg_freeresult($rid);
  }
  function errormessage() {
    return pg_errormessage($this->dbid);
  }
  function fetch_row($rid, $rowno) {
    return pg_fetch_row($rid, $rowno);
  }
  function fetch_array($rid, $rowno) {
    return pg_fetch_array($rid, $rowno);
  }
  function begin_transaction() {
    return $this->query("BEGIN");
  }
  function commit() {
    return $this->query("COMMIT");
  }
  function rollback() {
    return $this->query("ROLLBACK");
  }
}

// ----------------------------------------------------------------------
// Ensure Postgres Php module is present..
if (!extension_loaded("pgsql")) {
  if (!dl("pgsql.so")) {
    exit;
  }
}
// ----------------------------------------------------------------------
// Placeholder class
class response {
  var $datasource;
  function response() {}
}

// ----------------------------------------------------------------------
// Defines a database sequence.
class dbsequence {
  var $name = "";
  var $start = 1;
  var $increment = 1;
  var $minvalue = 1;
  var $cache = 1;
  // ....................................................................
  function dbsequence($name, $start=1, $inc=1, $min=1, $cache=1) {
    $this->name = strtolower($name);
    $this->start = $start;
    $this->increment = $inc;
    $this->minvalue = $min;
    $this->cache = $cache;
  }
  // ....................................................................
  // Dump ascii description of this sequence to stdout.
  function dump() {
    $s = "Sequence $this->name (start $this->start, inc $this->inc)\n";
    return $s;
  }
  // ....................................................................
  // Return SQL required to create this sequence.
  function create() {
    $s .= "create sequence \"$this->name\"";
    $s .= " start $this->start";
    $s .= " increment $this->increment";
    $s .= " minvalue $this->minvalue";
    $s .= " cache $this->cache";
    $s .= ";\n";
    return $s;
  }
  // ....................................................................
  // Return SQL to drop this sequence.
  function drop() {
    $s .= "drop sequence $this->name;\n";
    return $s;
  }
}

// ----------------------------------------------------------------------
// Defines a database function (procedure).
class dbfunction {
  var $name = "";
  var $return_type = "";
  var $src = "";
  var $arg_types = array();
  var $language = "";
  var $volatile = "v";
  var $strict = false;

  // ....................................................................
  function dbfunction($name, $returns, $src, $args="", $lang="plpgsql",
                      $volatile="v", $strict=false) {
    $this->name = strtolower($name);
    $this->return_type = $returns;
    $this->src = str_replace("'", "''", $src);
    if (is_array($args)) $this->arg_types = $args;
    $this->language = $lang;
    $this->volatile = $volatile;
    $this->strict = $strict;
  }
  // ....................................................................
  // Dump ascii description of this function to stdout.
  function dump() {
    $s = "Function $this->name returns $this->returns language='$this->language'\n";
    return $s;
  }
  // ....................................................................
  // Return the types parameter list, including brackets..
  function parameters() {
    $s = "";
    $s .= " (";
    if (count($this->arg_types) > 0) {
      foreach ($this->arg_types as $arg) {
        $s .= "$arg,";
      }
      $s = substr($s, 0, -1);
    }
    $s .= ")";
    return $s;
  }
  // ....................................................................
  // Return SQL required to create this function.
  function create() {
    $s .= "create function $this->name";
    $s .= $this->parameters();
    $s .= " returns $this->return_type";
    $s .= " as '$this->src'";
    $s .= " language '$this->language'";
    switch ($this->volatile) {
      case "i": $s .= " immutable"; break;
      case "s": $s .= " stable"; break;
    }
    if ($this->strict) $s .= " strict";
    $s .= ";\n";
    return $s;
  }
  // ....................................................................
  // Return SQL to drop this function..
  function drop() {
    $s .= "drop function $this->name";
    $s .= $this->parameters();
    $s .= ";\n";
    return $s;
  }
}

// ----------------------------------------------------------------------
// Defines a database index.
class dbindex {
  /** Name of this index */
  var $name = "";
  /** Table (object) to apply index to */
  var $table;
  /** Field numbers in index */
  var $fieldnums = array();
  /** True if index is for a primary key */
  var $primary = false;
  /** True if index is unique */
  var $unique = false;
  /** True if index is clustered */
  var $clustered = false;
  /** Access method, eg. btree, hash etc. */
  var $access_method = "";
  // ....................................................................
  function dbindex($name, $table="", $flds="", $primary=false,
                   $unique=false, $clustered=false, $access="") {
    $this->name = strtolower($name);
    $this->table = $table;
    if (is_array($flds)) $this->fieldnums = $flds;
    $this->primary = $primary;
    $this->unique = $unique;
    $this->clustered = $clustered;
    $this->access_method = $access;
  }
  // ....................................................................
  /** Dump ascii description of this index to stdout. */
  function dump() {
    $s = "Index $this->name on " . $this->table->name . " using $this->access_method";
    return $s;
  }
  // ....................................................................
  /** Return SQL required to create this index. */
  function create() {
    $s = "";
    $s .= "create";
    if ($this->unique) $s .= " unique";
    $s .= " index $this->name on " . $this->table->name;
    if ($this->access_method != "") {
      $s .= " using $this->access_method";
    }
    $s .= " (";
    if (count($this->fieldnums) > 0) {
      foreach ($this->fieldnums as $num) {
        $field = $this->table->getfieldbynum($num);
        $s .= "$field->name,";
      }
      $s = substr($s, 0, -1);
    }
    $s .= ");\n";
    if ($this->clustered) {
      $s .= "cluster $this->name on " . $this->table->name . ";\n";
    }
    return $s;
  }
  // ....................................................................
  // Return SQL to drop this index.
  function drop() {
    global $op_version;
    $s = "drop index $this->name";
    if ($op_version >= 7.3) {
      $s .= " cascade";
    }
    $s .= ";\n";
    return $s;
  }
} // class dbindex

// ----------------------------------------------------------------------
// Defines a database constraint.
class dbconstraint {
  /** Name of this constraint */
  var $name = "";
  /** Type of constraint 'c' - check, 'p' - pk, 'f' - fk */
  var $type = "";
  /** Table (object) to apply constraint to */
  var $table = "";
  /** Table (object) constraint refers to */
  var $fk_table = "";
  /** True if constraint is deferrable */
  var $deferrable = false;
  /** True if constraint is initially deferred */
  var $deferred = false;
  /** Array of ordinal numbers for table fields */
  var $fieldnums = array();
  /** Array of ordinal numbers for foreign key fields */
  var $fk_fieldnums = array();
  /** Action to take on update */
  var $update_action = "";
  /** Action to take on delete */
  var $delete_action = "";
  /** Match type for keys */
  var $match_type = "";
  /** Check constraint source */
  var $cksrc = "";
  // ....................................................................
  function dbconstraint($name, $type, $table="", $fktable="", $flds="",
                        $fkflds="", $updact="", $delact="", $match="", $cksrc="") {
    $this->name = strtolower($name);
    $this->set(
        $type,
        $table,
        $fktable,
        $flds,
        $fkflds,
        $updact,
        $delact,
        $match,
        $cksrc
        );
  }
  // ....................................................................
  /** Set constraint variables. */
  function set(
      $type,
      $table="",
      $fktable="",
      $flds="",
      $fkflds="",
      $updact="",
      $delact="",
      $match="",
      $cksrc="",
      $deferrable=false,
      $deferred=false
  ) {
    // The "!" implies 'leave as it is' ..
    if ($type != "!") $this->type = $type;
    if (is_object($table)) $this->table = $table;
    if (is_object($fktable)) $this->fk_table = $fktable;
    if (is_array($flds)) $this->fieldnums = $flds;
    if (is_array($fkflds)) $this->fk_fieldnums = $fkflds;
    if ($updact != "!") $this->update_action = $updact;
    if ($delact != "!") $this->delete_action = $delact;
    if ($match != "!") $this->match_type = $match;
    if ($cksrc != "!") $this->cksrc = $cksrc;
    if ($deferrable != "!") $this->deferrable = $deferrable;
    if ($deferred != "!") $this->deferred = $deferred;
  }
  // ....................................................................
  /** Dump ascii description of this constraint to stdout. */
  function dump() {
    $s = "Constraint $this->name";
    switch ($this->type) {
      case "p": $s .= " primary key"; break;
      case "f": $s .= " foreign key"; break;
      case "c": $s .= " check $this->cksrc"; break;
    }
    if (is_object($table)) {
      $s .= " $table->name";
    }
    if ($this->type == "f") {
      $s .= " references " . $this->fk_table->name;
    }
    return $s;
  }
  // ....................................................................
  /** Return SQL required to create this as an inline table constraint */
  function create_inline() {
    return $this->create(false);
  }
  // ....................................................................
  /** Return SQL required to create this constraint outside the table */
  function create($outside_table=true) {
    $s = "";
    switch ($this->type) {
      // PRIMARY KEY CONSTRAINT
      case "p":
        if ($outside_table) {
          if (!is_object($this->table)) return $s;
          $s .= "alter table " . $this->table->name . "\n";
          $s .= "  add ";
        }
        $s .= "constraint $this->name primary key";
        $s .= $this->fields();
        break;

      // CHECK CONSTRAINT
      case "c":
        if ($outside_table) {
          if (!is_object($this->table)) return $s;
          $s .= "alter table " . $this->table->name . "\n";
          $s .= "  add ";
        }
        $s .= "constraint $this->name check $this->cksrc";
        break;

      // FOREIGN KEY CONSTRAINT
      case "f":
        if ($outside_table) {
          if (!is_object($this->table) || !is_object($this->fk_table)) return $s;
          $s .= "alter table " . $this->table->name . "\n";
          $s .= "  add ";
        }
        $s .= "constraint $this->name foreign key";
        $s .= $this->fields();
        $s .= " references " . $this->fk_table->name;
        $s .= $this->fk_fields();

        // MATCH TYPE
        switch ($this->match_type) {
          case "f":
            $s .= " match full";
            break;
        }

        // UPDATE ACTION
        if ($this->update_action != "") {
          $act = " on update";
          switch ($this->update_action) {
            case "a": $act .= " no action";   break;
            case "c": $act .= " cascade";     break;
            case "n": $act .= " set null";    break;
            case "r": $act .= " restrict";    break;
            case "d": $act .= " set default"; break;
            default: $act = "";
          }
          $s .= $act;
        }

        // DELETE ACTION
        if ($this->delete_action != "") {
          $act = " on delete";
          switch ($this->delete_action) {
            case "a": $act .= " no action";   break;
            case "c": $act .= " cascade";     break;
            case "n": $act .= " set null";    break;
            case "r": $act .= " restrict";    break;
            case "d": $act .= " set default"; break;
            default: $act = "";
          }
          $s .= $act;
        }

        // DEFERRABLE MODES
        if ($this->deferrable) {
          $s .= " deferrable";
          if ($this->deferred) {
            $s .= " initially deferred";
          }
        }
        break;
    }
    if ($outside_table) {
      $s .= ";\n";
    }
    return $s;
  }
  // ....................................................................
  // Unpacks the table fields array and returns '(field1,field2, ...)'..
  // These are the fields on the table the constraint is on.
  function fields() {
    $s .= "";
    if (is_object($this->table)) {
      $s .= " (";
      if (count($this->fieldnums) > 0) {
        foreach ($this->fieldnums as $num) {
          $field = $this->table->getfieldbynum($num);
          $s .= "$field->name,";
        }
        $s = substr($s, 0, -1);
      }
      $s .= ")";
    }
    return $s;
  }
  // ....................................................................
  // Unpacks the foreign key fields array and returns '(field1,field2, ...)'..
  // These are the fields on the table that the constraint references.
  function fk_fields() {
    $s .= "";
    if (is_object($this->fk_table)) {
      $s .= " (";
      if (count($this->fk_fieldnums) > 0) {
        foreach ($this->fk_fieldnums as $num) {
          $field = $this->fk_table->getfieldbynum($num);
          $s .= "$field->name,";
        }
        $s = substr($s, 0, -1);
      }
      $s .= ")";
    }
    return $s;
  }
  // ....................................................................
  // Return SQL to drop this constraint.
  function drop() {
    global $op_version;
    switch ($this->type) {
      case "p":
      case "f":
      case "c":
        $s .= "alter table " . $this->table->name . "\n";
        $s .= "  drop constraint $this->name";
        $s .= " restrict;\n";
        break;
    }
    return $s;
  }
  // ....................................................................
  // Returns true if the given constraint matches this one in terms of
  // functionality. This allows you to identify constraints that are
  // the same apart from the naming.
  function matches($con) {
    if ($this->type != $con->type) return false;
    if ($this->table->name != $con->table->name)  return false;
    if ($this->fk_table->name != $con->fk_table->name) return false;
    if ($this->deferrable != $con->deferrable) return false;
    if ($this->deferred != $con->deferred) return false;
    if ($this->fields() != $con->fields()) return false;
    if ($this->fk_fields() != $con->fk_fields()) return false;
    if ($this->update_action != $con->update_action) return false;
    if ($this->delete_action != $con->delete_action) return false;
    if ($this->match_type != $con->match_type) return false;
    if ($this->cksrc != $con->cksrc) return false;
    return true;
  }
} // class dbconstraint

// ----------------------------------------------------------------------
// Defines a database trigger.
class dbtrigger {
  /** Name of this trigger */
  var $name = "";
  /** When trigger fires. If true BEFORE, else AFTER event */
  var $before = true;
  /** If true, fire trigger on INSERT */
  var $oninsert = false;
  /** If true, fire trigger on DELETE */
  var $ondelete = false;
  /** If true, fire trigger on UPDATE */
  var $onupdate = false;
  /** Table (object) to apply trigger to */
  var $table;
  /** If true, execute func for EACH ROW else EACH STATEMENT */
  var $eachrow = false;
  /** Name of function to call when triggered */
  var $funcname;
  /** Arguments to pass to the function */
  var $args = array();
  // ....................................................................
  function dbtrigger($name, $bitmask=0, $table="", $funcname="", $args="") {
    $this->name = strtolower($name);
    $this->eachrow  = ($bitmask & 0x01);
    $this->before   = ($bitmask & 0x02);
    $this->oninsert = ($bitmask & 0x04);
    $this->ondelete = ($bitmask & 0x08);
    $this->onupdate = ($bitmask & 0x10);
    $this->table = $table;
    $this->funcname = $funcname;
    if (is_array($args)) $this->args = $args;
  }
  // ....................................................................
  /** Dump ascii description of this trigger to stdout. */
  function dump() {
    $s = "trigger $this->name on " . $this->table->name . " executing function $this->funcname()";
    return $s;
  }
  // ....................................................................
  /** Return SQL required to create this trigger. */
  function create() {
    $s = "";
    $s .= "create";
    $s .= " trigger $this->name";
    if ($this->before) $s .= " before ";
    else $s .= " after ";
    $event = array();
    if ($this->oninsert) $event[] = "insert";
    if ($this->ondelete) $event[] = "delete";
    if ($this->onupdate) $event[] = "update";
    $s .= implode(" or ", $event);
    $s .= " on " . $this->table->name;
    if ($this->eachrow)  $s .= " for each row";
    else $s = " for each statement";
    $s .= " execute procedure $this->funcname";
    $s .= " (";
    if (count($this->args) > 0) {
      foreach ($this->args as $arg) {
        if ($arg != "00") {
          $s .= "'$arg',";
        }
      }
      $s = substr($s, 0, -1);
    }
    $s .= ");\n";
    return $s;
  }
  // ....................................................................
  // Return SQL to drop this trigger.
  function drop() {
    $s = "drop trigger $this->name on " . $this->table->name . ";\n";
    return $s;
  }
} // class dbtrigger

// ----------------------------------------------------------------------
// Class describing a database field of a table.
class dbfield {
  var $name = "";
  var $num = 0;
  var $type = "";
  var $defaultval = "";
  var $notnull = false;
  var $isarray = false;
  var $ispkey = false;
  var $constraints = array();
  // ....................................................................
  function dbfield($name, $num, $type, $defaultval="", $notnull=false, $isarray=false, $ispkey=false) {
    $this->name = strtolower($name);
    $this->num = $num;
    $this->type = $type;
    $this->defaultval = $defaultval;
    $this->notnull = $notnull;
    $this->ispkey = $ispkey;
  }
  // ....................................................................
  // Dump field description to stdout.
  function dump() {
    $s = "$this->name $this->type";
    if ($this->defaultval != "") $s .= " DEFAULT $this->defaultval";
    if ($this->notnull) $s .= " NOT NULL";
    if ($this->isarray) $s .= " (array)";
    if ($this->ispkey) $s .= " (pk)\n";
    return $s;
  }
  // ....................................................................
  // Return SQL to create this field in a table. This represents a
  // portion of the CREATE TABLE script pertaining to this field and
  // it comprises field name, type, and constraints.
  function create() {
    $s = "";
    $s .= "  \"$this->name\" $this->type";
    if ($this->notnull) {
      $s .= " not null";
    }
    if ($this->defaultval != "") {
      $s .= " default $this->defaultval";
    }
    $s .= $this->create_constraints();
    return $s;
  }
  // ....................................................................
  // Create constraints for this field..
  function create_constraints() {
    $s = "";
    foreach ($this->constraints as $con) {
      $s .= "\n" . $con->create_inline();
    }
    return $s;
  }
  // ....................................................................
  /** Return the SQL to drop this field. */
  function drop() {
    $s .= "drop column $this->name;";
    return $s;
  }
  // ....................................................................
  // Return true if field constraints match those passed in.
  function constraints_match($field) {
    $matched = true;
    if (count($this->constraints) != count($field->constraints)) {
      $matched = false;
    }
    else {
      foreach ($this->constraints as $con) {
        if (isset($field->constraints[$con->name])) {
          $fcon = $field->constraints[$con->name];
          if (!$fcon->matches($con)) {
            $matched = false;
            break;
          }
        }
        else {
          $matched = false;
          break;
        }
      } // foreach
    }
    return $matched;
  }
} // class dbfield

// ----------------------------------------------------------------------
// Class describing a database table.
class dbtable {
  /** Name of this table */
  var $name = "";
  /** Array of field attnum's which are primary keys in table */
  var $pkey = array();
  /** Array of field objects */
  var $fields = array();
  /** Array of constraints on this table */
  var $constraints = array();
  // ....................................................................
  /** Construct a table of given name and array of primary key fields.
  * The array of pkeys is actually a list of integers, each being
  * the enumerated order of the field which is part of the key.
  */
  function dbtable($name, $pkey) {
    global $op_version;
    $this->name = strtolower($name);
    $this->pkey = $pkey;
  }
  // ....................................................................
  /** Add a field to the table. */
  function addfield($field) {
    $this->fields[$field->name] = $field;
  }
  // ....................................................................
  /** Create a new field to the table with given parameters. */
  function newfield($name, $num, $type, $defaultval="", $notnull=false, $isarray=false) {
    $ispkey = (in_array($num, $this->pkey));
    $this->fields[$name] = new dbfield($name, $num, $type, $defaultval, $notnull, $isarray, $ispkey);
  }
  // ....................................................................
  /** Returns field object of given attnum (order number) */
  function getfieldbynum($num) {
    foreach ($this->fields as $field) {
      if ($field->num == $num) return $field;
    }
    return null;
  }
  // ....................................................................
  /** Returns field number of given field name */
  function getfieldnum($fieldname) {
    if (isset($this->fields[$fieldname])) {
      $field = $this->fields[$fieldname];
      return $field->num;
    }
    else return -1;
  }
  // ....................................................................
  /** Returns field object of given name */
  function getfield($name) {
    return $this->fields[$name];
  }
  // ....................................................................
  /** Dump the table description to stdout. */
  function dump() {
    $s = "\nTable: $this->name\n";
    foreach ($this->fields as $field) {
      $s .= $field->dump();
    }
    return $s;
  }
  // ....................................................................
  /** Return the SQL which will create this table. */
  function create() {
    global $op_version;
    $s = "";
    $s .= "create table $this->name (\n";
    foreach ($this->fields as $field) {
      $s .= $field->create() . ",\n";
    }
    if ($op_version < 7.3 && count($this->pkey > 0)) {
      $s .= "  constraint " . $this->name . "_pkey primary key (";
      foreach ($this->fields as $field) {
        if ($field->ispkey) {
          $s .= "$field->name,";
        }
      }
      $s = substr($s, 0, -1);
      $s .= ")\n";
    }
    else {
      $s = substr($s, 0, -2);
    }
    // Other inline table constraints..
    if (count($this->constraints) > 0) {
      foreach ($this->constraints as $con) {
        $s .= " " . $con->create_inline() . "\n";
      }
    }
    $s .= "\n);\n";
    return $s;
  }
  // ....................................................................
  /** Return SQL which will create a column in this table. The $column
  * passed in is actually a field object.
  */
  function addcolumn($column) {
    global $op_version;

    $s = "alter table \"$this->name\"\n";
    $s .= "  add column $column->name $column->type;\n";
    if ($column->defaultval != "") {
      $s .= $this->setdefault($column);
    }
    if ($column->notnull) {
      if ($op_version < 7.3) {
        $s .= "UPDATE pg_attribute SET attnotnull=true";
        $s .= " WHERE attname='$column->name'";
        $s .= "  AND attrelid = (SELECT oid FROM pg_class WHERE relname='$this->name');\n";
      }
      else {
        $s .= $this->setnullconstraint($column);
      }
    }
    if (count($column->constraints) > 0) {
      foreach ($column->constraints as $con) {
        $s .= $con->create() . "\n";
      }
    }
    return $s;
  }
  // ....................................................................
  /** Return SQL to set the default for given field on this table. */
  function setdefault($column) {
    $s = "alter table \"$this->name\"\n";
    $s .= "  alter column \"$column->name\"";
    if ($column->defaultval == "") {
      $s .= " drop default;\n";
    }
    else $s .= " set default $column->defaultval;\n";
    return $s;
  }
  // ....................................................................
  /** Return SQL to set the NULL/NOT NULL constraint.. */
  function setnullconstraint($column) {
    global $op_version;
    if ($op_version < 7.3) {
      $nullsetting = ($column->notnull) ? "true" : "false";
      $s .= "update pg_attribute set attnotnull=$nullsetting";
      $s .= " where attname='$column->name'";
      $s .= " and attrelid = (select oid from pg_class where relname='$this->name');\n";
    }
    else {
      $s = "alter table \"$this->name\"\n";
      $s .= "  alter column \"$column->name\"";
      if ($column->notnull) {
        $s .= " set not null;\n";
      }
      else {
        $s .= " drop not null;\n";
      }
    }
    return $s;
  }
  // ....................................................................
  /** Return SQL to drop a column from the table. The $column passed
  * is actually a field object.
  */
  function dropcolumn($column) {
    $s = "alter table \"$this->name\" ";
    $s .= $column->drop() . "\n";
    return $s;
  }
  // ....................................................................
  /** Return the SQL to drop this table. */
  function drop() {
    $s .= "drop table $this->name;\n";
    return $s;
  }
} // class dbtable

// ----------------------------------------------------------------------
// Class describing a database schema.
class dbschema {
  var $name = "";
  var $tables = array();
  var $sequences = array();
  var $constraints = array();
  var $indexes = array();
  var $functions = array();
  var $triggers = array();
  // ....................................................................
  // Create a schema (database) of given name. The name should be a
  // valid existing database name in postgres.
  function dbschema($name) {
    $this->name = strtolower($name);
  }
  // ....................................................................
  // Acquire all the details describing the database schema from the
  // Postgres catalog information. This fills the $tables array.
  function get() {
    global $pg_version, $RESPONSE;
    global $schemaQ, $seqQ, $fkQ, $indQ, $funQ, $trgQ;
    $RESPONSE->datasource->select($this->name);

    // Tables
    $schema = new dbrecords($schemaQ);
    if ($schema->hasdata) {
      $ctable = "";
      do {
        // Table information..
        $tablename    = strtolower(trim($schema->field("tablename")));
        $tableoid     = $schema->field("tableoid");
        $tablehaspkey = $schema->istrue("tablehaspkey");

        // Field information..
        $fieldname    = $schema->field("fieldname");
        $fieldnum     = $schema->field("fieldnum");
        $fieldtype    = $schema->field("fieldtype");
        $fieldisarray = ($schema->field("fieldarrdims") > 0);
        $fieldnotnull = $schema->istrue("fieldnotnull");
        $fieldhasdef  = $schema->istrue("fieldhasdef");

        // Table primary key fields..
        $tablepkey = array();
        if ($tablehaspkey) {
          $q  = "SELECT * FROM pg_index";
          $q .= " WHERE indrelid=$tableoid::int8";
          $q .= "   AND indisprimary";
          $pk = new dbrecords($q);
          if ($pk->hasdata) {
            $tablepkey = explode(" ", $pk->field("indkey"));
          }
        }

        // Field default..
        $fielddefault = "";
        if ($fieldhasdef) {
          $q  = "SELECT adsrc as fielddefault FROM pg_attrdef";
          $q .= " WHERE adrelid=$tableoid::int8";
          $q .= "   AND adnum=$fieldnum";
          $def = new dbrecords($q);
          if ($def->hasdata) {
            $fielddefault = $def->field("fielddefault");
          }
        }

        // Add current table then start new one..
        if ($tablename != $ctable) {
          if (isset($tableObj)) {
            $this->addtable($tableObj);
          }
          $tableObj = new dbtable($tablename, $tablepkey);
          $ctable = $tablename;
        }
        $tableObj->newfield(
            $fieldname,
            $fieldnum,
            $fieldtype,
            $fielddefault,
            $fieldnotnull,
            $fieldisarray
            );
      } while ($schema->get_next());
      // Add final table..
      if (isset($tableObj)) {
        $this->addtable($tableObj);
      }
    } // tables

    // Check, PK and FK Constraints
    if ($pg_version < 7.3) {
      foreach ($this->tables as $table) {
        // Check constraints stored on-table..
        $q  = "SELECT pgr.*";
        $q .= "  FROM pg_class pgc, pg_relcheck pgr";
        $q .= " WHERE pgc.relname='$table->name'";
        $q .= "   AND pgr.rcrelid=pgc.oid";
        $checks = new dbrecords($q);
        if ($checks->hasdata) {
          do {
            $conname = $checks->field("rcname");
            $consrc = $checks->field("rcsrc");
            $conbin = $checks->field("rcbin");
            if (preg_match("/.*?varattno ([0-9]+).*?/", $conbin, $matches)) {
              $attnum = $matches[1];
              $con = new dbconstraint($conname, "c");
              $con->cksrc = $consrc;
              $con->table = $table;
              // Store constraint..
              $field = $table->getfieldbynum($attnum);
              $field->constraints[$conname] = $con;
              $table->addfield($field);
              $this->addtable($table);
            }
          } while ($checks->get_next());
        }

        // Primary key constraints..
        $q  = "SELECT pgi.*,pgci.relname as indexname";
        $q .= "  FROM pg_class pgc, pg_class pgci, pg_index pgi";
        $q .= " WHERE pgc.relname='$table->name'";
        $q .= "   AND pgi.indrelid=pgc.oid";
        $q .= "   AND pgi.indisprimary";
        $q .= "   AND pgci.oid=pgi.indexrelid";
        $pks = new dbrecords($q);
        if ($pks->hasdata) {
          do {
            $conname = $pks->field("indexname");
            $fieldnums = explode(" ", $pks->field("indkey"));
            $con = new dbconstraint(
                      $conname,
                      "p",
                      $table,
                      "",
                      $fieldnums
                      );
            // Stash constraint in table..
            $this->addconstraint($con);
          } while ($pks->get_next());
        }

        // FK constraints stored globally..
        $q  = "SELECT pgt.*, pgp.prosrc";
        $q .= "  FROM pg_class pgc, pg_trigger pgt, pg_proc pgp";
        $q .= " WHERE pgc.relname='$table->name'";
        $q .= "   AND pgt.tgconstrrelid=pgc.oid";
        $q .= "   AND pgt.tgisconstraint";
        $q .= "   AND pgp.oid=pgt.tgfoid";
        $trigs = new dbrecords($q);
        if ($trigs->hasdata) {
          do {
            // Constraint information..
            $consrc    = $trigs->field("prosrc");
            $ribits = explode("_", $consrc);
            if ($ribits[1] == "FKey") {

              // cascade, restrict, setnull or  setdefault
              $operation = strtolower($ribits[2]);

              // Only interested in FK type constraints..
              if ($operation != "check") {
                // ins, upd or del
                $event = strtolower($ribits[3]);

                // Constraint operation & event..
                $cksrc = "!";
                switch ($operation) {
                  case "cascade":    $act = "c";  break;
                  case "setnull":    $act = "n";  break;
                  case "restrict":   $act = "r";  break;
                  case "setdefault": $act = "d";  break;
                  default: $act = "a";
                }

                // Constraint details..
                $fk_table = "!";
                $fieldnums = "!";
                $fk_fieldnums = "!";
                $updact = "!";
                $delact = "!";

                switch ($event) {
                  case "upd":
                    $updact = $act;
                    break;
                  case "del":
                    $delact = $act;
                    break;
                }
                // PK & FK table, and field numbers..
                $tgargs = $trigs->rawfield("tgargs");
                $argbits = explode("\\000", $tgargs);
                $fk_tablename = trim(strtolower($argbits[2]));
                $fk_table = $this->gettable($fk_tablename);

                $fieldnums = array();
                $fk_fieldnums = array();
                for ($ix=4; $ix < count($argbits); $ix+=2) {
                  $fname = trim(strtolower($argbits[$ix]));
                  $fk_fname = trim(strtolower($argbits[$ix + 1]));
                  if ($fname != "" && $fk_fname != "") {
                    $fieldnums[] = $table->getfieldnum($fname);
                    $fk_fieldnums[] = $fk_table->getfieldnum($fk_fname);
                  }
                }

                // Deferred settings..
                $deferrable = $trigs->istrue("tgdeferrable");
                $deferred   = $trigs->istrue("tginitdeferred");

                // Get the constraint..
                $conname   = $trigs->field("tgconstrname");
                if (isset($this->constraints[$conname])) {
                  $con = $this->getconstraint($conname);
                }
                else {
                  $con = new dbconstraint($conname, "f");
                }

                // Set constraint vars..
                $con->set(
                    "f",
                    $table,
                    $fk_table,
                    $fieldnums,
                    $fk_fieldnums,
                    $updact,
                    $delact,
                    "", // match type
                    $cksrc,
                    $deferrable,
                    $deferred
                    );

                // Stash constraint back in table..
                $this->addconstraint($con);
              }
            } // if foreign key
          } while ($trigs->get_next());
        }
      } // foreach
    }
    else {
      // Column constraints per table..
      foreach ($this->tables as $table) {
        // Check constraints stored on-table..
        $q  = "SELECT *";
        $q .= "  FROM pg_class pgc, pg_constraint pgcon";
        $q .= " WHERE pgc.relname='$table->name'";
        $q .= "   AND pgcon.conrelid=pgc.oid";
        $q .= "   AND pgcon.contype='c'";
        $checks = new dbrecords($q);
        if ($checks->hasdata) {
          do {
            $conname = $checks->field("conname");
            $consrc = $checks->field("consrc");
            $conbin = $checks->field("conbin");
            $condeferrable = $checks->istrue("condeferrable");
            $condeferred = $checks->istrue("condeferred");
            if (preg_match("/.*?varattno ([0-9]+).*?/", $conbin, $matches)) {
              $attnum = $matches[1];
              $con = new dbconstraint($conname, "c");
              $con->cksrc = $consrc;
              $con->deferrable = $deferrable;
              $con->deferred = $deferred;
              $con->table = $table;
              // Store constraint..
              $field = $table->getfieldbynum($attnum);
              $field->constraints[$conname] = $con;
              $table->addfield($field);
              $this->addtable($table);
            }
          } while ($checks->get_next());
        }
      }

      // Primary and Foreign Key Constraints..
      $schema = new dbrecords("SELECT * FROM pg_constraint WHERE contype IN ('p','f')");
      if ($schema->hasdata) {
        do {
          // Constraint information..
          $conname       = $schema->field("conname");
          $contype       = $schema->field("contype");
          $conrelid      = $schema->field("conrelid");
          $confrelid     = $schema->field("confrelid");
          $conkey        = $schema->field("conkey");
          $confkey       = $schema->field("confkey");
          $confupdtype   = $schema->field("confupdtype");
          $confdeltype   = $schema->field("confdeltype");
          $confmatchtype = $schema->field("confmatchtype");
          $consrc        = $schema->field("consrc");
          $table = "";
          $fk_table = "";

          // Get table for primary or foreign key constraints..
          $tQ = new dbrecords("SELECT relname FROM pg_class WHERE oid=$conrelid");
          if ($tQ->hasdata) {
            $table = $this->gettable( $tQ->field("relname") );
          }
          // Get foreign key table for foreign key constraints..
          if ($contype == "f") {
            $tQ = new dbrecords("SELECT relname FROM pg_class WHERE oid=$confrelid");
            if ($tQ->hasdata) {
              $fk_table = $this->gettable( $tQ->field("relname") );
            }
          }

          // Weird data for field numbers to identify the fields.
          // The array of integers is returned inside curly brackets
          // as in: '{1,3}'. So we get rid of the brackets and then
          // unpack the comma-delimited values..
          $conkey  = preg_replace("/[\{\}]/", "", $conkey);
          $confkey = preg_replace("/[\{\}]/", "", $confkey);

          // Add constraint object to schema..
          $this->addconstraint(
                new dbconstraint(
                    $conname,
                    $contype,
                    $table,
                    $fk_table,
                    explode(",", $conkey),
                    explode(",", $confkey),
                    $confupdtype,
                    $confdeltype,
                    $confmatchtype,
                    $consrc
                    )
                );
        } while ($schema->get_next());
      } // constraints
    }

    // Indexes
    $schema = new dbrecords($indQ);
    if ($schema->hasdata) {
      do {
        $indexname   = $schema->field("indexname");
        $tablename   = $schema->field("tablename");
        $table       = $this->gettable( $tablename );
        $fields      = explode(" ", $schema->field("fieldnums"));
        $isprimary   = $schema->istrue("indisprimary");
        $isunique    = $schema->istrue("indisunique");
        $isclustered = $schema->istrue("indisclustered");
        $access      = $schema->field("amname");
        $this->addindex(
            new dbindex(
                $indexname,
                $table,
                $fields,
                $isprimary,
                $isunique,
                $isclustered,
                $access
                )
            );
      } while ($schema->get_next());
    }

    // Functions
    $schema = new dbrecords($funQ);
    if ($schema->hasdata) {
      do {
        // Get return type..
        $returns = "opaque";
        $retoid = $schema->field("prorettype");
        if ($retoid != "") {
          $retQ = new dbrecords("SELECT typname FROM pg_type WHERE oid=$retoid");
          if ($retQ->hasdata) {
            $returns = $retQ->field("typname");
          }
        }
        $funcname  = $schema->field("funcname");
        $strict    = ($schema->field("strict") == "t");
        if ($pg_version >= 7.3) {
          $volatile  = $schema->field("volatile");
        }
        else {
          $volatile  = "v";
        }
        $src       = $schema->field("src");
        $language  = $schema->field("language");
        $argoids   = explode(" ", $schema->field("argoids"));
        $arg_types = array();
        foreach ($argoids as $oid) {
          $oid = trim($oid);
          if ($oid != "") {
            $argQ = new dbrecords("SELECT typname FROM pg_type WHERE oid=$oid");
            if ($argQ->hasdata) {
              $arg_types[] = $argQ->field("typname");
            }
          }
        }
        $this->addfunction(
            new dbfunction(
                $funcname,
                $returns,
                $src,
                $arg_types,
                $language,
                $volatile,
                $strict
                )
            );
      } while ($schema->get_next());
    }

    // Triggers
    $schema = new dbrecords($trgQ);
    if ($schema->hasdata) {
      do {
        $trigname    = $schema->field("trigname");
        $trigbitmask = $schema->field("trigbitmask");
        $tablename   = $schema->field("tablename");
        $table       = $this->gettable( $tablename );
        $funcname    = $schema->field("funcname");
        $numargs     = $schema->field("numargs");
        if ($numargs > 0) {
          $args = explode("\\000", $schema->rawfield("args"));
        }
        else {
          $args = array();
        }
        $this->addtrigger(
            new dbtrigger(
                $trigname,
                $trigbitmask,
                $table,
                $funcname,
                $args
                )
            );
      } while ($schema->get_next());
    }

    // Sequences
    $schema = new dbrecords($seqQ);
    if ($schema->hasdata) {
      do {
        $sequencename = $schema->field("sequencename");
        $this->addsequence( new dbsequence($sequencename) );
      } while ($schema->get_next());
    }

  } // get
  // ....................................................................
  // Add a sequence to the schema information.
  function addsequence($seqobject) {
    $this->sequences[$seqobject->name] = $seqobject;
  }
  // ....................................................................
  // Add an index to the schema information.
  function addindex($indexobject) {
    $this->indexes[$indexobject->name] = $indexobject;
  }
  // ....................................................................
  // Add a table to the schema information.
  function addtable($tableobject) {
    $this->tables[$tableobject->name] = $tableobject;
  }
  // ....................................................................
  // Add a constraint to the schema information.
  function addconstraint($conobject) {
    $this->constraints[$conobject->name] = $conobject;
  }
  // ....................................................................
  // Add a function to the schema information.
  function addfunction($funcobject) {
    $this->functions[$funcobject->name] = $funcobject;
  }
  // ....................................................................
  // Add a trigger to the schema information.
  function addtrigger($trigobject) {
    $this->triggers[$trigobject->name] = $trigobject;
  }
  // ....................................................................
  /** Returns table object of given name */
  function gettable($name) {
    return $this->tables[$name];
  }
  // ....................................................................
  /** Returns constraint object of given name */
  function getconstraint($name) {
    return $this->constraints[$name];
  }
  // ....................................................................
  /** Returns index object of given name */
  function getindex($name) {
    return $this->indexes[$name];
  }
  // ....................................................................
  /** Returns function object of given name */
  function getfunction($name) {
    return $this->functions[$name];
  }
  // ....................................................................
  /** Returns trigger object of given name */
  function gettrigger($name) {
    return $this->triggers[$name];
  }
  // ....................................................................
  /** Returns seqeuence object of given name */
  function getsequence($name) {
    return $this->sequences[$name];
  }
  // ....................................................................
  // Dump this entire schema description to stdout.
  function dump() {
    $s = "Database: $this->name\n";
    foreach ($this->tables as $table) {
      $s .= $table->dump();
    }
    return $s;
  } // dump
  // ....................................................................
  // Produce the SQL required to morph the schema described in the passed
  // dbschema object $db, into the schema we have in this current object.
  // The resulting SQL is commented.
  function SQLdiff($db) {
    global $pg_version, $PGdrop_column;
    global $nodrops;

    // Triggers to drop
    if (!$nodrops) {
      foreach ($db->triggers as $dbtrig) {
        if (!isset($this->triggers[$dbtrig->name])) {
          $diff .= "\n-- DROPPING NON-EXISTENT TRIGGER $dbtrig->name\n";
          $diff .= $dbtrig->drop();
        }
      }
    }

    // Functions to drop
    if (!$nodrops) {
      foreach ($db->functions as $dbfunc) {
        if (!isset($this->functions[$dbfunc->name])) {
          $diff .= "\n-- DROPPING NON-EXISTENT FUNCTION $dbfunc->name\n";
          $diff .= $dbfunc->drop();
        }
      }
    }

    // Check for entities we need to drop or change..
    $tables_dropped = array();
    foreach ($db->tables as $dbtable) {
      if (!isset($this->tables[$dbtable->name])) {
        if (!$nodrops) {
          $diff .= "\n-- DROPPING NON-EXISTENT TABLE $dbtable->name\n";
          $diff .= $dbtable->drop();
          $tables_dropped[] = $dbtable->name;
        }
      }
      else {
        $table = $this->gettable($dbtable->name);
        $dropfields = array();
        $dropdiff = "";
        $alterdiff = "";
        foreach ($dbtable->fields as $dbfield) {
          if (!isset($table->fields[$dbfield->name])) {
            if (!$nodrops) {
              // Field needs to be dropped..
              if ($PGdrop_column) {
                $dropdiff .= $table->dropcolumn($dbfield);
              }
              else {
                $dropfields[] = $dbfield->name;
              }
            }
          }
          else {
            // Field exists but might have been changed..
            $field = $table->getfield($dbfield->name);
            if ($field->create() != $dbfield->create()) {
              if (!$nodrops && $field->type != $dbfield->type) {
                // Must always recreate to change the type..
                if ($PGdrop_column) {
                  $dropdiff .= $dbtable->dropcolumn($dbfield);
                  $dropdiff .= $table->addcolumn($field);
                }
                else {
                  $dropfields[] = $dbfield->name;
                }
              }
              else {
                // Changed constraints..
                if (!$dbfield->constraints_match($field)) {
                  foreach ($dbfield->constraints as $dbcon) {
                    if (!isset($field->constraints[$dbcon->name])) {
                      $alterdiff .= $dbcon->drop();
                    }
                    else {
                      $con = $field->constraints[$dbcon->name];
                      if ($con->create_inline() != $dbcon->create_inline()) {
                        $alterdiff .= $dbcon->drop();
                        $alterdiff .= $con->create();
                      }
                    }
                  }
                  foreach ($field->constraints as $con) {
                    if (!isset($dbfield->constraints[$con->name])) {
                      $alterdiff .= $con->create();
                    }
                  }
                }
                // Changed default or NULL setting..
                if (preg_replace("/[()]/", "", $field->defaultval) != preg_replace("/[()]/", "", $dbfield->defaultval)) {
                  $alterdiff .= $table->setdefault($field);
                }
                if ($field->notnull !== $dbfield->notnull) {
                  $alterdiff .= $table->setnullconstraint($field);
                }
              }
            }
          }
        } // foreach field

        if ($PGdrop_column) {
          if ($dropdiff != "") {
            $diff .= "\n-- COLUMNS to DROP/RECREATE on TABLE $table->name\n";
            $diff .= $dropdiff;
          }
          if ($alterdiff != "") {
            $diff .= "\n-- COLUMNS to ALTER on TABLE $table->name\n";
            $diff .= $alterdiff;
          }
        }
        else {
          if (!$nodrops && count($dropfields) > 0) {
            $diff .= "\n-- RE-CREATING CHANGED TABLE $table->name\n";
            $diff .= "-- Dropped (altered) fields: " . implode(" ", $dropfields) . "\n";
            $diff .= $dbtable->drop();
            $diff .= $table->create();
          }
          elseif ($alterdiff != "") {
            $diff .= "\n-- COLUMNS to ALTER on TABLE $table->name\n";
            $diff .= $alterdiff;
          }
        }
      } // table exists
    } // foreach $db->tables

    // Check for entities we need to create..
    foreach ($this->tables as $table) {
      if (!isset($db->tables[$table->name])) {
        $diff .= "\n-- CREATING NEW TABLE $table->name\n";
        $diff .= $table->create();
      }
      else {
        $dbtable = $db->gettable($table->name);
        foreach ($table->fields as $field) {
          if (!isset($dbtable->fields[$field->name])) {
            $diff .= "\n-- ADDING NEW COLUMN $field->name\n";
            $diff .= $dbtable->addcolumn($field);
          }
        }
      }
    }

    // Constraints to drop
    $pkcons_dropped = array();
    if (!$nodrops) {
      foreach ($db->constraints as $dbcon) {
        if (!isset($this->constraints[$dbcon->name])) {
          if (!in_array($dbcon->table->name, $tables_dropped)) {
            $diff .= "\n-- DROPPING NON-EXISTENT CONSTRAINT $dbcon->name\n";
            $diff .= $dbcon->drop();
            if ($dbcon->type == "p") $pkcons_dropped[] = $dbcon->name;
          }
        }
      }
    }

    // Constraints to create, or re-create..
    $pkcons_created = array();
    foreach ($this->constraints as $con) {
      if (!isset($db->constraints[$con->name])) {
        // Constraint not present in target DB..
        if ($con->type == "p") {
          $tablename = $con->table->name;
          $dbpkexists = false;
          foreach ($db->constraints as $dbcon) {
            if ($dbcon->table->name == $tablename) {
              $dbpkexists = true;
              break;
            }
          }
          if (!$dbpkexists || in_array($dbcon->name, $pkcons_dropped)) {
            $diff .= "\n-- CREATING NEW PRIMARY KEY CONSTRAINT $con->name\n";
            $diff .= $con->create();
            $pkcons_created[] = $con->name;
          }
          else {
            $diff .= "\n-- NOTICE: Primary key constraint $dbcon->name on table $tablename ";
            $diff .= "is named $con->name in the reference database.\n";
          }
        }
        else {
          $diff .= "\n-- CREATING NEW CONSTRAINT $con->name\n";
          $diff .= $con->create();
        }
      }
      else {
        // Constraint exists in both Dbs..
        $dbcon = $db->getconstraint($con->name);
        if (!$con->matches($dbcon)) {
          $diff .= "\n-- RE-CREATING CHANGED CONSTRAINT $dbcon->name\n";
          $diff .= $dbcon->drop();
          $diff .= $con->create();
          if ($con->type == "p") $pkcons_created[] = $con->name;
        }
        else {
          if ($con->name != $dbcon->name) {
            $diff .= "\n-- NOTICE: Foreign key constraint $dbcon->name on table $tablename ";
            $diff .= "is named $con->name in the reference database.\n";
          }
        }
      }
    }

    // Indexes to drop
    if (!$nodrops) {
      foreach ($db->indexes as $dbindex) {
        if (!isset($this->indexes[$dbindex->name])) {
          if (!in_array($dbindex->table->name, $tables_dropped)) {
            $diff .= "\n-- DROPPING NON-EXISTENT INDEX $dbindex->name\n";
            if (in_array($dbindex->name, $pkcons_dropped)) {
              $diff .= "-- dropped already via constraint\n";
            }
            else {
              $diff .= $dbindex->drop();
            }
          }
        }
      }
    }

    // Indexes to create, or re-create..
    foreach ($this->indexes as $index) {
      if (!isset($db->indexes[$index->name])) {
        $diff .= "\n-- CREATING NEW INDEX $index->name\n";
        if (in_array($index->name, $pkcons_created)) {
          $diff .= "-- already implicitly created via constraint\n";
        }
        else {
          $diff .= $index->create();
        }
      }
      else {
        $dbindex = $db->getindex($index->name);
        if ($index->create() != $dbindex->create()) {
          $diff .= "\n-- RE-CREATING MODIFIED INDEX $dbindex->name\n";
          if (isset($this->constraints[$dbindex->name])) {
            $diff .= "-- index is part of a constraint - cannot drop/recreate\n";
          }
          else {
            $diff .= $dbindex->drop();
            $diff .= $index->create();
          }
        }
      }
    }

    // Triggers to drop..
    $dropped_trigs = array();
    foreach ($db->triggers as $trig) {
      if (!isset($this->triggers[$trig->name])) {
        $dropped_trigs[$trig->name] = $trig;
        $diff .= "\n-- DROPPING NON-EXISTENT TRIGGER $trig->name\n";
        $diff .= $trig->drop();
      }
    }

    // Triggers to create, or re-create. New triggers to be
    // created are deferred until after we have dealt to
    // functions (see below)..
    $trigs_tocreate = array();
    foreach ($this->triggers as $trig) {
      if (!isset($db->triggers[$trig->name])) {
        $trigs_tocreate[$trig->name] = $trig;
      }
      else {
        $dbtrig = $db->gettrigger($trig->name);
        if ($trig->create() != $dbtrig->create()) {
          $diff .= "\n-- RE-CREATING TRIGGER $dbtrig->name\n";
          $diff .= $dbtrig->drop();
          $diff .= $trig->create();
        }
      }
    }

    // Functions to create, or re-create..
    foreach ($this->functions as $func) {
      if (!isset($db->functions[$func->name])) {
        $diff .= "\n-- CREATING NEW FUNCTION $func->name\n";
        $diff .= $func->create();
      }
      else {
        // This is complicated by the fact that the function is most
        // probably referenced by one or more triggers. We must drop
        // these first, and then recreate them afterward..
        $dbfunc = $db->getfunction($func->name);
        if ($func->create() != $dbfunc->create()) {
          $diff .= "\n-- RE-CREATING FUNCTION $dbfunc->name\n";

          // Drop triggers referencing our function..
          $diff .= "\n-- Dropping triggers whch reference this function..\n";
          $trigs_to_recreate = array();
          foreach ($this->triggers as $trig) {
            if (!isset($trigs_tocreate[$trig->name])
             && !isset($dropped_trigs[$trig->name])) {
              if ($trig->funcname == $dbfunc->name) {
                  $trigs_to_recreate[] = $trig;
                  $diff .= $trig->drop();
              }
            }
          } // foreach

          // Recreate the function..
          $diff .= "\n-- Recreating function..\n";
          $diff .= $dbfunc->drop();
          $diff .= $func->create();

          // Recreate dropped triggers..
          $diff .= "\n-- Re-establishing dropped triggers..\n";
          foreach ($trigs_to_recreate as $trig) {
            $diff .= $trig->create();
          }
        }
      }
    }

    // Create new triggers deferred 'til we'd done functions..
    foreach ($trigs_tocreate as $trigname => $trig) {
      $diff .= "\n-- CREATING NEW TRIGGER $trig->name\n";
      $diff .= $trig->create();
    }

    // Sequences to drop
    if (!$nodrops) {
      foreach ($db->sequences as $dbseq) {
        if (!isset($this->sequences[$dbseq->name])) {
          $diff .= "\n-- DROPPING NON-EXISTENT SEQUENCE $dbseq->name\n";
          $diff .= $dbseq->drop();
        }
      }
    }

    // Sequences to create..
    foreach ($this->sequences as $seq) {
      if (!isset($db->sequences[$seq->name])) {
        $diff .= "\n-- CREATING NEW SEQUENCE $seq->name\n";
        $diff .= $seq->create();
      }
    }

    // Return the diffs SQL..
    return $diff;
  } // sqldiff

} // class dbschema

// ######################################################################
// MAIN PROGRAM
// ######################################################################
//debug_on(DBG_DIAGNOSTIC|DBG_SQL);
//debug_output(DBG_O_CLI);

$VERSION = "2.10b";

// Only do anything if there are args..
$dbtarg = "";
$dbref = "";
$user = "";
$password = "";
$nodrops = false;

// MAIN
$opts = new optlist();
if ($opts->optcount > 1) {
  // Program name..
  $progname = $opts->progname;

  // Option setting(s)
  $nodrops = $opts->opt_exists("no-drops");

  // Database names
  $dbref  = $opts->opt_value("ref");
  $dbtarg = $opts->opt_value("target");

  // Username
  if ($opts->opt_exists("user")) {
    $user = $opts->opt_value("user");
  }
  else {
    $ubits = explode(" ", exec("who -m"));
    if ($ubits[0] != "") $user = $ubits[0];
    else $user = "postgres";
  }

  // Password..
  if ($opts->opt_exists("password")) {
    $password = $opts->opt_value("password");
  }

  // Get Postgres version. This is the master version for the source
  // databases and the default version for generating diffs. If you have,
  // say, Postgres v7.3 databases, and want to generate diffs for a
  // Postgres 7.2 database, then use the --opversion option (see below).
  $pg_version = "";
  $vstr = 7.0;
  if ($opts->opt_exists("pgversion")) {
    $vstr = $opts->opt_value("pgversion");
  }
  else {
    $pg_paths = array(
        "/usr/lib/postgresql/bin",
        "/usr/bin",
        "/usr/local/pgsql/bin"
        );
    // Find Postgres executable directory..
    foreach ($pg_paths as $pg_path) {
      if (file_exists("$pg_path/pg_ctl")) {
        $pg_bin = $pg_path;
        break;
      }
    }
    // Find database version..
    if (file_exists("$pg_bin/pg_config")) {
      $vbits = explode(" ", shell_exec("$pg_bin/pg_config --version"));
      if ($vbits[1] != "") {
        $vstr = $vbits[1];
      }
    }
    elseif (file_exists("$pg_bin/psql")) {
      $vbits = explode(" ", shell_exec("$pg_bin/psql --version"));
      if ($vbits[2] != "") {
        $vvbits = explode(".", $vbits[2]);
        $vstr = $vvbits[0] . "." . (isset($vvbits[1]) ? $vvbits[1] : "0");
      }
    }
  }
  $pg_version = (float) $vstr;

  // The Postgres version for output of diffs. This defaults to the same
  // as pg_version, but allows the user to specify that diffs are generated
  // for a target version different from the version of the source databases.
  if ($opts->opt_exists("opversion")) {
    $op_version = (float) $opts->opt_value("opversion");
  }
  else {
    $op_version = $pg_version;
  }

  //  $pg_version = 6.5;
  $PGdrop_column = ($pg_version >= 7.3);

  $RESPONSE = new response();
  $RESPONSE->datasource = new datasources();

  $RESPONSE->datasource->add_database(
    "postgres",            // Database type eg: postgres, mssql_server, oracle etc.
    $dbtarg,               // Database name
    $user,                 // Name of user with access to the database
    $password,             // Password for this user
    "",                    // Host machine name
    "",                    // Port number
    DEFAULT_DATASOURCE     // DEFAULT_DATASOURCE, or omit for other databases
    );

  $RESPONSE->datasource->add_database(
    "postgres",            // Database type eg: postgres, mssql_server, oracle etc.
    $dbref,                // Database name
    $user,                 // Name of user with access to the database
    $password,             // Password for this user
    "",                    // Host machine name
    ""                     // Port number
    );

  // Check db's connected ok..
  $RESPONSE->datasource->select($dbref);
  $RESPONSE->datasource->connect();

  $stat1 = $RESPONSE->datasource->connected($dbtarg);
  $stat2 = $RESPONSE->datasource->connected($dbref);
  if (!$stat1 || !$stat2) {
    if (!$stat1) echo "connect to $dbtarg failed.\n";
    if (!$stat2) echo "connect to $dbref failed.\n";
    exit;
  }

  // TABLES and FIELDS
  // Only want database tables owned by the user..
  $schemaQ  = "SELECT pgc.relname as tablename,";
  $schemaQ .= " pgc.oid as tableoid,";
  $schemaQ .= " pgc.relhaspkey as tablehaspkey,";
  $schemaQ .= " pga.attname as fieldname,";
  $schemaQ .= " pga.attnum as fieldnum,";
  $schemaQ .= " pga.atthasdef as fieldhasdef,";
  $schemaQ .= " pga.attnotnull as fieldnotnull,";
  $schemaQ .= " pga.attndims as fieldarrdims,";
  $schemaQ .= " pgt.typname as fieldtype";
  $schemaQ .= "  FROM pg_class pgc,pg_attribute pga, pg_type pgt";
  $schemaQ .= " WHERE pgc.relkind='r'";
  $schemaQ .= "   AND pgc.relowner > 31";
  $schemaQ .= "   AND pga.attrelid=pgc.oid";
  $schemaQ .= "   AND pga.attnum > 0";
  if ($PGdrop_column) {
    $schemaQ .= "   AND NOT pga.attisdropped";
  }
  $schemaQ .= "   AND pgt.oid=pga.atttypid";
  $schemaQ .= " ORDER BY pgc.relname,pga.attnum";

  // SEQUENCES
  // Only sequences created by the user..
  $seqQ  = "SELECT relname as sequencename";
  $seqQ .= "  FROM pg_class";
  $seqQ .= " WHERE relkind='S'";
  $seqQ .= "   AND relowner > 31";
  $seqQ .= " ORDER BY relname";

  // INDEXES
  // Only want indexes on tables which belong to the user, and
  // which are not Postgres-created tables..
  $indQ  = "SELECT";
  $indQ .= " i.indexrelid,";
  $indQ .= " i.indrelid,";
  $indQ .= " i.indkey as fieldnums,";
  $indQ .= " i.indisclustered,";
  $indQ .= " i.indisunique,";
  $indQ .= " i.indisprimary,";
  $indQ .= " i.indclass,";
  $indQ .= " pgam.amname,";
  $indQ .= " pgctb.relname as tablename,";
  $indQ .= " pgcix.relname as indexname";
  $indQ .= " FROM pg_index i,pg_class pgcix,pg_class pgctb,pg_am pgam";
  $indQ .= " WHERE pgcix.oid=i.indexrelid";
  $indQ .= "   AND pgctb.oid=i.indrelid";
  $indQ .= "   AND pgam.oid=pgcix.relam";
  $indQ .= "   AND pgctb.relowner > 31";
  $indQ .= "   AND pgctb.relname not like 'pg_%'";
  $indQ .= " ORDER BY i.indrelid";

  // FUNCTIONS
  // Only functions which are user-owned..
  $funQ  = "SELECT";
  $funQ .= " p.proname as funcname,";
  $funQ .= " p.proisstrict as strict,";
  if ($pg_version >= 7.3) {
    $funQ .= " p.provolatile as volatile,";
  }
  $funQ .= " p.prosrc as src,";
  $funQ .= " p.proargtypes as argoids,";
  $funQ .= " l.lanname as language";
  $funQ .= "  FROM pg_proc p, pg_language l";
  $funQ .= " WHERE p.proowner > 31";
  $funQ .= "   AND l.oid=p.prolang";


  // TRIGGERS
  // We are intersted here, in triggers which are only on
  // our user tables, and NOT RI constraints..
  $trgQ  = "SELECT";
  $trgQ .= " pgt.tgname as trigname,";
  $trgQ .= " pgt.tgtype as trigbitmask,";
  $trgQ .= " pgt.tgnargs as numargs,";
  $trgQ .= " pgt.tgargs as args,";
  $trgQ .= " pgc.relname as tablename,";
  $trgQ .= " pgp.proname as funcname";
  $trgQ .= "  FROM pg_trigger pgt, pg_class pgc, pg_proc pgp";
  $trgQ .= " WHERE pgc.oid=pgt.tgrelid";
  $trgQ .= "   AND pgc.relowner > 31";
  $trgQ .= "   AND pgp.oid=tgfoid";
  $trgQ .= "   AND NOT pgt.tgisconstraint";
  $trgQ .= "   AND pgt.tgenabled";

  $DBTarg = new dbschema($dbtarg);
  $DBTarg->get();

  $DBRef = new dbschema($dbref);
  $DBRef->get();

  // Announce
  $tstamp = date("d/m/Y H:i:s");

  echo "----------------------------------------------------------------------------\n";
  echo "-- pgdiff $VERSION Postgres v$pg_version";
  if ($op_version != $pg_version) {
    echo " (script target: Postgres v$op_version)\n";
  }
  echo "\n";
  echo "-- connecting as user $user";
  if ($password != "") echo " (password supplied)";
  echo " at $tstamp\n";
  echo "--\n";
  if ($nodrops) {
    echo "-- Option override: elements absent in database $dbref will NOT be\n";
    echo "-- dropped from the target $dbtarg (--no-drops).\n";
    echo "--\n";
    echo "-- Instructions:\n";
    echo "-- apply this script to target database $dbtarg, This will then add\n";
    echo "-- any new elements found in reference database, $dbref.\n";
  }
  else {
    echo "-- Instructions:\n";
    echo "-- apply this script to target database $dbtarg, This will then\n";
    echo "-- make it identical to the reference database, $dbref.\n";
  }
  echo "--\n";
  echo "----------------------------------------------------------------------------\n";

  // Output diff SQL..
  echo $DBRef->SQLdiff($DBTarg);

} // main
else {
  echo "usage: $opts->progname --target=dbname --ref=dbname\n";
  echo "              [--user=user] [--password=passwd]\n";
  echo "              [--pgversion=n.nn]\n";
  echo "              [--opversion=n.nn]\n";
  echo "              [--no-drops]\n";
  echo "  Returns SQL which will turn the database schema of --target into --ref.\n";
  echo "  --no-drops returns SQL to create only entities missing from --ref db.\n";
  echo "  --pgversion forces the Postgres database version.\n";
  echo "  --opversion specifies a different Postgres version for generating output.\n";
}
// ----------------------------------------------------------------------
?>
