/*
 * Copyright 2000 Murray Cumming
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <bakery/App/GtkDialogs.h>
#include <bakery/App/Dialog_OfferSave.h>
#include <bakery/App/Dialog_OverwriteFile.h>
#include <bakery/App/App_Gtk.h>
#include <gtkmm/stock.h>
#include <gtkmm/messagedialog.h>
#include <gtkmm/filechooserdialog.h>
#include <libgnomevfsmm.h>
#include <libintl.h>


namespace Bakery
{

void GtkDialogs::ui_warning(App& app, const Glib::ustring& text, const Glib::ustring& secondary_text)
{
  Gtk::MessageDialog dialog(App_Gtk::util_bold_message(text), true /* use markup */, Gtk::MESSAGE_WARNING);
  dialog.set_secondary_text(secondary_text);

  Gtk::Window* pWindow = dynamic_cast<Gtk::Window*>(&app);
  if(pWindow)
    dialog.set_transient_for(*pWindow);

  dialog.set_title(""); //The HIG says that alert dialogs should not have titles. The default comes from the message type.

  dialog.run();
}

Glib::ustring GtkDialogs::ui_file_select_open(App& app, const Glib::ustring& starting_folder_uri)
{
  Gtk::FileChooserDialog fileChooser_Open(gettext("Open Document"), Gtk::FILE_CHOOSER_ACTION_OPEN);

  Gtk::Window* pWindow = dynamic_cast<Gtk::Window*>(&app);
  if(pWindow)
    fileChooser_Open.set_transient_for(*pWindow);

  fileChooser_Open.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
  fileChooser_Open.add_button(Gtk::Stock::OPEN, Gtk::RESPONSE_OK);

  fileChooser_Open.set_default_response(Gtk::RESPONSE_OK);

  if(!starting_folder_uri.empty())
    fileChooser_Open.set_current_folder_uri(starting_folder_uri);

  int response_id = fileChooser_Open.run();
  fileChooser_Open.hide();
  if(response_id != Gtk::RESPONSE_CANCEL)
  {
    return fileChooser_Open.get_uri();
  }
  else
    return "";
}

static bool uri_is_writable(const Glib::RefPtr<const Gnome::Vfs::Uri>& uri)
{
  if(!uri)
    return false;

  Glib::RefPtr<const Gnome::Vfs::FileInfo> file_info = uri->get_file_info(Gnome::Vfs::FILE_INFO_GET_ACCESS_RIGHTS);
  if(file_info)
  {
    const Gnome::Vfs::FilePermissions permissions = file_info->get_permissions();
    return ((permissions & Gnome::Vfs::PERM_ACCESS_WRITABLE) == Gnome::Vfs::PERM_ACCESS_WRITABLE);
  }
  else
    return true; //Not every URI protocol supports FILE_INFO_GET_ACCESS_RIGHTS, so assume that it's writable and complain later.
}

Glib::ustring GtkDialogs::ui_file_select_save(App& app, const Glib::ustring& old_file_uri)
{
  Gtk::FileChooserDialog fileChooser_Save(gettext("Save Document"), Gtk::FILE_CHOOSER_ACTION_SAVE);
  fileChooser_Save.set_do_overwrite_confirmation(); //Ask the user if the file already exists.

  Gtk::Window* pWindow = dynamic_cast<Gtk::Window*>(&app);
  if(pWindow)
    fileChooser_Save.set_transient_for(*pWindow);

  fileChooser_Save.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
  fileChooser_Save.add_button(Gtk::Stock::SAVE, Gtk::RESPONSE_OK);

  fileChooser_Save.set_default_response(Gtk::RESPONSE_OK);

  //Make the save dialog show the existing filename, if any:
  if(!old_file_uri.empty())
  {
    //Just start with the parent folder,
    //instead of the whole name, to avoid overwriting:
    Glib::RefPtr<Gnome::Vfs::Uri> vfs_uri = Gnome::Vfs::Uri::create(old_file_uri);
    if(vfs_uri)
    {
      Glib::ustring uri_parent = vfs_uri->extract_dirname();
      fileChooser_Save.set_uri(uri_parent);
    }
  }


  //bool tried_once_already = false;

  bool try_again = true;
  while(try_again)
  {
    try_again = false;

    //Work around bug #330680 "GtkFileChooserDialog is too small when shown a second time.":
    //(Commented-out because the workaround doesn't work)
    /*
    if(tried_once_already)
    {
      fileChooser_Save.set_default_size(-1, 600); 
    }
    else
      tried_once_already = true;
    */

    const int response_id = fileChooser_Save.run();
    fileChooser_Save.hide();
    if(response_id != Gtk::RESPONSE_CANCEL)
    {
      const Glib::ustring uri = fileChooser_Save.get_uri();

      Glib::RefPtr<Gnome::Vfs::Uri> vfs_uri = Gnome::Vfs::Uri::create(uri);
      if(!vfs_uri)
        return Glib::ustring(); //Failure.

      //If the file exists (the FileChooser offers a "replace?" dialog, so this is possible.):
      if(App_WithDoc::file_exists(uri))
      {
        //Check whether we have rights to the file to change it:
        //Really, GtkFileChooser should do this for us.
        if(!uri_is_writable(vfs_uri))
        {
           //Warn the user:
           ui_warning(app, gettext("Read-only File."), gettext("You may not overwrite the existing file, because you do not have sufficient access rights."));
           try_again = true; //Try again.
           continue;
        }
      }

      //Check whether we have rights to the directory, to create a new file in it:
      //Really, GtkFileChooser should do this for us.
      Glib::RefPtr<const Gnome::Vfs::Uri> vfs_uri_parent = vfs_uri->get_parent();
      if(vfs_uri_parent)
      {
        if(!uri_is_writable(vfs_uri_parent))
        {
          //Warn the user:
           ui_warning(app, gettext("Read-only Directory."), gettext("You may not create a file in this directory, because you do not have sufficient access rights."));
           try_again = true; //Try again.
           continue;
        }
      }

      if(!try_again)
        return uri;
    }
    else
      return Glib::ustring(); //The user cancelled.
  }
}

App_WithDoc::enumSaveChanges GtkDialogs::ui_offer_to_save_changes(App& app, const std::string& filepath)
{
  App_WithDoc::enumSaveChanges result = App_WithDoc::SAVECHANGES_Cancel;

  Bakery::Dialog_OfferSave* pDialogQuestion = new Bakery::Dialog_OfferSave(filepath);

  Gtk::Window* pWindow = dynamic_cast<Gtk::Window*>(&app);
  if(pWindow)
    pDialogQuestion->set_transient_for(*pWindow);

  Bakery::Dialog_OfferSave::enumButtons buttonClicked = (Bakery::Dialog_OfferSave::enumButtons)pDialogQuestion->run();
  delete pDialogQuestion;
  pDialogQuestion = 0;

  if(buttonClicked == Bakery::Dialog_OfferSave::BUTTON_Save)
     result = App_WithDoc::SAVECHANGES_Save;
  else if(buttonClicked == Bakery::Dialog_OfferSave::BUTTON_Discard)
     result = App_WithDoc::SAVECHANGES_Discard;
  else
     result = App_WithDoc::SAVECHANGES_Cancel;

  return result;
}

} //namespace
