/* archive-meta-info.c
 *
 ****************************************************************
 * Copyright (C) 2003 Tom Lord
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */


#include "config-options.h"
#include "po/gettext.h"
#include "hackerlab/bugs/exception.h"
#include "hackerlab/cmd/main.h"
#include "libarch/archive.h"
#include "libarch/debug.h"
#include "libarch/namespace.h"
#include "libarch/project-tree.h"
#include "libarch/my.h"
#include "commands/cmd.h"
#include "commands/archive-meta-info.h"
#include "commands/version.h"



static t_uchar * usage = N_("[options] [archive/]item-name");

/* FIXME: archive-meta-info needs a -d argument */
#define OPTS(OP) \
  OP (opt_help_msg, "h", "help", 0, \
      N_("Display a help message and exit.")) \
  OP (opt_long_help, "H", 0, 0, \
      N_("Display a verbose help message and exit.")) \
  OP (opt_version, "V", "version", 0, \
      N_("Display a release identifier string\n" \
      "and exit.")) 


t_uchar arch_cmd_archive_meta_info_help[] = N_("report meta-info from an archive\n"
                                             "Print the contents of a meta-info file from an archive.\n"
                                             "For example,\n"
                                             "\n"
                                             "    % baz archive-meta-info name\n"
                                             "\n"
                                             "prints the official name of your default archive.\n");

enum options
{
  OPTS (OPT_ENUM)
};

static struct opt_desc opts[] =
{
  OPTS (OPT_DESC)
    {-1, 0, 0, 0, 0}
};



int
arch_cmd_archive_meta_info (t_uchar *program_name, int argc, char * argv[])
{
  int o;
  int result=0;
  struct opt_parsed * option;


  safe_buffer_fd (1, 0, O_WRONLY, 0);

  option = 0;

  while (1)
    {
      o = opt_standard (lim_use_must_malloc, &option, opts, &argc, argv, program_name, usage, libarch_version_string, arch_cmd_archive_meta_info_help, opt_help_msg, opt_long_help, opt_version);
      if (o == opt_none)
        break;
      switch (o)
        {
        default:
          safe_printfmt (2, _("unhandled option `%s'\n"), option->opt_string);
          panic ("internal error parsing arguments");

        usage_error:
          opt_usage (2, argv[0], program_name, usage, 1);
          exit (1);

          /* bogus_arg: */
          safe_printfmt (2, _("ill-formed argument for `%s' (`%s')\n"),
                         option->opt_string, option->arg_string);
          goto usage_error;

        }
    }

  if (argc != 2)
    goto usage_error;


  {
    /* try to connect to the archive the user supplied */
    t_uchar * namespace = NULL;
    t_uchar * query_item = NULL;
    t_uchar * info;
    struct arch_archive * arch = NULL;
    arch_project_tree_t * tree = arch_project_tree_new (talloc_context, ".");
    
    if (tree && tree->root)
      {
	arch_project_tree_check_name (tree, &arch, &namespace, argv[1]);
	arch_project_tree_delete (tree);
      }
    
    if (!arch)
	arch = arch_archive_connect_branch (argv[1], &namespace);

    if (arch)
      {
	/* archive is always present in namespace, and never has / in it */
	debug (dbg_ui, 2, _("arch connected: '%s' at '%s'\n"), arch->official_name, arch->location);
	if (str_chr_index (namespace, '/'))
	  query_item = str_save (0, str_chr_index (namespace, '/') + 1);
	if (!str_length (query_item))
	  {
	    lim_free (0, query_item);
	    query_item = NULL;
	  }
	debug (dbg_ui, 2, _("querying for '%s' from '%s'\n"), query_item, arch->official_name);
      }
    
    if (!query_item || !arch)
      {
        safe_printfmt (2, _("%s: unable to connect to an appropriate archive, or invalid query. '%s'\n"), argv[0], argv[1]);
        exit(1);
      }

    info = arch_get_meta_info (arch, query_item);
    result = ! info;
    safe_printfmt (1, "%s", (info ? info : (t_uchar *)""));

    arch_archive_close (arch);
    lim_free (0, info);
    lim_free (0, query_item);
    lim_free (0, namespace);
  }

  return result;
}




/* tag: Tom Lord Mon Jun  9 22:33:13 2003 (cmd-archive-meta-info.c)
 */
