/* categories.c
 *
 ****************************************************************
 * Copyright (C) 2003 Tom Lord
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */


#include "config-options.h"
#include "po/gettext.h"
#include "hackerlab/cmd/main.h"
#include "libarch/namespace.h"
#include "libarch/project-tree.h"
#include "libarch/my.h"
#include "libarch/archive.h"
#include "commands/cmd.h"
#include "commands/categories.h"
#include "commands/version.h"



static t_uchar * usage = N_("[options] [archive]");

#define OPTS(OP) \
  OP (opt_help_msg, "h", "help", 0, \
      N_("Display a help message and exit.")) \
  OP (opt_long_help, "H", 0, 0, \
      N_("Display a verbose help message and exit.")) \
  OP (opt_version, "V", "version", 0, \
      N_("Display a release identifier string\n" \
      "and exit."))


t_uchar arch_cmd_categories_help[] = N_("list the categories in an archive\n"
                                      "Print a list of the categories present in an archive.\n");

enum options
{
  OPTS (OPT_ENUM)
};

static struct opt_desc opts[] =
{
  OPTS (OPT_DESC)
    {-1, 0, 0, 0, 0}
};



int
arch_cmd_categories (t_uchar * program_name, int argc, char * argv[])
{
  int o;
  struct opt_parsed * option;
  t_uchar * archive = 0;


  safe_buffer_fd (1, 0, O_WRONLY, 0);

  option = 0;

  while (1)
    {
      o = opt_standard (lim_use_must_malloc, &option, opts, &argc, argv, program_name, usage, libarch_version_string, arch_cmd_categories_help, opt_help_msg, opt_long_help, opt_version);
      if (o == opt_none)
        break;
      switch (o)
        {
        default:
          safe_printfmt (2, "unhandled option `%s'\n", option->opt_string);
          panic ("internal error parsing arguments");

        usage_error:
          opt_usage (2, argv[0], program_name, usage, 1);
          exit (1);

          /* bogus_arg: */
          safe_printfmt (2, "ill-formed argument for `%s' (`%s')\n", option->opt_string, option->arg_string);
          goto usage_error;

        }
    }

  if (argc > 2)
    goto usage_error;

  if (argc == 2)
    archive = str_save (0, argv[1]);
  else
    archive = arch_my_default_archive (NULL);

  if (!archive)
    {
      safe_printfmt (2, "%s: no default archive set\n", argv[0]);
      exit (1);
    }

  {
    struct arch_archive * arch = 0;
    rel_table cats = 0;

    arch = arch_archive_connect_branch (archive, NULL);
    if (!arch)
      {
	safe_printfmt (2, _("%s: cannot connect to archive (%s)\n"), argv[0], archive);
	exit(1);
      }
    cats = arch_archive_categories (arch);
    rel_print_table (1, cats);
  }

  return 0;
}




/* tag: Tom Lord Tue May 20 13:57:39 2003 (categories.c)
 */
