#!/bin/sh
# -*- mode: sh; coding: utf-8 -*-
# Simple branching tests
# Copyright © 2003 Colin Walters <walters@verbum.org>
#
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.

set -e

arg0="$0"
srcdir=`dirname "$arg0"`

. ${srcdir}/test-framework

setup_with_trivial_tla_archives

test_class "branch"

commit_mainline_patch_1()
{
  tla get jane@example.com--2003/hello-world--mainline--1.0--base-0 foo-source
  cd foo-source
  echo "patch-1" >> "patch-1"
  tla add patch-1
  tla commit -s "patch-1"
  cd ..
  rm -rf foo-source
}

commit_mainline_patch_1

begin_test "creating tag from new version"
tla branch jane@example.com--2003/hello-world--mainline--1.0 jane@example.com--2003/hello-world--mainline--1.1
test -d '{arch}' && test_fail "{arch} directory created incorrectly"
tla get jane@example.com--2003/hello-world--mainline--1.1 hello-world
test -f hello-world/hello-world.c
test -f hello-world/README
end_test 

begin_test "creating tag from new branch"
tla branch jane@example.com--2003/hello-world--mainline--1.0 jane@example.com--2003/hello-world--foobranch--3
tla get jane@example.com--2003/hello-world--foobranch--3 foobranch
test -f foobranch/hello-world.c
test -f foobranch/README
end_test 

begin_test "creating tag from new category"
tla branch jane@example.com--2003/hello-world--mainline--1.0 jane@example.com--2003/hi-world--mainline--1.3
tla get jane@example.com--2003/hi-world--mainline--1.3 hi-world
test -f hi-world/hello-world.c
test -f hi-world/README
end_test 

begin_test "creating tag from different archive"
tla branch jane@example.com--2003/hello-world--mainline--1.0 foo@example.org--2003/hello-world--devo--1.4
tla get foo@example.org--2003/hello-world--devo--1.4 foo-hello-world
test -f foo-hello-world/hello-world.c
test -f foo-hello-world/README
end_test 

begin_test "test branch in a tree obeys two param form"
tla branch jane@example.com--2003/hello-world--mainline--1.0 foo@example.org--2003/hello-world--devo--1.5
tla get foo@example.org--2003/hello-world--devo--1.5 foo-hello-world
cd foo-hello-world
tla branch jane@example.com--2003/hello-world--mainline--1.0 foo@example.org--2003/hello-world--devo--1.5.1
cd ..
tla get foo@example.org--2003/hello-world--devo--1.5.1 foo-hello-world-2
test -f foo-hello-world-2/hello-world.c
test -f foo-hello-world-2/README
cd foo-hello-world-2
tla cat-log foo@example.org--2003/hello-world--devo--1.5--base-0 && test_fail "tagged from wrong branch"
cd ..
end_test

begin_test "test branch with one argument gets source from tree revision"
tla get jane@example.com--2003/hello-world--mainline--1.0--base-0 foo-source
cd foo-source
tla branch foo@example.org--2003/hello-world--devo--1.6
cd ..
tla get foo@example.org--2003/hello-world--devo--1.6 foo-hello-world
test -f foo-hello-world/hello-world.c
test -f foo-hello-world/README
cd foo-hello-world
tla cat-log jane@example.com--2003/hello-world--mainline--1.0--patch-1 && test_fail "tagged from wrong reversion"
cd ..
end_test

begin_test "test branch with one argument gets source from tree revision using -d"
tla get jane@example.com--2003/hello-world--mainline--1.0--base-0 foo-source
tla branch -d foo-source foo@example.org--2003/hello-world--devo--1.6
[ $(tla tree-version -d foo-source) = "foo@example.org--2003/hello-world--devo--1.6" ]
end_test

begin_test "test one-arg branch with unresolved conflicts"
tla get jane@example.com--2003/hello-world--mainline--1.0--patch-1 foo-source
cd foo-source
echo conflict! > patch-1
tla commit -s "modified file to create conflict"
! tla replay jane@example.com--2003/hello-world--mainline--1.0--patch-2
! tla branch jane@example.com--2003/hello-world--devo--1.6
if [ $(tla tree-version) != "jane@example.com--2003/hello-world--mainline--1.0" ]; then
  test_fail "Tree-version was modified"
fi
! tla changes
cd ..
end_test


begin_test "test branch in a tree with patch level obeys tree version"
tla get jane@example.com--2003/hello-world--mainline--1.0 foo-source
cd foo-source
tla branch base-0 foo@example.org--2003/hello-world--devo--1.7
cd ..
tla get foo@example.org--2003/hello-world--devo--1.7 foo-hello-world
test -f foo-hello-world/hello-world.c
test -f foo-hello-world/README
cd foo-hello-world
tla cat-log jane@example.com--2003/hello-world--mainline--1.0--patch-1 && test_fail "tagged from wrong reversion"
cd ..
end_test

begin_test "test branch in a tree with source package obeys tree archive before default archive."
tla my-default-archive foo@example.example.org
tla get jane@example.com--2003/hello-world--mainline--1.0 foo-source
cd foo-source
tla branch hello-world--mainline--1.0 foo@example.org--2003/hello-world--devo--1.8
cd ..
tla get foo@example.org--2003/hello-world--devo--1.8 foo-hello-world
test -f foo-hello-world/hello-world.c
test -f foo-hello-world/README
tla my-default-archive -d
end_test

begin_test "test branch in a tree with target package obeys tree archive before default archive."
tla my-default-archive foo@example.example.org
tla get jane@example.com--2003/hello-world--mainline--1.0 foo-source
cd foo-source
tla branch hello-world--mainline--1.0 hello-world--devo--1.5
cd ..
tla get jane@example.com--2003/hello-world--devo--1.5 foo-hello-world
test -f foo-hello-world/hello-world.c
test -f foo-hello-world/README
tla my-default-archive -d
end_test

begin_test "test branch in a tree sets the tree version in one argument form."
tla get jane@example.com--2003/hello-world--mainline--1.0--base-0 foo-source
cd foo-source
tla branch foo@example.org--2003/hello-world--devo--1.9
test "foo@example.org--2003/hello-world--devo--1.9" = $(tla tree-version)
test -z "$(tla missing)"
cd ..
end_test

begin_test "test branch in a tree does not set the tree version in two argument form."
tla get jane@example.com--2003/hello-world--mainline--1.0 foo-source
cd foo-source
tla branch patch-1 foo@example.org--2003/hello-world--devo--1.10
test "jane@example.com--2003/hello-world--mainline--1.0" = $(tla tree-version)
test -z "$(tla missing)"
cd ..
end_test

fail_unless_log_present()
{
  tla cat-log $1 || test_fail "log $1 not present"
  # echo "    present" $1 1>&2
  return 0
}

fail_if_log_present()
{
  tla cat-log $1 && test_fail "log $1 present"
  # echo "not present" $1 1>&2
  return 0
}

begin_test "test branch from a branch with two continuations doesn't include the first one"
tla branch jane@example.com--2003/hello-world--mainline--1.0 jane@example.com--2003/hello-world--mainline--1.11
tla branch jane@example.com--2003/hello-world--mainline--1.0 jane@example.com--2003/hello-world--mainline--1.11
tla branch jane@example.com--2003/hello-world--mainline--1.11 jane@example.com--2003/hello-world--mainline--1.11.1
tla get jane@example.com--2003/hello-world--mainline--1.11.1 foo-source
cd foo-source
fail_unless_log_present jane@example.com--2003/hello-world--mainline--1.0--base-0
fail_unless_log_present jane@example.com--2003/hello-world--mainline--1.0--patch-1
fail_if_log_present     jane@example.com--2003/hello-world--mainline--1.11--base-0
fail_unless_log_present jane@example.com--2003/hello-world--mainline--1.11--patch-1
fail_unless_log_present jane@example.com--2003/hello-world--mainline--1.11.1--base-0
tla cat-log jane@example.com--2003/hello-world--mainline--1.11.1--base-0 | grep jane@example.com--2003/hello-world--mainline--1.11--base-0 && test_fail "jane@example.com--2003/hello-world--mainline--1.11--base-0 included in the tag base-0"
cd ..
end_test

begin_test "test branch from a branch with a patch deleted"
tla branch jane@example.com--2003/hello-world--mainline--1.0--base-0 jane@example.com--2003/hello-world--mainline--1.12
tla get jane@example.com--2003/hello-world--mainline--1.12 foo-source
cd foo-source
# TODO factor this out
echo "patch-1" >> "patch-1"
tla add patch-1
tla commit -s "patch-1"
rm {arch}/hello-world/hello-world--mainline/hello-world--mainline--1.12/jane@example.com--2003/patch-log/patch-1
tla rm patch-1
tla commit --out-of-date-ok -s "delete patch-1 log"
cd ..
rm -rf foo-source
tla branch jane@example.com--2003/hello-world--mainline--1.12 jane@example.com--2003/hello-world--mainline--1.12.1
tla get jane@example.com--2003/hello-world--mainline--1.12.1 foo-source
cd foo-source
fail_unless_log_present jane@example.com--2003/hello-world--mainline--1.0--base-0
fail_unless_log_present jane@example.com--2003/hello-world--mainline--1.12--base-0
fail_if_log_present     jane@example.com--2003/hello-world--mainline--1.12--patch-1
fail_unless_log_present jane@example.com--2003/hello-world--mainline--1.12--patch-2
fail_unless_log_present jane@example.com--2003/hello-world--mainline--1.12.1--base-0

tla cat-log jane@example.com--2003/hello-world--mainline--1.12.1--base-0 | grep jane@example.com--2003/hello-world--mainline--1.12--patch-1 && test_fail "jane@example.com--2003/hello-world--mainline--1.12--patch-1 included in the tag base-0"
cd ..
end_test

begin_test "test branch from a branch with the patch from a 3rd branch deleted"
tla branch jane@example.com--2003/hello-world--mainline--1.0--base-0 jane@example.com--2003/hello-world--mainline--1.13
tla branch jane@example.com--2003/hello-world--mainline--1.13 jane@example.com--2003/hello-world--mainline--1.13.2
tla get jane@example.com--2003/hello-world--mainline--1.13.2 foo-source
cd foo-source
# TODO factor this out
echo "patch-1" >> "patch-1"
tla add patch-1
tla commit -s "patch-1"
cd ..
rm -rf foo-source
tla get jane@example.com--2003/hello-world--mainline--1.13 foo-source
cd foo-source
tla merge jane@example.com--2003/hello-world--mainline--1.13.2
tla commit -s 'merge from 1.13.2'
rm {arch}/hello-world/hello-world--mainline/hello-world--mainline--1.13.2/jane@example.com--2003/patch-log/patch-1
tla rm patch-1
tla commit -s "delete merged patch-1 log and patch"
cd ..
rm -rf foo-source
tla branch jane@example.com--2003/hello-world--mainline--1.13 jane@example.com--2003/hello-world--mainline--1.13.1
tla get jane@example.com--2003/hello-world--mainline--1.13.1 foo-source
cd foo-source
fail_unless_log_present jane@example.com--2003/hello-world--mainline--1.0--base-0
fail_unless_log_present jane@example.com--2003/hello-world--mainline--1.13--base-0
fail_unless_log_present jane@example.com--2003/hello-world--mainline--1.13--patch-1
fail_unless_log_present jane@example.com--2003/hello-world--mainline--1.13--patch-2
fail_unless_log_present jane@example.com--2003/hello-world--mainline--1.13.1--base-0
fail_unless_log_present jane@example.com--2003/hello-world--mainline--1.13.2--base-0
fail_if_log_present     jane@example.com--2003/hello-world--mainline--1.13.2--patch-1

tla cat-log jane@example.com--2003/hello-world--mainline--1.13.1--base-0 | grep jane@example.com--2003/hello-world--mainline--1.13.2--patch-1 && test_fail "jane@example.com--2003/hello-world--mainline--1.13.2--patch-1 included in the tag base-0"
cd ..
end_test

begin_test "test branch from a branch with the merging patch from a 3rd branch deleted"
tla branch jane@example.com--2003/hello-world--mainline--1.0--base-0 jane@example.com--2003/hello-world--mainline--1.14
tla branch jane@example.com--2003/hello-world--mainline--1.14 jane@example.com--2003/hello-world--mainline--1.14.2
tla get jane@example.com--2003/hello-world--mainline--1.14.2 foo-source
cd foo-source
# TODO factor this out
echo "patch-1" >> "patch-1"
tla add patch-1
tla commit -s "patch-1"
cd ..
rm -rf foo-source
tla get jane@example.com--2003/hello-world--mainline--1.14 foo-source
cd foo-source
tla merge jane@example.com--2003/hello-world--mainline--1.14.2
tla commit -s 'merge from 1.14.2'
rm {arch}/hello-world/hello-world--mainline/hello-world--mainline--1.14/jane@example.com--2003/patch-log/patch-1
tla commit --out-of-date-ok -s "delete patch-1 log"
cd ..
rm -rf foo-source
tla branch jane@example.com--2003/hello-world--mainline--1.14 jane@example.com--2003/hello-world--mainline--1.14.1
tla get jane@example.com--2003/hello-world--mainline--1.14.1 foo-source
cd foo-source
fail_unless_log_present jane@example.com--2003/hello-world--mainline--1.0--base-0
fail_unless_log_present jane@example.com--2003/hello-world--mainline--1.14--base-0
fail_if_log_present     jane@example.com--2003/hello-world--mainline--1.14--patch-1
fail_unless_log_present jane@example.com--2003/hello-world--mainline--1.14--patch-2
fail_unless_log_present jane@example.com--2003/hello-world--mainline--1.14.1--base-0
fail_unless_log_present jane@example.com--2003/hello-world--mainline--1.14.2--base-0
fail_unless_log_present jane@example.com--2003/hello-world--mainline--1.14.2--patch-1

tla cat-log jane@example.com--2003/hello-world--mainline--1.14.1--base-0 
tla cat-log jane@example.com--2003/hello-world--mainline--1.14.1--base-0 | grep jane@example.com--2003/hello-world--mainline--1.14.1--patch-1 && test_fail "jane@example.com--2003/hello-world--mainline--1.14.1--patch-1 included in the tag base-0"
cd ..
end_test

clean_workdir

# tag: Colin Walters Wed, 17 Sep 2003 18:02:12 -0400 (test-tag.sh)
#
