/*
** Copyright (C) 2003-2006 Teus Benschop.
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/


#include "libraries.h"
#include "utilities.h"
#include <libgen.h>
#include <glib.h>
#include "templates.h"
#include <config.h>
#include "directories.h"
#include "gwrappers.h"
#include "versification.h"
#include "mapping.h"
#include "progresswindow.h"
#include "languages.h"
#include "keyterms.h"


#define MINIMUM_VERSIFICATIONS_COUNT 9


void check_templates (bool gui)
/*
At some point in time templates in the form of sqlite databases were distributed
with Bibeldit. But it was discovered that this could give problems when a 
database was produced on the developer's computer using a newer version of
sqlite. Users might still use older versions of sqlite, and the file format 
is not always compatible, so that older versions of sqlite could not read the 
templates produced with a newer version.
The solution os to distribute templates in text files, and to let Bibledit
produces databases out of them.
There are some triggers that cause Bibledit to (re)create the templates:
1. There are fewer templates than expected.
2. The version number of the templates differs from the version number of Bibledit.
3. There is no version number stored.
*/
{
  // Decide whether to (re)create the templates.
  bool createtemplates = false;
  ustring version_filename = gw_build_filename (directories_get_templates (), "version");
  ReadText rt (version_filename, true, true);
  if (rt.lines.empty()) {
    createtemplates = true;
  } else {
    if (VERSION != rt.lines[0])
      createtemplates = true;
  }

  vector<ustring> count;
  versification_get_ones_available (count);
  if (count.size () < MINIMUM_VERSIFICATIONS_COUNT)
    createtemplates = true;


  ustring user_template_version_filename = gw_build_filename (directories_get_templates_user(), "date");
  ReadText rt_user_template (user_template_version_filename, true, true);
  unsigned int reference_time = 0;
  if (!rt_user_template.lines.empty())
    reference_time = convert_to_int (rt_user_template.lines[0]);
  ReadFiles rf (directories_get_templates_user (), "", ".txt");
  for (unsigned int i = 0; i < rf.files.size(); i++) {
    ustring filename = gw_build_filename (directories_get_templates_user (), rf.files[i]);
    unsigned int modificationtime = file_get_modification_time (filename);
    if (modificationtime != reference_time) {
      createtemplates = true;
      ustring command = "touch '" + directories_get_templates_user () + "'/*";
      system (command.c_str());
      reference_time = file_get_modification_time (filename);
      rt_user_template.lines.clear();
      rt_user_template.lines.push_back (convert_to_string (reference_time));
      write_lines (user_template_version_filename, rt_user_template.lines);
      break;
    }
  }

  if (!createtemplates) 
    return;

  // Wait a little while to allow the splash screen to settle.
  g_usleep (200000);
  
  // GUI
  ProgressWindow * progresswindow = NULL;
  if (gui) progresswindow = new ProgressWindow ("Creating templates", false);

  // Clear the directory of possible incompatible old stuff.
  ustring command = "rm '" + directories_get_templates () + "'/*";
  system (command.c_str());
  
  // Create the versification templates.
  if (gui) progresswindow->set_text ("Versifications");
  vector <ustring> paths = versification_get_raw_templates_available ();
  if (gui) progresswindow->set_iterate (0, 1, paths.size());
  for (unsigned int i = 0; i < paths.size(); i++) {
    if (gui) progresswindow->iterate ();
    versification_import_textfile (versification_get_name (paths[i]), paths[i]);
  }

  // Create the mapping templates.
  if (gui) progresswindow->set_text ("Verse mappings");
  paths = mapping_get_raw_templates_available ();
  if (gui) progresswindow->set_iterate (0, 1, paths.size());
  for (unsigned int i = 0; i < paths.size(); i++) {
    if (gui) progresswindow->iterate ();
    mapping_import_textfile (mapping_get_name (paths[i]), paths[i]);
  }

  // Create the language templates.
  if (gui) progresswindow->set_text ("Languages");
  paths = language_get_raw_templates_available ();
  if (gui) progresswindow->set_iterate (0, 1, paths.size());
  for (unsigned int i = 0; i < paths.size(); i++) {
    if (gui) progresswindow->iterate ();
    language_import_textfile (language_get_name (paths[i]), paths[i]);
  }

  // Check that the "English" template is always there.
  vector<ustring> lang1 = language_get_ones_available ();
  set<ustring> lang2 (lang1.begin(), lang1.end());
  if (lang2.find ("English") == lang2.end()) {
    for (unsigned int i = 0; i < paths.size(); i++) {
      if (paths[i].find ("english") != string::npos) {
        language_import_textfile ("English", paths[i]);
        cout << "Adding English language" << endl;        
        break;
      }
    }
  }
 
  // Create the keyterms database.
  if (gui) progresswindow->set_text ("Keyterms");
  keyterms_create_database ();
  paths = keyterms_get_raw_templates_available ();
  if (gui) progresswindow->set_iterate (0, 1, paths.size() + 3);
  for (unsigned int i = 0; i < paths.size(); i++) {
    if (gui) progresswindow->iterate ();
    keyterms_import_textfile (paths[i]);
  }
  if (gui) progresswindow->iterate ();
  keyterms_import_otkey_db ();  
  if (gui) progresswindow->iterate ();
  keyterms_import_ktref_db ();  
  if (gui) progresswindow->iterate ();
  keyterms_import_ktbh_txt ();  
  
  // Destroy gui.
  if (progresswindow) delete progresswindow;
  
  // Store version number.
  rt.lines.clear();
  rt.lines.push_back (VERSION);
  write_lines (version_filename, rt.lines);
}
