//
// Author:
//   Mikael Hallendal <micke@imendio.com>
//
// (C) 2004 Imendio HB
//

using GConf;
using Glade;
using Gtk;
using System;
using System.Collections;

namespace Imendio.Blam {
    public class Preference {
        public static string REVERSE_ENTRIES   = "behaviour/reverse_entries";
        public static string AUTO_REFRESH      = "behaviour/auto_refresh";
        public static string AUTO_REFRESH_RATE = "behaviour/auto_refresh_rate";
        public static string IGNORE_SSL_ERR    = "behaviour/ignore_ssl_err";
        public static string BEHAVIOUR_PATH    = Conf.GetFullKey("behaviour");
        public static string FONT_PATH         = "/desktop/gnome/interface";
        public static string VARIABLE_FONT     = "/desktop/gnome/interface/font_name";
        public static string FIXED_FONT        = "/desktop/gnome/interface/monospace_font_name";
        public static string THEME             = "ui/theme";
    }

    public class PreferencesDialog {
        [Widget] Gtk.Dialog      preferencesDialog         = null;
        [Widget] Gtk.Label       boldBehaviourLabel        = null;
        [Widget] Gtk.CheckButton reverseEntriesCheckButton = null;
        [Widget] Gtk.CheckButton autoRefreshCheckButton    = null;
        [Widget] Gtk.CheckButton ignoreSSLErr              = null;
        [Widget] Gtk.SpinButton  refreshRateSpinButton     = null;
        [Widget] Gtk.Label       boldThemeLabel            = null;
        [Widget] Gtk.ComboBox	 themeComboBox             = null;

		public PreferencesDialog (Gtk.Window parentWindow)
		{
			Glade.XML gladeXML = Glade.XML.FromAssembly ("blam.glade",
								"preferencesDialog",
								null);
			gladeXML.Autoconnect (this);
			preferencesDialog.TransientFor = parentWindow;
			preferencesDialog.Icon = Gdk.Pixbuf.LoadFromResource ("blam.png");

			boldBehaviourLabel.Markup = "<b>" + boldBehaviourLabel.Text + "</b>";
			bool reverseEntries = Conf.Get (Preference.REVERSE_ENTRIES, false);
			reverseEntriesCheckButton.Active = reverseEntries;

			bool autoRefresh = Conf.Get (Preference.AUTO_REFRESH, false);
			autoRefreshCheckButton.Active = autoRefresh;

			int refreshRate = Conf.Get (Preference.AUTO_REFRESH_RATE, 15);
			refreshRateSpinButton.Value = refreshRate;

			boldThemeLabel.Markup = "<b>" + boldThemeLabel.Text + "</b>";

			themeComboBox.Changed += new EventHandler(ThemeSelectionChanged);

			Conf.AddNotify (Preference.BEHAVIOUR_PATH, new NotifyEventHandler (ConfNotifyHandler));
		}

		public void Show ()
		{
			/* This needs to be this late, otherwise mono will die. */
			fill_themes_combo(themeComboBox);
			preferencesDialog.ShowAll ();
		}

		public void CloseButtonClicked (object obj, EventArgs args)
		{
			preferencesDialog.Hide();
		}

		public void ReverseItemsCheckButtonToggled (object obj, EventArgs args)
		{
			Conf.Set (Preference.REVERSE_ENTRIES,
					  reverseEntriesCheckButton.Active);
		}

		public void AutoRefreshCheckButtonToggled (object obj, EventArgs args)
		{
			Conf.Set (Preference.AUTO_REFRESH, autoRefreshCheckButton.Active);
		}

		public void RefreshRateSpinButtonChanged (object obj, EventArgs args)
		{
			Conf.Set (Preference.AUTO_REFRESH_RATE,
					  (int) refreshRateSpinButton.Value);
		}
		
		public void IgnoreSSLErrToggled (object obj, EventArgs args)
		{
		    Conf.Set (Preference.IGNORE_SSL_ERR,
		               (bool) ignoreSSLErr.Active);
		}

		public void ThemeSelectionChanged (object obj, EventArgs args)
		{
			Gtk.ComboBox box = obj as Gtk.ComboBox;
			Conf.Set(Preference.THEME, Application.TheApp.ThemeManager.PathByName(box.ActiveText));
		}

		private void ConfNotifyHandler (object sender, NotifyEventArgs args)
		{
			if (args.Key == Conf.GetFullKey (Preference.REVERSE_ENTRIES)) {
				reverseEntriesCheckButton.Active = (bool) args.Value;
			} else if (args.Key == Conf.GetFullKey (Preference.AUTO_REFRESH)) {
				autoRefreshCheckButton.Active = (bool) args.Value;
			} else if (args.Key == Conf.GetFullKey (Preference.AUTO_REFRESH_RATE)) {
				refreshRateSpinButton.Value = (int) args.Value;
			} else if (args.Key == Conf.GetFullKey (Preference.IGNORE_SSL_ERR)) {
			    ignoreSSLErr.Active = (bool) args.Value;
			}
		}
		
		private void fill_themes_combo(Gtk.ComboBox box)
		{
			box.Clear();
			CellRendererText cell = new CellRendererText();
			box.PackStart(cell, false);
			box.AddAttribute(cell, "text", 0);
			ListStore store = new ListStore(typeof(string));
			box.Model = store;

			TreeIter iter;
			string cur_theme = Conf.Get(Preference.THEME, Defines.DEFAULT_THEME);
			cur_theme = cur_theme.Substring(cur_theme.LastIndexOf("/") + 1);
			IList themes = Application.TheApp.ThemeManager.GetThemeList();
			
			foreach(Theme t in themes){
				iter = store.AppendValues(t.Name);

				if(t.Name == cur_theme){
					box.SetActiveIter(iter);
				}
			}
		}
		
	}
}

