//  BMPx - The Dumb Music Player
//  Copyright (C) 2005 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <cstring>
#include <glibmm.h>

#include <gst/gst.h>
#include <gst/gstelement.h>
#include <gst/interfaces/mixer.h>
#include <gst/interfaces/mixertrack.h>
#include <gst/interfaces/mixeroptions.h>

#include <boost/algorithm/string.hpp>
#include <boost/format.hpp>

#include "uri++.hh"
#include "util.hh"

#include "debug.hh"
#include "audio.hh"

#define AUDIO_DEBUG_NAME "bmp-audio"

namespace
{
    bool
    test_element (const char *name)
    {
      GstElement *element = 0;
      element = gst_element_factory_make (name, "test0");
      bool exists = GST_IS_ELEMENT (element);
      if (GST_IS_ELEMENT (element))
        {
          gst_object_unref (element);
          element = 0;
        }
      return exists;
    }
}

namespace Bmp
{
  namespace Audio
  {

    Caps
    get_caps ()
    {
      Caps caps = CAPS_NONE;
      if (test_element ("bmpx-neonhttpsrc")) caps = Caps (caps | CAPS_HTTP);
      if (test_element ("cdparanoiasrc")) caps = Caps (caps | CAPS_CDDA);
      if (test_element ("mmssrc")) caps = Caps (caps | CAPS_MMS);
      return caps;
    }

    void
    have_type_handler (GstElement     *typefind,
                       guint           probability,
                       const GstCaps  *caps,
                       GstCaps       **p_caps)
    {
      if (p_caps)
      {
        *p_caps = gst_caps_copy (caps);
      }
    }

    std::string ElementNames[] =
    {
        "source",
        "decoder",  
        "tee",
        "queue",
        "convert",
        "volume",
        "sink"
    };

    void
    link_pad (GstElement *element,
              GstPad     *pad,
              gboolean    last,
              GstPad	 *sinkpad) 
    {
        gst_pad_link (pad, sinkpad);
    }

    void
    outfit_element  (GstElement			              *element,
                     const Element::Attributes&    attributes)		     
    {
      using namespace Element;

      debug (AUDIO_DEBUG_NAME, "Creating properties for element '%s' of type '%s'",
      gst_element_get_name (element),  G_OBJECT_CLASS_NAME (element));		 
      for (Attributes::const_iterator iter = attributes.begin (); iter != attributes.end (); ++iter)
      {
        const Attribute& attr = (*iter);

        switch (attr.type)
        {
          using namespace Variant;

          case BOOL: 
          {
            debug (AUDIO_DEBUG_NAME,
           "Adding property of type 'bool' with value '%s' for element '%s'",
           boost::get<bool>(attr.value) ? "true":"false",
           gst_element_get_name (element)); 

            g_object_set (G_OBJECT(element), attr.name.c_str(), boost::get<bool>(attr.value), NULL);
            break;
          }

          case INTEGER: 
          {
            debug (AUDIO_DEBUG_NAME,
           "Adding property of type 'int' with value '%d' for element '%s'",
           boost::get<int>(attr.value),
           gst_element_get_name (element)); 

            g_object_set (G_OBJECT(element), attr.name.c_str(), boost::get<int>(attr.value), NULL);
            break;
          }

          case DOUBLE: 
          {
            debug (AUDIO_DEBUG_NAME,
           "Adding property of type 'double' with value '%lf' for element '%s'",
           boost::get<double>(attr.value),
           gst_element_get_name (element)); 

            g_object_set (G_OBJECT(element), attr.name.c_str(), boost::get<double>(attr.value), NULL);
            break;
          }

          case STRING: 
          {
            debug (AUDIO_DEBUG_NAME,
           "Adding property of type 'string' with value '%s' for element '%s'",
           boost::get<std::string>(attr.value).c_str (),
           gst_element_get_name (element)); 

            g_object_set (G_OBJECT(element), attr.name.c_str(), boost::get<std::string>(attr.value).c_str (), NULL);
            break;
          }
        } 
      }
    }

    GstElement*
    create_pipeline (GstElement			                     *src,
		                 const Bmp::Audio::Element::Element&  sink)
    {
      GstElement *pipeline = 0;
      GstElement *e[N_ELEMENTS];

      e[SOURCE] = src;

      //create sink
      e[SINK] = gst_element_factory_make (sink.name.c_str(), ElementNames[SINK].c_str ()); 

      if (!e[SINK])
            throw PipelineError ((boost::format ("Pipeline could not be created: Unable to create sink element '%s'") % sink.name.c_str()).str());

      outfit_element (e[SINK], sink.attributes); 

      e[DECODER] =
	    gst_element_factory_make ("decodebin",
				      ElementNames[DECODER].c_str()); 
      e[CONVERT] =
	    gst_element_factory_make ("audioconvert",
				      ElementNames[CONVERT].c_str()); 
      e[VOLUME] =
	    gst_element_factory_make ("volume",
				      ElementNames[VOLUME].c_str()); 
      e[TEE] =
	    gst_element_factory_make ("tee",
				      ElementNames[TEE].c_str()); 
      e[QUEUE] =
	    gst_element_factory_make ("queue",
				      ElementNames[QUEUE].c_str()); 
		
      pipeline = gst_pipeline_new ("pipeline_file");
		
      gst_bin_add_many (GST_BIN (pipeline),
				 e[SOURCE],
				 e[DECODER],
				 e[TEE],
				 e[QUEUE],
				 e[CONVERT],
				 e[VOLUME],
				 e[SINK],
				 NULL);
		
      gst_element_link_many (e[SOURCE],
			     e[DECODER],
			     NULL);
		
      gst_element_link_many (e[TEE],
			     e[QUEUE],
			     e[CONVERT],
			     e[VOLUME],
			     e[SINK],
			     NULL);
	
      g_signal_connect (G_OBJECT(e[DECODER]),
			"new-decoded-pad",
			G_CALLBACK (Bmp::Audio::link_pad),
			gst_element_get_pad (e[TEE], "sink"));

      return pipeline; 
    }

    GstElement*
    create_pipeline (const Bmp::Audio::Element::Element& source,
		                 const Bmp::Audio::Element::Element& sink)
    {
      GstElement *e_source = 0;

      //create source
      e_source = gst_element_factory_make (source. name. c_str (), ElementNames[SOURCE].c_str ());
      if (!e_source)
            throw PipelineError ((boost::format ("Pipeline could not be created: Unable to create source element '%s'") % ElementNames[SOURCE].c_str()).str());
      outfit_element (e_source, source.attributes); 
      return create_pipeline (e_source, sink);
    }
  }
}

//ProcessorBase
namespace Bmp
{ 
  namespace Audio
  {
    bool 
    typefind (std::string const& filename, std::string & type)
    {
      GstStateChangeReturn state_change;
      GstElement  *   pipeline  = 0;
      GstElement  *   source    = 0;
      GstElement  *   typefind  = 0;
      GstElement  *   fakesink  = 0;
      GstState        state;
      GstCaps     *   caps      = 0;

      pipeline  = gst_pipeline_new ("pipeline");
      source    = gst_element_factory_make ("filesrc", "source");
      typefind  = gst_element_factory_make ("typefind", "typefind");
      fakesink  = gst_element_factory_make ("fakesink", "fakesink");

      gst_bin_add_many (GST_BIN (pipeline), source, typefind, fakesink, NULL);
      gst_element_link_many (source, typefind, fakesink, NULL);

      g_signal_connect (G_OBJECT (typefind), "have-type", G_CALLBACK (have_type_handler), &caps);

      g_object_set (source, "location", filename.c_str(), NULL);
      gst_element_set_state (GST_ELEMENT (pipeline), GST_STATE_PAUSED);

      state_change = gst_element_get_state (GST_ELEMENT (pipeline), &state, NULL, GST_CLOCK_TIME_NONE);

      switch (state_change)
      {
        case GST_STATE_CHANGE_FAILURE:
        {
          GstBus *bus;
          bus = gst_pipeline_get_bus (GST_PIPELINE (pipeline));
          gst_object_unref (bus);
          break;
        }

        case GST_STATE_CHANGE_SUCCESS:
        {
          if (caps)
            {
              char *caps_str;
              caps_str = gst_caps_to_string (caps);
              std::vector<std::string> subs;
              std::string aux = caps_str;
              boost::algorithm::split (subs, aux, boost::algorithm::is_any_of(","));
              if (!subs.size())
                {
                  type = aux; 
                }
              else
                {
                  type = subs[0];
                }
              g_free (caps_str);
              gst_caps_unref (caps);
              gst_element_set_state (pipeline, GST_STATE_NULL);
              gst_object_unref (pipeline);
              return true;
            }
          else
            {
              gst_element_set_state (pipeline, GST_STATE_NULL);
              gst_object_unref (pipeline);
              return false; 
            }
          break;
        }
        default: break;
      }
      gst_element_set_state (pipeline, GST_STATE_NULL);
      gst_object_unref (pipeline);
      return false;
    }

    /////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
  
    ProcessorBase::ProcessorBase ()
          : Glib::ObjectBase (typeid(ProcessorBase)),
	          pipeline			                    (0),
	          prop_stream_time_		              (*this, "stream-time", 0),
	          prop_stream_time_report_interval_ (*this, "stream-time-report-interval", 250),
	          prop_length_		                  (*this, "stream-length", 0),
            prop_volume_                      (*this, "volume", 50),
	          prop_state_			                  (*this, "processor-state", Bmp::Audio::STATE_STOPPED)
      {
        prop_stream_time_report_interval ().signal_changed ().connect
              (sigc::mem_fun (this, &Bmp::Audio::ProcessorBase::position_send_interval_change));

        prop_volume().signal_changed().connect 
              (sigc::mem_fun (this, &Bmp::Audio::ProcessorBase::prop_volume_changed));
        conn_state = prop_state ().signal_changed ().connect 
              (sigc::mem_fun (this, &Bmp::Audio::ProcessorBase::prop_state_changed));
      }

      void
      ProcessorBase::prop_volume_changed ()
      {
        gdouble volume;
        volume = prop_volume_.get_value()/100.; 
        g_object_set (gst_bin_get_by_name (GST_BIN (pipeline), ElementNames[VOLUME].c_str()), "volume", volume, NULL);
      }

      void
      ProcessorBase::prop_state_changed ()
      {
        switch (prop_state_.get_value ())
        {
            case Bmp::Audio::STATE_STOPPED:
            stop ();
            break;

            case Bmp::Audio::STATE_PAUSED:
            pause ();
            break;

            case Bmp::Audio::STATE_RUNNING:
            run ();
            break;
        }
      }

      ProcessorBase::~ProcessorBase ()
      {
      }

      void
      ProcessorBase::link_pad (GstElement *element,
                               GstPad     *srcpad,
                               gboolean    last,
                               gpointer	   data) 
      {
        GstPad *sinkpad;
        sinkpad = reinterpret_cast<GstPad*>(data);
        gst_pad_link (srcpad, sinkpad);
      }

      gboolean
      ProcessorBase::bus_watch (GstBus     *bus,
                                GstMessage *message,
                                gpointer    data)
      {
        Bmp::Audio::ProcessorBase *processor; 
      	processor = reinterpret_cast<Bmp::Audio::ProcessorBase*>(data);
        
        switch (GST_MESSAGE_TYPE (message))
	      { 
	        case GST_MESSAGE_TAG:
	        {
#if 0
            GstTagList *tag_list;
            gst_message_parse_tag (message, &tag_list);
            if (tag_list)
            {
              char *title = 0;
              unsigned int bitrate = 0;

              if (gst_tag_list_get_string (tag_list, GST_TAG_TITLE, &title));
              {
                  if (title)
                  {
                    play->signal_title_.emit (title);
                  }
              }

              if (gst_tag_list_get_uint (tag_list, GST_TAG_BITRATE, &bitrate));
              {
                if (bitrate)
                {
                  play->signal_bitrate_.emit (bitrate/1000);
                }
              }
            }
#endif
	        break;
	        }

          case GST_MESSAGE_STATE_CHANGED:
          {
          		GstState old_state, new_state, pending_state;

                gst_message_parse_state_changed (message,
					                                       &old_state,
                                                 &new_state,
                                                 &pending_state);
       
                if ((new_state == GST_STATE_PLAYING) && !processor->conn_position.connected())
                {
                  processor->position_send_stop ();
                  processor->position_send_start ();
#if 0
                  if (message->src == GST_OBJECT(play->elements[play->current_pipe][PIPELINE]))
                  {
                    GstPad *pad = gst_element_get_pad (play->elements[play->current_pipe][SINK], "sink");
                    GstCaps *caps = gst_pad_get_negotiated_caps (pad);
                    gst_object_unref (pad);

                    for (unsigned int n = 0; n < gst_caps_get_size (caps); ++n)
                    {
                      GstStructure *structure = gst_caps_get_structure (caps, n);
                      gst_structure_foreach (structure, GstStructureForeachFunc (Bmp::Play::foreach_structure), play);
                    }

                  }
#endif
                }
              else if ((old_state >= GST_STATE_PAUSED) && (new_state < GST_STATE_PAUSED))
		            {
            		  processor->position_send_stop ();
            		}
              break;
            }

            case GST_MESSAGE_ERROR:
            {
                GstElement    *element;
                GError	      *error = 0;
                std::string    name,
			                         location,
			                         message_str;
		            char	        *debug_string;
        
                element = GST_ELEMENT(GST_MESSAGE_SRC (message));
                name = gst_object_get_name (GST_OBJECT(element));
        
                message_str = "<b>ERROR [Element: <i>"+name+"</i>]</b>\n";

                gst_message_parse_error	(message,
                                        &error,
                                        &debug_string);
        
                if (error->domain == GST_CORE_ERROR)
                {
                            switch (error->code)
                                {
                                    case GST_CORE_ERROR_MISSING_PLUGIN:
                                    {
                              					message_str += "(E101) There is no plugin available to play this track\n";
                                        break;
                                    }
        
                                    case GST_CORE_ERROR_SEEK:
                                    {
                                        message_str += "(E102) An error occured during seeking\n";
                                        break;
                                    }
        
                                    case GST_CORE_ERROR_STATE_CHANGE:
                                    {
                                        typedef struct _StringPairs StringPairs;
                                        struct _StringPairs {
                                            std::string   state_pre;
                                            std::string   state_post;
                                        };
        
                                        static StringPairs string_pairs[] = {
                                            {"0",       "READY"},
                                            {"READY",	  "PAUSED"},
                                            {"PAUSED",	"PLAYING"},
                                            {"PLAYING",	"PAUSED"},
                                            {"PAUSED",	"READY"},
                                            {"READY",	  "0"},
                                        };
       
                                        message_str += "(E103) An error occured during changing the state from "+string_pairs[GST_STATE_PENDING(message->src)].state_pre+" to "+string_pairs[GST_STATE_PENDING(message->src)].state_post+" (note that these are internal states and don't neccesarily reflect what you might understand by e.g. 'play' or 'pause'\n";
                                        break;
                                    }
        
                                    case GST_CORE_ERROR_PAD:
                                    {
                                        message_str += "(E104) An error occured during pad linking/unlinking\n";
                                        break;
                                    }
        
                                    case GST_CORE_ERROR_NEGOTIATION:
                                    {
                                        message_str =+ "(E105) An error occured during element(s) negotiation\n";
                                        break;
                                    }
        
                                    default: break;
                                }
                }
                else if (error->domain == GST_RESOURCE_ERROR)
                {
                            switch (error->code)
                                {
        
                                    case GST_RESOURCE_ERROR_SEEK:
                                    {
                                        message_str += "(E201) An error occured during seeking\n";
                                        break;
                                    }
        
                                    case GST_RESOURCE_ERROR_NOT_FOUND:
                                    case GST_RESOURCE_ERROR_OPEN_READ:
                                    {
                                        message_str += "(E202) Couldn't open the stream/track\n";
                                        break;
                                    }
        
                                    default: break;
                                }
                    }
                else if (error->domain == GST_STREAM_ERROR)
                    {
                            switch (error->code)
                                {
                                    case GST_STREAM_ERROR_CODEC_NOT_FOUND:
                                    {
                                        message_str += "(E301) There is no codec installed to handle this stream\n";
                                        break;
                                    }
        
                                    case GST_STREAM_ERROR_DECODE:
                                    {
                                        message_str += "(E302) There was an error decoding the stream\n";
                                        break;
                                    }
        
                                    case GST_STREAM_ERROR_DEMUX:
                                    {
                                        message_str += "(E303) There was an error demultiplexing the stream\n";
                                        break;
                                    }
        
                                    case GST_STREAM_ERROR_FORMAT:
                                    {
                                        message_str += "(E304) There was an error parsing the format of the stream\n";
                                        break;
                                    }
        
                                    default: break;
                                }
                    }
        
                  message_str += "\n<b>Current URI:</b>\n"+location+"\n\n<b>Detailed debugging information:</b>\n"+debug_string; 
            	  	g_free (debug_string);
                  g_error_free (error);

              		processor->stop ();
                  processor->signal_error_.emit(message_str);

                  break;
            }
        
            case GST_MESSAGE_WARNING:
            {
                GError  *error = 0;
                gchar   *debug;
        
                gst_message_parse_warning (message, &error, &debug);
                g_print ("%s WARNING: %s (%s)\n", G_STRLOC, error->message, debug);
                g_error_free (error);
                break;
            }
        
            case GST_MESSAGE_EOS:
            {
      	        processor->conn_position.disconnect ();
                processor->signal_eos_.emit();
                break;
            }
        
            default: break;
	      }
        
        return TRUE;
      }

      bool
      ProcessorBase::emit_stream_position ()
      {
        GstQuery       *query;
        GstFormat       format = GST_FORMAT_TIME;
        gint64          time_in_nanoseconds;
        gint            time_in_seconds;

        if ((GST_STATE (pipeline) != GST_STATE_PLAYING))
          {
            return false;
          }

        query = gst_query_new_position (format);
        gst_element_query (pipeline, query);
        gst_query_parse_position (query, &format, &time_in_nanoseconds);
        time_in_seconds = time_in_nanoseconds / GST_SECOND;
        gst_query_unref (query);

        if ((time_in_seconds >= 0) && (GST_STATE(pipeline) >= GST_STATE_PLAYING))
          {
            signal_position_.emit(time_in_seconds);
            return true;
          }
        else
          {
            signal_position_.emit(0);
            return false;
          }
      }

      //Properties/Property proxies
      Glib::PropertyProxy<unsigned int>
      ProcessorBase::prop_stream_time ()
      {
        return prop_stream_time_.get_proxy ();
      }

      Glib::PropertyProxy<unsigned int>
      ProcessorBase::prop_stream_time_report_interval ()
      {
        return prop_stream_time_report_interval_.get_proxy ();
      }

      Glib::PropertyProxy<ProcessorState>
      ProcessorBase::prop_state ()
      {
        return prop_state_.get_proxy ();
      }

      Glib::PropertyProxy<int>
      ProcessorBase::prop_volume()
      {
        return prop_volume_.get_proxy();
      }

      Glib::PropertyProxy<unsigned int>
      ProcessorBase::prop_length()
      {
        GstQuery       *query;
        GstFormat       format = GST_FORMAT_TIME;
        gint64          length_in_nanoseconds;
        gint		        value = 0;

        query = gst_query_new_duration (format);
        gst_element_query (pipeline, query);
        gst_query_parse_duration (query, &format, &length_in_nanoseconds);
        value = length_in_nanoseconds / GST_SECOND;
        gst_query_unref (query);

        prop_length_ = value;
        return prop_length_.get_proxy();
      }

      void
      ProcessorBase::position_send_stop ()
      {
        if (conn_position.connected()) conn_position.disconnect ();
        signal_position_.emit(0);
      }

      void
      ProcessorBase::position_send_start ()
      {
        conn_position =
                  Glib::signal_timeout().connect(sigc::mem_fun (*this, &Bmp::Audio::ProcessorBase::emit_stream_position),
					                                                              prop_stream_time_report_interval_.get_value ());
      }

      void
      ProcessorBase::position_send_interval_change ()
      {
        position_send_stop ();
      	if (prop_state_.get_value () == Bmp::Audio::STATE_RUNNING)
          {
        	  position_send_start ();
        	}
      }

      //Signals
      Bmp::Audio::SignalEos&
      ProcessorBase::signal_eos ()
      { 
        return signal_eos_;
      }

      Bmp::Audio::SignalError&
      ProcessorBase::signal_error ()
      { 
        return signal_error_;
      }


      Bmp::Audio::SignalPosition&
      ProcessorBase::signal_position ()
      {
      	return signal_position_;
      }

      Bmp::Audio::SignalStreamProperties&
      ProcessorBase::signal_stream_properties ()
      {
      	return signal_stream_properties_;
      }


      GstStateChangeReturn
      ProcessorBase::stop ()
      {
        if (!pipeline) return GST_STATE_CHANGE_FAILURE; //FIXME: Exception
        conn_state.disconnect ();
      	conn_position.disconnect ();
      	prop_state_ = Bmp::Audio::STATE_STOPPED;
      	GstStateChangeReturn state_change = gst_element_set_state (pipeline, GST_STATE_READY);
        return state_change;
      }

      GstStateChangeReturn
      ProcessorBase::pause ()
      {
        if (!pipeline) return GST_STATE_CHANGE_FAILURE; //FIXME: Exception
        conn_state.disconnect ();
      	prop_state_ = Bmp::Audio::STATE_PAUSED;
        GstStateChangeReturn state_change = gst_element_set_state (pipeline, GST_STATE_PAUSED);
        conn_state = prop_state ().signal_changed ().connect (sigc::mem_fun (*this, &Bmp::Audio::ProcessorBase::prop_state_changed));
        return state_change; 
      }

      GstStateChangeReturn
      ProcessorBase::run ()
      {
      	if (!pipeline) return GST_STATE_CHANGE_FAILURE; //FIXME: Exception
      	conn_position.disconnect ();
        conn_state.disconnect ();
      	prop_state_ = Bmp::Audio::STATE_RUNNING;
      	GstStateChangeReturn state_change = gst_element_set_state (pipeline, GST_STATE_PLAYING);
        conn_state = prop_state ().signal_changed ().connect (sigc::mem_fun (*this, &Bmp::Audio::ProcessorBase::prop_state_changed));
        return state_change;
      }

      GstElement*
      ProcessorBase::tap ()
      {
        return gst_bin_get_by_name (GST_BIN(pipeline), ElementNames[TEE].c_str());	
      }

      bool
      ProcessorBase::verify_pipeline ()
      {
        GstIterator *iter = 0;
        GstIteratorResult result;

        iter = gst_bin_iterate_elements (GST_BIN (pipeline));

        while (iter)
          {
            gpointer elem;
            result = gst_iterator_next (iter, &elem); 
            switch (result)
              {
                  case GST_ITERATOR_DONE:
                      gst_iterator_free (iter);
                      return true;
                      break;

                  case GST_ITERATOR_OK:
                      if (!GST_IS_ELEMENT (elem))
                      {
                        gst_iterator_free (iter);
                        return false;
                      }
                      break;

                  case GST_ITERATOR_ERROR:
                      gst_iterator_free (iter);
                      return false;
                      break;

                  case GST_ITERATOR_RESYNC:
                      gst_iterator_resync (iter);
                      break;
              }
          }
        return false;
      }
  }
};

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

namespace Bmp
{ 
  namespace Audio
  {

      ProcessorURISink::ProcessorURISink  ()
      {
      }

      ProcessorURISink::~ProcessorURISink ()
      {
      }

      void
      ProcessorURISink::create_pipeline ()
      {
        if (pipeline)	
        	{
        	  gst_element_set_state (GST_ELEMENT(pipeline), GST_STATE_NULL);
        	  g_object_unref (pipeline);
        	}

        	pipeline = Bmp::Audio::create_pipeline (source, sink);
      }

      void
      ProcessorURISink::set_uri (Glib::ustring	const&  _uri,
                                 const Bmp::Audio::Element::Element& sink)
      {
        Bmp::URI u (_uri);
        Bmp::URI::Protocol protocol = u.get_protocol(); 

        stop ();
        this->stream = _uri;

        if (protocol != current_protocol)
        {
          switch (protocol)
          {
              case Bmp::URI::PROTOCOL_FILE:
              {
                debug (AUDIO_DEBUG_NAME,
                "Creating filesrc element");

                source = Element::Element ("filesrc");
                source.add (Element::Attribute ("location", Glib::filename_from_uri (_uri)));
                break;
              }

              case Bmp::URI::PROTOCOL_HTTP:
              {
                debug (AUDIO_DEBUG_NAME, "Creating neonhttpsrc element");

                source = Element::Element ("neonhttpsrc");
                source.add (Element::Attribute ("location", std::string (_uri)));
                source.add (Element::Attribute ("iradio-mode", true));
                break;
	            }

      	      case Bmp::URI::PROTOCOL_CDDA:
	            {
                int track = atoi ((u.path.c_str())+1);
                debug (AUDIO_DEBUG_NAME, "Creating cdparanoiasrc element");

                source = Element::Element ("cdparanoiasrc");
                source.add (Element::Attribute ("track", int(track)));
                break;
              }

	            default: break; //FIXME: error condition
	        }

	      this->sink = sink;
	      create_pipeline ();
	    }
	  else
	    {
	      switch (protocol)
	      {
	        case Bmp::URI::PROTOCOL_FILE:
	        {
		        g_object_set (gst_bin_get_by_name (GST_BIN (pipeline), ElementNames[SOURCE].c_str()), "location", Glib::filename_from_uri (_uri).c_str (), NULL);
		        break;
	        }

	        case Bmp::URI::PROTOCOL_HTTP:
	        {
		        g_object_set (gst_bin_get_by_name (GST_BIN (pipeline), ElementNames[SOURCE].c_str()), "location", _uri.c_str (), NULL);
		        break;
	        }

	        case Bmp::URI::PROTOCOL_CDDA:
	        {
		        int track = atoi ((u.path.c_str())+1);
		        g_object_set (gst_bin_get_by_name (GST_BIN (pipeline), ElementNames[SOURCE].c_str()), "track", track, NULL);
		        break;
	        }

	        default: break; //FIXME: error condition
	      }
	    } 
    }
  }
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#ifdef HAVE_OFA
namespace Bmp
{ 
  namespace Audio
  {
      void
      ProcessorPUID::stream_eos ()
      {
        stop ();
        char *puid = 0, *artist = 0, *title = 0;
        GstElement *elmt = gst_bin_get_by_name (GST_BIN (pipeline), "puid");
        g_object_get (G_OBJECT(elmt), "puid", &puid, "artist", &artist, "title", &title, NULL);

        if (puid && strlen (puid))
          {
            m_puid = puid;
            g_free (puid);
          }

        if (artist && strlen (artist))
          {
            m_artist = artist;
            g_free (artist);
          }

        if (title && strlen (title))
          {
            m_title = title;
            g_free (title);
          }
      }
    
      boost::optional<Glib::ustring>
      ProcessorPUID::get_puid () const
      {
        return m_puid;
      }
      boost::optional<Glib::ustring>
      ProcessorPUID::get_artist () const
      {
        return m_artist;
      }
      boost::optional<Glib::ustring>
      ProcessorPUID::get_title () const
      {
        return m_title;
      }

      ProcessorPUID::ProcessorPUID  () : Glib::ObjectBase(typeid(ProcessorPUID))
      {
        create_pipeline ();
        signal_eos().connect (sigc::mem_fun (this, &Bmp::Audio::ProcessorPUID::stream_eos));
      }

      ProcessorPUID::~ProcessorPUID ()
      {
        gst_element_set_state (GST_ELEMENT (pipeline), GST_STATE_NULL);
        gst_object_unref (GST_OBJECT (pipeline));
      }

      void
      ProcessorPUID::set_uri (Glib::ustring	const& uri)
      {
        m_stream = uri;
        Bmp::URI u (uri);
        Bmp::URI::Protocol protocol = u.get_protocol(); 

        if (protocol != Bmp::URI::PROTOCOL_FILE)
          {
            throw InvalidUriError ((boost::format ("Unable to parse URI '%s'") % uri.c_str()).str());
          }
  
        GstElement *src = gst_bin_get_by_name (GST_BIN (pipeline), "src");
        g_object_set (G_OBJECT (src), "location", Glib::filename_from_uri (m_stream).c_str(), NULL);
	    } 

      void
      ProcessorPUID::create_pipeline ()
      {
        GstElement *elements[N_ELEMENTS];

        elements[SOURCE] =
            gst_element_factory_make ("filesrc",
                                      "src");

        elements[DECODER] =
            gst_element_factory_make ("decodebin",
                                      "decoder");

        elements[CONVERT] =
            gst_element_factory_make ("audioconvert",
                                      "audioconvert");

        elements[PUID] =
            gst_element_factory_make ("puid-bmpx",
                                      "puid");

        elements[SINK] =
            gst_element_factory_make ("fakesink",
                                      "sink");

        g_object_set (G_OBJECT (elements[SINK]), "sync", FALSE, NULL);
        g_object_set (G_OBJECT (elements[PUID]), "musicdns-id", "a7f6063296c0f1c9b75c7f511861b89b", NULL);

        pipeline = gst_pipeline_new ("pipeline");

        gst_bin_add_many (GST_BIN (pipeline),
                          elements[SOURCE],
                          elements[DECODER],
                          elements[CONVERT],
                          elements[PUID],
                          elements[SINK],
                          NULL);

        gst_element_link_many (elements[SOURCE],
                               elements[DECODER], NULL);

        gst_element_link_many (elements[CONVERT],
                               elements[PUID],
                               elements[SINK], NULL);

        g_signal_connect (G_OBJECT(elements[DECODER]),
                          "new-decoded-pad",
                          G_CALLBACK (link_pad),
                          gst_element_get_pad (elements[CONVERT], "sink"));
        gst_bus_add_watch (gst_pipeline_get_bus (GST_PIPELINE (pipeline)), GstBusFunc(bus_watch), this); 
      }
  }
}
#endif

namespace Bmp
{ 
  namespace Audio
  {
      ProcessorCDDA_Vorbis::ProcessorCDDA_Vorbis  (std::string const& path, unsigned int track, std::string const& device, int quality)
          : Glib::ObjectBase (typeid(ProcessorCDDA_Vorbis)), m_path (path), m_track (track), m_device (device), m_quality (quality) 
      {
        ProcessorCDDA_Vorbis::create_pipeline ();
      }

      ProcessorCDDA_Vorbis::~ProcessorCDDA_Vorbis ()
      {
        gst_element_set_state (GST_ELEMENT (pipeline), GST_STATE_NULL);
        gst_object_unref (GST_OBJECT (pipeline));
      }

      void
      ProcessorCDDA_Vorbis::create_pipeline ()
      {
        GstElement *elements[N_ELEMENTS];

        elements[SOURCE]    = gst_element_factory_make ("cdparanoiasrc",  "src");
        elements[CONVERT]   = gst_element_factory_make ("audioconvert",   "audioconvert");
        elements[VORBISENC] = gst_element_factory_make ("vorbisenc",      "vorbisenc");
        elements[OGGMUX]    = gst_element_factory_make ("oggmux",         "oggmux");
        elements[SINK]      = gst_element_factory_make ("filesink",       "sink");

        if (!(elements[SOURCE] && elements[CONVERT] && elements[VORBISENC] && elements[OGGMUX] && elements[SINK]))
          {
            throw PipelineError ("Unable to create Pipeline");
          }
           
        pipeline = gst_pipeline_new ("pipeline");

        gst_bin_add_many (GST_BIN (pipeline),
                          elements[SOURCE],
                          elements[CONVERT],
                          elements[VORBISENC],
                          elements[OGGMUX],
                          elements[SINK],
                          NULL);

        gst_element_link_many (elements[SOURCE],
                               elements[CONVERT],
                               elements[VORBISENC],
                               elements[OGGMUX],
                               elements[SINK],
                               NULL); 

        if (!verify_pipeline())
          {
            throw PipelineError ("Pipeline could not be created");
          }

        switch (m_quality)
        {
            case 0:
              g_object_set (G_OBJECT (elements[VORBISENC]),  "quality", double(0.1), 
                                                             NULL);
              break;

            case 1:
              g_object_set (G_OBJECT (elements[VORBISENC]),  "quality", double(0.4), 
                                                             NULL);
              break;

            case 2:
              g_object_set (G_OBJECT (elements[VORBISENC]),  "quality", double(0.7),
                                                             NULL);
              break;
        }

        debug ("audio", "mp3-vorbis: path: %s", m_path.c_str());

        g_object_set (G_OBJECT (elements[SINK]),    "location", m_path.c_str(),
                                                     NULL);

        g_object_set (G_OBJECT (elements[SOURCE]),  "track",    m_track, 
                                                    "device",   m_device.c_str(), 
                                                     NULL);


        gst_bus_add_watch (gst_pipeline_get_bus (GST_PIPELINE (pipeline)), GstBusFunc(bus_watch), this); 
        prop_stream_time_report_interval() = 2000;
      }

      ///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

      ProcessorCDDA_MP3::ProcessorCDDA_MP3  (std::string const& path, unsigned int track, std::string const& device, int quality)
          : Glib::ObjectBase (typeid(ProcessorCDDA_MP3)), m_path (path), m_track (track), m_device (device), m_quality (quality)
      {
        ProcessorCDDA_MP3::create_pipeline ();
      }

      ProcessorCDDA_MP3::~ProcessorCDDA_MP3 ()
      {
        gst_element_set_state (GST_ELEMENT (pipeline), GST_STATE_NULL);
        gst_object_unref (GST_OBJECT (pipeline));
      }

      void
      ProcessorCDDA_MP3::create_pipeline ()
      {
        GstElement *elements[N_ELEMENTS];

        elements[SOURCE]  = gst_element_factory_make ("cdparanoiasrc",  "src");
        elements[CONVERT] = gst_element_factory_make ("audioconvert",   "audioconvert");
        elements[LAME]    = gst_element_factory_make ("lame",           "lame");
        elements[SINK]    = gst_element_factory_make ("filesink",       "sink");

        if (!(elements[SOURCE] && elements[CONVERT] && elements[LAME] && elements[SINK]))
          {
            throw PipelineError ("Pipeline could not be created");
          }

        debug ("audio", "mp3-cdda: path: %s", m_path.c_str());
           
        g_object_set (G_OBJECT (elements[SINK]),    "location", m_path.c_str(),
                                                     NULL);

        g_object_set (G_OBJECT (elements[SOURCE]),  "track",    m_track,
                                                    "device",   m_device.c_str(),
                                                     NULL);

        pipeline = gst_pipeline_new ("pipeline");
        gst_bin_add_many (GST_BIN (pipeline), elements[SOURCE], elements[CONVERT], elements[LAME], elements[SINK], NULL);
        gst_element_link_many (elements[SOURCE], elements[CONVERT], elements[LAME], elements[SINK], NULL); 

        if (!verify_pipeline())
          {
            throw PipelineError ("Pipeline could not be created");
          }

        switch (m_quality)
        {
            case 0:
              g_object_set (G_OBJECT (elements[LAME]), "vbr", int(4), "vbr-mean-bitrate", int(96), "mode", int(1), NULL);
              break;

            case 1:
              g_object_set (G_OBJECT (elements[LAME]), "vbr", int(4), "vbr-mean-bitrate", int(128), "mode", int(1), NULL);
              break;

            case 2:
              g_object_set (G_OBJECT (elements[LAME]), "vbr", int(4), "vbr-mean-bitrate", int(256), "mode", int(1), NULL);
              break;
        }

        gst_bus_add_watch (gst_pipeline_get_bus (GST_PIPELINE (pipeline)), GstBusFunc(bus_watch), this); 
        prop_stream_time_report_interval() = 2000;
      }

      ///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

      ProcessorCDDA_FLAC::ProcessorCDDA_FLAC  (std::string const& path, unsigned int track, std::string const& device, int quality)
          : Glib::ObjectBase (typeid(ProcessorCDDA_FLAC)), m_path (path), m_track (track), m_device (device), m_quality (quality)
      {
        ProcessorCDDA_FLAC::create_pipeline ();
      }

      ProcessorCDDA_FLAC::~ProcessorCDDA_FLAC ()
      {
        gst_element_set_state (GST_ELEMENT (pipeline), GST_STATE_NULL);
        gst_object_unref (GST_OBJECT (pipeline));
      }

      void
      ProcessorCDDA_FLAC::create_pipeline ()
      {
        GstElement *elements[N_ELEMENTS];

        elements[SOURCE]  = gst_element_factory_make ("cdparanoiasrc",  "src");
        elements[CONVERT] = gst_element_factory_make ("audioconvert",   "audioconvert");
        elements[FLACENC] = gst_element_factory_make ("flacenc",        "flacenc");
        elements[SINK]    = gst_element_factory_make ("filesink",       "sink");

        if (!(elements[SOURCE] && elements[CONVERT] && elements[FLACENC] && elements[SINK]))
          {
            throw PipelineError ("Pipeline could not be created");
          }

        debug ("audio", "mp3-flac: path: %s", m_path.c_str());
           
        g_object_set (G_OBJECT (elements[SINK]),    "location", m_path.c_str(),
                                                     NULL);

        g_object_set (G_OBJECT (elements[SOURCE]),  "track",    m_track,
                                                    "device",   m_device.c_str(),
                                                     NULL);

        pipeline = gst_pipeline_new ("pipeline");
        gst_bin_add_many (GST_BIN (pipeline),
                          elements[SOURCE],
                          elements[CONVERT],
                          elements[FLACENC],
                          elements[SINK], NULL);

        gst_element_link_many (elements[SOURCE],
                               elements[CONVERT],
                               elements[FLACENC],
                               elements[SINK], NULL); 

        if (!verify_pipeline())
          {
            throw PipelineError ("Pipeline could not be created");
          }

        switch (m_quality)
        {
            case 0:
              g_object_set (G_OBJECT (elements[FLACENC]), "quality", int(3), NULL); 
              break;

            case 1:
              g_object_set (G_OBJECT (elements[FLACENC]), "quality", int(5), NULL); 
              break;

            case 2:
              g_object_set (G_OBJECT (elements[FLACENC]), "quality", int(7), NULL); 
              break;
        }

        gst_bus_add_watch (gst_pipeline_get_bus (GST_PIPELINE (pipeline)), GstBusFunc(bus_watch), this); 
        prop_stream_time_report_interval() = 2000;
      }

  }
}

