/*  BMP - The Dumb Music Player
 *  Copyright (C) 2003-2006 BMP Project
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * The BMPx project hereby grant permission for non-gpl compatible GStreamer
 * plugins to be used and distributed together with GStreamer and BMPx. This
 * permission are above and beyond the permissions granted by the GPL license
 * BMPx is covered by.
 */

#include <iostream>
#include <sstream>

#include <glibmm.h>
#include <glibmm/markup.h>
#include <glibmm/i18n.h>
#include <glib.h>
#include <gtkmm.h>

#include <boost/format.hpp>

#include "xml.hh"
#include "util.hh"
#include "uri++.hh"
#include "stock.hh"
#include "paths.hh"
#include "main.hh"
#include "neon++/request.hh"

#include "x_bookmarks.hh"
#include "x_vfs.hh"

#include "ui_toolbox.hh"
#include "streams-icecast.hh"

namespace
{
    static char* icecast_headers[] =
    {
      N_("Stream Name"),
      N_("Bitrate"),
      N_("Stream Genre"),
    };
}

namespace Bmp
{
  namespace Streams
  {

#define STATE(e) ((state & e) != 0)
#define SET_STATE(e) ((state |= e))
#define CLEAR_STATE(e) ((state &= ~ e))

      void
      IcecastParser::check_sanity  ()
      {
        if (state)
          {
            g_warning (G_STRLOC ": State should be 0, but is %d", state);
          }
      }

      IcecastParser::IcecastParser (VIcecastEntries & entries) 
        : m_entries (entries),
          state     (0)
      {
      }

      IcecastParser::~IcecastParser () 
      {
      }
   
      void
      IcecastParser::on_start_element  (Glib::Markup::ParseContext& context,
                                        Glib::ustring const&        name,
                                        AttributeMap const&         attributes)
      {
        if (name == "directory")
        {
          SET_STATE(E_DIRECTORY);
          return;
        }

        if (name == "entry")
        {
          SET_STATE(E_ENTRY);
          entry = IcecastEntry ();
          return;
        }

        if (name == "server_name")
        {
          SET_STATE(E_SERVER_NAME);
          return;
        }

        if (name == "listen_url")
        {
          SET_STATE(E_LISTEN_URL);
          return;
        }

        if (name == "server_type")
        {
          SET_STATE(E_SERVER_TYPE);
          return;
        }

        if (name == "bitrate")
        {
          SET_STATE(E_BITRATE);
          return;
        }

        if (name == "channels")
        {
          SET_STATE(E_CHANNELS);
          return;
        }

        if (name == "samplerate")
        {
          SET_STATE(E_SAMPLERATE);
          return;
        }

        if (name == "genre")
        {
          SET_STATE(E_GENRE);
          return;
        }

        if (name == "current_song")
        {
          SET_STATE(E_CURRENT_SONG);
          return;
        }
      }

      void
      IcecastParser::on_end_element    (Glib::Markup::ParseContext& context,
                                        Glib::ustring const& name)
      {
        if (name == "directory")
        {
          CLEAR_STATE(E_DIRECTORY);
          return;
        }

        if (name == "entry")
        {
          CLEAR_STATE(E_ENTRY);
          m_entries.push_back (entry); 
          while (gtk_events_pending ()) gtk_main_iteration();
          return;
        }

        if (name == "server_name")
        {
          CLEAR_STATE(E_SERVER_NAME);
          return;
        }

        if (name == "listen_url")
        {
          CLEAR_STATE(E_LISTEN_URL);
          return;
        }

        if (name == "server_type")
        {
          CLEAR_STATE(E_SERVER_TYPE);
          return;
        }

        if (name == "bitrate")
        {
          CLEAR_STATE(E_BITRATE);
          return;
        }

        if (name == "channels")
        {
          CLEAR_STATE(E_CHANNELS);
          return;
        }

        if (name == "samplerate")
        {
          CLEAR_STATE(E_SAMPLERATE);
          return;
        }

        if (name == "genre")
        {
          CLEAR_STATE(E_GENRE);
          return;
        }

        if (name == "current_song")
        {
          CLEAR_STATE(E_CURRENT_SONG);
          return;
        }

      }

      void
      IcecastParser::on_text       (Glib::Markup::ParseContext& context,
                                    Glib::ustring const&  text)
      {
        if (!STATE(E_ENTRY))
          return;

        if (STATE(E_SERVER_NAME))
          {
            entry.server_name = text;
            return;
          }

        if (STATE(E_LISTEN_URL))
          {
            entry.listen_url = text;
            return;
          }

        if (STATE(E_SERVER_TYPE))
          {
            entry.server_type = text;
            return;
          }

        if (STATE(E_GENRE))
          {
            entry.genre = text;
            return;
          }

        if (STATE(E_CHANNELS))
          {
            entry.channels = g_ascii_strtoull (text.c_str(), NULL, 10); 
            return;
          }

        if (STATE(E_SAMPLERATE))
          {
            entry.samplerate = g_ascii_strtoull (text.c_str(), NULL, 10); 
            return;
          }

        if (STATE(E_BITRATE))
          {
            entry.bitrate = g_ascii_strtoull (text.c_str(), NULL, 10); 
            return;
          }
      }
    
      void
      Icecast::refresh ()
      {
        using namespace Gtk;
        using namespace Glib;

        Neon::Request r ("dir.xiph.org", "/yp.xml");
        r.set_check_server (false);

        std::string response;

        if (r.dispatch () != NE_OK)
          return;

        r >> response;

        m_streams->clear ();
        VIcecastEntries entries;
        try
          {
            IcecastParser parser (entries);
            Markup::ParseContext context (parser);
            context.parse (response);
            context.end_parse ();
            parser.check_sanity ();
          }
        catch (ConvertError& cxe)
          {
            return; 
          }
        catch (MarkupError& cxe)
          {
            return;
          }
        catch (...) {}

        for (VIcecastEntries::const_iterator i = entries.begin() ;  i != entries.end() ; ++i) 
          {
            IcecastEntry const& entry (*i);

            TreeModel::iterator m_iter = m_streams->append ();

            (*m_iter)[columns.genre]       = entry.genre;
            (*m_iter)[columns.bitrate]     = entry.bitrate;
            (*m_iter)[columns.name]        = entry.server_name;
            (*m_iter)[columns.uri]         = entry.listen_url;
          }
       while (gtk_events_pending ()) gtk_main_iteration();
      }

      bool
      Icecast::visible_func (Gtk::TreeModel::iterator const& m_iter)
      {
        if (!m_filter.length ())
          return true;

        Glib::ustring name ((*m_iter)[columns.name]);
        Glib::ustring genre ((*m_iter)[columns.genre]);
        Glib::ustring needle (m_filter.casefold ());

        name.casefold();
        genre.casefold();

        return ((Util::match_keys (name, needle)) || (Util::match_keys (genre, needle)));
      }

      void
      Icecast::filter (Glib::ustring const& filter)
      {
        m_filter = filter;
        m_filtered->refilter ();
      }

      void
      Icecast::column_clicked (int column)
      {
        Gtk::SortType sort_type, sort_type_new;
        int           sort_id;

        m_streams->get_sort_column_id (sort_id, sort_type);

        if ((sort_id >= 0) && (sort_id != column))
          {
            get_column (sort_id)->set_sort_indicator (false);
          }

        if (sort_id >= 0)
          sort_type_new = (sort_type == Gtk::SORT_ASCENDING) ? Gtk::SORT_DESCENDING : Gtk::SORT_ASCENDING;
        else
          sort_type_new = Gtk::SORT_ASCENDING;

        m_streams->set_sort_column_id (column, sort_type_new);
        get_column (column)->set_sort_indicator (true);
        get_column (column)->set_sort_order (sort_type_new);
      }

      void 
      Icecast::get_stream (Glib::ustring& title, Glib::ustring& uri)
      {
        Gtk::TreeModel::iterator m_iter = get_selection()->get_selected ();
        uri   = (*m_iter)[columns.uri];
        title = (*m_iter)[columns.name];
      }

      Icecast::Icecast (BaseObjectType                       * obj,
                        Glib::RefPtr<Gnome::Glade::Xml> const& xml)

          : Gtk::TreeView (obj),
            m_streams     (Gtk::ListStore::create (columns)),
            m_filtered    (Gtk::TreeModelFilter::create (m_streams))
      {
        m_filtered->set_visible_func (sigc::mem_fun (this, &Bmp::Streams::Icecast::visible_func));

        for (unsigned int n = 0; n < G_N_ELEMENTS (icecast_headers); n++)
        {
          Gtk::CellRendererText *cell;
          cell = Gtk::manage (new Gtk::CellRendererText());
          append_column (_(icecast_headers[n]), *cell);
          get_column (n)->add_attribute (*cell, "text", n);
          get_column (n)->set_resizable (true);
          get_column (n)->signal_clicked ().connect
                    (sigc::bind (sigc::mem_fun (this, &Bmp::Streams::Icecast::column_clicked),n));
        }
        get_selection()->set_mode (Gtk::SELECTION_SINGLE);
        set_headers_clickable (true);
        set_model (m_filtered);
      }
  }
};
