//  BMPx - The Dumb Music Player
//  Copyright (C) 2005 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License Version 2
//  as published by the Free Software Foundation.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifndef BMP_VFS_HH
#define BMP_VFS_HH

#include <iostream>
#include <sstream>
#include <glibmm.h>
#include <boost/shared_ptr.hpp>

#include <bmp/base_types.hh>

namespace Bmp
{
    namespace VFS
    {
#include "exception.hh"

      EXCEPTION(TransportError)
      EXCEPTION(ContainerError)
      EXCEPTION(ProcessingError)
      EXCEPTION(InvalidFlagsError)

      struct ExportData
      {
        std::string description;
        std::string extension;

        ExportData () {};
        ExportData (std::string description, std::string extension) : description (description), extension (extension) {};
      };

      class Handle 
      {
        public:

          Handle () {}
          Handle (Glib::ustring const& uri) : m_buffer (0), m_buffer_size (0), m_uri (uri) {}

          ~Handle ()
          {
            g_free (m_buffer);
          }

          const unsigned char*
          get_buffer () const
          {
            return m_buffer;
          }
      
          size_t 
          get_buffer_size () const
          {
            return m_buffer_size;
          }

          void
          set_buffer (const unsigned char* buffer, size_t buffer_size)
          {
            m_buffer = reinterpret_cast<unsigned char*>(malloc(buffer_size));
            m_buffer_size = buffer_size;
            memcpy (m_buffer, buffer, buffer_size);
          }

          Glib::ustring const
          get_uri () const
          {
            return m_uri;
          }

        private:

          unsigned char    *m_buffer;
          size_t	          m_buffer_size;
          Glib::ustring     m_uri;

      };

      class PluginTransportBase
      {
        public:

          PluginTransportBase () {};
          virtual ~PluginTransportBase () {}; 

          virtual bool
          can_process	  (Glib::ustring const& str) = 0;
      
          virtual bool
          handle_read	  (Handle & handle) = 0;

          virtual bool
          handle_write  (Handle const& handle) = 0;
      };

      class PluginContainerBase
      {
        public:

          PluginContainerBase () {};
          virtual ~PluginContainerBase () {}; 

          virtual bool
          can_process (Glib::ustring const& str) = 0;
      
          virtual bool 
          handle_read	(Handle & handle, VUri & list) = 0;

          virtual bool
          handle_write  (Handle & handle, VUri const& list) = 0; 

          virtual bool
          can_write	() = 0;

          virtual Bmp::VFS::ExportData
          get_export_data () = 0;
      };


      typedef std::map<std::string, PluginContainerBase*> ContainerPlugins;
      typedef std::map<std::string, PluginTransportBase*> TransportPlugins;
      typedef std::vector<ExportData>                     ExportDataList;	

      enum ProcessingFlags
      {
        FLAG_ZERO	 = 0, // _Don't_ use this
        TRANSPORT = 1 << 1,
        CONTAINER = 1 << 2,
      };

      class VFS
      {
        public:

          VFS ();
          ~VFS ();

          static void
          strv_to_uri_list (VUri & list, char **strv, bool free = false)
          {
            char **p = strv;

            for ( ; *p ; ++p)
              {
                list.push_back (*p);
              }

            if (free) g_strfreev (strv);
          } 

          void
          get_containers (ExportDataList & list);

          bool
          has_container (Glib::ustring const& uri);

          void 
          read  (Handle & handle,
                 VUri & list,
                 ProcessingFlags flags = ProcessingFlags(TRANSPORT | CONTAINER));

          void 
          read  (Handle & handle,
                 ProcessingFlags flags = ProcessingFlags(TRANSPORT));

          void
          write (VUri const& list,
                 Glib::ustring const&	uri,
                 std::string const&	container);

        private:
      
          ContainerPlugins  container_plugins;
          TransportPlugins  transport_plugins;

          bool
          load_plugins_transport (std::string const& entry,
                                  std::string const& type);

          bool
          load_plugins_container (std::string const& entry,
                                  std::string const& type);

      };
    } // class VFS 
}// namespace Bmp 
	    
#endif //!BMP_VFS_H 
