// Copyright 2003 Regents of the University of California

// SETI_BOINC is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free
// Software Foundation; either version 2, or (at your option) any later
// version.

// SETI_BOINC is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
// more details.

// Copyright 2003 Regents of the University of California

// SETI_BOINC is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free
// Software Foundation; either version 2, or (at your option) any later
// version.

// SETI_BOINC is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
// more details.

// You should have received a copy of the GNU General Public License along
// with SETI_BOINC; see the file COPYING.  If not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
// In addition, as a special exception, the Regents of the University of
// California give permission to link the code of this program with libraries
// that provide specific optimized fast Fourier transform (FFT) functions and
// distribute a linked executable.  You must obey the GNU General Public 
// License in all respects for all of the code used other than the FFT library
// itself.  Any modification required to support these libraries must be
// distributed in source code form.  If you modify this file, you may extend 
// this exception to your version of the file, but you are not obligated to 
// do so. If you do not wish to do so, delete this exception statement from 
// your version.


/*
 * $Id: pulsefind.cpp,v 1.16.2.10 2006/04/12 20:53:17 korpela Exp $
 *
 */

#include "config.h"

#include <cstdio>
#include <cstdlib>
#include <cmath>
#include <climits>
#include <cstring>
#include <ctime>

#include "diagnostics.h"
#include "util.h"
#include "s_util.h"
#ifdef BOINC_APP_GRAPHICS
#include "graphics_api.h"
#ifdef DYNAMIC_GRAPHICS
#include "graphics_lib.h"
#endif
#endif

#include "seti.h"
#include "seti_header.h"
#include "analyzePoT.h"
#include "analyzeReport.h"
#include "worker.h"
#include "malloc_a.h"
#include "lcgamm.h"

//#define DEBUG_TRIPLET_VERBOSE
//#define DEBUG_PULSE_VERBOSE
//#define DEBUG_PULSE_BOUNDS

#ifndef M_PI
#define M_PI (3.14159265358979323846)
#endif
#ifndef M_SQRT2
#define M_SQRT2 (1.41421356237309504880)
#endif
#ifndef M_SQRT3
#define M_SQRT3 (1.73205080756887719318)
#endif
#ifndef M_SQRT5
#define M_SQRT5 (2.23606797749978980505)
#endif
#ifndef M_SQRT1_2
#define M_SQRT1_2 (0.70710678118654752440)
#endif
#ifndef M_SQRT1_3
#define M_SQRT1_3 (0.57735026918962584208)
#endif
#ifndef M_SQRT1_5
#define M_SQRT1_5 (0.44721359549995792770)
#endif

/**********************
 *
 *   find_triplets - Analyzes the input signal for a triplet signal.
 *   
 *   Written by Eric Heien.
 */
int find_triplets( const float *power, int len_power, float triplet_thresh, int time_bin, int freq_bin ) {
  static int    *binsAboveThreshold;
  int            i,n,numBinsAboveThreshold=0,p,q;
  float        midpoint,mean_power=0,peak_power,period;

  if (!binsAboveThreshold) {
    binsAboveThreshold=(int *)malloc_a(PoTInfo.TripletMax*sizeof(int),MEM_ALIGN);
    if (!binsAboveThreshold) SETIERROR(MALLOC_FAILED, "!binsAboveThreshold");
  }

  /* Get all the bins that are above the threshold, and find the power array mean value */
#ifdef DEBUG_TRIPLET_VERBOSE
  fprintf(stderr, "In find_triplets()...   PulsePotLen = %d   triplet_thresh = %f   TOffset = %d   PoT = %d\n",  len_power,  triplet_thresh, time_bin, freq_bin);
#endif

  for( i=0;i<len_power;i++ ) {
    mean_power += power[i];
  }
  mean_power /= (float)len_power;
  triplet_thresh*=mean_power;
  for( i=0;i<len_power;i++ ) {
    if( power[i] >= triplet_thresh ) {
      binsAboveThreshold[numBinsAboveThreshold] = i;
      numBinsAboveThreshold++;
    }
  }
  analysis_state.FLOP_counter+=10.0+len_power;

  /* Check each bin combination for a triplet */
  if (numBinsAboveThreshold>2) {
    for( i=0;i<numBinsAboveThreshold-1;i++ ) {
      for( n=i+1;n<numBinsAboveThreshold;n++ ) {
        midpoint = (binsAboveThreshold[i]+binsAboveThreshold[n])/2.0f;
        period = (float)fabs((binsAboveThreshold[i]-binsAboveThreshold[n])/2.0f);

        /* Get the peak power of this triplet */
        peak_power = power[binsAboveThreshold[i]];

        if( power[binsAboveThreshold[n]] > peak_power )
          peak_power = power[binsAboveThreshold[n]];

        p = binsAboveThreshold[i];
        while( (power[p] >= triplet_thresh) && (p <= (static_cast<int>(floor(midpoint)))) ) {    /* Check if there is a pulse "off" in between init and midpoint */
          p++;
        }

        q = static_cast<int>(floor(midpoint))+1;
        while( (power[q] >= triplet_thresh) && (q <= binsAboveThreshold[n])) {    /* Check if there is a pulse "off" in between midpoint and end */
          q++;
        }

        if( p >= static_cast<int>(floor(midpoint)) || q >= binsAboveThreshold[n]) {
          /* if this pulse doesn't have an "off" between all the three spikes, it's dropped */
        } else if( (midpoint - floor(midpoint)) > 0.1f ) {    /* if it's spread among two bins */
          if( power[(int)midpoint] >= triplet_thresh ) {
            if( power[(int)midpoint] > peak_power )
              peak_power = power[(int)midpoint];

            ReportTripletEvent( peak_power/mean_power, mean_power, period, midpoint,time_bin, freq_bin, len_power, power, 1 );
          }

          if( power[(int)(midpoint+1.0f)] >= triplet_thresh ) {
            if( power[(int)(midpoint+1.0f)] > peak_power )
              peak_power = power[(int)(midpoint+1.0f)];

            ReportTripletEvent( peak_power/mean_power, mean_power, period, midpoint,time_bin, freq_bin, len_power, power, 1 );
          }
        } else {            /* otherwise just check the single midpoint bin */
          if( power[(int)midpoint] >= triplet_thresh ) {
            if( power[(int)midpoint] > peak_power )
              peak_power = power[(int)midpoint];

            ReportTripletEvent( peak_power/mean_power, mean_power, period, midpoint,time_bin, freq_bin, len_power, power, 1 );
          }
        }
      }
    }
  }
  analysis_state.FLOP_counter+=(10.0*numBinsAboveThreshold*numBinsAboveThreshold);
  return (0);
}

/**********************
 *
 * t_funct - Caching routine for calls to invert_lcgf, return cache value if present
 *
 */
float t_funct(int m, int n, int x) {
 static struct tftab {
   int n;
   float y;
 }
 *t_funct_tab;
 if (!t_funct_tab) {
   int tablen = (PoTInfo.PulseMax+2)/3 +
                3*(int)(log((float)PoTInfo.PulseMax)/log(2.0)-3) - 1;
   t_funct_tab=(struct tftab *)malloc_a(tablen*sizeof(struct tftab),MEM_ALIGN);
   if (!t_funct_tab) SETIERROR(MALLOC_FAILED, "!t_funct_tab");
   memset(t_funct_tab, 0, tablen*sizeof(struct tftab));
 }
 if (t_funct_tab[x].n!=n) {
   t_funct_tab[x].n=n;
   t_funct_tab[x].y=
     invert_lcgf((float)(-PoTInfo.PulseThresh-log((float)m)),(float)n,
                 (float)1e-4)/n;
 }
 return t_funct_tab[x].y;
}


/**********************
 *
 * find_pulse - uses a folding algorithm to find repeating pulses
 *   Initially folds by prime number then by powers of two.
 *
 * Initial code by Eric Korpela
 * Major revisions and optimization by Ben Herndon
 *
 */
int find_pulse(const float * fp_PulsePot, int PulsePotLen,
               float pulse_thresh, int TOffset, int FOffset) {
  static float *div,*FoldedPOT;
  float period;
  int ndivs;
  int i,j,di,dbins=PulsePotLen ,offset,nperiods;
  long p;
  float cperiod;
  float max=0,maxd=0,avg=0,rms=0,maxp=0, snr=0, fthresh=0;
  int num_adds;
  register float tmp_max,t1;
  int res=1;

  // debug possible heap corruption -- jeffc
#ifdef _WIN32
  BOINCASSERT(_CrtCheckMemory());
#endif

#ifdef DEBUG_PULSE_VERBOSE
  fprintf(stderr, "In find_pulse()...   PulsePotLen = %d   pulse_thresh = %f   TOffset = %d   PoT = %d\n",  PulsePotLen,  pulse_thresh, TOffset, FOffset);
#endif


  //  Create internal arrays....
  if (!div) {
    FoldedPOT=(float *)malloc_a((PoTInfo.PulseMax+1)*sizeof(float)/3,MEM_ALIGN);
    div=(float *)malloc_a(PoTInfo.PulseMax*2*sizeof(float),MEM_ALIGN);
    if ((!div) || (!FoldedPOT)) SETIERROR(MALLOC_FAILED, "(!div) || (!FoldedPOT)");
    if (t_funct(6,1000,0)<0) SETIERROR(MALLOC_FAILED, "t_funct(6,1000,0)<0");;
  }

  //  Rebin to lower resolution if required
  if (res <= 1) {
    memcpy(div,fp_PulsePot,PulsePotLen*sizeof(float));
  } else {
    for (j=0;j<PulsePotLen/res;j++) {
      div[j]=0;
      for (i=0;i<res;i++) {
        div[j]+=fp_PulsePot[j*res+i];
      }
      div[j]/=res;
    }
  }
  
  PulsePotLen/=res;



  //  Calculate average power and RMS deviation
  for (i=0;i<PulsePotLen;i++) {
    avg+=div[i];
  }
  avg/=PulsePotLen;
  for (i=0;i<PulsePotLen;i++) {
    rms+=(div[i]-avg)*(div[i]-avg) ;
  }

  rms=(float)sqrt(rms/PulsePotLen);
  for (i = 32, ndivs = 1; i <= PulsePotLen; ndivs++, i *= 2);
  // ndivs=(int)(log((float)PulsePotLen)/log(2.0)-3);

  int32_t thePotLen = PulsePotLen;
  nperiods =  (thePotLen*2)/3-(thePotLen*1)/2;
  nperiods += (thePotLen*3)/4-(thePotLen*3)/5;
  nperiods += (thePotLen*4)/5-(thePotLen*4)/6;
  nperiods *= ndivs;

  //  Periods from PulsePotLen/3 to PulsePotLen/4, and power of 2 fractions of.
  //   then (len/4 to len/5) and finally (len/5 to len/6)
  //
  int32_t firstP, lastP;
  for(num_adds = 3; num_adds <= 5; num_adds++) {
    switch(num_adds) {
      case 3: lastP = (thePotLen*2)/3;  firstP = (thePotLen*1)/2; break;
      case 4: lastP = (thePotLen*3)/4;  firstP = (thePotLen*3)/5; break;
      case 5: lastP = (thePotLen*4)/5;  firstP = (thePotLen*4)/6; break;
    }

    for (p = lastP ; p > firstP ; p--) {
      int  tmp0, tmp1, tmp2, tmp3;
      float cur_thresh , dis_thresh;
      int tabofst = ndivs*3 - 1 + 3 - num_adds;
      dbins = PulsePotLen; // Output storage
      period = ((float)p) / (num_adds-1);
      cperiod = period;
      tmp0 = (int)(cperiod+0.5f);
      tmp1 = (int)(cperiod*2.0f+0.5f);
      di  = (int)cperiod;
      if ((unsigned)di >= ((unsigned)PoTInfo.PulseMax)*2)
        SETIERROR(FP_ERROR,"FPU failure in pulse_find");

      cur_thresh = t_funct(di,num_adds,di+tabofst)*avg;
      dis_thresh = (cur_thresh-avg)*((float)swi.analysis_cfg.pulse_display_thresh) + avg;
      //      dis_thresh=(1.0+PULSE_DISPLAY_THRESH/sqrt(num_adds))*avg;

      switch(num_adds) {
        case 3:
          //   tmp_max = sum3tables(di, &div[0], &div[tmp0], &div[tmp1], &div[dbins]);
          tmp_max = 0.0f;
          for (i=0;i<di;i++) {
            register float tmpfloat=(div[i]+div[tmp0+i]+div[tmp1+i])/3;
            div[i+dbins]=tmpfloat;
            if (tmpfloat>tmp_max) {
              tmp_max=tmpfloat;
            }
          }
          break;
        case 4:
          //    tmp_max = sum4tables(di, &div[0], &div[tmp0], &div[tmp1], &div[tmp2], &div[dbins]);
          tmp_max = 0.0f;
          tmp2 = (int)(cperiod*3.0f+0.5f);
          for (i=0;i<di;i++) {
            register float tmpfloat=(div[i]+div[tmp0+i]+div[tmp1+i]+div[tmp2+i])/4;
            div[i+dbins]=tmpfloat;
            if (tmpfloat>tmp_max) {
              tmp_max=tmpfloat;
            }
          }
          break;
        case 5:
          //    tmp_max = sum5tables(di, &div[0], &div[tmp0], &div[tmp1], &div[tmp2], &div[tmp3], &div[dbins]);
          tmp_max = 0.0f;
          tmp2 = (int)(cperiod*3.0f+0.5f);
          tmp3 = (int)(cperiod*4.0f+0.5f);
          for (i=0;i<di;i++) {
            register float tmpfloat=(div[i]+div[tmp0+i]+div[tmp1+i]+div[tmp2+i]+div[tmp3+i])/5;
            div[i+dbins]=tmpfloat;
            if (tmpfloat>tmp_max) {
              tmp_max=tmpfloat;
            }
          }
          break;
      }

      if ((tmp_max-dis_thresh)>0) {
        ReportPulseEvent(tmp_max/avg,avg,cperiod*res,
                         TOffset+PulsePotLen/2,FOffset,
                         (tmp_max-avg)*(float)sqrt((float)num_adds)/avg,
                         (cur_thresh-avg)*(float)sqrt((float)num_adds)/avg, &div[dbins],0);
      }

      if ((t1=tmp_max-cur_thresh)>maxd) {
        maxp  = cperiod;
        maxd  = t1;
        max = tmp_max;
        snr = (float)((tmp_max-avg)*sqrt((float)num_adds)/avg);
        fthresh=(float)((cur_thresh-avg)*sqrt((float)num_adds)/avg);
        memcpy(FoldedPOT, &div[dbins], di*sizeof(float));
      }

      offset =PulsePotLen;
      dbins =PulsePotLen/2;
      cperiod /=2;
      int num_adds_2 = 2* num_adds;

      
      for (j = 1; j < ndivs ; j++) {
        tabofst -=3;
		tmp0  = (int)(cperiod+0.5f);
        di  = (int)cperiod;
        if ((unsigned)di>=(unsigned)(PoTInfo.PulseMax*2))
          SETIERROR(FP_ERROR,"FPU failure in pulse_find");
        cur_thresh= t_funct(di,num_adds_2,di+tabofst)*avg;
        dis_thresh= (cur_thresh-avg)*((float)swi.analysis_cfg.pulse_display_thresh) + avg;

        //  tmp_max = sum2tables(di, &div[offset], &div[offset+tmp0], &div[offset+dbins]);
        tmp_max = 0.0f;
        for (i=0;i<di;i++) {
          register float tmpfloat=(div[i+offset]+div[tmp0+i+offset])/2;
          div[i+offset+dbins]=tmpfloat;
          if (tmpfloat>tmp_max) {
            tmp_max=tmpfloat;
          }
        }

        if ((tmp_max-dis_thresh)>0) {
          ReportPulseEvent(tmp_max/avg,avg,cperiod*res,
                           TOffset+PulsePotLen/2,FOffset,
                           (tmp_max-avg)*(float)sqrt((float)num_adds_2)/avg,
                           (cur_thresh-avg)*(float)sqrt((float)num_adds_2)/avg, div+offset+dbins, 0);
        }
        if ((t1=tmp_max-cur_thresh)>maxd) {
          maxp = cperiod;
          maxd = t1;
          max  = tmp_max;
          snr  = (float)((tmp_max-avg)*sqrt((float)num_adds_2)/avg);
          fthresh = (float)((cur_thresh-avg)*sqrt((float)num_adds_2)/avg);
          memcpy(FoldedPOT, div+offset+dbins, di*sizeof(float));
        }

        cperiod  /=2;
        num_adds_2 *=2;
        offset  +=dbins;
        dbins   /=2;
      }  // for (j = 1; j < ndivs
    } // for (p = lastP
  } // for(num_adds =

  analysis_state.FLOP_counter+=(PulsePotLen*0.1818181818182+400.0)*PulsePotLen;
  if (maxp!=0)
    ReportPulseEvent(max/avg,avg,maxp*res,TOffset+PulsePotLen/2,FOffset,
                     snr, fthresh, FoldedPOT, 1);

  // debug possible heap corruption -- jeffc
#ifdef _WIN32
  BOINCASSERT(_CrtCheckMemory());
#endif

  return 0;
}

