/***************************************************************************
                          opera.cpp  -  Opera Browser Class
                             -------------------
    begin                : Tue Feb 18 2003
    copyright            : (C) 2003 by Ken Schenke
    email                : kschenke at users dot sourceforge dot net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU      *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the Free Software           *
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA         *
 *   02110-1301, USA                                                       *
 *                                                                         *
 ***************************************************************************/

#include "opera.h"

#include <QFile>
#include <QFileDialog>

#ifdef Q_WS_WIN
#include <windows.h>
#include <shlobj.h>
#endif

/***************************************************************************
 *                                                                         *
 *   Function Prototypes                                                   *
 *                                                                         *
 ***************************************************************************/

static void saveBookmark(QTextStream &t, BkBookmark &bkmark, BRWSNUM browserOrd)
	throw(BkException);
static void saveFolder(QTextStream &t, BkFolder &folder, BRWSNUM browserOrd)
	throw(BkException);

/***************************************************************************
 *                                                                         *
 *   Opera::AreBookmarksValid()                                            *
 *                                                                         *
 *   Parameters:                                                           *
 *      const QString &bookmarks                                           *
 *   Return:                                                               *
 *      true if valid, false otherwise                                     *
 *   Description:                                                          *
 *      This function is called to verify the filename given in the first  *
 *      first parameter is most likely an Opera file.  It first attempts   *
 *      to open the file and then reads the first line and matches it      *
 *      against what it thinks the first should be.                        *
 *                                                                         *
 ***************************************************************************/

bool Opera::AreBookmarksValid(const QString &bookmarks)
{
	QFile file(bookmarks);

	if(file.open(QIODevice::ReadOnly | QIODevice::Text) == false)
		return false;

	QTextStream stream(&file);

	QString line = stream.readLine();
	return line.startsWith("Opera Hotlist version 2.0");
}

/***************************************************************************
 *                                                                         *
 *   Opera::BrowseForBookmarks()                                           *
 *                                                                         *
 *   Parameters:                                                           *
 *      const BridgeCfg &cfg (WIN32 Only)                                  *
 *      QString &bookmarks                                                 *
 *      QWidget *parent                                                    *
 *   Return:                                                               *
 *      true if the user did not select a bookmark file, false otherwise   *
 *   Description:                                                          *
 *      This function is called to present the user with a file dialog box *
 *      allowing them to select the location of their bookmark file.       *
 *                                                                         *
 ***************************************************************************/

#if defined(Q_WS_WIN)
bool Opera::BrowseForBookmarks(const BridgeCfg &cfg, QString &bookmarks, QWidget *parent)
{
	if(bookmarks==QString::null || bookmarks=="")
		bookmarks = cfg.m_AppDataDir + "/Opera";
	bookmarks =
		QFileDialog::getOpenFileName(parent, "Select Your Bookmarks File",
									 bookmarks, "Bookmark File (opera*.adr)");

	return (bookmarks == QString::null);
}
#elif defined(Q_WS_X11)
bool Opera::BrowseForBookmarks(const BridgeCfg &, QString &bookmarks, QWidget *parent)
{
	if(bookmarks==QString::null || bookmarks=="")
		bookmarks = QDir::homePath() + "/.opera/default";
	bookmarks = QFileDialog::getOpenFileName(parent,
		"Select Your Bookmarks File",
		bookmarks, "Bookmark File (opera*.adr)");

	return (bookmarks == QString::null);
}
#else
#error "Must define platform-dependent Opera::BrowseForBookmarks() Function"
#endif

/***************************************************************************
 *                                                                         *
 *   Opera::classFactory()                                                 *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      the newly allocated class                                          *
 *   Description:                                                          *
 *      This is a static member function of the Opera class.  Its job is   *
 *      to allocate an instance of the Opera class for the caller.         *
 *                                                                         *
 ***************************************************************************/

BrowserBk *Opera::classFactory(void)
{
	return new Opera;
}

/***************************************************************************
 *                                                                         *
 *   CleanUpLine()                                                         *
 *                                                                         *
 *   Parameters:                                                           *
 *      QString &line                                                      *
 *   Return:                                                               *
 *      None                                                               *
 *   Description:                                                          *
 *      This function removes any non-printable characters on the string.  *
 *                                                                         *
 ***************************************************************************/

static void CleanUpLine(QString &line)
{
	for(int i=0; i<line.length(); i++)
	{
		if(!line.at(i).isPrint())
		{
			line.replace(i, 1, "");
			i--;
		}
	}
}

/***************************************************************************
 *                                                                         *
 *   Opera::DetectBrowser()                                                *
 *                                                                         *
 *   Parameters:                                                           *
 *      const BridgeCfg &cfg (WIN32 Only)                                  *
 *      QStringList &path                                                  *
 *   Return:                                                               *
 *      true if an installation of Opera was found, false otherwise        *
 *   Description:                                                          *
 *      This function attempts to locate Opera's bookmark database on the  *
 *      system.                                                            *
 *                                                                         *
 ***************************************************************************/

#if defined(Q_WS_WIN)
bool Opera::DetectBrowser(const BridgeCfg &cfg, QStringList &paths)
{
	QString basePath;

	paths.clear();

	// Start off with the base path.  On Unix it would be something like:
	// /home/ken/.opera
	// On Windows it would be something like:
	// C:\Documents and Settings\Ken\Application Data\Opera
	//   -- or --
	// C:\Program Files\Opera

	basePath = cfg.m_AppDataDir + "/Opera";

	QString	path;

	if(findBookmarkFile(basePath, path))
		paths.append(path);

	basePath = cfg.m_ProgFilesDir + "\\Opera";
	if(findBookmarkFile(basePath, path))
		paths.append(path);

	return (paths.count() > 0);
}
#elif defined(Q_WS_X11)
bool Opera::DetectBrowser(const BridgeCfg &, QStringList &paths)
{
	QString basePath;

	paths.clear();

	// Start off with the base path.  On Unix it would be something like:
	// /home/ken/.opera
	// On Windows it would be something like:
	// C:\Documents and Settings\Ken\Application Data\Opera
	//   -- or --
	// C:\Program Files\Opera

	basePath = QDir::homePath() + "/.opera";

	QString	path;

	if(findBookmarkFile(basePath, path))
		paths.append(path);

	return (paths.count() > 0);
}
#else
#error "Must define platform-dependent Opera Detection Code"
#endif

/***************************************************************************
 *                                                                         *
 *   Opera::findBookmarkFile()                                             *
 *                                                                         *
 *   Parameters:                                                           *
 *      const char *path                                                   *
 *      QString &bookmarkFile                                              *
 *   Return:                                                               *
 *      true if one and only one Opera bookmark file was found             *
 *   Description:                                                          *
 *      This function is called by DetectBrowser(), and by itself          *
 *      recursively.  It scans the path given in the first parameter for   *
 *      an Opera bookmark file.  If it finds one and only one it places    *
 *      the filename (with full path) in the second parameter and returns  *
 *      true.  It the path has any subdirectories it scans those as well.  *
 *                                                                         *
 ***************************************************************************/

bool Opera::findBookmarkFile(const QString &path, QString &bookmarkFile)
{
	QDir	dir(path, "opera*.adr");
	QString	subdirPath;

	// if no files matching filter were found in this directory,
	// look in subdirectories
	
	if(dir.count() < 1)
	{
		QDir subdirs(path, QString::null,
			QDir::Name | QDir::IgnoreCase,
			QDir::Dirs);

		for(uint i=0; i<subdirs.count(); i++)
		{
			if(subdirs[i]=="." || subdirs[i]=="..")
				continue;
			subdirPath = path + "/";
			subdirPath += subdirs[i];
			if(findBookmarkFile(subdirPath, bookmarkFile))
				return true;
		}

		// we looked in all the subdirectories. it's not here.

		return false;
	}
	else if(dir.count() > 1)
	{
		// more than one match.  it's probably not safe to make a guess.
		return false;
	}

	// a match was found, go with it

	bookmarkFile = path + "/";
	bookmarkFile += dir[0];
	return true;
}

/***************************************************************************
 *                                                                         *
 *   Opera::readAttributes()                                               *
 *                                                                         *
 *   Parameters:                                                           *
 *      QTextStream &stream                                                *
 *      BkNode &node                                                       *
 *      QString &line                                                      *
 *      BRWSNUM browserOrd                                                 *
 *   Return:                                                               *
 *      None.  BkException thrown in event of error                        *
 *   Description:                                                          *
 *      This function is called by readFolder() to read in attributes from *
 *      from the Opera bookmark file for URLs and folders.                 *
 *                                                                         *
 ***************************************************************************/

void Opera::readAttributes(QTextStream &stream, BkNode &node, QString &line,
	BRWSNUM browserOrd) throw(BkException)
{
	QString	field, value;
	int		idx;

	// loop through the lines for this node.  it could be a folder or a URL

	for(;;)
	{
		// are we at end of file?  if so, stop reading

		if(stream.atEnd())
			break;

		line = stream.readLine().simplified();
		if(line.length() < 1)
			break;
		
		CleanUpLine(line);

		if(line.startsWith("#"))
			break;		// beginning of #URL or #FOLDER. stop reading.
		if(line.startsWith("-"))
			break;		// end of folder

		idx = line.indexOf('=');
		if(idx < 0)
			continue;	// not FIELD=VALUE format.  skip this line

		field = line.left(idx).simplified().toUpper();
		value = line.right(line.length()-idx-1).simplified();

		if(field == "NAME")
		{
			node.setTitle(value);
			node.setValidField(BKVALID_TITLE);
		}
		else if(field == "DESCRIPTION")
		{
			node.setDesc(value);
			node.setValidField(BKVALID_DESC);
		}
		else if(field == "URL"
		   &&   node.type() == NODETYPE_BOOKMARK)
		{
			BkBookmark &bmark = static_cast<BkBookmark &>(node);
			bmark.setUrl(value);
			bmark.setValidField(BKVALID_URL);
		}
		else
		{
			// attribute not recognized.  save until later

			node.setAttr(field, value, browserOrd);
		}
	}
}

/***************************************************************************
 *                                                                         *
 *   Opera::readBookmarks()                                                *
 *                                                                         *
 *   Parameters:                                                           *
 *      const QString &filename                                            *
 *      BRWSNUM browserOrd                                                 *
 *   Return:                                                               *
 *      None.  BkException thrown if error occurs                          *
 *   Description:                                                          *
 *      This function attempts to read Opera's bookmark file.              *
 *                                                                         *
 ***************************************************************************/

void Opera::readBookmarks(const QString &path, BRWSNUM browserOrd)
	throw(BkException)
{
	QFile file(path);

	if(file.open(QIODevice::ReadOnly | QIODevice::Text) == false)
	{
		QString msg;

		msg = "Unable to Read Opera Bookmarks File\n";
		msg += path;
		BKEXCEPT(msg);
	}

	QTextStream stream(&file);
	
	stream.setCodec("UTF-8");

	QString line = stream.readLine();
	if(!line.startsWith("Opera Hotlist version 2.0"))
	{
		QString msg;

		msg = "The File ";
		msg += path;
		msg += "\nDoes Not Appear to be an Opera Bookmarks File";
		BKEXCEPT(msg);
	}
	line = stream.readLine();	// options line. ignore.

	m_Root = new BkFolder;
	if(m_Root == NULL)
		BKEXCEPT("Unable to Allocate Memory");

	readFolder(stream, *m_Root, browserOrd);
}

/***************************************************************************
 *                                                                         *
 *   Opera::readFolder()                                                   *
 *                                                                         *
 *   Parameters:                                                           *
 *      QTextStream &stream                                                *
 *      BkFolder &folder                                                   *
 *      BRWSNUM browserOrd                                                 *
 *   Return:                                                               *
 *      None.  BkException thrown if error occurs                          *
 *   Description:                                                          *
 *      This function is called by readBookmarks() and by itself           *
 *      recursively.  It reads all #URL sections in an Opera bookmark file *
 *      until it reaches end-of-file or until it encounters single dash    *
 *      character indicating the end of the folder.                        *
 *                                                                         *
 ***************************************************************************/

void Opera::readFolder(QTextStream &stream, BkFolder &folder, BRWSNUM browserOrd)
	throw(BkException)
{
	QString	line;
	bool	needLine=true;	// do we need to read a line from the file?
	
	for(;;)
	{
		// are we at the end of the file?

		if(stream.atEnd())
			break;

		// read a line from the bookmark file

		if(needLine)
		{		
			line = stream.readLine().simplified();
			if(line.length() < 1)
				continue;	// skip blank lines
		}
		needLine = true;

		// at this point, the only three valid lines are:
		//   #FOLDER
		//   #URL
		//   -  (a single dash)

		if(line.toUpper() == "#FOLDER")
		{
			BkFolder childFolder;

			childFolder.setBrowserFound(browserOrd);
			childFolder.setBrowserSaved(browserOrd);
			childFolder.setOrder(browserOrd, folder.folders()+folder.bookmarks()+1);
			readAttributes(stream, childFolder, line, browserOrd);
			readFolder(stream, folder.addFolder(childFolder), browserOrd);
		}
		else if(line.toUpper() == "#URL")
		{
			BkBookmark	bmark;

			bmark.setBrowserFound(browserOrd);
			bmark.setBrowserSaved(browserOrd);
			bmark.setOrder(browserOrd, folder.folders()+folder.bookmarks()+1);
			readAttributes(stream, bmark, line, browserOrd);
			folder.addBookmark(bmark);
			needLine = false;
		}
		
		if(line == "-")
			break;
	}
}

/***************************************************************************
 *                                                                         *
 *   Opera::saveBookmarks()                                                *
 *                                                                         *
 *   Parameters:                                                           *
 *      const QString &filename                                            *
 *      BkFolder &root                                                     *
 *      BRWSNUM browserOrd                                                 *
 *   Return:                                                               *
 *      None.  BkException thrown if error occurs                          *
 *   Description:                                                          *
 *      This function begins the process of saving the bookmark tree,      *
 *      specified in the second parameter to the file, specified in the    *
 *      first parameter.                                                   *
 *                                                                         *
 ***************************************************************************/

void Opera::saveBookmarks(const QString &filename, BkFolder &root, BRWSNUM browserOrd)
	throw(BkException)
{
	QFile	out;

	// open the file for writing

	out.setFileName(filename);
	if(out.open(QIODevice::WriteOnly | QIODevice::Text) == false)
		BKEXCEPT("Unable to Open Opera Bookmark File for Output");

	// set up a text stream

	QTextStream t(&out);

	t.setCodec("UTF-8");

	// write out Opera's boilerplate header

	t
		<< "Opera Hotlist version 2.0" << endl
		<< "Options: encoding = utf8, version=3" << endl
		<< endl;

	saveFolder(t, root, browserOrd);
}

/***************************************************************************
 *                                                                         *
 *   saveBookmark()                                                        *
 *                                                                         *
 *   Parameters:                                                           *
 *      QTextStream &stream                                                *
 *      BkBookmark &bkmark                                                 *
 *      BRWSNUM browserOrd                                                 *
 *   Return:                                                               *
 *      None.  BkException thrown if error occurs                          *
 *   Description:                                                          *
 *      This function is called by saveFolder() to save a single bookmark  *
 *      #URL section to the Opera bookmark file.                           *
 *                                                                         *
 ***************************************************************************/

static void saveBookmark(QTextStream &t, BkBookmark &bkmark, BRWSNUM browserOrd)
	throw(BkException)
{
	int		i;
	
	t << "#URL" << endl;
	
	t
		<< "\tNAME="
		<< bkmark.title()
		<< endl;
		
	t
		<< "\tURL="
		<< bkmark.url()
		<< endl;

	if(bkmark.isFieldValid(BKVALID_DESC))
	{
		t
			<< "\tDESCRIPTION="
			<< bkmark.desc()
			<< endl;
	}
	
	for(i=0; i<bkmark.nAttrs(); i++)
	{
		QString	name, value;
		BRWSNUM browser;

		bkmark.attr(i, name, value, browser);

		if(browser != browserOrd)
			continue;

		t
			<< '\t'
			<< name
			<< "="
			<< value
			<< endl;
	}

	t << endl;
}

/***************************************************************************
 *                                                                         *
 *   saveFolder()                                                          *
 *                                                                         *
 *   Parameters:                                                           *
 *      QTextStream &stream                                                *
 *      BkFolder &folder                                                   *
 *      BRWSNUM browserOrd                                                 *
 *   Return:                                                               *
 *      None.  BkException thrown if error occurs                          *
 *   Description:                                                          *
 *      This function is called by saveBookmarks() and by itself           *
 *      recursively to save a folder to the Opera bookmarks file.          *
 *                                                                         *
 ***************************************************************************/

static void saveFolder(QTextStream &t, BkFolder &folder, BRWSNUM browserOrd)
	throw(BkException)
{
	int i;
	std::vector<SortedNodeList> nodes;
	std::vector<SortedNodeList>::iterator it;

	SortNodes(folder, nodes, browserOrd);

	for(it=nodes.begin(); it!=nodes.end(); ++it)
	{
		if(it->m_type == NODETYPE_FOLDER)
		{
			// if this bookmark has been deleted from one of the
			// browsers, skip it...

			if(it->m_fit->isDeleted())
				continue;

			// Has the user requested to ignore this folder?
			// If so, we should skip it if it's not already stored in Opera

			if(it->m_fit->ignore() && !it->m_fit->browserFound(browserOrd))
				continue;

			t << "#FOLDER" << endl;
			t
				<< "\tNAME="
				<< it->m_fit->title()
				<< endl;
			if(it->m_fit->isFieldValid(BKVALID_DESC))
			{
				t
					<< "\tDESCRIPTION="
					<< it->m_fit->desc()
					<< endl;
			}

			for(i=0; i<it->m_fit->nAttrs(); i++)
			{
				QString	name, value;
				BRWSNUM browser;

				it->m_fit->attr(i, name, value, browser);

				if(browser != browserOrd)
					continue;

				t
					<< '\t'
					<< name
					<< "="
					<< value
					<< endl;
			}

			t << endl;

			saveFolder(t, *(it->m_fit), browserOrd);

			t << '-' << endl << endl;
		}
		else if(it->m_type == NODETYPE_BOOKMARK)
		{
			// if this bookmark has been deleted from one of the
			// browsers, skip it...

			if(it->m_bit->isDeleted())
				continue;

			// Has the user requested to ignore this bookmark?
			// If so, we should skip it if it's not already stored in Opera

			if(it->m_bit->ignore() && !it->m_bit->browserFound(browserOrd))
				continue;

			saveBookmark(t, *(it->m_bit), browserOrd);
			it->m_bit->setBrowserFound(browserOrd);
			it->m_bit->setBrowserSaved(browserOrd);
		}
		else if(it->m_type == NODETYPE_SEPARATOR)
			continue;	// Opera doesn't support separators
		else
			BKEXCEPT("Internal Error: Unrecognized Bookmark Type");
	}
		
	folder.setBrowserFound(browserOrd);
	folder.setBrowserSaved(browserOrd);
}
