/*************************************************
* SEAL Source File                               *
* (C) 1999-2005 The Botan Project                *
*************************************************/

#include <botan/seal.h>
#include <botan/sha160.h>
#include <botan/bit_ops.h>
#include <botan/parsing.h>

namespace Botan {

/*************************************************
* Combine cipher stream with message             *
*************************************************/
void SEAL::cipher(const byte in[], byte out[], u32bit length)
   {
   while(length >= buf.size() - position)
      {
      xor_buf(out, in, buf.begin() + position, buf.size() - position);
      length -= (buf.size() - position);
      in += (buf.size() - position);
      out += (buf.size() - position);
      generate(counter++);
      }
   xor_buf(out, in, buf.begin() + position, length);
   position += length;
   }

/*************************************************
* Generate cipher stream                         *
*************************************************/
void SEAL::generate(u32bit n)
   {
   u32bit A, B, C, D, P, Q;
   u32bit N0, N1, N2, N3;
   for(u32bit j = 0; j != buf.size() / 1024; j++)
      {
      A = n ^ R[4*j];                     B = rotate_right(n,  8) ^ R[4*j+1];
      C = rotate_right(n, 16) ^ R[4*j+2]; D = rotate_right(n, 24) ^ R[4*j+3];
      P = A & 0x7FC; A = rotate_right(A, 9); B += T[P/4];
      P = B & 0x7FC; B = rotate_right(B, 9); C += T[P/4];
      P = C & 0x7FC; C = rotate_right(C, 9); D += T[P/4];
      P = D & 0x7FC; D = rotate_right(D, 9); A += T[P/4];
      P = A & 0x7FC; A = rotate_right(A, 9); B += T[P/4];
      P = B & 0x7FC; B = rotate_right(B, 9); C += T[P/4];
      P = C & 0x7FC; C = rotate_right(C, 9); D += T[P/4];
      P = D & 0x7FC; D = rotate_right(D, 9); A += T[P/4];
      N0 = D; N1 = B; N2 = A; N3 = C;
      P = A & 0x7FC; A = rotate_right(A, 9); B += T[P/4];
      P = B & 0x7FC; B = rotate_right(B, 9); C += T[P/4];
      P = C & 0x7FC; C = rotate_right(C, 9); D += T[P/4];
      P = D & 0x7FC; D = rotate_right(D, 9); A += T[P/4];
      for(u32bit k = 0; k != 64; k += 2)
         {
         const u32bit OFFSET = 1024*j + 16*k;

         P = A       & 0x7FC; A = rotate_right(A, 9); B = (B + T[P/4]) ^ A;
         Q = B       & 0x7FC; B = rotate_right(B, 9); C = (C ^ T[Q/4]) + B;
         P = (P + C) & 0x7FC; C = rotate_right(C, 9); D = (D + T[P/4]) ^ C;
         Q = (Q + D) & 0x7FC; D = rotate_right(D, 9); A = (A ^ T[Q/4]) + D;
         P = (P + A) & 0x7FC; A = rotate_right(A, 9); B = (B ^ T[P/4]);
         Q = (Q + B) & 0x7FC; B = rotate_right(B, 9); C = (C + T[Q/4]);
         P = (P + C) & 0x7FC; C = rotate_right(C, 9); D = (D ^ T[P/4]);
         Q = (Q + D) & 0x7FC; D = rotate_right(D, 9); A = (A + T[Q/4]);
         u32bit X0 = B + S[4*k  ]; u32bit X1 = C ^ S[4*k+1];
         u32bit X2 = D + S[4*k+2]; u32bit X3 = A ^ S[4*k+3];

         buf[OFFSET+ 0] = get_byte(0, X0); buf[OFFSET+ 1] = get_byte(1, X0);
         buf[OFFSET+ 2] = get_byte(2, X0); buf[OFFSET+ 3] = get_byte(3, X0);
         buf[OFFSET+ 4] = get_byte(0, X1); buf[OFFSET+ 5] = get_byte(1, X1);
         buf[OFFSET+ 6] = get_byte(2, X1); buf[OFFSET+ 7] = get_byte(3, X1);
         buf[OFFSET+ 8] = get_byte(0, X2); buf[OFFSET+ 9] = get_byte(1, X2);
         buf[OFFSET+10] = get_byte(2, X2); buf[OFFSET+11] = get_byte(3, X2);
         buf[OFFSET+12] = get_byte(0, X3); buf[OFFSET+13] = get_byte(1, X3);
         buf[OFFSET+14] = get_byte(2, X3); buf[OFFSET+15] = get_byte(3, X3);
         A += N0; B += N1; C ^= N0; D ^= N1;

         P = A       & 0x7FC; A = rotate_right(A, 9); B = (B + T[P/4]) ^ A;
         Q = B       & 0x7FC; B = rotate_right(B, 9); C = (C ^ T[Q/4]) + B;
         P = (P + C) & 0x7FC; C = rotate_right(C, 9); D = (D + T[P/4]) ^ C;
         Q = (Q + D) & 0x7FC; D = rotate_right(D, 9); A = (A ^ T[Q/4]) + D;
         P = (P + A) & 0x7FC; A = rotate_right(A, 9); B = (B ^ T[P/4]);
         Q = (Q + B) & 0x7FC; B = rotate_right(B, 9); C = (C + T[Q/4]);
         P = (P + C) & 0x7FC; C = rotate_right(C, 9); D = (D ^ T[P/4]);
         Q = (Q + D) & 0x7FC; D = rotate_right(D, 9); A = (A + T[Q/4]);
         X0 = B + S[4*k+4]; X1 = C ^ S[4*k+5];
         X2 = D + S[4*k+6]; X3 = A ^ S[4*k+7];

         buf[OFFSET+16] = get_byte(0, X0); buf[OFFSET+17] = get_byte(1, X0);
         buf[OFFSET+18] = get_byte(2, X0); buf[OFFSET+19] = get_byte(3, X0);
         buf[OFFSET+20] = get_byte(0, X1); buf[OFFSET+21] = get_byte(1, X1);
         buf[OFFSET+22] = get_byte(2, X1); buf[OFFSET+23] = get_byte(3, X1);
         buf[OFFSET+24] = get_byte(0, X2); buf[OFFSET+25] = get_byte(1, X2);
         buf[OFFSET+26] = get_byte(2, X2); buf[OFFSET+27] = get_byte(3, X2);
         buf[OFFSET+28] = get_byte(0, X3); buf[OFFSET+29] = get_byte(1, X3);
         buf[OFFSET+30] = get_byte(2, X3); buf[OFFSET+31] = get_byte(3, X3);
         A += N2; B += N3; C ^= N2; D ^= N3;
         }
      }
   position = 0;
   }

/*************************************************
* SEAL Key Schedule                              *
*************************************************/
void SEAL::key(const byte key[], u32bit)
   {
   counter = START;

   Gamma gamma(key);
   for(u32bit j = 0; j != 512; j++)
      T[j] = gamma(j);
   for(u32bit j = 0; j != 256; j++)
      S[j] = gamma(0x1000 + j);
   for(u32bit j = 0; j != buf.size() / 256; j++)
      R[j] = gamma(0x2000 + j);
   generate(counter++);
   }

/*************************************************
* Seek a new position in the cipher stream       *
*************************************************/
void SEAL::seek(u32bit new_position)
   {
   counter = (new_position / buf.size()) + START;
   generate(counter++);
   position = new_position % buf.size();
   }

/*************************************************
* Clear memory of sensitive data                 *
*************************************************/
void SEAL::clear() throw()
   {
   buf.clear();
   T.clear();
   S.clear();
   R.clear();
   position = 0;
   counter = START;
   }

/*************************************************
* Return the name of this type                   *
*************************************************/
std::string SEAL::name() const
   {
   return "SEAL-3.0-BE(" + to_string(buf.size()) + ")";
   }

/*************************************************
* Return a clone of this object                  *
*************************************************/
StreamCipher* SEAL::clone() const
   {
   return new SEAL(buf.size());
   }

/*************************************************
* SEAL Constructor                               *
*************************************************/
SEAL::SEAL(u32bit Lbytes) : StreamCipher(1, 32),
                            buf(Lbytes), R(Lbytes/256)
   {
   if(Lbytes < 32 || Lbytes > 65536 || Lbytes % 32 != 0)
      throw Invalid_Argument("SEAL: Invalid Lbytes: " + to_string(Lbytes));
   if(Lbytes % 1024 != 0)
      throw Invalid_Argument("SEAL: L not a multiple of 1024 is unsupported");
   START = 0;
   clear();
   }

/*************************************************
* Gamma Constructor                              *
*************************************************/
Gamma::Gamma(const byte UK[20])
   {
   for(u32bit j = 0; j != 5; j++)
      K[j] = make_u32bit(UK[4*j], UK[4*j+1], UK[4*j+2], UK[4*j+3]);
   last_index = 0xFFFFFFFF;
   }

/*************************************************
* Gamma Function                                 *
*************************************************/
u32bit Gamma::operator()(u32bit index)
   {
   u32bit new_index = index / 5;
   if(new_index != last_index)
      {
      SecureBuffer<byte, 80> W;
      for(u32bit j = 0; j != 4; j++)
         W[j] = get_byte(j, new_index);
      SHA_160 sha;
      sha.digest = K;
      sha.hash(W);
      digest = sha.digest;
      last_index = new_index;
      }
   return digest[index % 5];
   }

}
