package org.bouncycastle.asn1;

import java.io.*;
import java.util.*;
import java.io.*;

/**
 * Generalized time object.
 */
public class DERGeneralizedTime extends DERObject
{
    String time;

    /**
     * return a generalized time from the passed in object
     * 
     * @exception IllegalArgumentException
     *                if the object cannot be converted.
     */
    public static DERGeneralizedTime getInstance(Object obj)
    {
        if (obj == null || obj instanceof DERGeneralizedTime)
        {
            return (DERGeneralizedTime)obj;
        }

        if (obj instanceof ASN1OctetString)
        {
            return new DERGeneralizedTime(((ASN1OctetString)obj).getOctets());
        }

        throw new IllegalArgumentException("illegal object in getInstance: "
                + obj.getClass().getName());
    }

    /**
     * return a Generalized Time object from a tagged object.
     * 
     * @param obj
     *            the tagged object holding the object we want
     * @param explicit
     *            true if the object is meant to be explicitly tagged false
     *            otherwise.
     * @exception IllegalArgumentException
     *                if the tagged object cannot be converted.
     */
    public static DERGeneralizedTime getInstance(ASN1TaggedObject obj, boolean explicit)
    {
        return getInstance(obj.getObject());
    }

    /**
     * The correct format for this is YYYYMMDDHHMMSSZ, or without the Z for
     * local time, or Z+-HHMM on the end, for difference between local time and
     * UTC time.
     * <p>
     * 
     * @param time
     *            the time string.
     */
    public DERGeneralizedTime(String time)
    {
        this.time = time;
    }

    DERGeneralizedTime(byte[] bytes)
    {
        //
        // explicitly convert to characters
        //
        char[] dateC = new char[bytes.length];

        for (int i = 0; i != dateC.length; i++)
        {
            dateC[i] = (char)(bytes[i] & 0xff);
        }

        this.time = new String(dateC);
    }

    /**
     * return the time - always in the form of YYYYMMDDhhmmssGMT(+hh:mm|-hh:mm).
     * <p>
     * Normally in a certificate we would expect "Z" rather than "GMT", however
     * adding the "GMT" means we can just use:
     * 
     * <pre>
     * dateF = new SimpleDateFormat(&quot;yyyyMMddHHmmssz&quot;);
     * </pre>
     * 
     * To read in the time and get a date which is compatible with our local
     * time zone.
     */
    public String getTime()
    {
        //
        // standardise the format.
        //             
        if (time.charAt(time.length() - 1) == 'Z')
        {
            return time.substring(0, time.length() - 1) + "GMT+00:00";
        }
        else
        {
            int signPos = time.length() - 5;
            char sign = time.charAt(signPos);
            if (sign == '-' || sign == '+')
            {
                return time.substring(0, signPos) + "GMT" + time.substring(signPos, signPos + 3)
                        + ":" + time.substring(signPos + 3);
            }
            else
            {
                signPos = time.length() - 3;
                sign = time.charAt(signPos);
                if (sign == '-' || sign == '+')
                {
                    return time.substring(0, signPos) + "GMT" + time.substring(signPos) + ":00";
                }
            }
        }

        return time;
    }

    private byte[] getOctets()
    {
        char[] cs = time.toCharArray();
        byte[] bs = new byte[cs.length];

        for (int i = 0; i != cs.length; i++)
        {
            bs[i] = (byte)cs[i];
        }

        return bs;
    }

    void encode(DEROutputStream out) throws IOException
    {
        out.writeEncoded(GENERALIZED_TIME, this.getOctets());
    }

    public boolean equals(Object o)
    {
        if ((o == null) || !(o instanceof DERGeneralizedTime))
        {
            return false;
        }

        return time.equals(((DERGeneralizedTime)o).time);
    }

    public int hashCode()
    {
        return time.hashCode();
    }
}
